/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.services.security.turbine;


// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.om.profile.Entry;
import org.apache.jetspeed.om.profile.psml.PsmlEntry;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.registry.base.BaseSecurity;
import org.apache.jetspeed.om.registry.base.BasePortletEntry;
import org.apache.jetspeed.om.registry.base.BaseParameter;
import org.apache.jetspeed.om.registry.Parameter;
import org.apache.jetspeed.om.registry.PortletEntry;
import org.apache.jetspeed.om.registry.Security;

import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.JetspeedPortalAccessController;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.jetspeed.services.security.PortalResource;

// Turbine imports
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

// Junit imports
import junit.awtui.TestRunner;
import junit.framework.Test;
import junit.framework.TestSuite;

/**
 * TestAccessController
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TestAccessController.java,v 1.1 2004/04/07 22:02:42 jford Exp $
 */
public class TestAccessController extends JetspeedTestCase
{
    private static String ADMIN_PORTLET = "GlobalAdminPortlet"; // Portlet accessable by Admin user, role = admin
    private static String ALL_PORTLET = "HelloVelocity";  // Portlet accessable by Anonymous user
    private static String TEST_GROUP = "Jetspeed";
    private static String TEST_SECURITY_PAGE = "SecurityTest";
    private static String USER_PORTLET = "SkinBrowser"; // Portlet accessable by general user, role = user

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestAccessController( String name )
    {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[]
        { TestAccessController.class.getName() } );
    }
    
    public void setup()
    {
        System.out.println("Setup: Testing categories of Profiler Service");
    }
    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestAccessController.class );
    }
    
    public void testVerifyEnvironment() throws Exception
    {
        assertEquals( "Using TurbineAccessController",
        "org.apache.jetspeed.services.security.turbine.TurbineAccessController",
        JetspeedResources.getString("services.PortalAccessController.classname"));
        
        assertNotNull( "Getting admin user", JetspeedSecurity.getUser("admin"));
        assertNotNull( "Getting turbine user", JetspeedSecurity.getUser("turbine"));
        assertNotNull( "Getting anonymous user", JetspeedSecurity.getAnonymousUser());
    }
        
    public void testRequiredActions() throws Exception
    {
        JetspeedUser adminUser = (JetspeedUser) JetspeedSecurity.getUser("admin");
        assertNotNull( "Getting admin user", adminUser);
        adminUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser turbineUser = (JetspeedUser) JetspeedSecurity.getUser("turbine");
        assertNotNull( "Getting turbine user", turbineUser);
        turbineUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser anonymousUser =  (JetspeedUser) JetspeedSecurity.getAnonymousUser();
        assertNotNull( "Getting anonymous user", anonymousUser);
        Entry adminEntry = createEntry( ADMIN_PORTLET, "ST_01.admin");
        Entry userEntry = createEntry(USER_PORTLET, "ST_01.user");
        Entry allEntry = createEntry(ALL_PORTLET, "ST_01.all");
        
        assertEquals( "Admin user has view access to " + ADMIN_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, adminEntry, "view"));
//        assertEquals( "Turbine user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( turbineUser, adminEntry, "view"));

//        assertEquals( "Anonymous user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, adminEntry, "view"));
//        assertEquals( "null user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, adminEntry, "view"));
        
        assertEquals( "Admin user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, userEntry, "view"));
        assertEquals( "Turbine user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, userEntry, "view"));
 //       assertEquals( "Anonymous user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, userEntry, "view"));
//        assertEquals( "null user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, userEntry, "view"));
        
        assertEquals( "Admin user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, allEntry, "view"));
        assertEquals( "Turbine user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, allEntry, "view"));
        assertEquals( "Anonymous user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( anonymousUser, allEntry, "view"));
        assertEquals( "null user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, allEntry, "view"));
    }

    public void testRegistryActions() throws Exception
    {
        JetspeedUser adminUser = (JetspeedUser) JetspeedSecurity.getUser("admin");
        assertNotNull( "Getting admin user", adminUser);
        adminUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser turbineUser = (JetspeedUser) JetspeedSecurity.getUser("turbine");
        assertNotNull( "Getting turbine user", turbineUser);
        turbineUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser anonymousUser =  (JetspeedUser) JetspeedSecurity.getAnonymousUser();
        assertNotNull( "Getting anonymous user", anonymousUser);
        // Create security objects
        Security adminSecurity = new BaseSecurity("admin");
        assertNotNull( "Have admin security", adminSecurity);
        Security userSecurity = new BaseSecurity("user");
        assertNotNull( "Have user security", userSecurity);

        PortletEntry userPortletEntry = new BasePortletEntry();
        assertNotNull( "Have userPortletEntry", userPortletEntry);
        userPortletEntry.setName( USER_PORTLET);
        userPortletEntry.setSecurity( userSecurity);
        Parameter adminParam = new BaseParameter();
        assertNotNull( "Have adminParameter", adminParam);
        adminParam.setName("AdminParam");
        adminParam.setValue("adminValue");
        adminParam.setSecurity(adminSecurity);
        userPortletEntry.addParameter(adminParam);

        Parameter userParam = new BaseParameter();
        assertNotNull( "Have userParameter", userParam);
        userParam.setName("UserParam");
        userParam.setValue("userValue");
        userParam.setSecurity(userSecurity);
        userPortletEntry.addParameter(userParam);
        assertEquals( "Admin user customize access to admin parameter", true, JetspeedPortalAccessController.checkPermission( adminUser, new PortalResource( userPortletEntry, adminParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        assertEquals( "Turbine user customize access to admin parameter", false, JetspeedPortalAccessController.checkPermission( turbineUser, new PortalResource( userPortletEntry, adminParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        assertEquals( "Admin user customize access to admin parameter", true, JetspeedPortalAccessController.checkPermission( adminUser, new PortalResource( userPortletEntry, userParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        assertEquals( "Turbine user customize access to admin parameter", true, JetspeedPortalAccessController.checkPermission( turbineUser, new PortalResource( userPortletEntry, userParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        
/*
        RegistryEntry adminEntry = createRegistryEntry( ADMIN_PORTLET, "ST_01.admin");
        RegistryEntry userEntry = createRegistryEntry( USER_PORTLET, "ST_01.user");
        RegistryEntry allEntry = createRegistryEntry( ALL_PORTLET, "ST_01.all");
        
        assertEquals( "Admin user has view access to " + ADMIN_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, adminEntry, "view"));
        assertEquals( "Turbine user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( turbineUser, adminEntry, "view"));
        assertEquals( "Anonymous user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, adminEntry, "view"));
        assertEquals( "null user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, adminEntry, "view"));
        
        assertEquals( "Admin user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, userEntry, "view"));
        assertEquals( "Turbine user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, userEntry, "view"));
        assertEquals( "Anonymous user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, userEntry, "view"));
        assertEquals( "null user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, userEntry, "view"));
        
        assertEquals( "Admin user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, allEntry, "view"));
        assertEquals( "Turbine user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, allEntry, "view"));
        assertEquals( "Anonymous user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( anonymousUser, allEntry, "view"));
        assertEquals( "null user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, allEntry, "view"));
*/
    }
    /*
     * Setup Turbine environment
     */
    
    /*
     * Configuration object to run Turbine outside a servlet container
     * ( uses turbine.properties )
     */
    private static TurbineConfig config = null;
    
    /*
     * Sets up TurbineConfig using the system property:
     * <pre>turbine.properties</pre>
     */
    static
    {
        try
        {
            config = new TurbineConfig( "webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }
    
    private PsmlEntry createEntry(java.lang.String parent, java.lang.String id)
    {
        PsmlEntry entry = new PsmlEntry();
        entry.setParent( parent);
        if (id != null)
            entry.setId( id);
        return entry;
    }
}
