/***************************************************************************
 *            cxp-profile.c
 *
 *  Mon Nov 29 23:50:12 2004
 *  Copyright  2004  Yasumichi Akahoshi
 *  yasumichi@users.sourceforge.jp
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "cxp-profile.h"

/**
 * @brief private member of CxpProfile
 */

struct _CxpProfilePrivate
{
	gchar *appname;			/**< application name */
	gchar *section;			/**< section name */
	GHashTable *config;		/**< hash table of configuration */
	gboolean dispose_has_run;	/**< Is dispose funciton executed */
};

/*
 * privete methods
 */
static void cxp_profile_init (GTypeInstance * instance, gpointer g_class);
static void cxp_profile_class_init (gpointer g_class, gpointer g_class_data);
static void cxp_profile_dispose (GObject * obj);
static void cxp_profile_finalize (GObject * obj);
static void cxp_profile_save_file (gpointer key, gpointer value,
				   gpointer channel);
static void cxp_profile_load_file (CxpProfile * profile);

/**
 * regist CxpProfile to GTypeTable.
 * @return GType
 */
GType cxp_profile_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpProfileClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			cxp_profile_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpProfile),
			0,	/* n_preallocs */
			cxp_profile_init	/* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT, "CxpProfileType",
					       &info, 0);
	}

	return type;
}

/**
 * base init CxpProfile
 *
 */
static void cxp_profile_init (GTypeInstance * instance, gpointer g_class)
{
	CxpProfile *self = CXP_PROFILE (instance);

	self->priv = g_new (CxpProfilePrivate, 1);
	self->priv->appname = NULL;
	self->priv->section = NULL;
	self->priv->config =
		g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	self->priv->dispose_has_run = FALSE;
}

/**
 * class init CxpProfileClass
 *
 */
static void cxp_profile_class_init (gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);

	gobject_class->dispose = cxp_profile_dispose;
	gobject_class->finalize = cxp_profile_finalize;
}

static void cxp_profile_dispose (GObject * obj)
{
	CxpProfile *self = CXP_PROFILE (obj);
	GIOChannel *channel;
	gchar *filename;

	if (self->priv->dispose_has_run)
	{
		return;
	}

	self->priv->dispose_has_run = TRUE;
	filename = g_build_filename (
			g_get_home_dir (),
			".cxp", 
			self->priv->appname,
			self->priv->section,
			NULL );
	if ((channel =
	     g_io_channel_new_file (filename, "w", NULL)) != NULL)
	{
		g_hash_table_foreach (self->priv->config, cxp_profile_save_file,
				      channel);
		g_io_channel_flush (channel, NULL);

		g_io_channel_unref (channel);
	}
	g_free (filename);
}

static void cxp_profile_finalize (GObject * obj)
{
	CxpProfile *self = CXP_PROFILE (obj);

	g_free (self->priv->appname);
	g_free (self->priv->section);
	g_hash_table_destroy (self->priv->config);
	g_free (self->priv);
}

CxpProfile *cxp_profile_new (const gchar * appname, const gchar *section)
{
	CxpProfile *profile;
	gchar *cxpdir;
	gchar *appdir;

	profile = CXP_PROFILE (g_object_new (CXP_TYPE_PROFILE, NULL));
	profile->priv->appname = g_strdup (appname);
	profile->priv->section = g_strdup (section);
	cxpdir = g_build_filename (
			g_get_home_dir (),
			".cxp",
			NULL);
	if (!g_file_test (cxpdir, G_FILE_TEST_EXISTS))
	{
		if (mkdir (cxpdir, S_IFDIR | 0777) != 0)
		{
			g_error ("%s couldn't be created.\n", cxpdir);
		}
	}
	appdir = g_build_filename (
			cxpdir,
			appname,
			NULL);
	if (!g_file_test (appdir, G_FILE_TEST_EXISTS))
	{
		if (mkdir (appdir, S_IFDIR | 0777) != 0)
		{
			g_error ("%s couldn't be created.\n", appdir);
		}
	}
	cxp_profile_load_file (profile);

	g_free (cxpdir);
	g_free (appdir);
	return profile;
}

gint cxp_profile_get_integer (CxpProfile * profile, const gchar * name)
{
	gint retval;
	gchar *value;

	value = (gchar *) g_hash_table_lookup (profile->priv->config, name);
	if (value == NULL)
	{
		retval = 0;
		g_hash_table_replace (profile->priv->config, g_strdup (name),
				      g_strdup ("0"));
	}
	else
	{
		retval = atoi (value);
	}

	return retval;
}

gchar *cxp_profile_get_string (CxpProfile * profile, const gchar * name)
{
	gchar *value;

	value = (gchar *) g_hash_table_lookup (profile->priv->config, name);

	return g_strdup (value);
}

void cxp_profile_set_integer (CxpProfile * profile, const gchar * key,
			      gint value)
{
	g_hash_table_replace (profile->priv->config, g_strdup (key),
			      g_strdup_printf ("%d", value));
}

void cxp_profile_set_string (CxpProfile * profile, const gchar * key,
			     const gchar * value)
{
	g_hash_table_replace (profile->priv->config, g_strdup (key),
			      g_strdup (value));
}


static void cxp_profile_load_file (CxpProfile * profile)
{
	gchar *filename;
	GHashTable *config = profile->priv->config;
	gchar *line;
	gchar **array;
	gchar *name;
	gchar *value;
	GIOChannel *channel;

	filename = g_build_filename (
			g_get_home_dir (),
			".cxp", 
			profile->priv->appname,
			profile->priv->section,
			NULL );
	if ((channel = g_io_channel_new_file (filename, "r", NULL)) == NULL)
	{
		g_free(filename);
		filename = g_build_filename (CONFDIR, profile->priv->appname, profile->priv->section, NULL);
		channel = g_io_channel_new_file (filename, "r", NULL);
	}

	if (channel != NULL)
	{
		while (g_io_channel_read_line
		       (channel, &line, NULL, NULL, NULL) == G_IO_STATUS_NORMAL)
		{
			g_strstrip (line);
			if (strlen (line) > 0)
			{
				array = g_strsplit (line, "=", 2);
				name = g_strdup (array[0]);
				value = g_strdup (array[1]);
				g_hash_table_replace(config, name, value);
			}
			g_free (line);
		}
		g_io_channel_unref(channel);
	}
	g_free (filename);
}

static void cxp_profile_save_file (gpointer key, gpointer value,
				   gpointer channel)
{
	GIOChannel *chan = (GIOChannel *) channel;
	gchar *line;

	line = g_strdup_printf ("%s=%s\n", (gchar *) key, (gchar *) value);
	g_io_channel_write_chars (chan, line, -1, NULL, NULL);
	g_free (line);
}
