/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include <gdk/gdkkeysyms.h>
#include <gconf/gconf-client.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include "cxp-right-pane.h"
#include "cxp-file-list-store.h"

/* If you use Pimpls, include the private structure 
 * definition here. Some people create a cxp-right_pane-private.h header
 * which is included by the cxp-right_pane.c file and which contains the
 * definition for this private structure.
 */

typedef struct
{
	GConfClient *client;
	GtkWidget *file_list;
	GtkWidget *expander;
	GtkWidget *preview;
	GtkWidget *popup_menu;
	gboolean dispose_has_run;
	gchar *cur_dir;
	CxpHandler *handler;
} CxpRightPanePrivate;

#define CXP_RIGHT_PANE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXP_TYPE_RIGHT_PANE, CxpRightPanePrivate))

/*
 * ץѥƥ̤뤿
 * This enumeration define property_id.
 */
enum
{
	CXP_RIGHT_PANE_GCONF_CLIENT = 1,
};

/**
 * This enumeration define signal.
 */
enum
{
	DIR_DOUBLE_CLICKED_SIGNAL,
	UP_DIR_REQUEST_SIGNAL,
	LAST_SIGNAL
};

enum
{
	TARGET_CXP_FILE_INFO,
	TARGET_STRING
};

static GtkTargetEntry cxp_target_types[] = {
	{"_CXP_FILE_INFO", 0, TARGET_CXP_FILE_INFO},
	{"STRING", 0, TARGET_STRING}
};

static GObjectClass *parent_class = NULL;
static guint cxp_right_pane_signals[LAST_SIGNAL] = { 0 };

/* Declare prototype */
static void cxp_right_pane_class_init (CxpRightPaneClass * klass);
static void cxp_right_pane_instance_init (CxpRightPane * self);
static void cxp_right_pane_dispose (GObject * obj);
static void cxp_right_pane_finalize (GObject * obj);
static void cxp_right_pane_set_property (GObject * object,
					 guint property_id,
					 const GValue * value,
					 GParamSpec * pspec);
static void cxp_right_pane_get_property (GObject * object,
					 guint property_id,
					 GValue * value, GParamSpec * pspec);
static GtkWidget *cxp_right_pane_file_list_new (CxpRightPane * self);
static void cxp_right_pane_on_cursor_changed (GtkTreeView * treeview,
					      gpointer user_data);
static void cxp_right_pane_on_row_activated (GtkTreeView * treeview,
					     GtkTreePath * path,
					     GtkTreeViewColumn * column,
					     gpointer user_data);
static void cxp_right_pane_on_key_pressed (GtkWidget * widget,
					   GdkEventKey * event,
					   gpointer user_data);
static void cxp_right_pane_draw_icon (GtkTreeViewColumn * tree_column,
				      GtkCellRenderer * cell,
				      GtkTreeModel * tree_model,
				      GtkTreeIter * iter, gpointer data);
static void cxp_right_pane_draw_filename (GtkTreeViewColumn * tree_column,
					  GtkCellRenderer * cell,
					  GtkTreeModel * tree_model,
					  GtkTreeIter * iter, gpointer data);
static void cxp_right_pane_draw_filetype (GtkTreeViewColumn * tree_column,
					  GtkCellRenderer * cell,
					  GtkTreeModel * tree_model,
					  GtkTreeIter * iter, gpointer data);
static void cxp_right_pane_draw_update_time (GtkTreeViewColumn * tree_column,
					     GtkCellRenderer * cell,
					     GtkTreeModel * tree_model,
					     GtkTreeIter * iter, gpointer data);
static void cxp_right_pane_cell_edited (GtkCellRendererText * cell,
					const gchar * path_string,
					const gchar * new_text, gpointer data);
static gboolean cxp_right_pane_search_equal_func (GtkTreeModel *model, gint column, const gchar *key, GtkTreeIter *iter, gpointer search_data);
static gint cxp_right_pane_compare_by_default (GtkTreeModel * model,
					       GtkTreeIter * iter_a,
					       GtkTreeIter * iter_b,
					       gpointer user_data);
static GtkTreeIter cxp_right_pane_append_file (CxpRightPane * self,
					       CxpFileListStore * store,
					       const gchar * fullpath);
static void cxp_right_pane_show_preview (CxpRightPane * self,
					 GtkTextBuffer * text_buffer,
					 const gchar * filename);
static void cxp_right_pane_preview_by_command (CxpRightPane * self,
					       GtkTextBuffer * text_buffer,
					       const gchar * command);
static gint cxp_right_pane_popup_handler (GtkWidget * widget, GdkEvent * event);
static void cxp_right_pane_preview_expanded (GObject * object,
					     GParamSpec * param_spec,
					     gpointer user_data);
static void cxp_right_pane_delete_selected_file_foreach (GtkTreeModel * model,
							 GtkTreePath * path,
							 GtkTreeIter * iter,
							 gpointer userdata);
static gchar *cxp_right_pane_selection_to_string (CxpRightPane *obj);
static void cxp_right_pane_clipboard_get (GtkClipboard *clipboard, GtkSelectionData *selection_data, guint info, gpointer user_data_or_owner);
static void cxp_right_pane_clipboard_received (GtkClipboard *clipboard, GtkSelectionData *selection_data, gpointer data);
static void cxp_right_pane_drag_data_get(GtkWidget *widget, GdkDragContext *dc, GtkSelectionData *selection_data, guint info, guint t, gpointer data);
static void cxp_right_pane_drag_data_delete (GtkWidget *widget, GdkDragContext *dc, gpointer data);
static void cxp_right_pane_drag_data_received (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, GtkSelectionData *selection_data, guint info, guint t, gpointer data);
static gboolean cxp_right_pane_drag_drop (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, guint t, gpointer data);
static gchar *cxp_right_pane_get_drag_dest_dir (CxpRightPane *obj, gint x, gint y);

GType cxp_right_pane_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpRightPaneClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			(GClassInitFunc) cxp_right_pane_class_init,
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpRightPane),
			0,	/* n_preallocs */
			(GInstanceInitFunc) cxp_right_pane_instance_init
		};
		type = g_type_register_static (GTK_TYPE_VBOX,
					       "CxpRightPaneType", &info, 0);
	}
	return type;
}

static void cxp_right_pane_class_init (CxpRightPaneClass * klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	GParamSpec *pspec;

	gobject_class->dispose = cxp_right_pane_dispose;
	gobject_class->finalize = cxp_right_pane_finalize;
	gobject_class->set_property = cxp_right_pane_set_property;
	gobject_class->get_property = cxp_right_pane_get_property;

	g_type_class_add_private (klass, sizeof (CxpRightPanePrivate));

	/* install properties. */
	pspec = g_param_spec_object ("gconf_client",
				     "GConfClient",
				     "Set GConfClient",
				     GCONF_TYPE_CLIENT,
				     G_PARAM_CONSTRUCT_ONLY |
				     G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 CXP_RIGHT_PANE_GCONF_CLIENT, pspec);

	cxp_right_pane_signals[DIR_DOUBLE_CLICKED_SIGNAL] =
		g_signal_new ("dir_double_clicked",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
			      G_STRUCT_OFFSET (CxpRightPaneClass,
					       dir_double_clicked), NULL,
			      NULL, g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	cxp_right_pane_signals[UP_DIR_REQUEST_SIGNAL] =
		g_signal_new ("up_dir_request",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
			      G_STRUCT_OFFSET (CxpRightPaneClass,
					       up_dir_request), NULL,
			      NULL, g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);


	parent_class = g_type_class_peek_parent (klass);
}

static void cxp_right_pane_instance_init (CxpRightPane * self)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (self);
	GtkWidget *scrolled_window;
	PangoFontDescription *font_desc;

	priv->client = NULL;

	//gtk_paned_set_position (GTK_PANED (self), 250);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_set_size_request (GTK_WIDGET (scrolled_window), 300, 200);
	gtk_box_pack_start (GTK_BOX (self), scrolled_window, TRUE, TRUE, 0);
	gtk_widget_show (scrolled_window);
	priv->file_list = cxp_right_pane_file_list_new (self);
	gtk_container_add (GTK_CONTAINER (scrolled_window), priv->file_list);
	gtk_widget_show (priv->file_list);
	gtk_drag_source_set(priv->file_list, GDK_BUTTON1_MASK, cxp_target_types, G_N_ELEMENTS(cxp_target_types), GDK_ACTION_MOVE);
	//gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW(priv->file_list), GDK_BUTTON1_MASK, cxp_target_types, G_N_ELEMENTS(cxp_target_types), GDK_ACTION_MOVE);
	//gtk_drag_dest_set (priv->file_list, GTK_DEST_DEFAULT_ALL, cxp_target_types, G_N_ELEMENTS(cxp_target_types), GDK_ACTION_MOVE);
	gtk_tree_view_enable_model_drag_dest(GTK_TREE_VIEW(priv->file_list), cxp_target_types, G_N_ELEMENTS(cxp_target_types), GDK_ACTION_MOVE);
	gtk_tree_view_set_search_column (GTK_TREE_VIEW(priv->file_list), FILE_LIST_COL_FULLPATH);
	gtk_tree_view_set_search_equal_func (GTK_TREE_VIEW(priv->file_list), cxp_right_pane_search_equal_func, NULL, NULL);

	priv->expander = gtk_expander_new (_("<b>Preview</b>"));
	gtk_expander_set_use_markup (GTK_EXPANDER (priv->expander), TRUE);
	gtk_box_pack_start (GTK_BOX (self), priv->expander, FALSE, TRUE, 0);
	gtk_widget_show (priv->expander);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_set_size_request (GTK_WIDGET (scrolled_window), -1, 200);
	gtk_container_add (GTK_CONTAINER (priv->expander), scrolled_window);
	gtk_widget_show (scrolled_window);

	priv->preview = gtk_text_view_new ();
	font_desc = pango_font_description_new ();
	pango_font_description_set_family (font_desc, "monospace");
	gtk_widget_modify_font (GTK_WIDGET (priv->preview), font_desc);
	pango_font_description_free (font_desc);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (priv->preview), FALSE);
	gtk_container_add (GTK_CONTAINER (scrolled_window), priv->preview);
	gtk_widget_show (priv->preview);

	priv->dispose_has_run = FALSE;
	priv->cur_dir = NULL;
	priv->handler = cxp_handler_new ();

	g_signal_connect (priv->file_list, "cursor_changed",
			  G_CALLBACK (cxp_right_pane_on_cursor_changed), self);
	g_signal_connect (priv->file_list, "row_activated",
			  G_CALLBACK (cxp_right_pane_on_row_activated), self);
	g_signal_connect (priv->file_list, "drag_data_get",
			G_CALLBACK (cxp_right_pane_drag_data_get), self);
	g_signal_connect (priv->file_list, "drag_data_delete",
			G_CALLBACK (cxp_right_pane_drag_data_delete), self);
	g_signal_connect (priv->file_list, "drag_drop",
			G_CALLBACK (cxp_right_pane_drag_drop), self);
	g_signal_connect (priv->file_list, "drag_data_received",
			G_CALLBACK (cxp_right_pane_drag_data_received), self);
	g_signal_connect (priv->file_list, "key-press-event",
			  G_CALLBACK (cxp_right_pane_on_key_pressed), self);
	g_signal_connect (priv->expander, "notify::expanded",
			  G_CALLBACK (cxp_right_pane_preview_expanded), self);
}

static void cxp_right_pane_dispose (GObject * obj)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (obj);

	if (priv->dispose_has_run)
	{
		/* If dispose did already run, return. */
		return;
	}
	/* Make sure dispose does not run twice. */
	priv->dispose_has_run = TRUE;

	/* 
	 * In dispose, you are supposed to free all types referenced from this
	 * object which might themselves hold a reference to self. Generally,
	 * the most simple solution is to unref all members on which you own a 
	 * reference.
	 */
	g_object_unref (priv->client);
	g_object_unref (priv->handler);


	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void cxp_right_pane_finalize (GObject * obj)
{
	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static void cxp_right_pane_set_property (GObject * object,
					 guint property_id,
					 const GValue * value,
					 GParamSpec * pspec)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (object);

	switch (property_id)
	{
	case CXP_RIGHT_PANE_GCONF_CLIENT:
		priv->client = GCONF_CLIENT (g_value_dup_object (value));
		break;
	default:
		/* We don't have any other property... */
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static void cxp_right_pane_get_property (GObject * object,
					 guint property_id,
					 GValue * value, GParamSpec * pspec)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (object);

	switch (property_id)
	{
	case CXP_RIGHT_PANE_GCONF_CLIENT:
		g_value_set_object (value, priv->client);
		break;
	default:
		/* We don't have any other property... */
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static GtkWidget *cxp_right_pane_file_list_new (CxpRightPane * self)
{
	GtkWidget *file_list;
	CxpFileListStore *store;
	GtkTreeViewColumn *column;
	GtkTreeSelection *selection;
	GtkCellRenderer *renderer;
	GtkTreeSortable *sortable;

	file_list = gtk_tree_view_new ();
	store = cxp_file_list_store_new ();
	gtk_tree_view_set_model (GTK_TREE_VIEW (file_list),
				 GTK_TREE_MODEL (store));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (file_list));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

	/* Sort setting */
	   sortable = GTK_TREE_SORTABLE (store);
	/*
	   gtk_tree_sortable_set_sort_func (sortable, SORT_ID_NAME,
	   cxp_right_pane_compare_by_default,
	   NULL, NULL);
	 */
	   gtk_tree_sortable_set_sort_column_id (sortable, SORT_ID_NAME,
	   GTK_SORT_ASCENDING);

	/* 1st column */
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Filename"));
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (file_list), column);
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column, renderer,
						 cxp_right_pane_draw_icon,
						 self, NULL);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_set_sort_column_id (column, SORT_ID_NAME);
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, renderer,
					     "editable",
					     FILE_LIST_COL_EDITABLE, NULL);
	gtk_tree_view_column_set_cell_data_func (column, renderer,
						 cxp_right_pane_draw_filename,
						 self, NULL);
	g_signal_connect (renderer, "edited",
			  G_CALLBACK (cxp_right_pane_cell_edited), self);

	/* 2nd column */
	renderer = gtk_cell_renderer_text_new ();
	g_object_set (renderer, "xalign", 1.0, NULL);
	column = gtk_tree_view_column_new_with_attributes (_("Size"), renderer,
							   "text",
							   FILE_LIST_COL_FILESIZE,
							   NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sort_column_id (column, SORT_ID_SIZE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (file_list), column);

	/* 3rd column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Type"), renderer,
							   NULL);
	gtk_tree_view_column_set_cell_data_func (column, renderer,
						 cxp_right_pane_draw_filetype,
						 self, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (column, 100);
	gtk_tree_view_column_set_sort_column_id (column, SORT_ID_TYPE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (file_list), column);

	/* 4th column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Update Time"),
							   renderer, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (file_list), column);
	gtk_tree_view_column_set_sort_column_id (column, SORT_ID_TIME);
	gtk_tree_view_column_set_cell_data_func (column, renderer,
						 cxp_right_pane_draw_update_time,
						 NULL, NULL);

	g_object_unref (store);

	return file_list;
}

/*
 * Internal event handlers
 */

/**
 * This function is called back when cursor is changed on file list view.
 * @param treeview [in] Pointer to instance of GtkTreeView.
 * @param user_data [in] Pointer to instance of Cxplorer which is casted by
 * gpointer.
 */
static void cxp_right_pane_on_cursor_changed (GtkTreeView * treeview,
					      gpointer user_data)
{
	CxpRightPane *self = CXP_RIGHT_PANE (user_data);
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (self);
	GtkTextBuffer *text_buffer;
	GtkTreePath *treepath;
	GtkTreeViewColumn *column;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar *fullpath;

	if (gtk_expander_get_expanded (GTK_EXPANDER (priv->expander)) == FALSE)
	{
		return;
	}

	text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->preview));
	gtk_text_buffer_set_text (text_buffer, "", -1);

	gtk_tree_view_get_cursor (treeview, &treepath, &column);
	if (treepath)
	{
		model = gtk_tree_view_get_model (treeview);
		if (gtk_tree_model_get_iter (model, &iter, treepath))
		{
			gtk_tree_model_get (model, &iter,
					    FILE_LIST_COL_FULLPATH, &fullpath,
					    -1);
			cxp_right_pane_show_preview (self, text_buffer,
						     fullpath);
			g_free (fullpath);
		}
	}
}


/**
 * This function is called back when any row is activate.
 * @param treeview [in] Pointer to instance of GtkTreeView.
 * @param path [in] Pointer to instance of GtkTreePath.
 * @param column [in] Pointer to instance of GtkTreeViewColumn which is
 * activate.
 * @param user_data [in] Pointer to data which is defined by user.
 * @todo fixed me to process deep link and none absolute link.
 */
static void cxp_right_pane_on_row_activated (GtkTreeView * treeview,
					     GtkTreePath * path,
					     GtkTreeViewColumn * column,
					     gpointer user_data)
{
	CxpRightPane *right_pane = CXP_RIGHT_PANE (user_data);
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);
	gchar *fullpath;
	GtkTreeModel *model;
	GtkTreeIter iter;

	if (priv->dispose_has_run)
	{
		return;
	}

	model = gtk_tree_view_get_model (treeview);
	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, FILE_LIST_COL_FULLPATH, &fullpath,
			    -1);

	if (g_file_test (fullpath, G_FILE_TEST_IS_DIR))
	{
		g_signal_emit (right_pane,
			       cxp_right_pane_signals
			       [DIR_DOUBLE_CLICKED_SIGNAL], 0);
	}
	else
	{
		cxp_handler_launch (priv->handler, fullpath);
	}

	g_free (fullpath);
}

static void cxp_right_pane_on_key_pressed (GtkWidget * widget,
					   GdkEventKey * event,
					   gpointer user_data)
{
	CxpRightPane *self = CXP_RIGHT_PANE (user_data);
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (self);

	switch (event->keyval)
	{
	case GDK_F2:
		cxp_right_pane_rename_file_request (self);
		break;
	case GDK_Delete:
		cxp_right_pane_delete_selected_files (self);
		break;
	case GDK_BackSpace:
		g_signal_emit (self,
			       cxp_right_pane_signals
			       [UP_DIR_REQUEST_SIGNAL], 0);
		break;
	default:
		GTK_WIDGET_GET_CLASS (priv->file_list)->key_press_event (widget,
									 event);
	}
}

/**
 * This function is called back before pixbuf cell is painted.
 * @param tree_column [in] Pointer to instance fo GtkTreeViewColumn.
 * @param cell [in] Pointer to instance of GtkCellRender which render pixbuf.
 * @param tree_model [in] Pointer to memory model of GtkTreeView.
 * @param iter [in] Iterator of node which want to be paint by GtkCellRender.
 * @param data [in] Pointer to data which is defined by user.
 */
static void cxp_right_pane_draw_icon (GtkTreeViewColumn * tree_column,
				      GtkCellRenderer * cell,
				      GtkTreeModel * tree_model,
				      GtkTreeIter * iter, gpointer data)
{
	gchar *mimetype;
	gchar *fullpath;
	gchar *linkend;
	gchar *linkmime;
	GdkPixbuf *icon;
	GdkPixbuf *mask;
	mode_t mode;

	gtk_tree_model_get (tree_model, iter, FILE_LIST_COL_MIMETYPE, &mimetype, 
			FILE_LIST_COL_FULLPATH, &fullpath,
			FILE_LIST_COL_FILEMODE, &mode,
			-1);

	switch (mode & S_IFMT)
	{
	case S_IFLNK:
		linkend = cxp_path_get_link_end (fullpath);
		linkmime = cxp_get_mime_type_for_file (linkend);
		icon = gdk_pixbuf_copy (cxp_lookup_icon_from_mime (linkmime));
		mask = cxp_lookup_icon_from_mime (mimetype);
		gdk_pixbuf_composite (mask, icon, 0, 0, 16, 16, 0.0, 0.0, 1.0, 1.0, GDK_INTERP_NEAREST, 255);
		break;
	default:
		icon = cxp_lookup_icon_from_mime (mimetype);
	}

	g_object_set (cell, "pixbuf", icon, NULL);
	g_free (mimetype);
}

static void cxp_right_pane_draw_filename (GtkTreeViewColumn * tree_column,
					  GtkCellRenderer * cell,
					  GtkTreeModel * tree_model,
					  GtkTreeIter * iter, gpointer data)
{
	gchar *fullpath;
	gchar *filename;

	gtk_tree_model_get (tree_model, iter, FILE_LIST_COL_FULLPATH, &fullpath,
			    -1);
	if ((filename = cxp_path_get_basename_of_utf8 (fullpath)) == NULL)
	{
		filename = g_strdup (_("(Invalid filename)"));
	}

	g_object_set (cell, "text", filename, NULL);

	g_free (fullpath);
}

static void cxp_right_pane_draw_filetype (GtkTreeViewColumn * tree_column,
					  GtkCellRenderer * cell,
					  GtkTreeModel * tree_model,
					  GtkTreeIter * iter, gpointer data)
{
	gchar *mimetype;
	gchar *comment = NULL;
	mode_t mode;

	gtk_tree_model_get (tree_model, iter, FILE_LIST_COL_MIMETYPE, &mimetype,
			    FILE_LIST_COL_FILEMODE, &mode, -1);
	switch (mode & S_IFMT)
	{
	case S_IFDIR:
		comment = g_strdup (_("directory"));
		break;
	default:
		comment = cxp_get_mime_comment (mimetype);
		break;
	}
	g_object_set (cell, "text", comment, NULL);

	g_free (mimetype);
	g_free (comment);
}

/**
 * This function is called back before update time is shown.
 * @param tree_column [in] Pointer to instance fo GtkTreeViewColumn.
 * @param cell [in] Pointer to instance of GtkCellRender which render text.
 * @param tree_model [in] Pointer to memory model of GtkTreeView.
 * @param iter [in] Iterator of node which want to be paint by GtkCellRender.
 * @param data [in] Pointer to data which is defined by user.
 */
static void cxp_right_pane_draw_update_time (GtkTreeViewColumn * tree_column,
					     GtkCellRenderer * cell,
					     GtkTreeModel * tree_model,
					     GtkTreeIter * iter, gpointer data)
{
	gchar *time_string;
	time_t updated;
	struct tm *tm;

	gtk_tree_model_get (tree_model, iter, FILE_LIST_COL_FILETIME,
			    &updated, -1);
	tm = localtime (&updated);
	time_string = g_new (gchar, 20);
	strftime (time_string, 20, "%Y/%m/%d %H:%M:%S", tm);
	g_object_set (cell, "text", time_string, NULL);
	g_free (time_string);
}

static void cxp_right_pane_cell_edited (GtkCellRendererText * cell,
					const gchar * path_string,
					const gchar * new_text, gpointer data)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (data);
	GtkTreeModel *model =
		gtk_tree_view_get_model (GTK_TREE_VIEW (priv->file_list));
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	GtkTreeIter iter;
	GtkWidget *dialog;
	gchar *dirname;
	gchar *srcpath;
	gchar *srcbase;
	gchar *destpath;
	gchar *destbase_locale;
	gchar *error_msg;

	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, FILE_LIST_COL_FULLPATH, &srcpath, -1);
	srcbase = cxp_path_get_basename_of_utf8 (srcpath);
	if (srcbase == NULL)
	{
		srcbase = g_strdup(_("(Invalid filename)"));
	}
	if ((strcmp (srcbase, new_text) == 0) || (strcmp(new_text, "") == 0))
	{
		cxp_file_list_store_set_editable (CXP_FILE_LIST_STORE (model),
						  &iter, FALSE);
		g_free (srcbase);
		g_free (srcpath);
		gtk_tree_path_free (path);
		return;
	}

	dirname = g_path_get_dirname (srcpath);
	destbase_locale = g_locale_from_utf8 (new_text, -1, NULL, NULL, NULL);
	destpath = g_build_filename (dirname, destbase_locale, NULL);

	if (g_file_test (destpath, G_FILE_TEST_EXISTS) == FALSE)
	{
		errno = 0;
		if (rename (srcpath, destpath) == 0)
		{
			cxp_file_list_store_rename_file (CXP_FILE_LIST_STORE
							 (model), &iter,
							 destpath);
			cxp_file_list_store_set_editable (CXP_FILE_LIST_STORE
							  (model), &iter,
							  FALSE);
		}
		else
		{
			cxp_file_list_store_set_editable (CXP_FILE_LIST_STORE
							  (model), &iter,
							  FALSE);
			cxp_error_dialog_run_about_file (srcbase);
		}
	}
	else
	{
		cxp_file_list_store_set_editable (CXP_FILE_LIST_STORE (model),
						  &iter, FALSE);
		error_msg =
			g_strdup_printf (_("%s is already exists."), new_text);
		dialog = gtk_message_dialog_new (NULL,
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE, "%s",
						 error_msg);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		g_free (error_msg);
	}

	g_free (destpath);
	g_free (destbase_locale);
	g_free (dirname);
	g_free (srcpath);
	g_free (srcbase);
	/* test */

	gtk_tree_path_free (path);
}

/*
 * for intaractive search
 */
static gboolean cxp_right_pane_search_equal_func (GtkTreeModel *model, gint column, const gchar *key, GtkTreeIter *iter, gpointer search_data)
{
	gchar *fullpath;
	gchar *base_utf8;
	gint length;
	gboolean retval;

	gtk_tree_model_get (model, iter, column, &fullpath, -1);
	base_utf8 = cxp_path_get_basename_of_utf8 (fullpath);
	length = strlen (key);
	if (g_strncasecmp(key, base_utf8, length) == 0)
	{
		retval = FALSE;
	}
	else
	{
		retval = TRUE;
	}

	g_free (base_utf8);

	return	retval;
}

/**
 * compare function for default sort method.
 * @param model [in] A instance of GtkTreeModel.
 * @param iter_a [in] A iterator of tree node.
 * @param iter_b [in] A iterator of tree node.
 * @param user_data [in] SortID
 */
static gint cxp_right_pane_compare_by_default (GtkTreeModel * model,
					       GtkTreeIter * iter_a,
					       GtkTreeIter * iter_b,
					       gpointer user_data)
{
	gint result = 0;
	gint mode_a, mode_b;
	gchar *name_a, *name_b;

	gtk_tree_model_get (model, iter_a, FILE_LIST_COL_FULLPATH, &name_a,
			    FILE_LIST_COL_FILEMODE, &mode_a, -1);
	gtk_tree_model_get (model, iter_b, FILE_LIST_COL_FULLPATH, &name_b,
			    FILE_LIST_COL_FILEMODE, &mode_b, -1);
	mode_a &= S_IFMT;
	mode_b &= S_IFMT;
	if (mode_a == mode_b)
	{
		if (name_a == NULL || name_b == NULL)
		{
			if (name_a == NULL && name_b == NULL)
				result = 0;

			result = (name_a == NULL) ? -1 : 1;
		}
		else
		{
			result = g_utf8_collate (name_a, name_b);
		}
	}
	else
	{
		result = mode_a - mode_b;
	}

	g_free (name_a);
	g_free (name_b);

	return result;
}

/*
 * Internal API
 */

/**
 * This appends file to GtkListStore as a new row.
 * @param glStore [in] A GtkListStore.
 * @param gcFullPath [in] Current locale string.
 * @param gcLabel [in] UTF-8 string.
 * @return GtkTreeIter of appendded row.
 */
static GtkTreeIter cxp_right_pane_append_file (CxpRightPane * self,
					       CxpFileListStore * store,
					       const gchar * fullpath)
{
	GtkTreeIter iter;

	cxp_file_list_store_append_file (store, fullpath, &iter);

	return iter;
}

/**
 * This function shows 100 byte data in file When any file is selected.
 * @param cxplorer [in] An instance of Cxplorer.
 * @param gcFileName [in] Full path of file which is selected.
 */
static void cxp_right_pane_show_preview (CxpRightPane * self,
					 GtkTextBuffer * text_buffer,
					 const gchar * filename)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (self);
	GdkPixbuf *pixbuf;
	GtkTextIter iter;
	gchar *preview_cmd = NULL;
	gchar *cmd = NULL;
	gchar *filetype;
	gchar *key;
	struct stat status;

	gtk_text_buffer_set_text (text_buffer, "", -1);	/* clear text buffer */
	stat (filename, &status);
	switch (status.st_mode & S_IFMT)
	{
	case S_IFREG:
		filetype = cxp_get_mime_type_for_file (filename);
		if (g_str_has_prefix (filetype, "image") == TRUE)
		{
			if ((pixbuf =
			     gdk_pixbuf_new_from_file (filename, NULL)) != NULL)
			{
				gtk_text_buffer_get_iter_at_line (text_buffer,
								  &iter, 0);
				gtk_text_buffer_insert_pixbuf (text_buffer,
							       &iter, pixbuf);
				g_free (filetype);

				return;
			}
		}

		key = g_strdup_printf ("/apps/cxp/mime-types/%s/preview",
				       filetype);
		g_strdelimit (key, "+", '@');	/* example: application/xhtml+xml => /apps/cxp/mime-types/application/xhtml@xml/preview */
		if (((preview_cmd = gconf_client_get_string (priv->client, key, NULL)) == NULL) || (strcmp(preview_cmd,"") == 0))
		{
			if (g_str_has_prefix (filetype, "text") == TRUE)
			{
				preview_cmd =
					gconf_client_get_string (priv->client,
								 "/apps/cxp/cxplorer/PreviewText",
								 NULL);
			}
			else
			{
				preview_cmd =
					gconf_client_get_string (priv->client,
								 "/apps/cxp/cxplorer/PreviewBinary",
								 NULL);
			}
		}
		g_free (key);
		g_free (filetype);
		break;
	case S_IFDIR:
		preview_cmd =
			gconf_client_get_string (priv->client,
						 "/apps/cxp/cxplorer/PreviewDir",
						 NULL);
		break;
	case S_IFCHR:
	case S_IFBLK:
	case S_IFIFO:
	case S_IFSOCK:
	default:
		return;
	}

	cmd = g_strdup_printf ("%s '%s'", preview_cmd, filename);
	cxp_right_pane_preview_by_command (self, text_buffer, cmd);
	g_free (preview_cmd);
	g_free (cmd);
}


static void cxp_right_pane_preview_by_command (CxpRightPane * self,
					       GtkTextBuffer * text_buffer,
					       const gchar * command)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (self);
	const gchar *encoding;
	gchar **args;
	gint sout;
	gint loop;
	gint max;
	GPid pid;
	GIOChannel *channel;
	GString *str;
	GError *error = NULL;

	max = gconf_client_get_int (priv->client,
				    "/apps/cxp/cxplorer/PreviewLines", NULL);
	g_shell_parse_argv (command, NULL, &args, NULL);

	if (g_spawn_async_with_pipes
	    (NULL, args, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL, &pid, NULL,
	     &sout, NULL, &error))
	{
		channel = g_io_channel_unix_new (sout);
		if (g_get_charset (&encoding) == FALSE)
		{
			g_io_channel_set_encoding (channel, encoding, NULL);
		}
		str = g_string_new ("");
		for (loop = 0;
		     (loop < max)
		     &&
		     (g_io_channel_read_line_string (channel, str, NULL, NULL)
		      == G_IO_STATUS_NORMAL); loop++)
		{
			gtk_text_buffer_insert_at_cursor (text_buffer, str->str,
							  str->len);
		}
		g_io_channel_shutdown (channel, FALSE, NULL);
		g_io_channel_unref (channel);
		g_string_free (str, TRUE);
		g_spawn_close_pid (pid);
	}
	else
	{
		gtk_text_buffer_set_text (text_buffer, error->message, -1);
	}

	g_strfreev (args);
}

static gint cxp_right_pane_popup_handler (GtkWidget * widget, GdkEvent * event)
{
	GtkMenu *menu;
	GdkEventButton *event_button;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_MENU (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	/* The "widget" is the menu that was supplied when 
	 * g_signal_connect_swapped() was called.
	 */
	menu = GTK_MENU (widget);

	if (event->type == GDK_BUTTON_PRESS)
	{
		event_button = (GdkEventButton *) event;
		if (event_button->button == 3)
		{
			gtk_menu_popup (menu, NULL, NULL, NULL, NULL,
					event_button->button,
					event_button->time);
			return TRUE;
		}
	}

	return FALSE;
}

static void cxp_right_pane_preview_expanded (GObject * object,
					     GParamSpec * param_spec,
					     gpointer user_data)
{
	CxpRightPane *self = CXP_RIGHT_PANE (user_data);
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (self);
	GtkTextBuffer *buffer;
	gchar *filename;

	if (gtk_expander_get_expanded (GTK_EXPANDER (priv->expander)))
	{
		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW
						   (priv->preview));
		if ((filename =
		     cxp_right_pane_get_active_file_name (self)) != NULL)
		{
			cxp_right_pane_show_preview (self, buffer, filename);
			g_free (filename);
		}
		else
		{
			gtk_text_buffer_set_text (buffer, "", -1);
		}
	}
}

/**
 * \if japanese
 * ե򤽤줾롣
 * \endif
 */
static void cxp_right_pane_delete_selected_file_foreach (GtkTreeModel * model,
							 GtkTreePath * path,
							 GtkTreeIter * iter,
							 gpointer user_data)
{
	GList **rowref_list = (GList **) user_data;
	GtkTreeRowReference *rowref;

	g_assert (rowref_list != NULL);

	rowref = gtk_tree_row_reference_new (model, path);
	*rowref_list = g_list_append (*rowref_list, rowref);
}

/*
 * API
 */

GtkWidget *cxp_right_pane_new (GConfClient * client)
{
	return GTK_WIDGET (g_object_new
			   (CXP_TYPE_RIGHT_PANE, "gconf-client", client, NULL));
}

/**
 * This function show file list in directory to File List View.
 * @param cxplorer [in] Pointer to instance of Cxplorer.
 */
void cxp_right_pane_change_directory (CxpRightPane * self,
				      const gchar * dirname)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (self);
	const gchar *filename;
	gchar *cur_dir;
	gchar *fullpath;
	GDir *dir;
	GtkTreeView *file_list;
	GtkTreeModel *model;
	GtkTextBuffer *text_buffer;
	gboolean show_dot_file;

	cur_dir = g_strdup (dirname);
	show_dot_file =
		gconf_client_get_bool (priv->client,
				       "/apps/cxp/cxplorer/ShowDotFile", NULL);

	text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->preview));
	gtk_text_buffer_set_text (text_buffer, "", -1);

	file_list = GTK_TREE_VIEW (priv->file_list);
	model = gtk_tree_view_get_model (file_list);
	g_object_ref (model);
	gtk_tree_view_set_model (file_list, NULL);
	cxp_file_list_store_clear (CXP_FILE_LIST_STORE (model));
	dir = g_dir_open (dirname, 0, NULL);
	if (dir)
	{
		while ((filename = g_dir_read_name (dir)) != NULL)
		{
			if ((filename[0] == '.') && (show_dot_file == FALSE))
			{
				continue;
			}
			fullpath = g_build_filename (dirname, filename, NULL);
			cxp_right_pane_append_file (self,
						    CXP_FILE_LIST_STORE (model),
						    fullpath);
			g_free (fullpath);
		}
		g_dir_close (dir);
	}
	cxp_file_list_store_resort (CXP_FILE_LIST_STORE(model));

	gtk_tree_view_set_model (file_list, model);
	g_object_unref (model);

	g_free (priv->cur_dir);
	priv->cur_dir = cur_dir;
}

gchar *cxp_right_pane_get_active_file_name (CxpRightPane * right_pane)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);
	gchar *fullpath;
	GtkTreePath *treepath;
	GtkTreeView *treeview;
	GtkTreeViewColumn *column;
	GtkTreeModel *model;
	GtkTreeIter iter;

	treeview = GTK_TREE_VIEW (priv->file_list);
	gtk_tree_view_get_cursor (treeview, &treepath, &column);
	if (treepath)
	{
		model = gtk_tree_view_get_model (treeview);
		if (gtk_tree_model_get_iter (model, &iter, treepath))
		{
			gtk_tree_model_get (model, &iter,
					    FILE_LIST_COL_FULLPATH, &fullpath,
					    -1);

			return fullpath;
		}
	}

	return NULL;
}

void cxp_right_pane_make_file (CxpRightPane * right_pane,
			       const gchar * base_name)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);
	gchar *fullpath;
	gchar *basename_locale;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *treepath;
	GtkWidget *dialog;
	GError *error = NULL;

	if (strlen (base_name) <= 0)
		return;

	basename_locale =
		g_locale_from_utf8 (base_name, -1, NULL, NULL, &error);
	if (error == NULL)
	{
		fullpath =
			g_build_filename (priv->cur_dir, basename_locale, NULL);
		g_free (basename_locale);
		if (mknod (fullpath, S_IFREG | 0666, 0) == 0)
		{
			model = gtk_tree_view_get_model (GTK_TREE_VIEW
							 (priv->file_list));
			iter = cxp_right_pane_append_file (right_pane,
							   CXP_FILE_LIST_STORE
							   (model), fullpath);
			treepath = gtk_tree_model_get_path (model, &iter);
			gtk_tree_view_set_cursor (GTK_TREE_VIEW
						  (priv->
						   file_list), treepath, NULL,
						  FALSE);
			gtk_tree_path_free (treepath);
		}
		else
		{
			dialog = gtk_message_dialog_new (NULL,
							 GTK_DIALOG_DESTROY_WITH_PARENT,
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_CLOSE,
							 "%s",
							 g_strerror (errno));
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
		}
		g_free (fullpath);
	}
}

/**
 * \if japanese
 * ե
 * \endif
 */
void cxp_right_pane_delete_selected_files (CxpRightPane * right_pane)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);
	GList *rr_list = NULL;	/* list of GtkTreeRowReferences to remove */
	GList *node;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeSelection *selection;
	gchar *fullpath;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (priv->file_list));
	selection =
		gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->file_list));
	gtk_tree_selection_selected_foreach (selection,
					     cxp_right_pane_delete_selected_file_foreach,
					     &rr_list);

	for (node = rr_list; node != NULL; node = node->next)
	{
		path = gtk_tree_row_reference_get_path ((GtkTreeRowReference
							 *) (node->data));
		if (path)
		{
			if (gtk_tree_model_get_iter (model, &iter, path))
			{
				gtk_tree_model_get (model, &iter,
						    FILE_LIST_COL_FULLPATH,
						    &fullpath, -1);
				cxp_remove_file (fullpath);
				if (g_file_test(fullpath, G_FILE_TEST_EXISTS) == FALSE)
				{
					cxp_file_list_store_remove
						(CXP_FILE_LIST_STORE (model),
						 &iter);
				}
				g_free (fullpath);
			}
		}
	}
}

void cxp_right_pane_copy_to_clipboard (CxpRightPane *right_pane)
{
	GtkClipboard *clipboard;
	gchar *copy_files;

	clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);

	copy_files = cxp_right_pane_selection_to_string (right_pane);
	gtk_clipboard_set_with_data (clipboard, cxp_target_types, G_N_ELEMENTS(cxp_target_types), cxp_right_pane_clipboard_get, NULL, copy_files);
}

void cxp_right_pane_paste_from_clipboard (CxpRightPane * right_pane)
{
	GtkClipboard *clipboard;

	clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
	gtk_clipboard_request_contents (clipboard, GDK_TARGET_STRING, cxp_right_pane_clipboard_received, right_pane);
}

void cxp_right_pane_rename_file_request (CxpRightPane * right_pane)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *treepath;
	GtkTreeSelection *selection;
	GtkTreeViewColumn *column;

	selection =
		gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->file_list));
	if (gtk_tree_selection_count_selected_rows (selection) == 1)
	{
		gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->file_list),
					  &treepath, NULL);
		column = gtk_tree_view_get_column (GTK_TREE_VIEW (priv->file_list), 0);	/* FIXME: */
		model = gtk_tree_view_get_model (GTK_TREE_VIEW
						 (priv->file_list));
		gtk_tree_model_get_iter (model, &iter, treepath);
		cxp_file_list_store_set_editable (CXP_FILE_LIST_STORE (model),
						  &iter, TRUE);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->file_list),
					  treepath, column, TRUE);
		gtk_tree_path_free (treepath);
	}
}

void cxp_right_pane_send_file_to_command (CxpRightPane * right_pane,
					  const gchar * cmd)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);
	GError *gerror = NULL;
	GList *list, *cur;
	GString *fullcmd;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkWidget *dialog;
	gchar *fullpath = NULL;

	g_return_if_fail (cmd != NULL);

	selection =
		gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->file_list));
	list = gtk_tree_selection_get_selected_rows (selection, &model);

	g_return_if_fail (list != NULL);

	fullcmd = g_string_new (cmd);
	cur = list;
	while (cur != NULL)
	{
		gtk_tree_model_get_iter (model, &iter, cur->data);
		gtk_tree_model_get (model, &iter, FILE_LIST_COL_FULLPATH,
				    &fullpath, -1);
		g_string_append_printf (fullcmd, " '%s'", fullpath);
		g_free (fullpath);
		cur = g_list_next (cur);
	}
	g_list_foreach (list, (GFunc)gtk_tree_path_free, NULL);
	g_list_free (list);

	if (g_spawn_command_line_async (fullcmd->str, &gerror) == FALSE)
	{
		dialog = gtk_message_dialog_new (NULL,
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_CLOSE,
						 "%s", gerror->message);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
	}

	g_string_free (fullcmd, TRUE);
}

void cxp_right_pane_set_cmd_request (CxpRightPane *right_pane)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *treepath;
	GtkTreeSelection *selection;
	GtkWidget *dialog;
	gchar *fullpath;
	gchar *mime_type;
	gchar *linkend;
	mode_t mode;

	selection =
		gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->file_list));
	if (gtk_tree_selection_count_selected_rows (selection) == 1)
	{
		gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->file_list),
					  &treepath, NULL);
		model = gtk_tree_view_get_model (GTK_TREE_VIEW
						 (priv->file_list));
		gtk_tree_model_get_iter (model, &iter, treepath);
		gtk_tree_model_get (model, &iter, FILE_LIST_COL_FULLPATH, &fullpath, FILE_LIST_COL_FILEMODE, &mode, -1);
		switch(mode  & S_IFMT)
		{
		case S_IFLNK:
			linkend = cxp_path_get_link_end (fullpath);
			mime_type = cxp_get_mime_type_for_file (linkend);
			break;
		case S_IFREG:
			mime_type = cxp_get_mime_type_for_file (fullpath);
			break;
		default:
			return;
		}
		g_free (fullpath);

		dialog = cxp_mime_cmd_dialog_new (mime_type);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

		gtk_tree_path_free (treepath);
	}
}

void cxp_right_pane_set_popup_menu (CxpRightPane * right_pane,
				    GtkWidget * popup_menu)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (right_pane);

	priv->popup_menu = popup_menu;
	g_signal_connect_swapped (priv->file_list, "button_press_event",
				  G_CALLBACK (cxp_right_pane_popup_handler),
				  priv->popup_menu);
}

/**
 * convert GtkTreeSelection to string
 */
static gchar *cxp_right_pane_selection_to_string (CxpRightPane *obj)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (obj);
	GList *list, *cur;
	GString *select_files;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	gchar *fullpath;
	gchar *result;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->file_list));

	list = gtk_tree_selection_get_selected_rows (selection, &model);

	g_return_val_if_fail (list != NULL, "");

	select_files = g_string_new ("");
	cur = list;
	while (cur != NULL)
	{
		gtk_tree_model_get_iter (model, &iter, cur->data);
		gtk_tree_model_get (model, &iter, FILE_LIST_COL_FULLPATH, &fullpath, -1);
		g_string_append (select_files, fullpath);
		g_string_append_c (select_files, '\n');
		g_free (fullpath);
		cur = g_list_next (cur);
	}
	g_list_foreach (list, (GFunc)gtk_tree_path_free, NULL);
	g_list_free (list);

	result = g_string_free (select_files, FALSE);

	return	result;
}

/*
 * Virtually sets the contents of the specified
 */
static void cxp_right_pane_clipboard_get (GtkClipboard *clipboard, GtkSelectionData *selection_data, guint info, gpointer data)
{
	gtk_selection_data_set (selection_data, GDK_TARGET_STRING, 8, data, strlen(data)); 
}

static void cxp_right_pane_clipboard_received (GtkClipboard *clipboard, GtkSelectionData *selection_data, gpointer data)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (data);
	GtkWidget *dialog;
	gchar *cmd;
	gchar *cur_dir;
	gchar *copy_files;
	gchar **copy_file;
	gchar *serr;
	gchar *error_msg;
	guint index;
	gint exit_status;

	copy_files = selection_data->data;
	if (copy_files == NULL)
	{
		g_warning ("faild get text from clipboard\n");
		return;
	}
	copy_file = g_strsplit (copy_files, "\n", 0);
	for(index=0; copy_file[index] != NULL; index++)
	{
		cmd = NULL;
		if (g_file_test (copy_file[index], G_FILE_TEST_IS_REGULAR))
		{
			cmd = g_strdup_printf ("cp --backup=t '%s' '%s'", copy_file[index],
					priv->cur_dir);
		}
		else if (g_file_test (copy_file[index], G_FILE_TEST_IS_DIR))
		{
			cmd = g_strdup_printf ("cp -R --backup=t '%s' '%s'", copy_file[index],
					priv->cur_dir);
		}

		if(cmd != NULL)
		{
			g_spawn_command_line_sync (cmd, NULL, &serr,
					&exit_status, NULL);
			if (exit_status != 0)
			{
				error_msg = g_locale_to_utf8 (serr, -1, NULL, NULL, NULL);
				dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "%s", error_msg);
				gtk_dialog_run (GTK_DIALOG (dialog));
				gtk_widget_destroy (dialog);
				g_free (error_msg);
			}
			g_free (cmd);
		}
	}
	cur_dir = g_strdup (priv->cur_dir);
	cxp_right_pane_change_directory (CXP_RIGHT_PANE(data), cur_dir);
	g_free (cur_dir);
	g_strfreev (copy_file);
}

static void cxp_right_pane_drag_data_get(GtkWidget *widget, GdkDragContext *dc, GtkSelectionData *selection_data, guint info, guint t, gpointer data)
{
	gchar *move_files;

	move_files = cxp_right_pane_selection_to_string (CXP_RIGHT_PANE(data));
	gtk_selection_data_set (selection_data, GDK_TARGET_STRING, 8, move_files, strlen(move_files)); 
}

static void cxp_right_pane_drag_data_delete (GtkWidget *widget, GdkDragContext *dc, gpointer data)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (data);
	GList *rr_list = NULL;	/* list of GtkTreeRowReferences to remove */
	GList *node;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeSelection *selection;
	gchar *fullpath;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (priv->file_list));
	selection =
		gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->file_list));
	gtk_tree_selection_selected_foreach (selection,
					     cxp_right_pane_delete_selected_file_foreach,
					     &rr_list);

	for (node = rr_list; node != NULL; node = node->next)
	{
		path = gtk_tree_row_reference_get_path ((GtkTreeRowReference
							 *) (node->data));
		if (path)
		{
			if (gtk_tree_model_get_iter (model, &iter, path))
			{
				gtk_tree_model_get (model, &iter,
						    FILE_LIST_COL_FULLPATH,
						    &fullpath, -1);
				if (g_file_test(fullpath, G_FILE_TEST_EXISTS) == FALSE)
				{
					cxp_file_list_store_remove
						(CXP_FILE_LIST_STORE (model),
						 &iter);
				}
				g_free (fullpath);
			}
		}
	}

	g_signal_stop_emission_by_name (priv->file_list, "drag_data_delete");
}

static void cxp_right_pane_drag_data_received (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, GtkSelectionData *selection_data, guint info, guint t, gpointer data)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (data);
	gchar *move_files;
	gchar **move_file;
	gchar *src_base;
	gchar *dest_dir;
	gchar *dest_path;
	guint index;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter iter;

	dest_dir = cxp_right_pane_get_drag_dest_dir (CXP_RIGHT_PANE(data), x, y);
	if ((strcmp(dest_dir, priv->cur_dir) == 0) && (gtk_drag_get_source_widget (dc) != NULL))
	{
		gtk_drag_finish (dc, FALSE, FALSE, t);
	}
	else
	{
		model = gtk_tree_view_get_model (GTK_TREE_VIEW(priv->file_list));
		move_files = selection_data->data;
		move_file = g_strsplit (move_files, "\n", 0);
		for(index=0; move_file[index] != NULL; index++)
		{
			if(g_file_test(move_file[index], G_FILE_TEST_EXISTS))
			{
				src_base = g_path_get_basename (move_file[index]);
				dest_path = g_build_filename (dest_dir, src_base, NULL);
				errno = 0;
				if(rename(move_file[index], dest_path) == -1)
				{
					cxp_error_dialog_run_about_file (src_base);
				}
				else if(strcmp(dest_dir, priv->cur_dir) == 0)
				{
					cxp_file_list_store_append_file (CXP_FILE_LIST_STORE(model), dest_path, &iter);
					path = gtk_tree_model_get_path (model, &iter);
					gtk_tree_view_set_cursor (GTK_TREE_VIEW(priv->file_list), path, NULL, FALSE);
					gtk_tree_path_free (path);
				}
				g_free (dest_path);
				g_free (src_base);
			}
		}
		g_strfreev (move_file);

		gtk_drag_finish (dc, TRUE, TRUE, t);
	}
	g_free (dest_dir);
	g_signal_stop_emission_by_name (priv->file_list, "drag_data_received");
}

static gboolean cxp_right_pane_drag_drop (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, guint t, gpointer data)
{
	gtk_drag_get_data (widget, dc, GDK_TARGET_STRING, t);

	return	TRUE;
}

static gchar *cxp_right_pane_get_drag_dest_dir (CxpRightPane *obj, gint x, gint y)
{
	CxpRightPanePrivate *priv = CXP_RIGHT_PANE_GET_PRIVATE (obj);
	gchar *dest_dir;
	GtkTreeViewDropPosition pos;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter iter;
	gchar *result;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW(priv->file_list));
	if(gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW(priv->file_list), x, y, &path, &pos))
	{
		switch (pos)
		{
			case GTK_TREE_VIEW_DROP_BEFORE:
			case GTK_TREE_VIEW_DROP_AFTER:
				result = g_strdup (priv->cur_dir);
				break;
			default:
				if (gtk_tree_model_get_iter(model, &iter, path))
				{
					gtk_tree_model_get (model, &iter, FILE_LIST_COL_FULLPATH, &dest_dir, -1);
					if(g_file_test(dest_dir, G_FILE_TEST_IS_DIR))
					{
						result = dest_dir;
					}
					else
					{
						result = g_strdup (priv->cur_dir);
					}
				}
				else
				{
					result = g_strdup (priv->cur_dir);
				}
		}
	}
	else
	{
			result = g_strdup (priv->cur_dir);
	}

	return	result;
}
