/* $Id: cxp-dir-view.c,v 1.28 2005/09/26 13:03:04 yasumichi Exp $ */
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gconf/gconf-client.h>
#include <glib/gi18n.h>
#include <string.h>
#include <cxp.h>
#include <errno.h>
#include "cxp-dir-view.h"

/*
 * define column number of directory view.
 */
enum
{
	COL_TERMINATOR = -1,	/**< terminator */
	COL_DISP_NAME,		/**< display name */
	COL_FULL_PATH,		/**< full path of directory */
	COL_IS_DUMMY,		/**< Is node dummy? */
	NUM_COLS		/**< Count of columns */
};

typedef struct
{
	GConfClient *client;
	GtkWidget *dirview;
	gboolean dispose_has_run;
} CxpDirViewPrivate;

#define CXP_DIR_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXP_TYPE_DIR_VIEW, CxpDirViewPrivate))

enum
{
	CXP_DIR_VIEW_GCONF_CLIENT = 1,
};

/**
 * This enumeration define signal.
 */
enum
{
	DIRECTORY_CHANGED_SIGNAL,
	LAST_SIGNAL
};

enum
{
	TARGET_CXP_FILE_INFO,
	TARGET_STRING
};

static GtkTargetEntry cxp_target_types[] = {
	{"_CXP_FILE_INFO", 0, TARGET_CXP_FILE_INFO},
	{"STRING", 0, TARGET_STRING}
};

static GObjectClass *parent_class = NULL;
static guint cxp_dir_view_signals[LAST_SIGNAL] = { 0 };

/*
 * Prototype declaration of private method
 */
static void cxp_dir_view_init (GTypeInstance * instance, gpointer g_class);
static void cxp_dir_view_class_init (CxpDirViewClass * klass);
static void cxp_dir_view_dispose (GObject * obj);
static void cxp_dir_view_finalize (GObject * object);
static void cxp_dir_view_set_property (GObject * object,
					 guint property_id,
					 const GValue * value,
					 GParamSpec * pspec);
static void cxp_dir_view_get_property (GObject * object,
					 guint property_id,
					 GValue * value, GParamSpec * pspec);
static GtkWidget *cxp_dir_view_tree_view_new (CxpDirView * self);
static GtkTreeModel *cxp_dir_view_model_new (void);
static gint cxp_dir_view_compare_node (GtkTreeModel * model,
				       GtkTreeIter * iterA,
				       GtkTreeIter * iterB, gpointer data);
static void cxp_dir_view_prepend_dummy (GtkTreeStore * store,
					GtkTreeIter * parent_iter);
static void cxp_dir_view_set_child_dirctory (CxpDirView *self, 
		GtkTreeStore * treestore,
		GtkTreeIter * parent,
		gchar * parentPath);
static void cxp_dir_view_on_row_expanded (GtkTreeView * treeview,
					  GtkTreeIter * iter,
					  GtkTreePath * path,
					  gpointer user_data);
static void cxp_dir_view_on_cursor_changed (GtkTreeView * treeview,
					    gpointer user_data);
static void cxp_dir_view_draw_icon_cell (GtkTreeViewColumn * tree_column,
					 GtkCellRenderer * cell,
					 GtkTreeModel * tree_model,
					 GtkTreeIter * iter, gpointer data);
static GtkTreeIter *cxp_dir_view_search_child_node (GtkTreeModel * model,
						    GtkTreeIter * parent,
						    gchar * label);
void cxp_dir_view_drag_data_received (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, GtkSelectionData *selection_data, guint info, guint t, gpointer data);

/**
 * cxp_dir_view_get_type
 *
 * Register CxpDirViewType.
 **/
GType cxp_dir_view_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		/* ֥Ȥη˴ؤޤ */
		static const GTypeInfo info = {
			sizeof (CxpDirViewClass),
			NULL,	/* base_init (饹νԤؿ) ̵ */
			NULL,	/* base_finalize (饹νλԤؿ) ̵ */
			(GClassInitFunc) cxp_dir_view_class_init,
			NULL,	/* class_finalize (饹νλԤؿ) ̵ */
			NULL,	/* class_data (饹Υǡ) ̵ */
			sizeof (CxpDirView),
			0,	/* n_preallocs (󥹥󥹤ݤΥݥꥷ) ̵ */
			(GInstanceInitFunc) cxp_dir_view_init
		};
		type = g_type_register_static (GTK_TYPE_SCROLLED_WINDOW,
					       "CxpDirViewType", &info, 0);
	}
	return type;
}

/**
 * cxp_dir_view_init
 * @instance:
 * @g_class:
 *
 * Initialize instance of CxpDirView.
 **/
static void cxp_dir_view_init (GTypeInstance * instance, gpointer g_class)
{
	CxpDirView *self = (CxpDirView *) instance;
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (self);

	gtk_scrolled_window_set_hadjustment (GTK_SCROLLED_WINDOW (self), NULL);
	gtk_scrolled_window_set_vadjustment (GTK_SCROLLED_WINDOW (self), NULL);
	gtk_widget_set_size_request (GTK_WIDGET(self), 200, -1);
	priv->dirview = cxp_dir_view_tree_view_new (self);
	gtk_container_add (GTK_CONTAINER (self), priv->dirview);
	priv->dispose_has_run = FALSE;
	//gtk_drag_dest_set (priv->dirview, GTK_DEST_DEFAULT_ALL, cxp_target_types, G_N_ELEMENTS(cxp_target_types), GDK_ACTION_MOVE);
	gtk_tree_view_enable_model_drag_dest(GTK_TREE_VIEW(priv->dirview), cxp_target_types, G_N_ELEMENTS(cxp_target_types), GDK_ACTION_MOVE);

	/* signal connect */
	g_signal_connect (priv->dirview, "row_expanded",
			  G_CALLBACK (cxp_dir_view_on_row_expanded), self);
	g_signal_connect (priv->dirview, "cursor_changed",
			  G_CALLBACK (cxp_dir_view_on_cursor_changed), self);
	g_signal_connect (priv->dirview, "drag_data_received",
			  G_CALLBACK (cxp_dir_view_drag_data_received), self);
}

/**
 * cxp_dir_view_class_init
 * @klass:
 *
 * Initialize CxpDirViewClass.
 **/
static void cxp_dir_view_class_init (CxpDirViewClass * klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	GParamSpec *pspec;

	/* ᥽å dispose()/finalize() 򥪡С饤ɤޤ */
	gobject_class->dispose = cxp_dir_view_dispose;
	gobject_class->finalize = cxp_dir_view_finalize;
	gobject_class->set_property = cxp_dir_view_set_property;
	gobject_class->get_property = cxp_dir_view_get_property;

	/* install properties. */
	pspec = g_param_spec_object ("gconf_client",
				     "GConfClient",
				     "Set GConfClient",
				     GCONF_TYPE_CLIENT,
				     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 CXP_DIR_VIEW_GCONF_CLIENT, pspec);

	cxp_dir_view_signals[DIRECTORY_CHANGED_SIGNAL] =
		g_signal_new ("directory_changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
			      G_STRUCT_OFFSET (CxpDirViewClass,
					       directory_changed), NULL,
			      NULL, g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	g_type_class_add_private (klass, sizeof (CxpDirViewPrivate));

	parent_class = g_type_class_peek_parent (klass);
}

/**
 * cxp_dir_view_dispose
 * @obj: 
 *
 * Dispose instance of CxpDirView.
 **/
static void cxp_dir_view_dispose (GObject * obj)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (obj);

	if (priv->dispose_has_run)
	{
		/* When dispose has run, return. */
		return;
	}
	/* ᥽å dispose() ټ¹Ԥ뤳Ȥʤ褦åޤ */
	priv->dispose_has_run = TRUE;

	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

/**
 * cxp_dir_view_finalize
 * @object:
 *
 * Finalize instance of CxpDirView.
 **/
static void cxp_dir_view_finalize (GObject * object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void cxp_dir_view_set_property (GObject * object,
					 guint property_id,
					 const GValue * value,
					 GParamSpec * pspec)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (object);

	switch (property_id)
	{
	case CXP_DIR_VIEW_GCONF_CLIENT:
		priv->client = GCONF_CLIENT(g_value_dup_object (value));
		break;
	default:
		/* We don't have any other property... */
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static void cxp_dir_view_get_property (GObject * object,
					 guint property_id,
					 GValue * value, GParamSpec * pspec)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (object);

	switch (property_id)
	{
	case CXP_DIR_VIEW_GCONF_CLIENT:
		g_value_set_object (value, priv->client);
		break;
	default:
		/* We don't have any other property... */
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static GtkWidget *cxp_dir_view_tree_view_new (CxpDirView * self)
{
	GtkWidget *treeview;
	GtkTreeModel *model;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	model = cxp_dir_view_model_new ();
	treeview = gtk_tree_view_new_with_model (model);
	gtk_widget_show (treeview);

	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Directory"));
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column, renderer,
						 cxp_dir_view_draw_icon_cell,
						 self, NULL);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer, "text",
					     COL_DISP_NAME, NULL);

	gtk_tree_view_set_enable_search (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_search_column (GTK_TREE_VIEW (treeview), 0);

	return treeview;
}

/**
 * cxp_dir_view_model_new
 * 
 * Creates a new tree store for CxpDirView.
 **/
static GtkTreeModel *cxp_dir_view_model_new (void)
{
	GtkTreeStore *store;
	GtkTreeIter root;
	GtkTreeSortable *sortable;

	store = gtk_tree_store_new (NUM_COLS, G_TYPE_STRING, G_TYPE_STRING,
				    G_TYPE_BOOLEAN);

	sortable = GTK_TREE_SORTABLE (store);
	gtk_tree_sortable_set_default_sort_func (sortable,
						 cxp_dir_view_compare_node,
						 NULL, NULL);
	gtk_tree_sortable_set_sort_column_id (sortable, COL_DISP_NAME,
					      GTK_SORT_ASCENDING);

	gtk_tree_store_prepend (store, &root, NULL);
	gtk_tree_store_set (store, &root, COL_DISP_NAME, "/",
			    COL_FULL_PATH, "/", COL_IS_DUMMY, FALSE, -1);
	cxp_dir_view_prepend_dummy (store, &root);

	return GTK_TREE_MODEL (store);
}

/**
 * (GtkTreeIterCompareFunc)
 *
 */
static gint cxp_dir_view_compare_node (GtkTreeModel * model,
				       GtkTreeIter * iterA,
				       GtkTreeIter * iterB, gpointer data)
{
	gint retval = 0;

	gchar *nameA, *nameB;

	gtk_tree_model_get (model, iterA, COL_DISP_NAME, &nameA,
			    COL_TERMINATOR);
	gtk_tree_model_get (model, iterB, COL_DISP_NAME, &nameB,
			    COL_TERMINATOR);

	if (nameA == NULL || nameB == NULL)
	{
		if (nameA == NULL && nameB == NULL)
		{
			retval = 0;
		}
		else
		{
			retval = (nameA == NULL) ? -1 : 1;
		}
	}
	else
	{
		retval = g_utf8_collate (nameA, nameB);
	}

	g_free (nameA);
	g_free (nameB);

	return retval;
}

/**
 *
 *
 **/
static void cxp_dir_view_prepend_dummy (GtkTreeStore * store,
					GtkTreeIter * parent_iter)
{
	GtkTreeIter iter;
	gboolean success;

	g_return_if_fail (store);

	gtk_tree_store_prepend (store, &iter, parent_iter);
	gtk_tree_store_set (store, &iter, COL_DISP_NAME, "(null)", COL_IS_DUMMY,
			    TRUE, -1);

	success = gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
						 &iter, parent_iter, 1);
	if (!success)
		return;

	while (gtk_tree_store_is_ancestor (store, parent_iter, &iter))
	{
		gtk_tree_store_remove (store, &iter);
	}
}

static void cxp_dir_view_set_child_dirctory (CxpDirView *self,
					     GtkTreeStore * treestore,
					     GtkTreeIter * parent,
					     gchar * parentPath)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (self);
	const gchar *childName;
	gchar *fullpath;
	gchar *dispname;
	GDir *gdir;
	GtkTreeIter child;
	gboolean showDotFile;

	showDotFile = gconf_client_get_bool (priv->client, "/apps/cxp/cxplorer/ShowDotFile", NULL);

	/* open directory */
	gdir = g_dir_open (parentPath, 0, NULL);
	g_return_if_fail (gdir);

	while ((childName = g_dir_read_name (gdir)) != NULL)
	{
		/* ToDo: when ShowDotFile is turned off, reconstruct tree.
		   And Cxplorer is referenced. */
		if ((childName[0] == '.') && (showDotFile == FALSE))
		{
			continue;
		}
		fullpath = g_build_filename (parentPath, childName, NULL);
		/* Child node is added if it is directory */
		if (g_file_test (fullpath, G_FILE_TEST_IS_DIR))
		{
			if((dispname = cxp_path_get_basename_of_utf8 (fullpath)) == NULL)
			{
				dispname = g_strdup (_("(Invalid filename)"));
			}
			gtk_tree_store_prepend (treestore, &child, parent);
			gtk_tree_store_set (treestore, &child, COL_DISP_NAME,
					    dispname, COL_FULL_PATH, fullpath,
					    COL_IS_DUMMY, FALSE,
					    COL_TERMINATOR);
			cxp_dir_view_prepend_dummy (treestore, &child);
		}
		g_free (fullpath);
	}

	/* Close directory */
	g_dir_close (gdir);
}

static void cxp_dir_view_draw_icon_cell (GtkTreeViewColumn * tree_column,
					 GtkCellRenderer * cell,
					 GtkTreeModel * tree_model,
					 GtkTreeIter * iter, gpointer data)
{
	GdkPixbuf *icon = cxp_lookup_icon_from_mime("inode/directory");

	g_object_set (cell, "pixbuf", icon, NULL);
}

/*
 * private signal handler.
 */

/**
 * \if japanese
 * ǥ쥯ȥŸΥХå
 * @param treeview ʥȯԤGtkTreeView
 * @param iter Ÿ줿Ρɤؤƥ졼
 * @param path Ÿ줿Ρɤؤѥ
 * @param user_data 桼ΥǡΥץǤϡCxpDirViewؤ
 * ݥ󥿤ϤƤ롣
 * \endif
 * \if english
 * Callback when directory is expanded
 * @param treeview GtkTreeView that emitted signal.
 * @param iter GtkTreeIter that indicates expanded node
 * @param path GtkTreePath that indicates expanded node
 * @param user_data Data of user definition. In this program, the pointer to
 * CxpDirView has been passed.
 * \endif
 */
static void cxp_dir_view_on_row_expanded (GtkTreeView * treeview,
					  GtkTreeIter * iter,
					  GtkTreePath * path,
					  gpointer user_data)
{
	GtkTreeStore *store;
	GtkTreeIter child;
	gchar *fullpath;
	gboolean isDummy;

	/* Ÿ줿ΡɤκǽλҥΡɤ */
	store = GTK_TREE_STORE (gtk_tree_view_get_model (treeview));
	gtk_tree_model_iter_children (GTK_TREE_MODEL (store), &child, iter);
	gtk_tree_model_get (GTK_TREE_MODEL (store), &child, COL_IS_DUMMY,
			    &isDummy, COL_TERMINATOR);

	if (isDummy)
	{
		/* ҥΡɤߡΡɤξ硢ºݤλҥǥ쥯ȥɲ */
		gtk_tree_model_get (GTK_TREE_MODEL (store), iter,
				    COL_FULL_PATH, &fullpath, COL_TERMINATOR);
		cxp_dir_view_set_child_dirctory (CXP_DIR_VIEW(user_data), store, iter, fullpath);
		gtk_tree_store_remove (store, &child);

		g_free (fullpath);
	}

}

/*
 * \if japanese
 * 뤬ѹ줿ΥХå
 * \endif
 * \if english
 * Callback when cursor is changed
 * \endif
 */
static void cxp_dir_view_on_cursor_changed (GtkTreeView * treeview,
					    gpointer user_data)
{
	CxpDirView *self = CXP_DIR_VIEW (user_data);

	/* ǥ쥯ȥ꤬ѹ줿Τ */
	g_signal_emit (self, cxp_dir_view_signals[DIRECTORY_CHANGED_SIGNAL], 0);
}

/**
 * \if japanese
 * label˰פҥΡɤõ
 * \endif
 */
static GtkTreeIter *cxp_dir_view_search_child_node (GtkTreeModel * model,
						    GtkTreeIter * parent,
						    gchar * label)
{
	gchar *cur_label;
	GtkTreeIter *current;

	current = g_new (GtkTreeIter, 1);
	if (gtk_tree_model_iter_children (model, current, parent))
	{
		do
		{
			gtk_tree_model_get (model, current,
					    COL_DISP_NAME, &cur_label, -1);
			if (strcmp (label, cur_label) == 0)
			{
				g_free (cur_label);
				return current;
			}
			g_free (cur_label);
		}
		while (gtk_tree_model_iter_next (model, current));
	}

	g_free (current);
	return NULL;
}

/*
 * public methods.
 *
 */

/**
 * \if japanese
 * CxpDirViewåȤ
 * \endif
 * \if english
 * Creates a new CxpDirView widget.
 * \endif
 */
GtkWidget *cxp_dir_view_new (GConfClient *client)
{
	CxpDirView *object;

	object = CXP_DIR_VIEW (g_object_new (CXP_TYPE_DIR_VIEW, "gconf-client", client, NULL));

	return GTK_WIDGET (object);
}

/**
 * \if japanese
 * ΥΡɤбǥ쥯ȥ̾
 * @return Υǥ쥯ȥ̾
 * \endif
 * \if english
 * The directory name corresponding to the node selecting it now is obtained.
 * @return Directory name selecting it now
 * \endif
 */
gchar *cxp_dir_view_get_current_directory (CxpDirView * instance)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (instance);
	gchar *fullpath = NULL;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	GtkTreeIter iter;
	GtkTreeModel *model;

	gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->dirview),
				  &path, &column);

	if (path)
	{
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (priv->dirview));
		if (gtk_tree_model_get_iter (model, &iter, path))
		{
			gtk_tree_model_get (model, &iter, COL_FULL_PATH,
					    &fullpath, -1);
		}

		gtk_tree_path_free (path);
	}

	return fullpath;
}

/**
 * \if japanese
 * ǥ쥯ȥfullpathбΡɤ򤹤
 * \endif
 * \if english
 * The node corresponding to directory fullpath is selected.
 * \endif
 */
gboolean cxp_dir_view_change_directory (CxpDirView * instance,
					const gchar * fullpath)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (instance);
	gchar **elements;
	gint index;
	GtkWidget *treeview;
	GtkTreeModel *model;
	GtkTreeIter *parent;
	GtkTreeIter *current;
	GtkTreePath *currentPath;

	if (g_path_is_absolute (fullpath) != TRUE)
	{
		g_warning ("%s isn't absolute path.\n", fullpath);
		return FALSE;
	}

	treeview = priv->dirview;

	parent = g_new (GtkTreeIter, 1);
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
	gtk_tree_model_get_iter_first (model, parent);
	currentPath = gtk_tree_model_get_path (model, parent);
	gtk_tree_view_expand_to_path (GTK_TREE_VIEW (treeview), currentPath);
	gtk_tree_path_free (currentPath);

	elements = g_strsplit (fullpath, "/", 0);
	for (index = 0; elements[index] != NULL; index++)
	{
		if (strlen (elements[index]) == 0)
			continue;
		if ((current = cxp_dir_view_search_child_node (model, parent,
							       elements[index]))
		    != NULL)
		{
			currentPath = gtk_tree_model_get_path (model, current);
			gtk_tree_view_expand_to_path (GTK_TREE_VIEW
						      (treeview), currentPath);
			gtk_tree_path_free (currentPath);
			g_free (parent);
			parent = current;
		}
		else
		{
			g_warning ("%s is not exists\n", fullpath);
			break;
		}
	}
	g_strfreev (elements);
	currentPath = gtk_tree_model_get_path (model, parent);
	gtk_tree_view_set_cursor (GTK_TREE_VIEW (treeview),
				  currentPath, NULL, FALSE);
	gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				      currentPath, NULL, FALSE, 0.0, 0.0);
	gtk_tree_path_free (currentPath);
	g_free (parent);

	return TRUE;
}

/**
 * \if japanese
 * ǥ쥯ȥĥ꡼ƹۤ
 * @param instance ƹۤCxpDirView
 * \endif
 * \if english
 * The directory tree is restructured.
 * @param instance CxpDirView that wants to restructure
 * \endif
 */
void cxp_dir_view_refresh (CxpDirView * instance)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (instance);
	gchar *currentDirectory;
	GtkWidget *dirview;
	GtkTreeIter root, current;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeViewColumn *column;

	dirview = priv->dirview;
	gtk_tree_view_get_cursor (GTK_TREE_VIEW (dirview), &path, &column);
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (dirview));
	gtk_tree_model_get_iter (model, &current, path);
	gtk_tree_model_get (model, &current, COL_FULL_PATH,
			    &currentDirectory, -1);

	gtk_tree_store_clear (GTK_TREE_STORE (model));
	gtk_tree_store_prepend (GTK_TREE_STORE (model), &root, NULL);
	gtk_tree_store_set (GTK_TREE_STORE (model), &root, COL_DISP_NAME, "/",
			    COL_FULL_PATH, "/", COL_IS_DUMMY, FALSE, -1);
	cxp_dir_view_set_child_dirctory (CXP_DIR_VIEW(instance), GTK_TREE_STORE (model), &root, "/");
	cxp_dir_view_change_directory (instance, currentDirectory);
	g_free (currentDirectory);
}

void cxp_dir_view_drag_data_received (GtkWidget *widget, GdkDragContext *dc, gint x, gint y, GtkSelectionData *selection_data, guint info, guint t, gpointer data)
{
	CxpDirViewPrivate *priv = CXP_DIR_VIEW_GET_PRIVATE (data);
	gchar *move_files;
	gchar **move_file;
	gchar *src_base;
	gchar *dest_dir;
	gchar *dest_path;
	guint index;
	GtkTreeViewDropPosition pos;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter iter;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW(priv->dirview));
	if(gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW(priv->dirview), x, y, &path, &pos))
	{
		if (gtk_tree_model_get_iter(model, &iter, path))
		{
			gtk_tree_model_get (model, &iter, COL_FULL_PATH, &dest_dir, -1);
			move_files = selection_data->data;
			move_file = g_strsplit (move_files, "\n", 0);
			for(index=0; move_file[index] != NULL; index++)
			{
				if(g_file_test(move_file[index], G_FILE_TEST_EXISTS))
				{
					src_base = g_path_get_basename (move_file[index]);
					dest_path = g_build_filename (dest_dir, src_base, NULL);
					errno = 0;
					if(rename(move_file[index], dest_path) == -1)
					{
						cxp_error_dialog_run_about_file (src_base);
					}
					g_free (dest_path);
					g_free (src_base);
				}
			}
			g_free (dest_dir);
			g_strfreev (move_file);
			gtk_drag_finish (dc, TRUE, TRUE, t);
			return;
		}
	}

	gtk_drag_finish (dc, FALSE, FALSE, t);
}
