/**
 * @file
 * @brief A widget of Property dialog
 *
 * A widget of Property dialog
 * @author Yasumichi Akahoshi <yasumichi@users.sourceforge.jp>
 * @date Sun Apr 3 02:53:00 2005
 * $Revision: 1.9 $
 ****************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <cxp.h>
#include <errno.h>
#include <glib/gi18n.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <grp.h>
#include <pwd.h>
#include "cxp-property-dialog.h"

enum
{
	INFO_FILENAME = 0,
	INFO_DIRECTORY,
	INFO_SIZE,
	INFO_FILETYPE,
	INFO_OWNER,
	INFO_GROUP,
	INFO_LAST_STATUS_CHANGE,
	INFO_LAST_MODIFICATION,
	INFO_LAST_ACCESS,
	INFO_COUNT,
};

/**
 * the private structure 
 */
typedef struct
{
	gchar *filename;
	GtkWidget *info_labels[INFO_COUNT];
	GtkWidget *check_boxs[9];
	gboolean dispose_has_run;
} CxpPropertyDialogPrivate;

#define CXP_PROPERTY_DIALOG_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXP_TYPE_PROPERTY_DIALOG, CxpPropertyDialogPrivate))

static GObjectClass *parent_class = NULL;

enum
{
	CXP_PROPERTY_DIALOG_FILENAME = 1,
};

static void cxp_property_dialog_class_init (CxpPropertyDialogClass *klass);
static void cxp_property_dialog_instance_init (GTypeInstance * instance,
					       gpointer g_class);
static void cxp_property_dialog_set_property (GObject * object,
					      guint property_id,
					      const GValue * value,
					      GParamSpec * pspec);
static void cxp_property_dialog_get_property (GObject * object,
					      guint property_id, GValue * value,
					      GParamSpec * pspec);
static void cxp_property_dialog_dispose (GObject * obj);
static void cxp_property_dialog_finalize (GObject * obj);
static void cxp_property_dialog_set_values (CxpPropertyDialog *self);
static void cxp_property_dialog_chmod (GtkButton *button, gpointer user_data);

static void cxp_property_dialog_class_init (CxpPropertyDialogClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	GParamSpec *pspec;

	gobject_class->set_property = cxp_property_dialog_set_property;
	gobject_class->get_property = cxp_property_dialog_get_property;
	gobject_class->dispose = cxp_property_dialog_dispose;
	gobject_class->finalize = cxp_property_dialog_finalize;

	parent_class = g_type_class_peek_parent (klass);

	pspec = g_param_spec_string ("filename",
			"CxpPropertyDialog construct prop",
			"Set filename",
			NULL,	
			G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
			CXP_PROPERTY_DIALOG_FILENAME,
			pspec);

	g_type_class_add_private (klass, sizeof (CxpPropertyDialogPrivate));
}

static void cxp_property_dialog_instance_init (GTypeInstance * instance,
					       gpointer g_class)
{
	CxpPropertyDialog *self = CXP_PROPERTY_DIALOG (instance);
	CxpPropertyDialogPrivate *private = CXP_PROPERTY_DIALOG_GET_PRIVATE(self);
	GtkWidget *close_button;
	GtkWidget *refresh_button;
	GtkWidget *table;
	GtkWidget *table2;
	GtkWidget *frame;
	GtkWidget *title_label;
	gchar *info_title[] = {
		N_("Filename:"),
		N_("Place:"),
		N_("Size:"),
		N_("Filetype:"),
		N_("Owner:"),
		N_("Group:"),
		N_("last status change:"),
		N_("last modification:"),
		N_("last access:")
	};
	const gchar *permit_type[] = {
		N_("read"),
		N_("write"),
		N_("execute")
	};
	const gchar *person_type[] = {
		N_("owner:"),
		N_("group:"),
		N_("other:")
	};
	gint index;
	gint col;
	gint row;

	private->filename = NULL;
	private->dispose_has_run = FALSE;

	gtk_window_set_destroy_with_parent(GTK_WINDOW(self), TRUE);
	table = gtk_table_new(2, INFO_COUNT, FALSE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(self)->vbox), table, FALSE, FALSE, 0);
	for(index=0; index<INFO_COUNT; index++)
	{
		title_label = gtk_label_new(_(info_title[index]));
		gtk_misc_set_alignment (GTK_MISC(title_label), 1, 0);
		gtk_table_attach (GTK_TABLE(table), title_label, 0, 1, index, index+1, GTK_FILL, 0, 2, 2);
		gtk_widget_show (title_label);

		private->info_labels[index] = gtk_label_new ("(null)");
		gtk_misc_set_alignment (GTK_MISC(private->info_labels[index]), 0, 0);
		gtk_table_attach (GTK_TABLE(table), private->info_labels[index], 1, 2, index, index+1, GTK_FILL, 0, 2, 2);
		gtk_widget_show (private->info_labels[index]);
	}
	gtk_widget_show (table);


	frame = gtk_frame_new (_("permissions"));
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(self)->vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show (frame);

	table2 = gtk_table_new (5, 3, FALSE);
	gtk_container_add (GTK_CONTAINER(frame), table2);
	for(row = 0; row < 3; row++)
	{
		title_label = gtk_label_new(_(person_type[row]));
		gtk_table_attach (GTK_TABLE(table2), title_label, 0, 1, row, row+1, GTK_FILL, 0, 2, 2);
		gtk_widget_show (title_label);
		for(col = 1; col < 4; col++)
		{
			index = col - 1 + row * 3;
			private->check_boxs[index] = gtk_check_button_new_with_label (_(permit_type[col-1]));
			gtk_table_attach (GTK_TABLE(table2), private->check_boxs[index], col, col+1, row, row+1, GTK_FILL, 0, 2, 2);
			gtk_widget_show (private->check_boxs[index]);
		}
	}
	refresh_button = gtk_button_new_from_stock(GTK_STOCK_REFRESH);
	gtk_table_attach (GTK_TABLE(table2), refresh_button, 4, 5, 2, 3, GTK_FILL, 0, 2, 2);
	gtk_widget_show (refresh_button);
	gtk_widget_show (table2);
	g_signal_connect (refresh_button, "clicked", G_CALLBACK(cxp_property_dialog_chmod), self);

	close_button = gtk_button_new_from_stock("gtk-close");
	gtk_dialog_add_action_widget (GTK_DIALOG (self), close_button, GTK_RESPONSE_CLOSE);
	gtk_widget_show (close_button);
}

static void cxp_property_dialog_set_property (GObject * object,
					      guint property_id,
					      const GValue * value,
					      GParamSpec * pspec)
{
	CxpPropertyDialog *self = CXP_PROPERTY_DIALOG (object);
	CxpPropertyDialogPrivate *private = CXP_PROPERTY_DIALOG_GET_PRIVATE(self);

	switch (property_id)
	{
	case CXP_PROPERTY_DIALOG_FILENAME:
		if (private->filename != NULL)
		{
			g_free (private->filename);
		}
		private->filename = g_value_dup_string (value);
		cxp_property_dialog_set_values (self);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static void cxp_property_dialog_get_property (GObject * object,
					      guint property_id, GValue * value,
					      GParamSpec * pspec)
{
	CxpPropertyDialogPrivate *private = CXP_PROPERTY_DIALOG_GET_PRIVATE(object);

	switch (property_id)
	{
	case CXP_PROPERTY_DIALOG_FILENAME:
		g_value_set_string (value, private->filename);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static void cxp_property_dialog_dispose (GObject * obj)
{
	CxpPropertyDialogPrivate *private = CXP_PROPERTY_DIALOG_GET_PRIVATE(obj);

	if (private->dispose_has_run)
	{
		/* If dispose did already run, return. */
		return;
	}
	/* Make sure dispose does not run twice. */
	private->dispose_has_run = TRUE;

	/* 
	 * In dispose, you are supposed to free all types referenced from this
	 * object which might themselves hold a reference to self. Generally,
	 * the most simple solution is to unref all members on which you own a 
	 * reference.
	 */

	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void cxp_property_dialog_finalize (GObject * obj)
{
	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

GType cxp_property_dialog_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpPropertyDialogClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			(GClassInitFunc) cxp_property_dialog_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpPropertyDialog),
			0,	/* n_preallocs */
			cxp_property_dialog_instance_init	/* instance_init */
		};
		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "CxpPropertyDialogType",
					       &info, 0);
	}
	return type;
}

/**
 * Creates a new dialog box which show detail of file.
 */
GtkWidget *cxp_property_dialog_new(const gchar *fullpath)
{
	CxpPropertyDialog *dialog;

	dialog = g_object_new(CXP_TYPE_PROPERTY_DIALOG, NULL);
	gtk_window_set_title(GTK_WINDOW(dialog), _("Property"));
	g_object_set(dialog, "filename", fullpath, NULL);

	return	GTK_WIDGET(dialog);
}

static void cxp_property_dialog_set_values (CxpPropertyDialog *self)
{
	CxpPropertyDialogPrivate *private = CXP_PROPERTY_DIALOG_GET_PRIVATE(self);
	gchar *filename = private->filename;
	gchar *mimetype;
	gchar *filetype;
	gchar buf[1024];
	GtkWidget **labels = private->info_labels;
	gchar *value;
	struct stat status;
	struct passwd *spasswd;
	struct group *sgroup;
	gint index;
	gint length;
	mode_t mask = S_IRUSR;

	g_return_if_fail (filename != NULL);

	lstat (filename, &status);

	value = cxp_path_get_basename_of_utf8(filename);
	if (value == NULL)
	{
		value = g_strdup(_("(Invalid filename)"));
	}
	gtk_label_set_text(GTK_LABEL(labels[INFO_FILENAME]), value);
	g_free (value);

	value = cxp_path_get_dirname_of_utf8(filename);
	gtk_label_set_text(GTK_LABEL(labels[INFO_DIRECTORY]), value);
	g_free (value);

	value = g_strdup_printf(_("%10ld bytes"), status.st_size);
	gtk_label_set_text(GTK_LABEL(labels[INFO_SIZE]), value);
	g_free (value);

	switch (status.st_mode & S_IFMT)
	{
		case S_IFLNK:
			if ((length = readlink(filename, buf, 1024)) > 0)
			{
				buf[length] = '\0';
				filetype = g_strdup_printf (_("symbolic link to %s"), buf);
			}
			else
			{
				filetype = g_strdup (_("symbolic link"));
			}
			break;
		case S_IFDIR:
			filetype = g_strdup (_("directory"));
			break;
		default:
			mimetype = cxp_get_mime_type_for_file (filename);
			filetype = cxp_get_mime_comment (mimetype);
			g_free (mimetype);
			break;
	}
	gtk_label_set_text(GTK_LABEL(labels[INFO_FILETYPE]), filetype);
	g_free (filetype);

	if((spasswd = getpwuid (status.st_uid)) != NULL)
	{
		value = g_strdup (spasswd->pw_name);
	}
	else
	{
		value = g_strdup_printf("(uid)%d", status.st_uid);
	}
	gtk_label_set_text(GTK_LABEL(labels[INFO_OWNER]), value);
	g_free (value);

	if((sgroup = getgrgid (status.st_gid)) != NULL)
	{
		value = g_strdup (sgroup->gr_name);
	}
	else
	{
		value = g_strdup_printf("(gid)%d", status.st_gid);
	}
	gtk_label_set_text(GTK_LABEL(labels[INFO_GROUP]), value);
	g_free (value);

	value = cxp_strftime (&status.st_ctime);
	gtk_label_set_text(GTK_LABEL(labels[INFO_LAST_STATUS_CHANGE]), value);
	g_free (value);

	value = cxp_strftime (&status.st_mtime);
	gtk_label_set_text(GTK_LABEL(labels[INFO_LAST_MODIFICATION]), value);
	g_free (value);

	value = cxp_strftime (&status.st_atime);
	gtk_label_set_text(GTK_LABEL(labels[INFO_LAST_ACCESS]), value);
	g_free (value);

	for(index=0; index<9; index++)
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(private->check_boxs[index]), status.st_mode & mask >> index);
	}
}

static void cxp_property_dialog_chmod (GtkButton *button, gpointer user_data)
{
	CxpPropertyDialogPrivate *private = CXP_PROPERTY_DIALOG_GET_PRIVATE(user_data);
	struct stat status;
	mode_t old_mode;
	mode_t new_mode;
	gint index;

	stat (private->filename, &status);

	old_mode = status.st_mode;
	new_mode = old_mode & 07000;

	for(index=0; index<9; index++)
	{
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(private->check_boxs[index])))
		{
			new_mode = new_mode | S_IRUSR >> index;
		}
	}

	errno = 0;
	if(chmod(private->filename, new_mode) == -1)
	{
		cxp_error_dialog_run_about_file ("change permissions of a file");
	}

	stat (private->filename, &status);
	for(index=0; index<9; index++)
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(private->check_boxs[index]), status.st_mode & S_IRUSR >> index);
	}
}
