/***************************************************************************
 *            cxplorer-window.c
 *
 *  Sun Feb 13 23:50:30 2005
 *  Copyright  2005  Yasumichi Akahoshi
 *  yasumichi@users.sourceforge.jp
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gconf/gconf-client.h>
#include <cxp.h>
#include <glib/gi18n.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include "cxp-dir-view.h"
#include "cxp-right-pane.h"
#include "cxp-property-dialog.h"
#include "cxp-pref-dialog.h"
#include "cxplorer-window.h"

/**
 * \if japanese
 * ToDo: ޥӽ
 * \endif
 */
#define CXP_GCONF_DIR	"/apps/cxp"
#define CXPLORER_GCONF_DIR	CXP_GCONF_DIR"/cxplorer"
#define BOOKMARK_GCONF_DIR	CXPLORER_GCONF_DIR"/bookmark"
#define SENDTO_GCONF_DIR	CXP_GCONF_DIR"/SendTo"

/* If you use Pimpls, include the private structure 
 * definition here. Some people create a cxplorer-property_dialog-private.h header
 * which is included by the cxplorer-property_dialog.c file and which contains the
 * definition for this private structure.
 */

typedef struct
{
	GConfClient *client;
	GtkActionGroup *action_group;
	GtkUIManager *ui_manager;
	GtkWidget *menubar;
	GtkWidget *toolbar;
	GtkWidget *scombo;
	GtkWidget *sentry;
	GHashTable *dynamic_widgets;
	GtkWidget *entry;
	GtkWidget *dirview;
	GtkWidget *right_pane;
	GtkWidget *statusbar;
	gboolean dispose_has_run;
} CxplorerWindowPrivate;

#define CXPLORER_WINDOW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXPLORER_TYPE_WINDOW, CxplorerWindowPrivate))

static GObjectClass *parent_class = NULL;


/*
 * Prototypes of private methods.
 */
static void cxplorer_window_class_init (gpointer g_class,
					gpointer g_class_data);
static void cxplorer_window_instance_init (GTypeInstance * instance,
					   gpointer g_class);
static void cxplorer_window_sendto_menu_init (CxplorerWindow *self);
static GSList* cxplorer_window_menu_item_new_from_sendto (const gchar *caption, const gchar *command, gpointer user_data);
static void cxplorer_window_bookmark_menu_init (CxplorerWindow *self);
static GSList* cxplorer_window_menu_item_new_from_bookmark (const gchar *key, const gchar *value, gpointer user_data);
static void cxplorer_window_dispose (GObject * obj);
static void cxplorer_window_finalize (GObject * obj);
static void cxplorer_window_entry_activate (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_search_entry_activate  (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_search_end (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_on_directory_changed (CxpDirView * dirview,
						  gpointer user_data);
static void cxplorer_window_on_dir_double_clicked (CxpRightPane * right_pane,
						   gpointer user_data);

static void cxplorer_window_new_win_action (GtkWidget *widget, gpointer user_data);
void cxplorer_window_new_file_action (GtkWidget * widget, gpointer user_data);
void cxplorer_window_new_directory_action (GtkWidget *widget, gpointer user_data);
void cxplorer_window_delete_action (GtkWidget * widget, gpointer user_data);
static void cxplorer_window_rename_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_send_mail_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_set_cmd_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_property_action (GtkWidget * widget, gpointer user_data);
static void cxplorer_window_copy_action (GtkWidget * widget, gpointer user_data);
static void cxplorer_window_paste_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_pref_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_show_dot_file_toggled (GtkToggleAction *action, gpointer user_data);
static void cxplorer_window_refresh_views_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_go_up_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_go_home_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_bookmark_add_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_bookmark_edit_action (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_about_action (GtkWidget *widget, gpointer user_data);

static void cxplorer_window_sendto_activate (GtkWidget *widget, gpointer user_data);
static void cxplorer_window_user_bookmark_activate (GtkWidget *widget, gpointer user_data);

static void cxplorer_window_gconf_value_changed (GConfClient* client, const gchar* key, GConfValue* value, gpointer user_data);

/**
 * Actions
 */
GtkActionEntry actions[] = {
	/* name, stock-id, label, accelerator, tooltip, callback */
	/* File menu */
	{"FileMenuAction", NULL, N_("_File"), NULL, NULL, NULL},
	{"NewMenuAction", NULL, N_("_New"), NULL, NULL, NULL},
	{"NewWinAction", NULL, N_("_Window"), NULL, NULL, G_CALLBACK(cxplorer_window_new_win_action)},
	{"NewFileAction", "gtk-new", N_("_File..."), NULL, NULL, G_CALLBACK(cxplorer_window_new_file_action)},
	{"NewDirAction", NULL, N_("_Directory..."), NULL, NULL, G_CALLBACK(cxplorer_window_new_directory_action)},
	{"DeleteAction", "gtk-delete", N_("_Delete"), NULL, NULL, G_CALLBACK(cxplorer_window_delete_action)},
	{"RenameAction", NULL, N_("_Rename..."), NULL, NULL, G_CALLBACK(cxplorer_window_rename_action)},
	{"SendAction", NULL, N_("_Send to"), NULL, NULL, NULL},
	{"MailAction", NULL, N_("_Mail"), NULL, NULL, G_CALLBACK(cxplorer_window_send_mail_action)},
	{"SetCmdAction", NULL, N_("Set related command..."), NULL, NULL, G_CALLBACK(cxplorer_window_set_cmd_action)},
	{"PropAction", "gtk-properties", N_("_Property"), NULL, NULL, G_CALLBACK(cxplorer_window_property_action)},
	{"QuitAction", "gtk-quit", N_("_Quit"), NULL, NULL, gtk_main_quit},
	/* Edit menu */
	{"EditAction", NULL, N_("_Edit"), NULL, NULL, NULL},
	{"CopyAction", "gtk-copy", N_("_Copy"), NULL, NULL, G_CALLBACK(cxplorer_window_copy_action)},
	{"PasteAction", "gtk-paste", N_("_Paste"), NULL, NULL, G_CALLBACK(cxplorer_window_paste_action)},
	{"PrefAction", "gtk-preferences", N_("_Preferences"), NULL, NULL, G_CALLBACK(cxplorer_window_pref_action)},
	/* View menu */
	{"ViewAction", NULL, N_("_View"), NULL, NULL, NULL},
	{"RefreshAct", "gtk-refresh", N_("_Refresh"), NULL, NULL, G_CALLBACK(cxplorer_window_refresh_views_action)},
	/* Move menu */
	{"BookmarkAction", NULL, N_("_Bookmark"), NULL, NULL, NULL},
	{"BookmarkAddAction", "gtk-add", N_("_Add Bookmark..."), NULL, NULL, G_CALLBACK(cxplorer_window_bookmark_add_action)},
	{"BookmarkEditAction", NULL, N_("_Edit Bookmark..."), NULL, NULL, G_CALLBACK(cxplorer_window_bookmark_edit_action)},
	{"UpAction", "gtk-go-up", N_("_Up"), "<Alt>Up", NULL, G_CALLBACK(cxplorer_window_go_up_action)},
	{"HomeAct", "gtk-home", N_("_Home"), "<Alt>Home", NULL, G_CALLBACK(cxplorer_window_go_home_action)},
	/* Help menu */
	{"HelpAction", NULL, N_("_Help"), NULL, NULL, NULL},
	{"AboutAction", NULL, N_("_About"), NULL, NULL, G_CALLBACK(cxplorer_window_about_action)}
};

const gint actions_count = G_N_ELEMENTS(actions);

GtkToggleActionEntry toggle_actions[] = {
	{"DotFileAct", NULL, N_("_Show dot file"), NULL, NULL, NULL},
};

const gint toggle_actions_count = G_N_ELEMENTS(toggle_actions);


/**
 * Class init
 */
static void cxplorer_window_class_init (gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);
	CxplorerWindowClass *klass = CXPLORER_WINDOW_CLASS (g_class);

	gobject_class->dispose = cxplorer_window_dispose;
	gobject_class->finalize = cxplorer_window_finalize;

	g_type_class_add_private (klass, sizeof (CxplorerWindowPrivate));

	parent_class = g_type_class_peek_parent (klass);
}

static void cxplorer_window_instance_init (GTypeInstance * instance,
					   gpointer g_class)
{
	CxplorerWindow *self = CXPLORER_WINDOW (instance);
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (instance);
	GtkAction *dot_file_act;
	GtkWidget *menubar;
	GtkWidget *toolbar;
	GtkWidget *vbox;
	GtkWidget *hpaned;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *dirview;
	GtkWidget *right_pane;
	GtkWidget *statusbar;
	gboolean show_dot_file;
	gchar *current_dir;
	gchar *current_dir_utf8;
	gchar *filter;

	gtk_window_set_icon_from_file (GTK_WINDOW(self), PACKAGE_PIXMAPS_DIR"/cxplorer-icon.png", NULL);
	private->client = gconf_client_get_default ();
	gconf_client_add_dir (private->client, BOOKMARK_GCONF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_add_dir (private->client, SENDTO_GCONF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
	show_dot_file = gconf_client_get_bool (private->client, CXPLORER_GCONF_DIR"/ShowDotFile", NULL);
	filter = gconf_client_get_string (private->client, CXPLORER_GCONF_DIR"/FilenameFilter", NULL);
	if(filter != NULL)
	{
		cxp_utils_set_filename_filter (filter);
		g_free (filter);
	}

	private->dynamic_widgets = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, (GDestroyNotify)g_slist_free);
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (self), vbox);

	private->action_group = gtk_action_group_new ("Cxplorer");
	gtk_action_group_set_translation_domain (private->action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (private->action_group, actions, actions_count, self);
	gtk_action_group_add_toggle_actions (private->action_group, toggle_actions, toggle_actions_count, self);
	dot_file_act = gtk_action_group_get_action (private->action_group, "DotFileAct");
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION(dot_file_act), show_dot_file);

	private->ui_manager = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (private->ui_manager,
					    private->action_group, 0);

	gtk_ui_manager_add_ui_from_file (private->ui_manager,
					 PACKAGE_DATA_DIR"/cxplorer-ui.xml",
					 NULL);
	gtk_window_add_accel_group (GTK_WINDOW(self), gtk_ui_manager_get_accel_group (private->ui_manager));

	menubar =
		gtk_ui_manager_get_widget (private->ui_manager, "/menubar");
	gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);
	gtk_widget_show (menubar);

	/*  ˥塼ۤ롣 */
	cxplorer_window_sendto_menu_init (self);

	/* ǥ֥åޡ˥塼ۤ롣 */
	cxplorer_window_bookmark_menu_init (self);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	toolbar = gtk_ui_manager_get_widget (private->ui_manager, "/toolbar");
	gtk_box_pack_start (GTK_BOX (hbox), toolbar, TRUE, TRUE, 0);
	gtk_widget_show (toolbar);
	private->scombo = gtk_combo_box_new_text ();
	gtk_box_pack_start (GTK_BOX (hbox), private->scombo, FALSE, FALSE, 0);
	gtk_combo_box_append_text (GTK_COMBO_BOX(private->scombo), "find");
	gtk_combo_box_append_text (GTK_COMBO_BOX(private->scombo), "locate");
	/*
	gtk_combo_box_append_text (GTK_COMBO_BOX(private->scombo), "namazu");
	*/
	gtk_combo_box_set_active (GTK_COMBO_BOX(private->scombo), 0);
	gtk_widget_show (private->scombo);
	private->sentry = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (hbox), private->sentry, FALSE, FALSE, 0);
	gtk_widget_show (private->sentry);
	gtk_widget_show (hbox);

	hbox = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 5);
	label = gtk_label_new (_("Adress:"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
	gtk_widget_show (label);
	entry = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
	gtk_widget_show (entry);
	gtk_widget_show (hbox);

	hpaned = gtk_hpaned_new ();
	gtk_paned_set_position (GTK_PANED (hpaned), 200);
	gtk_box_pack_start_defaults (GTK_BOX (vbox), hpaned);

	dirview = cxp_dir_view_new (private->client);
	gtk_paned_pack1 (GTK_PANED (hpaned), dirview, FALSE, FALSE);
	gtk_widget_show (dirview);

	right_pane = cxp_right_pane_new (private->client);
	gtk_paned_pack2 (GTK_PANED (hpaned), right_pane, TRUE, FALSE);
	gtk_widget_show (right_pane);
	gtk_widget_show (hpaned);

	cxp_right_pane_set_popup_menu (CXP_RIGHT_PANE (right_pane), gtk_ui_manager_get_widget (private->ui_manager, "/popup"));

	statusbar = gtk_statusbar_new ();
	gtk_box_pack_start (GTK_BOX (vbox), statusbar, FALSE, FALSE, 0);
	gtk_widget_show (statusbar);
	gtk_widget_show (vbox);

	current_dir = g_get_current_dir ();
	current_dir_utf8 = g_locale_to_utf8 (current_dir, -1 ,NULL, NULL, NULL);
	gtk_entry_set_text (GTK_ENTRY (entry), current_dir_utf8);
	cxp_dir_view_change_directory (CXP_DIR_VIEW (dirview), current_dir_utf8);
	cxp_right_pane_change_directory (CXP_RIGHT_PANE (right_pane),
					 current_dir);
	g_free (current_dir);
	g_free (current_dir_utf8);

	private->menubar = menubar;
	private->toolbar = toolbar;
	private->entry = entry;
	private->dirview = dirview;
	private->right_pane = right_pane;
	private->statusbar = statusbar;
	private->dispose_has_run = FALSE;

	g_signal_connect (dot_file_act, "toggled", G_CALLBACK(cxplorer_window_show_dot_file_toggled), self);
	g_signal_connect (entry, "activate",
			  G_CALLBACK (cxplorer_window_entry_activate),
			  self);
	g_signal_connect (private->sentry, "activate",
			  G_CALLBACK (cxplorer_window_search_entry_activate),
			  self);
	g_signal_connect (dirview, "directory_changed",
			  G_CALLBACK (cxplorer_window_on_directory_changed),
			  self);
	g_signal_connect (right_pane, "dir_double_clicked",
			  G_CALLBACK (cxplorer_window_on_dir_double_clicked),
			  self);
	g_signal_connect (right_pane, "up_dir_request",
			  G_CALLBACK (cxplorer_window_go_up_action),
			  self);
	g_signal_connect (right_pane, "search_end",
			  G_CALLBACK (cxplorer_window_search_end),
			  self);
	g_signal_connect (private->client, "value_changed", G_CALLBACK(cxplorer_window_gconf_value_changed), self);
}

/**
 * \if japanese
 * ˥塼ν
 * \endif
 */
static void cxplorer_window_sendto_menu_init (CxplorerWindow *self)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (self);
	GSList *sendto_list;
	GSList *node;
	GSList *widget_list = NULL;
	GConfEntry *entry;
	GConfValue *value;
	gchar *key;

	sendto_list = gconf_client_all_entries (private->client, SENDTO_GCONF_DIR, NULL);
	node = sendto_list;
	while (node != NULL)
	{
		entry = node->data;
		key = g_strdup (gconf_entry_get_key(entry));
		value = gconf_entry_get_value (entry);
		if (value && value->type == GCONF_VALUE_PAIR)
		{
			widget_list = cxplorer_window_menu_item_new_from_sendto (
					gconf_value_get_string(gconf_value_get_car(value)),
					gconf_value_get_string(gconf_value_get_cdr(value)),
					self);
			if (widget_list)
			{
				g_hash_table_insert (private->dynamic_widgets, key, widget_list);
			}
		}
		gconf_entry_free (entry);
		node = g_slist_next(node);
	}

	g_slist_free (sendto_list);
}

/**
 * \if japanese
 * ˥塼˹ܤɲä
 * @param [in] key ϥåΥ
 * @param [in] value ϥå
 * @param [in] user_data CxplorerWindow
 * \endif
 */
static GSList* cxplorer_window_menu_item_new_from_sendto (const gchar *caption, const gchar *command, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	GtkWidget *menu;
	GtkWidget *menuitem = NULL;
	GtkWidget *senditem = NULL;
	GSList *widget_list = NULL;
	const gchar *path[] = {
		"/menubar/FileMenu/SendMenu",
		"/popup/SendMenu",
	};
	guint idx;

	g_return_val_if_fail (caption != NULL, NULL);
	g_return_val_if_fail (command != NULL, NULL);

	for (idx=0; idx<G_N_ELEMENTS(path); idx++)
	{
		if((menuitem = gtk_ui_manager_get_widget (private->ui_manager, path[idx])) != NULL)
		{
			senditem = gtk_menu_item_new_with_label (caption);
			g_object_set_data (G_OBJECT (senditem), "config_value", g_strdup(command));
			g_signal_connect (senditem, "activate", G_CALLBACK(cxplorer_window_sendto_activate), user_data);
			gtk_widget_show (senditem);
			widget_list = g_slist_prepend (widget_list, senditem);

			menu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menuitem));
			gtk_container_add (GTK_CONTAINER(menu), senditem);
		}
	}

	return widget_list;
}

/**
 * \if japanese
 * ֥åޡ˥塼ν
 * \endif
 */
static void cxplorer_window_bookmark_menu_init (CxplorerWindow *self)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (self);
	GSList *bookmark_list;
	GSList *node;
	GSList *widget_list = NULL;
	GConfEntry *entry;
	GConfValue *value;
	gchar *key;

	bookmark_list = gconf_client_all_entries (private->client, BOOKMARK_GCONF_DIR, NULL);
	node = bookmark_list;
	while (node != NULL)
	{
		entry = node->data;
		key = g_strdup(gconf_entry_get_key(entry));
		value = gconf_entry_get_value (entry);
		if (value && value->type == GCONF_VALUE_PAIR)
		{
			widget_list = cxplorer_window_menu_item_new_from_bookmark (
					gconf_value_get_string(gconf_value_get_car(value)),
					gconf_value_get_string(gconf_value_get_cdr(value)),
					self);
			if (widget_list)
			{
				g_hash_table_insert (private->dynamic_widgets, key, widget_list);
			}
		}
		gconf_entry_free (entry);
		node = g_slist_next(node);
	}

	g_slist_free (bookmark_list);
}

/**
 * \if japanese
 * ֥åޡ˥˥塼롣
 * @param [in] key ϥåΥ
 * @param [in] value ϥå
 * @param [in] user_data CxplorerWindow
 * \endif
 */
static GSList *cxplorer_window_menu_item_new_from_bookmark (const gchar *key, const gchar *value, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	GtkWidget *menu;
	GtkWidget *menuitem = NULL;
	GSList *widget_list = NULL;

	g_return_val_if_fail (key != NULL, NULL);
	g_return_val_if_fail (value != NULL, NULL);

	if((menuitem = gtk_ui_manager_get_widget (private->ui_manager, "/menubar/BookmarkMenu")) != NULL)
	{
		menu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menuitem));
		menuitem = gtk_menu_item_new_with_label (key);
		g_object_set_data (G_OBJECT (menuitem), "config_value", g_strdup(value));
		gtk_container_add (GTK_CONTAINER(menu), menuitem);
		g_signal_connect (menuitem, "activate", G_CALLBACK(cxplorer_window_user_bookmark_activate), user_data);
		gtk_widget_show (menuitem);
		widget_list = g_slist_prepend (widget_list, menuitem);
	}

	return	widget_list;
}

static void cxplorer_window_dispose (GObject * obj)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (obj);

	if (private->dispose_has_run)
	{
		/* If dispose did already run, return. */
		return;
	}
	/* Make sure dispose does not run twice. */
	private->dispose_has_run = TRUE;

	/* 
	 * In dispose, you are supposed to free all types referenced from this
	 * object which might themselves hold a reference to self. Generally,
	 * the most simple solution is to unref all members on which you own a 
	 * reference.
	 */
	g_object_unref (private->client);
	g_object_unref (private->action_group);
	g_object_unref (private->ui_manager);
	g_hash_table_destroy (private->dynamic_widgets);
	cxp_utils_free_filename_filter ();

	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void cxplorer_window_finalize (GObject * obj)
{
	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

GType cxplorer_window_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxplorerWindowClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			cxplorer_window_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxplorerWindow),
			0,	/* n_preallocs */
			cxplorer_window_instance_init	/* instance_init */
		};
		type = g_type_register_static (GTK_TYPE_WINDOW,
					       "CxplorerWindowType", &info, 0);
	}
	return type;
}

GtkWidget *cxplorer_window_new (void)
{
	GtkWidget *window;

	window = GTK_WIDGET (g_object_new (CXPLORER_TYPE_WINDOW, NULL));

	return window;
}

static void cxplorer_window_search_entry_activate  (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	const gchar *pattern;
	gchar *pat_locale;
	gchar *cmd;
	gchar *cmd_utf8;
	gchar *cur_dir;
	gint index;
	gint cid;

	pattern = gtk_entry_get_text (GTK_ENTRY(private->sentry));
	pat_locale = g_locale_from_utf8 (pattern, -1, NULL, NULL, NULL);
	index = gtk_combo_box_get_active (GTK_COMBO_BOX(private->scombo));
	switch (index)
	{
		case 0:
			cur_dir = cxp_dir_view_get_current_directory (CXP_DIR_VIEW(private->dirview));
			cmd = g_strdup_printf("find '%s' -iname '%s'", cur_dir, pat_locale);
			g_free (cur_dir);
			break;
		case 1:
			cmd = g_strdup_printf("locate '%s'", pat_locale);
			break;
		/*
		case 2:
			cmd = g_strdup_printf("namazu --list '%s'", pat_locale);
			break;
		*/
		default:
			return;
	}
	cmd_utf8 = g_locale_to_utf8(cmd, -1, NULL, NULL, NULL);
	cid = gtk_statusbar_get_context_id (GTK_STATUSBAR(private->statusbar), "SearchContext");
	gtk_statusbar_push (GTK_STATUSBAR(private->statusbar), cid, cmd_utf8);
	cxp_right_pane_search_file (CXP_RIGHT_PANE(private->right_pane), cmd);
	//gtk_statusbar_pop (GTK_STATUSBAR(private->statusbar), cid);
	g_free (cmd);
	g_free (cmd_utf8);
	g_free (pat_locale);
}

static void cxplorer_window_search_end (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gint cid;

	cid = gtk_statusbar_get_context_id (GTK_STATUSBAR(private->statusbar), "SearchContext");
	gtk_statusbar_pop (GTK_STATUSBAR(private->statusbar), cid);
}

/**
 * This function is called back when value of TextEntry is changed by user.
 * @param gwCaller [in] GtkWidget which call this function.
 * @param gpUserData [in] pointer to instance of Cxplorer.
 * @todo Fixed me to process symlink.
 */
static void cxplorer_window_entry_activate (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	const gchar *address;

	address = gtk_entry_get_text (GTK_ENTRY (widget));
	switch (address[0])
	{
	case '/':
		cxp_dir_view_change_directory (CXP_DIR_VIEW(private->dirview), address);
		break;
	case '.':
		break;
	default:
		g_spawn_command_line_async (address, NULL);
		break;
	}
}

/**
 * This function is called back when cursor is changed on directory view.
 * It have file list view shown file list.
 * @param treeview [in] Pointer to directory view.
 * @param user_data [in] Pointer to instance of Cxplorer is casted.
 */
static void cxplorer_window_on_directory_changed (CxpDirView * dirview,
						  gpointer user_data)
{
	CxplorerWindowPrivate *private =
		CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *locale_path;
	gchar *utf8_path;

	locale_path = cxp_dir_view_get_current_directory (dirview);
	utf8_path = g_locale_to_utf8(locale_path, -1, NULL, NULL, NULL);
	gtk_entry_set_text (GTK_ENTRY (private->entry), utf8_path);
	cxp_right_pane_change_directory (CXP_RIGHT_PANE (private->right_pane),
					 locale_path);
	g_free (locale_path);
	g_free (utf8_path);
}

static void cxplorer_window_on_dir_double_clicked (CxpRightPane * right_pane,
						   gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *locale_path;
	gchar *utf8_path;

	locale_path = cxp_right_pane_get_active_file_name (right_pane);
	utf8_path = g_locale_to_utf8(locale_path, -1, NULL, NULL, NULL);
	cxp_dir_view_change_directory (CXP_DIR_VIEW (private->dirview),
				       utf8_path);
	gtk_entry_set_text (GTK_ENTRY (private->entry), utf8_path);
	g_free (locale_path);
	g_free (utf8_path);
}

/**
 * This function is called back when user click "File->New->Window".
 * @param menuitem [in] Pointer to menu item "File->New->Window"
 * @param user_data [in] Pointer to data which is defined by user.
 * @todo Error handling
 */
static void cxplorer_window_new_win_action (GtkWidget *widget, gpointer user_data)
{
	gchar *prgname = g_get_prgname();

	g_spawn_command_line_async (prgname, NULL);
}

/**
 * This function generate new file in directory when user click
 * "File->New->File".
 * @param menuitem [in] Pointer to menu item "File->New->File".
 * @param user_data [in] Pointer to data which is defined by user.
 */
void cxplorer_window_new_file_action (GtkWidget * widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *user_input;
	GtkWidget *dialog;

	dialog = cxp_entry_dialog_new(_("New file"), _("Please input the name of a new file."), _("NewFile"));
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
	{
		user_input = cxp_entry_dialog_get_entry_text (CXP_ENTRY_DIALOG(dialog));
		cxp_right_pane_make_file (CXP_RIGHT_PANE (private->right_pane),
					  user_input);
		g_free(user_input);
	}
	gtk_widget_destroy(dialog);
}

/**
 * This function generate new directory when user click "File->New->Directory".
 * @param menuitem [in] Pointer to menu item "File->New->Directory".
 * @param user_data [in] Pointer to data which is defined user_data.
 */
void cxplorer_window_new_directory_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *current_dir;
	gchar *fullpath;
	gchar *label;
	gchar *locale_name;
	GtkWidget *dialog;

	dialog = cxp_entry_dialog_new(_("New directory"), _("Please input the name of a new directory."), _("NewDirectory"));
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
	{
		label = cxp_entry_dialog_get_entry_text(CXP_ENTRY_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		locale_name = g_locale_from_utf8 (label, -1, NULL, NULL, NULL);
		if (locale_name != NULL)
		{
			current_dir = cxp_dir_view_get_current_directory (CXP_DIR_VIEW(private->dirview));
			fullpath = g_build_filename (current_dir, locale_name, NULL);
			g_free (current_dir);
			if (mkdir (fullpath, S_IFDIR | 0777) == 0)
			{
				cxp_dir_view_refresh(CXP_DIR_VIEW(private->dirview));
			}
			else
			{
				dialog = gtk_message_dialog_new (NULL,
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_CLOSE,
						"%s",
						g_strerror (errno));
				gtk_dialog_run (GTK_DIALOG (dialog));
				gtk_widget_destroy (dialog);
			}
			g_free (label);
		}
	}
	else
	{
		gtk_widget_destroy(dialog);
	}
}

/**
 * This function delete selected file when user clicked "File->Delete".
 * @param menuitem [in] Pointer to menu item "File->Delete".
 * @param user_data [in] Pointer to instance of Cxplorer.
 * @todo Implement code to remove directory.
 */
void cxplorer_window_delete_action (GtkWidget * widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);

	cxp_right_pane_delete_selected_files (CXP_RIGHT_PANE(private->right_pane));
}

/**
 * This function rename selected file when user click "File->Rename".
 * @param menuitem [in] Pointer to menu item "File->Rename".
 * @param user_data [in] Pointer to instance of Cxplorer.
 * @todo Implement code.
 */
static void cxplorer_window_rename_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);

	cxp_right_pane_rename_file_request (CXP_RIGHT_PANE(private->right_pane));
}

/**
 * This function attach file to mail when user click "File->Send to->Mail".
 * @param menuitem [in] Pointer to menu item "File->Send to->Mail".
 * @param user_data [in] Pointer to instance of Cxplorer.
 * @todo A user enables it to input command format.
 */
static void cxplorer_window_send_mail_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *cmd;
	gchar *cmd_format;
	gchar *fullpath;
	GError *gerror = NULL;
	GtkWidget *dialog;


	cmd_format = gconf_client_get_string (private->client, CXPLORER_GCONF_DIR"/AttachCommand", NULL);
	g_return_if_fail (cmd_format != NULL);
	if ((fullpath = cxp_right_pane_get_active_file_name (CXP_RIGHT_PANE(private->right_pane))) != NULL)
	{
		cmd = g_strdup_printf(cmd_format, fullpath);
		if(g_spawn_command_line_async (cmd, &gerror) == FALSE)
		{
			dialog = gtk_message_dialog_new (NULL,
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_CLOSE,
					"%s",
					gerror->message);
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
		}
		
		g_free(fullpath);
		g_free(cmd);
	}

	g_free (cmd_format);
}

static void cxplorer_window_set_cmd_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);

	cxp_right_pane_set_cmd_request (CXP_RIGHT_PANE(private->right_pane));
}

/**
 * This function show property of selected file when user click
 * "File->Property".
 * @param menuitem [in] Pointer to menu item "File->Property".
 * @param user_data [in] Pointer to instance of Cxplorer.
 * @todo Implement code.
 */
static void cxplorer_window_property_action (GtkWidget * widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *fullpath;
	GtkWidget *dialog;

	if ((fullpath = cxp_right_pane_get_active_file_name (CXP_RIGHT_PANE (private->right_pane))) != NULL)
	{
		dialog = cxp_property_dialog_new (fullpath);

		gtk_dialog_run (GTK_DIALOG (dialog));

		gtk_widget_destroy (dialog);

		g_free (fullpath);
	}
}

/**
 * This function copy file to clipboard when user click "Edit->Copy".
 * @param menuitem [in] Pointer to menu item "Edit->Copy".
 * @param user_data [in] Pointer to instance of Cxplorer.
 * @todo Implement code.
 */
static void cxplorer_window_copy_action (GtkWidget * widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);

	cxp_right_pane_copy_to_clipboard (CXP_RIGHT_PANE(private->right_pane));
}

/**
 * This function paste file from clipboard when user click "Edit->Paste".
 * @param menuitem [in] Pointer to menu item "Edit->Paste".
 * @param user_data [in] Pointer to instance of Cxplorer.
 * @todo Implement code.
 */
static void cxplorer_window_paste_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);

	cxp_right_pane_paste_from_clipboard (CXP_RIGHT_PANE(private->right_pane));
}

static void cxplorer_window_pref_action (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;

	dialog = cxp_pref_dialog_new (GTK_WINDOW(user_data));
	gtk_dialog_run (GTK_DIALOG(dialog));
	gtk_widget_destroy (dialog);
}

/**
 * This function called back when "Show dot file" is toggled.
 * @param gcmItem [in] Pointer to menu item "View->Show dot file".
 * @param gpData [in] Pointer to data which is defined by user.
 */
static void cxplorer_window_show_dot_file_toggled (GtkToggleAction *action, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gboolean show_dot_file;

	show_dot_file = gtk_toggle_action_get_active(action);
	gconf_client_set_bool (private->client, CXPLORER_GCONF_DIR"/ShowDotFile", show_dot_file, NULL);
	cxp_dir_view_refresh (CXP_DIR_VIEW(private->dirview));
}

/**
 * This function refresh views When user request.
 * @param widget [in] Pointer to GtkWidget which calls this function.
 * @param user_data [in] Pointer to data which is defined by user.
 */
static void cxplorer_window_refresh_views_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);

	cxp_dir_view_refresh (CXP_DIR_VIEW(private->dirview));
}

/**
 * This function select parent directory when user request.
 * @param widget [in] Pointer to GtkWidget which calls this function.
 * @param user_data [in] Pointer to data which is defined by user.
 */
static void cxplorer_window_go_up_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *up_dir;
	gchar *current_dir;

	current_dir = cxp_dir_view_get_current_directory (CXP_DIR_VIEW(private->dirview));
	if (current_dir != NULL)
	{
		up_dir = cxp_path_get_dirname_of_utf8 (current_dir);
		cxp_dir_view_change_directory (CXP_DIR_VIEW(private->dirview), up_dir);
		g_free (up_dir);
		g_free (current_dir);
	}
}

/**
 * This function select home directory when user request.
 * @param widget [in] Pointer to GtkWidget which calls this function.
 * @param user_data [in] Pointer to data which is defined by user.
 */
static void cxplorer_window_go_home_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *utf8_path;

	utf8_path = g_locale_to_utf8 (g_get_home_dir (), -1, NULL, NULL, NULL);
	cxp_dir_view_change_directory (CXP_DIR_VIEW(private->dirview), utf8_path);
	g_free (utf8_path);
}

/**
 * \if japanese
 * ֥åޡɲä׵ᤵ줿ν
 * \endif
 */
static void cxplorer_window_bookmark_add_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	GtkWidget *dialog;
	GSList *widget_list = NULL;
	gchar *cur_dir;
	gchar *dir_utf8;
	gchar *message;
	gchar *alias;
	gchar *bookmark_name;
	gchar *bookmark_key;

	cur_dir = cxp_dir_view_get_current_directory (CXP_DIR_VIEW(private->dirview));
	dir_utf8 = g_locale_to_utf8 (cur_dir, -1, NULL, NULL, NULL);
	message = g_strdup_printf (_("Please input the title of the bookmark to directory '%s'."), dir_utf8);

	dialog = cxp_entry_dialog_new (_("Add Bookmark"), message, dir_utf8);

	if (gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
	{
		alias = cxp_entry_dialog_get_entry_text (CXP_ENTRY_DIALOG(dialog));
		bookmark_name = cxp_utils_make_unique_key ();
		bookmark_key = g_build_path ("/", BOOKMARK_GCONF_DIR, bookmark_name, NULL);

		/* ֥åޡ̾ȥǥ쥯ȥ̾ΥڥϿ */
		gconf_client_set_pair (private->client, bookmark_key, GCONF_VALUE_STRING, GCONF_VALUE_STRING, &alias, &dir_utf8, NULL);
		widget_list = cxplorer_window_menu_item_new_from_bookmark (alias, cur_dir, user_data);
		g_hash_table_insert (private->dynamic_widgets, bookmark_key, widget_list);
		g_free (alias);
		g_free (bookmark_name);
	}

	gtk_widget_destroy (dialog);

	g_free (message);
	g_free (dir_utf8);
	g_free (cur_dir);
}

static void cxplorer_window_bookmark_edit_action (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindow *self = CXPLORER_WINDOW (user_data);
	GtkWidget *window;

	window = cxp_gconf_pair_editor_new (GTK_WINDOW(self), _("Edit Bookmark"), "/apps/cxp/cxplorer/bookmark", _("Caption"), _("Directory"));
	gtk_widget_show (window);
	gtk_window_set_modal (GTK_WINDOW(window), TRUE);
}

/**
 * This function show about dialog when user click "Help->About".
 * @param menuitem [in] Pointer to menu item "Help->About".
 * @param user_data [in] Pointer to data which is defined by user, but it isn't
 * used.
 */
static void cxplorer_window_about_action (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *dialog;

	dialog = cxp_about_dialog_new ("Cxplorer", VERSION, "http://cxplorer.sourceforge.jp/", "Copyright 2003-2005 Cxplorer Project", PACKAGE_PIXMAPS_DIR"/cxplorer.png");
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

/**
 * \if japanese
 * ˥塼ιܤ¹Ԥ줿
 * @param [in] widget 桼å˥塼
 * @param [in] user_data CxplorerWindow
 * \endif
 */
static void cxplorer_window_sendto_activate (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *cmd;

	if((cmd = g_object_get_data (G_OBJECT (widget), "config_value")) != NULL)
	{
		cxp_right_pane_send_file_to_command (CXP_RIGHT_PANE(private->right_pane), cmd);
	}
}

/**
 * \if japanese
 * 桼ɲä֥åޡذư
 * @param [in] widget 桼å˥塼
 * @param [in] user_data CxplorerWindow
 * \endif
 */
static void cxplorer_window_user_bookmark_activate (GtkWidget *widget, gpointer user_data)
{
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	gchar *dirname;
	GtkWidget *label;

	label = gtk_bin_get_child (GTK_BIN(widget));
	if (GTK_IS_LABEL(label))
	{
		dirname = g_object_get_data (G_OBJECT(widget), "config_value");
		cxp_dir_view_change_directory (CXP_DIR_VIEW(private->dirview), dirname);
	}
}

/**
 * \if japanese
 * GConfƻΥǥ쥯ȥͤѲ򸡽ФΥХå
 * \endif
 */
static void cxplorer_window_gconf_value_changed (GConfClient* client, const gchar* key, GConfValue* value, gpointer user_data)
{
	CxplorerWindow *self = CXPLORER_WINDOW (user_data);
	CxplorerWindowPrivate *private = CXPLORER_WINDOW_GET_PRIVATE (user_data);
	GSList *widget_list;
	GSList *node;
	GtkWidget *menu_item;
	GtkWidget *label;
	gchar *dir;

	if (value == NULL)
	{
		widget_list = g_hash_table_lookup (private->dynamic_widgets, key);
		for (node = widget_list; node != NULL; node = node->next)
		{
			gtk_widget_destroy (GTK_WIDGET(node->data));
		}
		g_hash_table_remove (private->dynamic_widgets, key);
	}
	else if (value->type == GCONF_VALUE_PAIR)
	{
		widget_list = g_hash_table_lookup (private->dynamic_widgets, key);
		if (widget_list)
		{
			for (node = widget_list; node != NULL; node = node->next)
			{
				menu_item = GTK_WIDGET(node->data);
				label = gtk_bin_get_child (GTK_BIN(menu_item));
				gtk_label_set_text (GTK_LABEL(label), gconf_value_get_string(gconf_value_get_car(value)));
				g_object_set_data (G_OBJECT(menu_item), "config_value", g_strdup(gconf_value_get_string(gconf_value_get_cdr(value))));
			}
		}
		else
		{
			dir = g_path_get_dirname (key);
			if (strcmp(dir, BOOKMARK_GCONF_DIR) == 0)
			{
				widget_list = cxplorer_window_menu_item_new_from_bookmark (
						gconf_value_get_string(gconf_value_get_car(value)),
						gconf_value_get_string(gconf_value_get_cdr(value)),
						self);
			}
			else if (strcmp(dir, SENDTO_GCONF_DIR) == 0)
			{
				widget_list = cxplorer_window_menu_item_new_from_sendto (
						gconf_value_get_string(gconf_value_get_car(value)),
						gconf_value_get_string(gconf_value_get_cdr(value)),
						self);
			}

			if (widget_list)
			{
				g_hash_table_insert (private->dynamic_widgets, g_strdup(key), widget_list);
			}

			g_free (dir);
		}
	}
}
