#include <Class/DAQBuilderClass.hh>
#include <Class/DAQBuilderField.hh>
#include <Class/DAQBuilderMethod.hh>
#include <System/IO/DAQBuilderByteArrayOutputStream.hh>
#include <System/IO/DAQBuilderObjectOutputStream.hh>
#include <System/IO/DAQBuilderByteArrayInputStream.hh>
#include <System/IO/DAQBuilderObjectInputStream.hh>

namespace DAQBuilder
{
  inline void  Class::_finalize()
  {
    _fields.clear();
    _methods.clear();
  }   
  
  const DString Class::toString() const 
  {

    DString returnString = "Class : ";
    returnString += this -> _className;
    returnString += "{\n";
    returnString += "Fields : \n";
    returnString += this -> getFieldTableString();
    returnString += "Methods : \n";
    returnString += this -> getMethodTableString();
    returnString += "}";
    return returnString;
  }
  
  void              Class::_setType()
  {
    this -> _type ="Class";
  }
  inline void Class::_setParentClasses()
  {
    _parentClasses.insert("Object");
    _parentClasses.insert("PubliclyCloneable");
    _parentClasses.insert("Serializable");
    _parentClasses.insert("SerializableObject");
  }
  

  Class::Class(const DString& className )
    :SerializableObject(),_className(className)
  {
    this -> _setType();
    this -> _setParentClasses();    
  }
  Class::Class( const Class& right )
    :SerializableObject(*((SerializableObject*)&right))
     ,_className(right._className),_fields(right._fields),_methods(right._methods)
  {
  }
  Class&                    Class::operator=(const Class & right )
  {
    *((Object*)this)  = *((Object*)&right);
    _className        = right._className ;
    _fields           = right._fields;
    _methods          = right._methods;
    return *this;
  }
  Class::~Class()
  {
    this -> _finalize();
  }

  void                  Class::serialize( DByte *& data , DULong& dataSize )
  {
    ByteArrayOutputStream aBAO;   
    ObjectOutputStream aOOS(&aBAO);
    aOOS.writeString(_className);
    aOOS.writeULong(_fields.size());
    for(  std::map<DString,Field>::const_iterator iter = _fields.begin();
	  iter != _fields.end();
	  iter++)
      {
	aOOS.writeString(iter->first);
	aOOS.writeObject((SerializableObject&)iter->second);
      }
    aOOS.writeULong(_methods.size());
    for(  std::map<DString,Method>::const_iterator iter = _methods.begin();
	  iter != _methods.end();
	  iter++)
      {
	aOOS.writeString(iter->first);
	aOOS.writeObject((SerializableObject&)iter->second);
      }
    return aBAO.toByteArray(data,dataSize);
  }
  ByteArray                  Class::serialize()
  {
    ByteArrayOutputStream aBAO;   
    ObjectOutputStream aOOS(&aBAO);
    aOOS.writeString(_className);
    aOOS.writeULong(_fields.size());
    for(  std::map<DString,Field>::const_iterator iter = _fields.begin();
	  iter != _fields.end();
	  iter++)
      {
	aOOS.writeString(iter->first);
	aOOS.writeObject((SerializableObject&)iter->second);
      }
    for(  std::map<DString,Method>::const_iterator iter = _methods.begin();
	  iter != _methods.end();
	  iter++)
      {
	aOOS.writeString(iter->first);
	aOOS.writeObject((SerializableObject&)iter->second);
      }
    return aBAO.toByteArray();
  }
  void                  Class::unSerialize( DByte * data , const DULong & dataSize )
  {
    ByteArrayInputStream     aBAI(data,dataSize);   
    ObjectInputStream        aOIS(&aBAI);
    _fields.clear();
    _methods.clear();
    _className = 	aOIS.readString();
    DULong    aSize = aOIS.readULong();
    for(   DULong i = 0 ; i <  aSize ; i++)
      {
	_fields.insert(std::pair<DString,Field>(aOIS.readString(),*((Field*)aOIS.readObject())));
      }
    while(aBAI.available()!=0)
      {
	_methods.insert(std::pair<DString,Method>(aOIS.readString(),*((Method*)aOIS.readObject())));
      }
  }
  Cloneable *                Class::clone() const 
  {
    return (Cloneable*)new Class(*this);
  }
  
  
  const DString& Class::getClassName()const 
  {
    return this->_className;
  }    
  
  
  void  Class::setField( Field& field ) 
    throw( SameNameBindedException *)
  {
    std::map<DString,Field>::const_iterator iter = _fields.find(field.getFieldName());
    if(iter != _fields.end())
      throw new SameNameBindedException("Class::setField( Field field )"+field.toString());
    field.setAffiliationClassName(_className);
    _fields.insert(std::pair<DString,Field>(field.getFieldName(),field));
  }
  
  
  const Field& Class::getField(const  DString& fieldName ) const 
  throw( FieldNotFoundException *)
  {
    std::map<DString ,Field >::const_iterator iter = this->_fields.find(fieldName);
    if( iter == this->_fields.end() )
      throw new FieldNotFoundException(this -> getClassName());
    return iter -> second;
  }    
  
  
  void  Class::setMethod( Method& method ) 
    throw( SameNameBindedException  *)
  {
    std::map<DString,Method>::const_iterator iter = _methods.find(method.getMethodName());
    if(iter != _methods.end())
      throw new SameNameBindedException("Class::setMethod( Method method )"+method.toString());
    method.setAffiliationClassName(_className);
    _methods.insert(std::pair<DString,Method>(method.getMethodName(),method));
  }
  
  
  const Method& Class::getMethod(const  DString& methodName ) const 
    throw( MethodNotFoundException *)
  {
    std::map<DString ,Method >::const_iterator iter = this->_methods.find(methodName);
    if( iter == this->_methods.end() ){
      MethodNotFoundException * e = new MethodNotFoundException(this -> getClassName());
       e->insert("Location","Class::getMethod(const  DString& methodName ) const ");
       e->insert("Method",methodName);
       e->insert("I hope ",toString());
       throw e;
    }
    return iter -> second;
  }    
  
  
  const DString Class::getFieldTableString() const 
  { 
    DString messageBuffer = "";
    std::map<DString ,Field >::const_iterator iter = this -> _fields.begin();
    for(;
	iter != this -> _fields.end();
	iter ++)
      {
	DString buf = (iter->second).toString();
	buf += "\n";
	messageBuffer += buf;
      }
    return messageBuffer;
  }
  
  
  const DString Class::getMethodTableString() const 
  { 
    DString messageBuffer = "";
    std::map<DString ,Method >::const_iterator iter =  this -> _methods.begin();
    for( ;
	 iter != this -> _methods.end();
	 iter ++)
      {
	DString buf = (iter->second).toString();
	buf += "\n";
	messageBuffer += buf;
      }
    return messageBuffer;
  }
    StringList            Class::getMethodList() const 
    {
     StringList aList;
    std::map<DString ,Method >::const_iterator iter =  this -> _methods.begin();
    for( ;
	 iter != this -> _methods.end();
	 iter ++)
      {
         aList.push_back(iter->first);
      }
      return aList;
    }
    StringList            Class::getFieldList() const 
    {
     StringList aList;
    std::map<DString ,Field >::const_iterator iter =  this -> _fields.begin();
    for( ;
	 iter != this -> _fields.end();
	 iter ++)
      {
         aList.push_back(iter->first);
      }
      return aList;
    }
  DBool   Class::operator   ==(const Class & right ) const 
  {
    DBool aBool = *((SerializableObject*)this) ==  *((SerializableObject*)&right);
    aBool |= _className==right._className;
    aBool |= _fields==right._fields;
    aBool |= _methods==right._methods;
    return aBool;
  }
  DBool   Class::operator   !=(const Class & right ) const 
  {
    DBool aBool = *((SerializableObject*)this) !=  *((SerializableObject*)&right);
    aBool |= _className!=right._className;
    aBool |= _fields!=right._fields;
    aBool |= _methods!=right._methods;
    return aBool;
  }
}
