#include <Run/DAQBuilderRun.hh>
#include <Run/DAQBuilderAction.hh>
#include <Primitives/DAQBuilderString.hh>
#include <System/DAQBuilderObjectRegistry.hh>
#include <System/DAQBuilderSystem.hh>
namespace DAQBuilder
{
  const DString Run::toString() const 
  {
    DString returnString ="Status      : ";
    switch(_status)
      {
      case Running:
	returnString += "Running";
	break;
      case Pause:
	returnString += "Pause";
	break;
      default: 
	returnString += "Shutdown";
	break;
      }
    returnString +="\n";
    returnString +="NumEvents   : ";
    returnString += String::digitToString((DLong)_numEvents);
    returnString +="\n";
    returnString +="EventNumber : ";
    returnString += String::digitToString((DLong)_eventNumber);
    returnString +="\n";
    returnString +="BeginAction : \n";
    returnString +=_runBeginActions.toString();
    returnString +="\n";
    returnString +="LoopAction : \n";
    returnString +=_runLoopActions.toString();
    returnString +="\n";
    returnString +="EndAction : \n";
    returnString +=_runEndActions.toString();
    returnString +="\n";
    return returnString;
  }
  void              Run::_setType()
  {
    _type ="Run";
  }
  void              Run::_setParentClasses()
  {
    _parentClasses.insert("Object");
    _parentClasses.insert("Cloneable");
    _parentClasses.insert("PubliclyCloneable");
    _parentClasses.insert("Thread");
  }
  
  Run::Run( const DInt& numEvents)
    :Thread(),_numEvents(numEvents),_status(Run::Shutdown),_eventNumber(0)
  {
    _setType();
    _setParentClasses();
  }
  Run::Run( const Run& right )
    :Thread(*((Thread*)&right)),_numEvents(right._numEvents),
     _runBeginActions(right._runBeginActions),_runLoopActions(right._runLoopActions),
     _runEndActions(right._runEndActions),
     _status(right._status),_eventNumber(right._eventNumber)
  {
  }
  Run& Run::operator=( const Run& right )
  {
    *((Thread*)this)= *((Thread*)&right);
    _numEvents      = right._numEvents;
    _runBeginActions= right._runBeginActions;
    _runLoopActions = right._runLoopActions;
    _runEndActions  = right._runEndActions;
    _status         = right._status;
    _eventNumber    = right._eventNumber;
    return *this;    
  }
  Run::~Run()
  {
    //_finalize();
  }
  
  void              Run::run()
  { 
    _status=Run::Running;
    _eventNumber=0;
    try{
      _runBeginActions.execute();
    }catch(Exception *e){
      System::err.println(e);
      delete e;
      return ;
    }
    if( _numEvents <= 0 )
      {
	DBool isEnd = false;
	while(!isEnd)
	  {
	    _lock();
	    switch(_status)
	      {
	      case Run::Running :
		{
		  try{
		    _runLoopActions.execute();
		  }catch(Exception *e){
		    System::err.println(e);
		    delete e;
		    _unLock();
		    _status= Run::Shutdown;
		    isEnd=true;
		    break;
		  }
		  _eventNumber++;
		  break;
		}	     
	      case Run::Pause :
		break;
	      case Run::Shutdown :
		isEnd=true;
		break;
	      }
	    _unLock();
	  }
      }
    else
      {
	DBool isEnd = false;
	for( _eventNumber = 0 ; _eventNumber < _numEvents && !isEnd; )
	  {
	    _lock();
	    switch(_status)
	      {
	      case Run::Running :
		{
		  try{
		    _runLoopActions.execute();
		  }catch(Exception *e){
		    System::err.println(e);
		    delete e;
		    _unLock();
		    _status= Run::Shutdown;
		    isEnd=false;
		    break;
		  }
		  _eventNumber++;
		  break;
		}	     
	      case Run::Pause :
		break;
	      case Run::Shutdown :
		isEnd=false;
		break;
	      }
	    _unLock();
	  }
      }
    try{
      _runEndActions.execute();
    }catch(Exception *e){
      System::err.println(e);
      delete e;
      return ;
    }
  }  
  
  DInt   Run::getNumEvents() const
  {
    return _numEvents;
  }
  void  Run::setNumEvents( const DInt&  numEvents )
  {
    _numEvents  = numEvents;
  }
 
  void  Run::addBeginAction( const Action& beginAction )
  {
    _runBeginActions.push_back( beginAction );
  }
 void  Run::addBeginAction( const ActionList& beginAction )
  {
    std::copy(beginAction.begin(),beginAction.end(),std::back_inserter(_runBeginActions));
  }

  void  Run::addLoopAction( const Action& loopAction )
  {
    _runLoopActions.push_back( loopAction );
  }
 void  Run::addLoopAction( const ActionList& loopAction )
  {
    std::copy(loopAction.begin(),loopAction.end(),std::back_inserter(_runLoopActions));
  }
  void  Run::clearLoopActions() 
  {
    _runLoopActions.clear();
  }


  void  Run::addEndAction( const Action& endAction )
  {
    _runEndActions.push_back( endAction );
  }
 void  Run::addEndAction( const ActionList& endAction )
  {
    std::copy(endAction.begin(),endAction.end(),std::back_inserter(_runEndActions));
  }
  void  Run::clearEndActions() 
  {
    _runEndActions.clear();
  }
  DLong Run::getEventNumber() const 
  {
    return _eventNumber;
  }  
  void  Run::pause()
  {
    _lock();
    _status = Run::Pause;
    _unLock();
  }
  void  Run::resume()
  {
    _lock();
    _status = Run::Running;
    _unLock();
  }
  DBool  Run::shutdown()
  {
    DBool retV = false;
    _lock();
    if(_status == Run::Pause) 
      {
	_status= Run::Shutdown;
        retV=true;
      }
    _unLock();
    return retV;
  }
  DBool    Run::isRunning()
  {
    DBool result;
    _lock();
    result=_status == Run::Running;
    _unLock();
    return result;
  }
}
