#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
import os
import getopt
import glob
import urllib
import logging

try:
    import pygtk
    pygtk.require("2.0")
    import gtk
    assert gtk.gtk_version >= (2, 16, 0)
    assert gtk.pygtk_version >= (2, 16, 0)
except:
    print "Error: PyGTK 2.16.0 or later required."
    sys.exit(1)

import dat2html


class Dat2HtmlGtk():
    def __init__(self):
        self.builder = gtk.Builder()
        self.builder.add_from_file(
            os.path.join(os.path.dirname(os.path.realpath(__file__)),
                         "dat2html_gtk.ui"))
        self.builder.connect_signals(self)
        self.window = self._get("window1")
        self.window.connect("delete_event", self.terminate_program)
        self.window.show()

        drag_targets = [("text/uri-list", 0, 0)]
        self._get("entry_input_path").drag_dest_set(
            gtk.DEST_DEFAULT_MOTION, drag_targets, gtk.gdk.ACTION_COPY)
        self._get("entry_output_path").drag_dest_set(
            gtk.DEST_DEFAULT_MOTION, drag_targets, gtk.gdk.ACTION_COPY)

        self.loggerwindow = self._get("window2")
        self.loggerwindow.connect("delete_event", self.close_loggerwindow)
        loggerview = self._get("textview1")
        loggerview.modify_base(gtk.STATE_NORMAL, gtk.gdk.color_parse("black"))
        loggerview.modify_text(gtk.STATE_NORMAL, gtk.gdk.color_parse("white"))
        self.loggerbuffer = loggerview.get_buffer()

    def terminate_program(self, *args):
        self.window.hide()
        if gtk.main_level() > 0:
            gtk.main_quit()
        sys.exit()

    def close_loggerwindow(self, *args):
        self.loggerwindow.hide()
        self._get("menuitem_loggerwindow").set_active(False)
        return True

    def _get(self, name):
        return self.builder.get_object(name)

    def on_menuitem_open_activate(self, widget):
        self._get("button_input_path").clicked()

    def on_menuitem_convert_activate(self, widget):
        self._get("button_convert").clicked()

    def on_menuitem_quit_activate(self, widget):
        self.terminate_program()

    def on_menuitem_loggerwindow_toggled(self, widget):
        if widget.get_active():
            self.loggerwindow.show()
        else:
            self.loggerwindow.hide()

    def on_menuitem_about_activate(self, widget):
        dialog = gtk.AboutDialog()
        dialog.set_transient_for(self.window)
        dialog.set_name("Dat2Html-Gtk")
        dialog.set_version(dat2html.VERSION)
        dialog.set_comments("2ch dat file to HTML converter")
        dialog.set_website("http://sourceforge.jp/projects/dat2html-gtk/")
        dialog.set_copyright("Copyright (C) 2010-2012 Dat2Html-Gtk Project")
        dialog.set_position(gtk.WIN_POS_CENTER)
        try:
            dialog.set_logo(gtk.gdk.pixbuf_new_from_file(
                    os.path.join(os.path.dirname(os.path.realpath(__file__)),
                                 "dat2html_gtk.svg")))
        except:
            pass
        dialog.run()
        dialog.destroy()

    def on_radio_format_html_toggled(self, widget):
        if widget.get_active():
            self._get("check_option_template").set_sensitive(True)
            self._get("check_option_index").set_sensitive(True)
        if self._get("check_option_template").get_active():
            self._get("button_template_dir").set_sensitive(True)

    def on_radio_format_text_toggled(self, widget):
        if widget.get_active():
            self._get("check_option_template").set_sensitive(False)
            self._get("button_template_dir").set_sensitive(False)
            self._get("check_option_index").set_sensitive(False)

    def on_radio_select_file_toggled(self, widget):
        input_path = self._get("entry_input_path").get_text()
        if widget.get_active() and os.path.isfile(input_path):
            title = dat2html.get_title(open(input_path).readline())
            self._get("label_input_title").set_text(
                title.decode("shift-jis").encode("utf-8"))

    def on_entry_input_path_changed(self, widget):
        input_path = self._get("entry_input_path").get_text()
        if os.path.isfile(input_path):
            title = dat2html.get_title(open(input_path).readline())
            self._get("label_input_title").set_text(
                title.decode("shift-jis").encode("utf-8"))
        else:
            self._get("label_input_title").set_text("")

    def on_entry_input_path_drag_data_received(
        self, widget, drag_context, x, y, selection_data, info, timestamp):
        uris = selection_data.get_uris()
        if len(uris) > 1:
            self.error_dialog(u"複数のファイルは入力できません")
            return
        if not uris[0].startswith("file://"):
            self.error_dialog(u"ファイル名を入力してください")
            return
        widget.set_text(urllib.url2pathname(uris[0]).replace("file://", ""))

    def on_entry_output_path_drag_data_received(
        self, widget, drag_context, x, y, selection_data, info, timestamp):
        uris = selection_data.get_uris()
        if len(uris) > 1:
            self.error_dialog(u"複数のファイルは入力できません")
            return
        if (not uris[0].startswith("file://") or
            not os.path.isdir(uris[0].replace("file://", ""))):
            self.error_dialog(
                u"ディレクトリ名を入力してください")
            return
        widget.set_text(urllib.url2pathname(uris[0]).replace("file://", ""))

    def on_radio_select_directory_toggled(self, widget):
        if widget.get_active():
            self._get("label_input_title").set_text("")

    def on_check_option_template_toggled(self, widget):
        if widget.get_active():
            self._get("button_template_dir").set_sensitive(True)
        else:
            self._get("button_template_dir").set_sensitive(False)

    def on_button_input_path_clicked(self, widget):
        title = u"ファイルの選択"
        action = gtk.FILE_CHOOSER_ACTION_OPEN
        if self._get("radio_select_directory").get_active():
            title = u"ディレクトリの選択"
            action = gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER

        dialog = gtk.FileChooserDialog(title, self.window, action,
                                       (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                        gtk.STOCK_OK, gtk.RESPONSE_OK))
        dialog.set_default_response(gtk.RESPONSE_OK)

        filter = gtk.FileFilter()
        filter.set_name("All files")
        filter.add_pattern("*")
        dialog.add_filter(filter)

        filter = gtk.FileFilter()
        filter.set_name("dat files")
        filter.add_pattern("*.dat")
        dialog.add_filter(filter)
        dialog.set_filter(filter)

        input_path = self._get("entry_input_path").get_text()
        if input_path:
            dialog.set_current_folder_uri(
                "file://" + urllib.pathname2url(os.path.dirname(input_path)))

        response = dialog.run()
        input_path = dialog.get_filename()
        dialog.destroy()

        if response != gtk.RESPONSE_OK:
            return
        if not os.path.exists(input_path):
            logging.debug("Could not open selected path: %s" % input_path)
            return

        logging.debug("Selected path: %s" % input_path)
        self._get("entry_input_path").set_text(input_path)

    def on_button_output_path_clicked(self, widget):
        dialog = gtk.FileChooserDialog(
            u"ディレクトリの選択", self.window,
            gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
             gtk.STOCK_OK, gtk.RESPONSE_OK))
        dialog.set_default_response(gtk.RESPONSE_OK)

        output_path = self._get("entry_output_path").get_text()
        if output_path:
            dialog.set_current_folder_uri(
                "file://" + urllib.pathname2url(os.path.dirname(output_path)))

        response = dialog.run()
        output_path = dialog.get_filename()
        dialog.destroy()

        if response != gtk.RESPONSE_OK:
            return
        if not os.access(output_path, os.W_OK):
            logging.debug("Could not open selected dir: %s" % output_path)
            return

        logging.debug("Selected dir: %s" % output_path)
        self._get("entry_output_path").set_text(output_path)

    def on_button_convert_clicked(self, widget):
        input_path = self._get("entry_input_path").get_text()
        if input_path == "":
            self.error_dialog(u"対象ファイルを選択してください")
            return

        logging.debug("input_path=%s" % input_path)

        template_dir = self.get_template_dir()
        logging.debug("template_dir=%s" % template_dir)
        use_template = self._get("check_option_template").get_active()
        if (use_template and template_dir == ""):
            self.error_dialog(u"テンプレートを選択してください")
            return

        if (template_dir != "*text*" and use_template and
            (not dat2html.template_exists(template_dir))):
            self.error_dialog(u"テンプレートが見つかりません")
            return

        index = (self._get("check_option_index").get_active() and
                 self._get("check_option_index").is_sensitive())

        output_path = self._get("entry_output_path").get_text()
        if output_path == "" and os.path.isfile(input_path):
            output_path = os.path.dirname(os.path.abspath(input_path))
        elif output_path == "" and os.path.isdir(input_path):
            output_path = os.path.abspath(input_path)
        elif output_path == "":
            try:
                output_path = os.path.dirname(
                    os.path.abspath(glob.glob(input_path)[0]))
            except IndexError:
                self.error_dialog(
                    u"出力先ディレクトリを特定できません")
                return

        logging.debug("output_path=%s" % output_path)

        success = self.convert_files([input_path], output_path,
                                     template_dir, index)
        if success:
            dialog = gtk.MessageDialog(self.window, gtk.DIALOG_MODAL,
                                       gtk.MESSAGE_INFO, gtk.BUTTONS_OK)
            dialog.set_title(u"完了しました")
            dialog.set_markup(u"完了しました")
            dialog.run()
            dialog.destroy()
        else:
            self.error_dialog(u"エラーが発生しました")
            self._get("menuitem_loggerwindow").set_active(True)

    def get_template_dir(self):
        if self._get("radio_format_text").get_active():
            return "*text*"

        use_template = self._get("check_option_template").get_active()
        if not use_template:
            return None

        template_dir = self._get("button_template_dir").get_filename()
        return template_dir

    def convert_files(self, input_files, output_dir, template_dir=None,
                      index=False, subject=False):
        success = True
        filenames = dat2html.get_filenames(input_files)
        logging.debug("filenames=%s" % filenames)

        if len(filenames) <= 0:
            logging.error("No input files")
            return False

        if not os.path.exists(output_dir):
            logging.info("Creating directory ...")
            try:
                os.makedirs(output_dir)
            except OSError, (errorno, strerror):
                logging.error("Could not create %s: %s" %
                              (output_dir, strerror))
                return False
        if not os.access(output_dir, os.W_OK):
            logging.error("Could not open %s: permission denied" %
                          output_dir)
            return False

        for filename in filenames:
            success &= dat2html.convert_file(
                filename, output_dir, template_dir)
        if index:
            dat2html.make_index(filenames, output_dir)
        if subject:
            dat2html.make_subject(filenames, output_dir)
        return success

    def error_dialog(self, message):
        dialog = gtk.MessageDialog(self.window, gtk.DIALOG_DESTROY_WITH_PARENT,
                                   gtk.MESSAGE_ERROR, gtk.BUTTONS_OK, message)
        dialog.set_title(u"エラー")
        dialog.set_markup(message)
        dialog.run()
        dialog.destroy()


class LoggerWindowHandler(logging.Handler):
    def __init__(self, loggerbuffer):
        logging.Handler.__init__(self)
        self.loggerbuffer = loggerbuffer

    def emit(self, record):
        self.loggerbuffer.insert(
            self.loggerbuffer.get_end_iter(),
            "%s: %s\n" % (record.levelname, record.message))


def print_help():
    print "Usage: dat2html-gtk [OPTIONS...]\n"
    print "Options:"
    print "  -q, --quiet       suppress warning and info messages"
    print "  -v, --verbose     print debugging messages"
    print "  -h, --help        display this help and exit"
    print "  -V, --version     display version information and exit"


def main():
    app = Dat2HtmlGtk()
    log_level = logging.INFO

    try:
        opts, args = getopt.getopt(
            sys.argv[1:],
            "qhvV",
            ["quiet", "verbose", "help", "version"])
    except getopt.GetoptError:
        print_help()
        sys.exit(2)

    for opt, value in opts:
        if opt in ("-q", "--quiet"):
            log_level = logging.ERROR
        if opt in ("-v", "--verbose"):
            log_level = logging.DEBUG
        if opt in ("-h", "--help"):
            print_help()
            sys.exit()
        if opt in ("-V", "--version"):
            dat2html.print_version()
            sys.exit()

    logging.basicConfig(level=log_level,
                        format="%(levelname)s: %(message)s")
    logging.getLogger().addHandler(LoggerWindowHandler(app.loggerbuffer))

    try:
        gtk.main()
    except KeyboardInterrupt:
        app.terminate_program()


if __name__ == "__main__":
    main()
