﻿using System;
using System.Text;
using System.IO;
using System.Xml.Serialization;
using System.Data;
using System.Data.Common;
using System.Data.OleDb;
using System.Data.SqlClient;
using System.Data.Odbc;
using System.Diagnostics;
using log4net;
using System.Reflection;
using System.Data.OracleClient;

namespace DataUnit.Core
{
    public class Host
    {

        private static readonly ILog _logger
            = LogManager.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        public Host()
        {
        }

        public Host(string configPath)
        {
            _configPath = configPath;
        }

        public Host(Config config)
        {
            _config = config;
        }

        //イベント
        public delegate void StatusEventHandler(object sender, StatusEventArgs e);
        public event StatusEventHandler Status;
        protected virtual void OnExecute(StatusEventArgs e)
        {
            if (Status != null)
            {
                Status(this, e);
            }
        }

        string _configPath = "";
        public string ConfigPath
        {
            get { return _configPath; }
            set { _configPath = value; }
        }

        private Config _config = new Config();
        public Config Config
        {
            get { return _config; }
            set { _config = value; }
        }


        public void CheckData()
        {

            StatusEventArgs e = new StatusEventArgs();
            DirectoryInfo dir = new DirectoryInfo(_config.SqlPath);
            e.AllCount = dir.GetFiles("*.sql").Length;
            e.CurrentCount = 0;
            e.Message = "チェックを開始します。";
            e.Kind = StatusKind.Start;
            OnExecute(e);  //イベント通知

            using (DbConnection con = GetFactory().CreateConnection())
            {
                con.ConnectionString = GetConnectionString();
                con.Open();
                foreach (FileInfo f in dir.GetFiles("*.sql"))
                {
                    e.CurrentCount++;
                    e.Message = String.Format("「{0}」を実行します。", f.Name.Split('.')[0]);
                    e.Kind = StatusKind.Checking;
                    OnExecute(e); //イベント通知
                    string sql = ReadSqlFromFile(f);
                    _logger.Debug(String.Format("SQL実行開始 [{0}]", f.Name));
                    DataTable dt = GetDataTable(con, sql);
                    _logger.Debug(String.Format("SQL実行完了 [{0}]", f.Name));

                    if (0 < dt.Rows.Count)
                    {
                        CreateOutputDirectory();
                        string fileFullPath = String.Format(@"{0}\{1}_{2:yyyyMMdd_HHmm_ss}.csv", _config.OutputPath, f.Name.Split('.')[0], DateTime.Now);
                        e.Message = String.Format("「{0}」に該当するデータが存在します。\r\n          {1}", f.Name.Split('.')[0], fileFullPath);
                        e.Kind = StatusKind.ExistData;
                        OnExecute(e); //イベント通知

                        Csv csv = new Csv();
                        csv.Output(dt, fileFullPath);
                        if (_config.MailConfig.IsSend)
                        {
                            e.Message = String.Format("{0}宛てにメールを送信します。", _config.MailConfig.To);
                            e.Kind = StatusKind.MailSend;
                            OnExecute(e); //イベント通知
                            _config.MailConfig.Send(fileFullPath, f.Name);
                        }
                    }
                }
                e.Message = String.Format("チェックを完了しました。\r\n ");
                e.Kind = StatusKind.End;
                OnExecute(e); //イベント通知

            }
        }

        private void CreateOutputDirectory()
        {
            if (!Directory.Exists(_config.OutputPath))
            {
                Directory.CreateDirectory(_config.OutputPath);
            }
        }

        private string GetConnectionString()
        {
            DbConnectionStringBuilder builder = GetFactory().CreateConnectionStringBuilder();
            foreach (string conStr in _config.ConnectionString)
            {
                string key = conStr.Split('=')[0];
                string value = conStr.Split('=')[1];
                builder.Add(key, value);
            }
            return builder.ConnectionString;
        }


        private DataTable GetDataTable(DbConnection con, string sql)
        {
            DbCommand cmd = GetFactory().CreateCommand();
            cmd.Connection = con;
            cmd.CommandText = sql;
            DbDataAdapter adapter = GetFactory().CreateDataAdapter();
            adapter.SelectCommand = cmd;
            DataTable dt = new DataTable();
            adapter.Fill(dt);
            return dt;
        }

        private string ReadSqlFromFile(FileInfo f)
        {
            string sql = "";
            using (StreamReader sr = new StreamReader(f.FullName, Encoding.GetEncoding("Shift_JIS")))
            {
                sql = sr.ReadToEnd();
            }
            return sql;
        }

        private DbProviderFactory GetFactory()
        {
            DbProviderFactory factory = null;
            switch (_config.DbKind)
            {
                case DbKind.SqlServer:
                    factory = SqlClientFactory.Instance;
                    break;
                case DbKind.Oracle:
                    factory = OracleClientFactory.Instance;
                    break;
                case DbKind.Mdb:
                    factory = OleDbFactory.Instance;
                    break;
                case DbKind.Odbc:
                    factory = OdbcFactory.Instance;
                    break;
                default:
                    Debug.Assert(false, "DB種別が未設定");
                    break;
            }
            return factory;
        }

        public void ReadConfig()
        {
            using (FileStream fs = new FileStream(_configPath, FileMode.Open))
            {
                XmlSerializer serial = new XmlSerializer(typeof(Config));
                _config = (Config)serial.Deserialize(fs);
            }
        }

        public void WriteConfig()
        {
            using (FileStream fs = new FileStream(_configPath, FileMode.Create))
            {
                XmlSerializer serial = new XmlSerializer(typeof(Config));
                serial.Serialize(fs, _config);
            }
        }

    }
}
