/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  dav.c - disk allocation viewer                                           */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2004-2005                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include "dav.h"

#include "xpm/ublk.xpm"
#include "xpm/sblk.xpm"
#include "xpm/cblk.xpm"
#include "xpm/fblk.xpm"
#include "xpm/fblk_lf.xpm"
#include "xpm/fblk_lsf.xpm"



static const gchar	*note_data[] = {
	"unused blocks",
	"system blocks",
	"continuous blocks (continuous file)",
	"continuous blocks (fragmented file)",
	"fragmented blocks (fragmented file)",
	"system fragmented blocks (fragmented file)",
};

/* must be same order to note_data */
static const gchar	**xpm_pathes[] = {
	ublk,
	sblk,
	cblk,
	fblk,
	fblk_lf,
	fblk_lsf,
};

static const gchar	*result_chk_tbl[] = {
	TAG_FSTYPE,
	TAG_MOUNT,
	TAG_PERCENT,
	TAG_BLOCKS,
	TAG_SBLOCKS,
	TAG_FBLOCKS,
	TAG_FRAGS,
	TAG_SFRAGS,
	TAG_CFILES,
	TAG_FFILES,
	TAG_DEPTH,
};


void display_err_dialog(gchar* msg) {
	GtkWidget	*dlg, *label, *btn;

	dlg = gtk_dialog_new();
	label = gtk_label_new(msg);
	btn = gtk_button_new_with_label("OK");
	gtk_signal_connect_object(GTK_OBJECT(btn), "clicked",
				  GTK_SIGNAL_FUNC(gtk_widget_destroy),
				  (gpointer)dlg);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->action_area), btn,
			   TRUE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), label,
			   TRUE, TRUE, 5);
  	gtk_window_set_title(GTK_WINDOW(dlg), "Error");
#ifdef	DEBUG
	gtk_window_set_position(GTK_WINDOW(dlg), GTK_WIN_POS_CENTER);
#endif

	gtk_widget_show_all(dlg);
}



gint on_version_close(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;

	if (app->version_dlg) {
		gtk_widget_destroy(app->version_dlg);
		app->version_dlg = NULL;
	}
	return FALSE;
}
void display_version_dialog(gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	GtkWidget	*dlg, *label, *btn;
	GString		*ver = g_string_new("\nDAV\n\n");

	if (app->version_dlg)
		return;

	g_string_sprintfa(ver, " version %s \n %s ", DAV_VER, COPYRIGHT);
	dlg = gtk_dialog_new();
	app->version_dlg = dlg;

	label = gtk_label_new(ver->str);
	btn = gtk_button_new_with_label("OK");
	gtk_signal_connect(GTK_OBJECT(btn), "clicked",
			   GTK_SIGNAL_FUNC(on_version_close), tmp);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->action_area), btn,
			   TRUE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), label,
			   TRUE, TRUE, 5);
  	gtk_window_set_title(GTK_WINDOW(dlg), "version");
#ifdef	DEBUG
	gtk_window_set_position(GTK_WINDOW(dlg), GTK_WIN_POS_CENTER);
#endif
	g_string_free(ver, TRUE);

	gtk_widget_show_all(dlg);
}



/* draw note */
gint draw_note(t_draw *note) {
	GdkGC		*gc = note->widget->style->black_gc;
	gint		i, x, y;

	gdk_draw_rectangle(note->pixmap,
			   note->widget->style->bg_gc[GTK_STATE_NORMAL],
			   TRUE, 0, 0, note->width, note->height);

	x = MARGIN;
	y = MARGIN;
	for (i = 0; i < MAX_BLK_ID; i++) {
		gdk_draw_pixmap(note->pixmap, gc, note->xpm[i], 0, 0,
				x, y, -1, -1);
		gdk_draw_string(note->pixmap, note->font, gc,
				x + 14, y + 10, note_data[i]);
		y += 14;
	}
	return FALSE;
}

gint on_expose_note(GtkWidget *w, GdkEventExpose *event, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;

	gdk_draw_pixmap(w->window,
			w->style->fg_gc[GTK_WIDGET_STATE(w)],
			app->note.pixmap,
			event->area.x, event->area.y,
			event->area.x, event->area.y,
			event->area.width, event->area.height);
	return FALSE;
}





guint get_page_ln(t_page* page, guint total_ln, guint ln_per_page,
		  gint ev_type) {

	switch (ev_type) {
	case EV_TOTAL_PAGE:
	case EV_FFILE_PAGE:
	case EV_FRAG_PAGE:
		/* do nothing */
		break;
	default:
		page->page = 0;
		page->page_max = (total_ln + ln_per_page - 1) / ln_per_page;
		page->ln_per_page = ln_per_page;
		break;
	}

	if (total_ln > ln_per_page) {
		if (page->page == page->page_max - 1)
			return total_ln - (ln_per_page * page->page);
		else
			return ln_per_page;
	}
	return total_ln;
}

void parse_tcnts(t_tcnts* tcnts) {
	gchar		*p = tcnts->s, *p_tmp, *p_max;
	t_bseq_dt	dt;
	guint		bn = 0;

	if (!p)		/* empty parse string */
		goto ERR_RETURN;

	if (tcnts->dts)	/* already parsed */
		return;

	tcnts->dts = g_array_new(TRUE, FALSE, sizeof(t_bseq_dt));

	p_max = p + strlen(p);
	for (;;) {
		p_tmp = strchr(p, ',');
		if (p_tmp) {
			if (p_tmp >= p_max)
				goto ERR_RETURN;
			*p_tmp = '\0';
		}
		switch (*p++) {
		case '.': dt.type = UBLK; break;
		case 'S':
		case 's':
		case 'i':
		case 'I': dt.type = SBLK; break;
		case 'o': dt.type = CBLK; break;
		case 'x': dt.type = FBLK; break;
		case 'X': dt.type = FBLK_LF; break;
		case 'Y': dt.type = FBLK_LSF; break;
		default:
			goto ERR_RETURN;
		}
		dt.n = strtoul(p, NULL, 10);
		if (dt.n == ULONG_MAX)
			goto ERR_RETURN;
		bn += dt.n;

		g_array_append_val(tcnts->dts, dt);
		if (p_tmp)
			p = p_tmp + 1;
		else
			break;
	}
	tcnts->bn = bn;
	return;

ERR_RETURN:
	g_array_free(tcnts->dts, TRUE);
	tcnts->dts = NULL;
	return;
}

gint build_tcnts_dts(t_result *r, gint ev_type) {
	t_frag_info	*dt;

	switch (ev_type) {
	case EV_START:
		parse_tcnts(&r->fsys_tcnts);
		if (!r->fsys_tcnts.dts) {
			display_err_dialog("invalid block-count sequence.");
			return FALSE;
		}
		if (r->fb_tcnts.s) {
			parse_tcnts(&r->fb_tcnts);
			if (!r->fb_tcnts.dts) {
				display_err_dialog("invalid fbseq format.");
				return FALSE;
			}
		}
		break;
	case EV_FRAG_VIEW:
		if (r->ffile_row < 0)
			return TRUE;
		dt = (t_frag_info*)g_slist_nth_data(r->frag_info, r->ffile_row);
		parse_tcnts(&dt->tcnts);
		if (!dt->tcnts.dts) {
			display_err_dialog("invalid fragmented file info.");
			return FALSE;
		}
		break;
	}
	return TRUE;
}

void check_draw_area(t_draw* blks, t_tcnts* tcnts, gint summary_cnt,
		     gint ev_type) {
	gint		bpl, width, height;
	guint		bn, ln, max_ln_per_page;
	t_page		*page = &blks->page;

	bpl = summary_cnt * BLKS_PER_LINE;
	bn = tcnts->bn;
	ln = (bn + bpl - 1) / bpl;

	if (ev_type != EV_TOTAL_PAGE && ev_type != EV_FRAG_PAGE)
		blks->bnum_width = gdk_string_width(blks->font, "9999999999");
	width = (BLK_WIDTH - 1) * BLKS_PER_LINE + 1
					+ blks->bnum_width + MARGIN * 3;
	max_ln_per_page =
		(MAX_DRAW_PIXEL - 1 - (MARGIN * 2)) / (BLK_HEIGHT - 1);
	ln = get_page_ln(page, ln, max_ln_per_page, ev_type);
	height = (BLK_HEIGHT - 1) * ln + 1 + MARGIN * 2;

	if (blks->pixmap)
		gdk_pixmap_unref(blks->pixmap);
	blks->pixmap = gdk_pixmap_new(blks->widget->window, width, height, -1);
	blks->width = width;
	blks->height = height;
}

gint get_summarized_type(gint cur_type, gint new_type) {
	return (cur_type > new_type ? cur_type : new_type);
}

void get_summary_cnt(t_spin* summary) {
	gint		i, n, cnt;

	n = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(summary->widget));
	cnt = 1;
	for (i = 0; i < n; i++)
		cnt *= 2;
	summary->cnt = cnt;
}

void update_page_btn(t_page* p) {
	if (p->page_max > 1) {
		GString *s = g_string_sized_new(0);

		g_string_sprintfa(s, "(%d/%d)", p->page + 1, p->page_max);
		gtk_label_set_text(GTK_LABEL(p->info), s->str);
		g_string_free(s, TRUE);

		gtk_widget_set_sensitive(p->prev, (p->page > 0));
		gtk_widget_set_sensitive(p->next, (p->page < p->page_max - 1));
		gtk_widget_show_all(p->box);
	} else
		gtk_widget_hide_all(p->box);
}

void draw_update(t_draw* blks) {
	GdkRectangle		rect;

	rect.x = 0;
	rect.y = 0;
	rect.width = blks->width;
	rect.height = blks->height;
	gtk_widget_draw(blks->widget, &rect);
}

/* draw blocks */
void draw_blocks(GArray* ary, gint summary_cnt, gboolean is_fbseq,
		 guint start, t_draw* blks) {
	GtkStyle	*style = blks->widget->style;
	GdkGC		*gc = style->black_gc;
	t_page		*page = &blks->page;
	t_bseq_dt	*dt;
	gint		x, y, width, bpl, type, type_lowest, cnt, ln;
	guint		i, j, bn, target_bn;
	gboolean	loop, is_left_sum_proc;
	gchar		bnum[10 + 1];


	/* first, resize draw area, and fill with background color */
	gtk_drawing_area_size(GTK_DRAWING_AREA(blks->widget),
			      blks->width, blks->height);
	gdk_draw_rectangle(blks->pixmap, style->bg_gc[GTK_STATE_NORMAL],
			   TRUE, 0, 0, blks->width, blks->height);

	type_lowest = UBLK;
	type = type_lowest;
	cnt = 0;
	y = MARGIN;
	x = MARGIN * 2 + blks->bnum_width;
	bn = 0;
	bpl = summary_cnt * BLKS_PER_LINE;
	target_bn = page->page * page->ln_per_page * bpl;
	ln = 0;
	loop = TRUE;
	is_left_sum_proc = FALSE;

	for (i = 0; loop; i++) {
		dt = &g_array_index(ary, t_bseq_dt, i);
		if (!dt->n)
			is_left_sum_proc = TRUE;
		for (j = 0; j < dt->n || is_left_sum_proc; j++) {
			if (bn < target_bn) {
				bn++;
				continue;
			}
			if (bn % bpl == 0) {
				if (ln >= page->ln_per_page) {
					loop = FALSE;
					break;
				}
				if (!is_fbseq) {
					sprintf(bnum, "%10u", (start + bn));
					width = gdk_string_width(blks->font,
								 bnum);
					x = MARGIN + blks->bnum_width - width;
					gdk_draw_string(blks->pixmap,
							blks->font, gc,
							x, y + 10, bnum);
					x += width + MARGIN;
				}
				ln++;
			}
			if (!is_left_sum_proc) {
				type = get_summarized_type(type, dt->type);
				cnt++;
			}
			if (cnt == summary_cnt || (cnt && is_left_sum_proc)) {
				gdk_draw_pixmap(blks->pixmap, gc,
						blks->xpm[type],
						0, 0, x, y, -1, -1);
				x += BLK_WIDTH - 1;
				type = type_lowest;
				cnt = 0;
			}
			if (is_left_sum_proc) {
				loop = FALSE;
				break;
			}
			if (bn % bpl == bpl - 1) {
				x = MARGIN * 2 + blks->bnum_width;
				y += BLK_HEIGHT - 1;
			}
			bn++;
		}
	}
	draw_update(blks);
}

/* draw view */
void draw_view(gpointer tmp, gint ev_type) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_result	*r = ((t_all_data*)tmp)->result;
	t_draw		*blks;
	t_frag_info	*dt;
	t_tcnts		*tcnts;
	gboolean	is_total = TRUE, is_fbseq = TRUE;
	GtkWidget	*fbview;

	if (!r->fsys_tcnts.s)	/* dac output correct ? */
		return;

	if (!build_tcnts_dts(r, ev_type))
		return;

	switch (ev_type) {
	case EV_FRAG_VIEW:
	case EV_FRAG_PAGE:
		is_total = FALSE;
		break;
	case EV_SWITCH_TAB:
	case EV_FBVIEW:
	case EV_SUMMARY:
		is_total = (app->tab_page == NOTEPAGE_TOTAL);
		break;
	}
	if (is_total) {
		fbview = app->fbview[NOTEPAGE_TOTAL];
		is_fbseq = r->fb_tcnts.s && GTK_TOGGLE_BUTTON(fbview)->active;
		tcnts = is_fbseq ? &r->fb_tcnts : &r->fsys_tcnts;
		blks = &app->blks[NOTEPAGE_TOTAL];
	} else {
		if (r->ffile_row < 0)
			return;
		dt = (t_frag_info*)g_slist_nth_data(r->frag_info, r->ffile_row);
		tcnts = &dt->tcnts;
		blks = &app->blks[NOTEPAGE_FRAG];
	}

	/* already drawn ? */
	switch (ev_type) {
	case EV_SWITCH_TAB:
	case EV_SUMMARY:
		if (blks->summary_cnt == app->summary.cnt)
			return;
	}

	check_draw_area(blks, tcnts, app->summary.cnt, ev_type);
	update_page_btn(&blks->page);
	draw_blocks(tcnts->dts, app->summary.cnt, is_fbseq, r->start, blks);
	blks->summary_cnt = app->summary.cnt;

	return;
}

gint on_expose_total_blocks(GtkWidget *w, GdkEventExpose *event, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_draw		*blks = &app->blks[NOTEPAGE_TOTAL];

	if (!blks->pixmap)
		return FALSE;

	gdk_draw_pixmap(w->window,
			w->style->fg_gc[GTK_WIDGET_STATE(w)],
			blks->pixmap,
			event->area.x, event->area.y,
			event->area.x, event->area.y,
			event->area.width, event->area.height);
	return FALSE;
}

gint on_expose_frag_blocks(GtkWidget *w, GdkEventExpose *event, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_draw		*blks = &app->blks[NOTEPAGE_FRAG];

	if (!blks->pixmap)
		return FALSE;

	gdk_draw_pixmap(w->window,
			w->style->fg_gc[GTK_WIDGET_STATE(w)],
			blks->pixmap,
			event->area.x, event->area.y,
			event->area.x, event->area.y,
			event->area.width, event->area.height);
	return FALSE;
}





/* dac io function */
gint g_pipe(gint fd[2]) {
	gint		rc;

	fd[0] = fd[1] = -1;
	rc = pipe(fd);
	if (rc < 0) {
		display_err_dialog("pipe create failed.");
		return rc;
	}
	return 0;
}

void g_fd_close(gint fd[2], gint i) {
	if (fd[i] < 0)
		return;
	close(fd[i]);
	fd[i] = -1;
}

void g_pipe_close(gint fd[2]) {
	gint	i;

	for (i = 0; i < 2; i++)
		g_fd_close(fd, i);
}

gint g_dup(gint fd) {
	gint		rc;

	rc = dup(fd);
	if (rc < 0) {
		display_err_dialog("dup failed.");
		return -1;
	}
	return 0;
}

gint exec_dac(gint out, gint err, t_app_status* app) {
	gint		rc;
	gchar*		argv[5];

	close(STDOUT_FILENO);
	rc = g_dup(out);
	if (rc < 0)
		return -1;
	close(out);

	close(STDERR_FILENO);
	rc = g_dup(err);
	if (rc < 0)
		return -1;
	close(err);

	argv[0] = "dac";
	argv[1] = app->not_use_devdrv ? "-Tvs" : "-Tv";
	argv[2] = app->dev.s;
	argv[3] = app->path.s;
	argv[4] = NULL;

	rc = execvp("dac", argv);
	if (rc < 0) {
		fprintf(stderr,
			"dac not found in PATH environment variable.\n");
		return -1;
	}
	return 0;
}

void clean_up(pid_t* pid, t_io_info *out, t_io_info *err) {
	gint		status;

	if (out->ch) {
		g_io_channel_unref(out->ch);
		out->ch = NULL;
	}
	if (err->ch) {
		g_io_channel_unref(err->ch);
		err->ch = NULL;
	}
	if (*pid >= 0) {
		kill(*pid, SIGTERM);
		waitpid(*pid, &status, 0);
		*pid = -1;
	}
	if (out->s) {
		g_string_free(out->s, TRUE);
		out->s = NULL;
	}
	if (err->s) {
		g_string_free(err->s, TRUE);
		err->s = NULL;
	}
}





/* output result (text data) */
void output_tag_str(t_text* text, gchar* tag, gchar* s) {
	GdkFont		*font = text->font;
	GtkWidget	*w = text->widget;

	gtk_text_insert(GTK_TEXT(w), font, NULL, NULL, " ", -1);
	gtk_text_insert(GTK_TEXT(w), font, NULL, NULL, tag, -1);
	gtk_text_insert(GTK_TEXT(w), font, NULL, NULL, s, -1);
	gtk_text_insert(GTK_TEXT(w), font, NULL, NULL, "\n", -1);
}

void output_tag_value(t_text* text, gchar* tag, guint value) {
	GdkFont		*font = text->font;
	GtkWidget	*w = text->widget;
	GString		*s = g_string_new(tag);

	if (value == ULONG_MAX)
		return;
	g_string_sprintfa(s, "%u\n", value);
	gtk_text_insert(GTK_TEXT(w), font, NULL, NULL, " ", -1);
	gtk_text_insert(GTK_TEXT(w), font, NULL, NULL, s->str, -1);
	g_string_free(s, TRUE);
}

void output_result(t_text* text, t_result* r) {
	output_tag_str(text, "filesystem: ", r->fstype);
	output_tag_str(text, "mount status: ", r->mount_st);
	output_tag_str(text, "fragmented percentage: ", r->f_per);
	output_tag_value(text, "total blocks: ", r->values[ID_BLOCKS]);
	output_tag_value(text, "system blocks: ", r->values[ID_SBLOCKS]);
	output_tag_value(text, "file blocks: ", r->values[ID_FBLOCKS]);
	output_tag_value(text, "fragmented blocks: ", r->values[ID_FRAGS]);
	output_tag_value(text, "sys-fragmented blocks: ", r->values[ID_SFRAGS]);
	output_tag_value(text, "continuous files: ", r->values[ID_CFILES]);
	output_tag_value(text, "fragmented files: ", r->values[ID_FFILES]);
	output_tag_value(text, "max dir depth: ", r->values[ID_DEPTH]);

	gtk_text_backward_delete(GTK_TEXT(text->widget), 1); /* delete last lf*/
	gtk_text_set_point(GTK_TEXT(text->widget), 0);
	gtk_adjustment_set_value(text->adjust, 0.0);
}

void fragment_info_free(t_result* r) {
	gint			i;
	t_frag_info		*dt;

	for (i = 0; i < g_slist_length(r->frag_info); i++) {
		dt = g_slist_nth_data(r->frag_info, i);
		if (dt->tcnts.dts)
			g_array_free(dt->tcnts.dts, TRUE);
		g_chunk_free(dt, r->finfo_mem_chunk);
	}
	g_slist_free(r->frag_info);
	g_mem_chunk_destroy(r->finfo_mem_chunk);
	r->frag_info = NULL;
	r->finfo_mem_chunk = NULL;
}

gboolean init_draw_area(gint ev_type, gint i, gpointer p) {
	t_draw			*blks = &((t_app_status*)p)->blks[i];
	GtkScrolledWindow	*win;
	GtkAdjustment		*adj;

	win = GTK_SCROLLED_WINDOW(blks->scrl_win);
	adj = gtk_scrolled_window_get_vadjustment(win);
	gtk_adjustment_set_value(adj, 0.0);

	if (blks->pixmap) {
		gdk_draw_rectangle(blks->pixmap,
				   blks->widget->style->
					bg_gc[GTK_STATE_NORMAL],
				   TRUE, 0, 0, blks->width, blks->height);
		draw_update(blks);
	}
	switch (ev_type) {
	case EV_TOTAL_PAGE:
	case EV_FFILE_PAGE:
	case EV_FRAG_PAGE:
		break;
	default:
		blks->page.page_max = 0;
		update_page_btn(&blks->page);
		break;
	}
	return TRUE;
}

void init_gui(t_app_status* app, t_result* r, gint ev_type) {
	t_text		*text = &app->text;
	gint		i, j, max;

	if (ev_type == EV_START) {
		gtk_widget_set_sensitive(app->fbview[NOTEPAGE_TOTAL], FALSE);
		gtk_text_forward_delete(GTK_TEXT(text->widget),
					gtk_text_get_length
						(GTK_TEXT(text->widget)));
		g_string_truncate(app->out.s, 0);
		g_string_truncate(app->err.s, 0);
		if (r->fsys_tcnts.dts)
			g_array_free(r->fsys_tcnts.dts, TRUE);
		if (r->fb_tcnts.dts)
			g_array_free(r->fb_tcnts.dts, TRUE);
		if (r->frag_info)
			fragment_info_free(r);
		bzero(r, sizeof(t_result));
		r->ffile_row = -1;
		app->is_abort = FALSE;

		app->ffrag.page.page_max = 0;
		update_page_btn(&app->ffrag.page);
		gtk_clist_clear(GTK_CLIST(app->ffrag.widget));
	}

	i = max = 0;
	switch (ev_type) {
	case EV_START:
		i = NOTEPAGE_TOTAL;
		max = 2;
		break;
	case EV_TOTAL_PAGE:
		i = NOTEPAGE_TOTAL;
		max = 1;
		break;
	case EV_FRAG_VIEW:
	case EV_FRAG_PAGE:
		i = NOTEPAGE_FRAG;
		max = 1;
		break;
	case EV_SWITCH_TAB:
	case EV_SUMMARY:
		/* already drawn ? */
		if (app->blks[app->tab_page].summary_cnt == app->summary.cnt)
			return;
		/* else, fallthrough */
	case EV_FBVIEW:
		i = app->tab_page;
		max = 1;
		break;
	}
	for (j = 0; j < max; j++)
		if (!init_draw_area(ev_type, i++, app))
			break;
}

void update_frag_files(gpointer tmp, gint ev_type) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_result	*r = ((t_all_data*)tmp)->result;
	GtkWidget	*w = app->ffrag.widget;
	t_page		*page = &app->ffrag.page;
	GSList		*l = r->frag_info;
	guint		i, j, ln, top_ln;
	t_frag_info	*dt;
	gchar		*p[FFILE_INFO_CNT +1 +1];
					/* +1 for path, termination */

	ln = get_page_ln(page, g_slist_length(l), MAX_FFILES_PER_PAGE, ev_type);

	update_page_btn(page);

	top_ln = page->page * page->ln_per_page;
	p[FFILE_INFO_CNT + 1] = "";

	gtk_clist_freeze(GTK_CLIST(w));
	gtk_clist_clear(GTK_CLIST(w));
	for (i = 0; i < ln; i++) {
		dt = (t_frag_info*)g_slist_nth_data(l, i + top_ln);
		p[0] = dt->path;
		for (j = 0; j < FFILE_INFO_CNT; j++)
			p[j + 1] = dt->tcnts.info[j];
		gtk_clist_append(GTK_CLIST(w), p);
	}
	gtk_clist_thaw(GTK_CLIST(w));

	/* clear the draw area of fragmented file blocks */
	r->ffile_row = -1;
	init_gui(app, r, EV_FRAG_VIEW);
}



void update_popdown(t_combo* combo) {
	GList		*l;
	gchar		*s;

	if (!combo->s)
		return;

	for (l = combo->list; l; l = l->next)
		if (strcmp(l->data, combo->s) == 0)
			break;
	if (!l) {
		if (g_list_length(combo->list) >= MAX_POPDOWN_CNT) {
			/* del */
			l = g_list_last(combo->list);
			if (l) {
				l->prev->next = NULL;
				l->prev = NULL;
				if (l->data)
					g_free(l->data);
				g_list_free(l);
			}
		}
		/* add */
		s = g_strdup(combo->s);
		combo->list = g_list_prepend(combo->list, s);
	} else if (l->prev){
		/* mov */
		l->prev->next = l->next;
		if (l->next)
			l->next->prev = l->prev;
		l->prev = NULL;
		l->next = combo->list;
		if (l->next)
			l->next->prev = l;
		combo->list = l;
	}
#ifdef	DEBUG
	g_print("... dump popup-history ...\n");
	if (combo->list) {
		for (l = combo->list; l; l = l->next)
			g_print("\t%s\n", (gchar*)l->data);
	} else
		g_print("\t(NULL)\n");
#endif
	gtk_combo_set_popdown_strings(GTK_COMBO(combo->widget), combo->list);
}

gint compare_fraginfo(gconstpointer a, gconstpointer b) {
	t_frag_info	*x, *y;

	x = (t_frag_info*)a;
	y = (t_frag_info*)b;

	return strcmp(x->path, y->path);
}

gchar* parse_ffile_info(gchar* p, gchar* p_max,
			gchar* tag_start, gchar* tag_end, gchar** set) {
	p = strstr(p + 1, tag_start);
	CHK_POINTER(p, p_max);
	*set = p + strlen(tag_start);
	p = strstr(*set, tag_end);
	CHK_POINTER(p, p_max);
	*p++ = '\0';
	return p;

ERR_RETURN:
	return NULL;
}

gchar* parse_ffiles(gchar* p, gint len, t_result* r) {
	GSList		*l = NULL;
	GMemChunk	*mem_chunk;
	t_frag_info	*dt = NULL;
	gchar		*p_max, *p_frag_max;
	gchar		*tag_path_start = "\n\"";
	gchar		*tag_path_end = "\"(ino: ";
	gchar		*tag_fbseq_start = "\nblock sequence: ";
	gint		i;

	mem_chunk = g_mem_chunk_create(t_frag_info, 1024, G_ALLOC_AND_FREE);
	r->finfo_mem_chunk = mem_chunk;

	p_max = p + len;
	p_frag_max = strstr(p, TAG_RESULT_START);
	CHK_POINTER(p_frag_max, p_max);

	while (p < p_frag_max) {
		/* check path */
		p = strstr(p, tag_path_start);
		if (!p || p >= p_max)
			break;
		*p = '\0';
		dt = g_chunk_new(t_frag_info, mem_chunk);
		dt->path = p + strlen(tag_path_start);
		p = strstr(dt->path, tag_path_end);
		CHK_POINTER(p, p_max);
		*p++ = '\0';

		/* check additional-info */
		i = 0;
		if ((p = parse_ffile_info(p, p_max, " (", "%, ",
					  &dt->tcnts.info[i++])) == NULL)
			goto ERR_RETURN;
		if ((p = parse_ffile_info(p, p_max, "total: ", "/",
					  &dt->tcnts.info[i++])) == NULL)
			goto ERR_RETURN;
		if ((p = parse_ffile_info(p, p_max, "frag: ", ",",
					  &dt->tcnts.info[i++])) == NULL)
			goto ERR_RETURN;
		if ((p = parse_ffile_info(p, p_max, "sfrag: ", ")",
					  &dt->tcnts.info[i++])) == NULL)
			goto ERR_RETURN;

		/* check fbseq */
		p = strstr(p, tag_fbseq_start);
		CHK_POINTER(p, p_max);
		dt->tcnts.s = p + strlen(tag_fbseq_start);

		dt->tcnts.dts = NULL;
		dt->tcnts.bn = 0;
		l = g_slist_insert_sorted(l, dt, compare_fraginfo);
	}
	if (g_slist_length(l)) {	/* fragmented file exists */
		p = strchr(dt->tcnts.s, '\n');
		CHK_POINTER(p, p_max);
		*p++ = '\0';
	}
	r->frag_info = l;
	return p_frag_max;

ERR_RETURN:
	r->frag_info = l;
	fragment_info_free(r);
	return NULL;
}

gboolean chk_result_line(guint ln, gchar* tag, gchar* p, t_result *r) {
	gchar		**chk = (gchar**)result_chk_tbl;
	gint		max = ARRAY_SIZE(result_chk_tbl);
	guint		val;

	/* check tag */
	if (ln < max) {
		if (strcmp(chk[ln], tag) != 0)
			return FALSE;
	} else if (strcmp(TAG_FBSEQ, tag) == 0) {
		if (r->fb_tcnts.s)
			return FALSE;
		r->fb_tcnts.s = p;
	} else {
		if (r->fb_tcnts.s)
			return FALSE;
		if (!r->fsys_tcnts.s) {
			val = strtoul(tag, NULL, 10);
			if (val == ULONG_MAX)
				return FALSE;
			r->start = val;
			r->fsys_tcnts.s = p;
		}
	}

	/* check value */
	if (ln >= max)
		return TRUE;
	switch (ln) {
	case 0:
		if (strcmp(p, "ext2") != 0 && strcmp(p, "ext3") != 0)
			return FALSE;
		r->fstype = p;
		break;
	case 1:
		r->mount_st = p;
		break;
	case 2:
		r->f_per = p;
		break;
	default:
		if (strcmp(p, "-") == 0)
			val = ULONG_MAX;
		else {
			val = strtoul(p, NULL, 10);
			if (val == ULONG_MAX)
				return FALSE;
		}
		r->values[ln-3] = val;
		break;
	}
	return TRUE;
}

gchar* parse_result(gchar* p, gint len, t_result* r) {
	gchar		*p_max, *p_tmp, *p_new;
	guint		ln;

	p_max = p + len;
	p = strstr(p, TAG_RESULT_START);
	if (p == NULL || p >= p_max) {
		display_err_dialog(EMSG_DAC_RESULT_INVALID);
		return NULL;
	}
	*p = '\0';
	p += strlen(TAG_RESULT_START);

	ln = 0;
	while (p < p_max) {
		p_tmp = strchr(p, '\t');
		CHK_POINTER(p_tmp, p_max);
		*p_tmp = '\0';
		p_new = strchr(p_tmp + 1, '\n');
		CHK_POINTER(p_new, p_max);
		*p_new = '\0';
		if (!chk_result_line(ln, p, p_tmp + 1, r)) {
			display_err_dialog(EMSG_DAC_RESULT_INVALID);
			return NULL;
		}
		p = p_new + 1;
		ln++;
	}
	return p_max;

ERR_RETURN:
	return NULL;
}

void get_diff_time(struct timeval *start, struct timeval *end,
		   struct timeval *result) {

	result->tv_sec = end->tv_sec - start->tv_sec;
	if (end->tv_usec > start->tv_usec)
		result->tv_usec = end->tv_usec - start->tv_usec;
	else {
		result->tv_sec--;
		result->tv_usec = 1000000 - start->tv_usec + end->tv_usec;
	}
}

void set_statusbar_abort(t_sbar* sbar) {
	gtk_statusbar_push(GTK_STATUSBAR(sbar->widget), sbar->cid,
			   "check aborted.");
}

void set_statusbar_done(t_sbar* sbar, t_time_info* time) {
	GString		*s = g_string_sized_new(0);
	struct timeval	full_time;
	struct timeval	chk_time;

	get_diff_time(&time->start, &time->end, &full_time);
	get_diff_time(&time->start, &time->dac_end, &chk_time);
	g_string_sprintfa(s,
			  "check done.      time: %u.%usec (check: %u.%usec)",
			  (int)full_time.tv_sec, (int)full_time.tv_usec / 1000,
			  (int)chk_time.tv_sec, (int)chk_time.tv_usec / 1000);
	gtk_statusbar_push(GTK_STATUSBAR(sbar->widget), sbar->cid, s->str);
	g_string_free(s, TRUE);
}

gint output_data_proc(gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_result	*r = ((t_all_data*)tmp)->result;
	gchar		*p;
	t_io_info	*out = &app->out;
	GtkWidget	*fbview;

	/* block display */
	p = parse_ffiles(out->s->str, out->s->len, r);
	if (!p)
		return FALSE;

	p = parse_result(p, out->s->len - (p - out->s->str), r);
	if (!p)
		return FALSE;

	if (r->fb_tcnts.s) {
		fbview =  app->fbview[NOTEPAGE_TOTAL];
		gtk_widget_set_sensitive (fbview, TRUE);
	}
	update_popdown(&app->dev);
	update_popdown(&app->path);

	output_result(&app->text, r);
	update_frag_files(tmp, EV_START);
	draw_view(tmp, EV_START);

	return TRUE;
}

guint on_input(GIOChannel *c, GIOCondition condition, gpointer tmp,
	       gboolean is_out) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	GIOError	rc;
	gint		max = 64, cnt, status;
	gchar		buf[max];
	GIOChannel	**c_target;
	GString		*s;
	t_io_info	*out, *err;

	out = &app->out;
	err = &app->err;
	c_target = is_out ? &out->ch : &err->ch;
	if (!*c_target) {
		/* already closed */
		/*g_warning("already closed.\n");*/
		return FALSE;
	}
	/*
	if (g_io_channel_unix_get_fd(c) != g_io_channel_unix_get_fd(*c_target)){
		g_print("fd unmatch.\n");
		return FALSE;
	}
	*/

	rc = g_io_channel_read(c, buf, max, &cnt);
	if (rc == G_IO_ERROR_AGAIN)
		return TRUE;
	if (rc != G_IO_ERROR_NONE || cnt == 0) {
		g_io_channel_unref(c);
		*c_target = NULL;
		if (!out->ch && !err->ch && app->dac_pid >= 0) {
			waitpid(app->dac_pid, &status, 0);
			app->dac_pid = -1;
			gettimeofday(&app->time.dac_end, NULL);

			if (err->s->len) {
				g_string_truncate(err->s,
						  strlen(err->s->str) - 1);
				display_err_dialog(err->s->str);
			} else if (!app->is_abort)
				output_data_proc(tmp);

			gettimeofday(&app->time.end, NULL);
			if (app->is_abort)
				set_statusbar_abort(&app->sbar);
			else
				set_statusbar_done(&app->sbar, &app->time);
		}
		return FALSE;
	}
	s = is_out ? out->s : err->s;
	g_string_sprintfa(s, "%.*s", cnt, buf);

	return TRUE;
}

guint on_input_from_out(GIOChannel *c, GIOCondition condition, gpointer tmp) {
	return on_input(c, condition, tmp, TRUE);
}

guint on_input_from_err(GIOChannel *c, GIOCondition condition, gpointer tmp) {
	return on_input(c, condition, tmp, FALSE);
}





gint proc_dav_out_file(gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	gchar		*p = "data for dav (dac ver: ";
	ssize_t		size, max = 128, len = strlen(p);
	gint		fd, result = FALSE;
	gchar		buf[max];
	t_io_info	*out = &app->out;

	/* check device file */
	fd = open(app->dev.s, O_RDONLY);
	if (fd < 0)
		goto RETURN;

	while ((size = read(fd, buf, len)) < 0 && errno == EINTR);
	if (size != len)
		goto RETURN;

	if (strncmp(p, buf, len) != 0)
		goto RETURN;

	result = TRUE;
	gettimeofday(&app->time.start, NULL);
	gettimeofday(&app->time.dac_end, NULL);
	gtk_statusbar_push(GTK_STATUSBAR(app->sbar.widget),
			   app->sbar.cid, "checking fragmentation...");

	while ((size = read(fd, buf, max))) {
		if (size < 0) {
			if (errno == EINTR)
				continue;
			else {
				display_err_dialog("dac out-file read failed.");
				break;
			}
		} else
			g_string_sprintfa(out->s, "%.*s", size, buf);
	}
	if (size >= 0)
		output_data_proc(tmp);

	gettimeofday(&app->time.end, NULL);
	set_statusbar_done(&app->sbar, &app->time);

RETURN:
	if (fd >= 0)
		close(fd);

	return result;
}

gint on_exec_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_result	*r = ((t_all_data*)tmp)->result;
	gint		rc;
	gint		pipe_out[2], pipe_err[2];
	t_io_info	*out, *err;

	if (app->dac_pid >= 0)
		return TRUE;
	if (!app->dev.s) {
		display_err_dialog("device not defined.");
		return TRUE;
	}

	init_gui(app, r, EV_START);

	rc = proc_dav_out_file(tmp);
	if (rc)
		return TRUE;

	rc = g_pipe(pipe_out);
	if (rc < 0)
		return TRUE;
	rc = g_pipe(pipe_err);
	if (rc < 0) {
		g_pipe_close(pipe_out);
		return TRUE;
	}

	gettimeofday(&app->time.start, NULL);
	app->dac_pid = fork();
	switch (app->dac_pid) {
	case -1:
		display_err_dialog("fork failed.");
		g_pipe_close(pipe_out);
		g_pipe_close(pipe_err);
		return TRUE;
	case 0:
		g_fd_close(pipe_out, 0);
		g_fd_close(pipe_err, 0);
		rc = exec_dac(pipe_out[1], pipe_err[1], app);
		if (rc < 0)
			_exit(0);
		_exit(1);
		break;
	default:
		gtk_statusbar_push(GTK_STATUSBAR(app->sbar.widget),
				   app->sbar.cid,
				   "checking fragmentation...");
		g_fd_close(pipe_out, 1);
		g_fd_close(pipe_err, 1);
		out = &app->out;
		err = &app->err;
		out->ch = g_io_channel_unix_new(pipe_out[0]);
		err->ch = g_io_channel_unix_new(pipe_err[0]);
		g_io_add_watch(out->ch,
			       G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL,
			       (GIOFunc)on_input_from_out, tmp);
		g_io_add_watch(err->ch,
			       G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL,
			       (GIOFunc)on_input_from_err, tmp);
		break;
	}
	return TRUE;
}

gint on_abort_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;

	if (app->dac_pid < 0)
		return TRUE;

	app->is_abort = TRUE;
	kill(app->dac_pid, SIGTERM);
	return TRUE;
}

void on_delete(GtkWidget *w, GdkEvent *event, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;

	clean_up(&app->dac_pid, &app->out, &app->err);
	gtk_main_quit();
}

/*
void destroy(GtkWidget *widget, gpointer data) {
	g_print("destroy event occurred\n");
	gtk_main_quit();
}
*/

gint on_device_changed(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;

	app->dev.s = gtk_entry_get_text(GTK_ENTRY(w));
	if (app->dev.s[0] == '\0')
		app->dev.s = NULL;
	return TRUE;
}

gint on_path_changed(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;

	app->path.s = gtk_entry_get_text(GTK_ENTRY(w));
	if (app->path.s[0] == '\0')
		app->path.s = NULL;
	return TRUE;
}

void update_view(gpointer tmp, gint ev_type) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_result	*r = ((t_all_data*)tmp)->result;

	init_gui(app, r, ev_type);
	draw_view(tmp, ev_type);
}

void page_button_proc(t_page* page, gboolean is_next,
		      t_page_func func, gpointer tmp, gint ev_type) {
	if (is_next) {
		if (page->page < page->page_max - 1) {
			page->page++;
			func(tmp, ev_type);
		}
	} else {
		if (page->page > 0) {
			page->page--;
			func(tmp, ev_type);
		}
	}
}

gint on_total_prev_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_page		*page = &app->blks[NOTEPAGE_TOTAL].page;

	page_button_proc(page, FALSE, update_view, tmp, EV_TOTAL_PAGE);
	return FALSE;
}
gint on_total_next_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_page		*page = &app->blks[NOTEPAGE_TOTAL].page;

	page_button_proc(page, TRUE, update_view, tmp, EV_TOTAL_PAGE);
	return FALSE;
}

gint on_frag_prev_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_page		*page = &app->blks[NOTEPAGE_FRAG].page;

	page_button_proc(page, FALSE, update_view, tmp, EV_FRAG_PAGE);
	return FALSE;
}
gint on_frag_next_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_page		*page = &app->blks[NOTEPAGE_FRAG].page;

	page_button_proc(page, TRUE, update_view, tmp, EV_FRAG_PAGE);
	return FALSE;
}

gint on_ffile_prev_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_page		*page = &app->ffrag.page;

	page_button_proc(page, FALSE, update_frag_files, tmp, EV_FFILE_PAGE);
	return FALSE;
}
gint on_ffile_next_clicked(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_page		*page = &app->ffrag.page;

	page_button_proc(page, TRUE, update_frag_files, tmp, EV_FFILE_PAGE);
	return FALSE;
}

void on_switch_tab(GtkNotebook *w, GtkNotebookPage *page,
		   gint pnum, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	gint		pnum_other;

	pnum_other = pnum == NOTEPAGE_TOTAL ? NOTEPAGE_FRAG : NOTEPAGE_TOTAL;
	gtk_widget_hide(app->fbview[pnum_other]);
	gtk_widget_show(app->fbview[pnum]);

	app->tab_page = pnum;
	update_view(tmp, EV_SWITCH_TAB);
}

gint on_fbview_toggled(GtkWidget *w, gpointer tmp) {
	update_view(tmp, EV_FBVIEW);
	return FALSE;
}

gint on_summary_changed(GtkWidget *w, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;

	get_summary_cnt(&app->summary);
	update_view(tmp, EV_SUMMARY);
	return FALSE;
}

gint on_select_ffile(GtkWidget *w, gint row, gint column,
		     GdkEventButton *event, gpointer tmp) {
	t_app_status	*app = ((t_all_data*)tmp)->app;
	t_result	*r = ((t_all_data*)tmp)->result;
	t_page		*page = &app->ffrag.page;

	r->ffile_row = row += page->page * page->ln_per_page;
	update_view(tmp, EV_FRAG_VIEW);
	return FALSE;
}

gint on_unselect_ffile(GtkWidget *w, gint row, gint column,
		     GdkEventButton *event, gpointer tmp) {
	t_result	*r = ((t_all_data*)tmp)->result;

	r->ffile_row = -1;
	update_view(tmp, EV_FRAG_VIEW);
	return FALSE;
}

gint on_menu_version(GtkWidget *w, gpointer tmp) {
	display_version_dialog(tmp);
	return FALSE;
}



void usage(void) {
	fprintf(stderr, "dav %s\n", DAV_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "dav [-hs] [devname [path]]\n");
	fprintf(stderr, "  -h: show this help message\n");
	fprintf(stderr, "  -s: do not use device-driver interface\n");
	fprintf(stderr, "      (this option takes effect" \
			      " only if the partition is mounted)\n");
}

int main(int argc, char *argv[]) {
	t_all_data	all;
	t_app_status	app;
	t_result	result;
	int		i;

	bzero(&app, sizeof(app));
	app.dac_pid = -1;
	bzero(&result, sizeof(result));
	all.app = &app;
	all.result = &result;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			if (strcmp(argv[i], "-s") == 0)
				app.not_use_devdrv = 1;
			else {
				usage();
				return(1);
			}
		} else if (!app.dev.s)
			app.dev.s = argv[i];
		else if (!app.path.s)
			app.path.s = argv[i];
		else {
			usage();
			return(1);
		}
	}

	gtk_init(&argc, &argv);

	app.out.s = g_string_sized_new(OUT_DEFAULT_LEN);
	app.err.s = g_string_sized_new(ERR_DEFAULT_LEN);

	app.blks[NOTEPAGE_TOTAL].font
	= app.blks[NOTEPAGE_FRAG].font
	= app.note.font
	= app.text.font
		= gdk_font_load(DAV_FONT);

	create_window(&all);

	for (i = 0; i < MAX_BLK_ID; i++) {
		app.blks[NOTEPAGE_TOTAL].xpm[i]
		= app.blks[NOTEPAGE_FRAG].xpm[i]
		= app.note.xpm[i]
			= gdk_pixmap_create_from_xpm_d(app.note.widget->window,
						       NULL, NULL,
						       (gchar**)xpm_pathes[i]);
	}
  	draw_note(&app.note);

	if (app.dev.s)
		update_popdown(&app.dev);
	if (app.path.s)
		update_popdown(&app.path);

	get_summary_cnt(&app.summary);
	init_gui(&app, &result, EV_START);
	gtk_main();

	return(0);
}
