package jp.ac.fun.db.data;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Diff結果オブジェクトです。
 * @param <T1> カラムを一意に識別する値の型
 * @param <T2> カラム値の型
 */
public class DiffResult<T1, T2> {

    /** 追加された要素のリスト */
    private List<Map<T1, T2>> addedList = new ArrayList<>();
    /** 変更された要素のリスト */
    private List<ModifiedPair<T1, T2>> modifiedList = new ArrayList<>();
    /** 削除された要素のリスト */
    private List<Map<T1, T2>> deletedList = new ArrayList<>();
    /** 変更のなかった要素のリスト */
    private List<Map<T1, T2>> invariantList = new ArrayList<>();
    /** 変更のなかった要素数 */
    private volatile long invariantCount = 0L;

    /**
     * 差分があったかを示すフラグを取得します。<br>
     * 差分があった場合は{@code true}を返します。
     * @return 差分があったかを示すフラグ
     */
    public boolean isDiff() {
        return !(addedList.isEmpty() && modifiedList.isEmpty() && deletedList.isEmpty());
    }

    /**
     * 変更前の要素数を取得します。
     * @return 変更前の要素数
     */
    public long getBeforeCount() {
        return modifiedList.size() + invariantCount + deletedList.size();
    }

    /**
     * 変更後の要素数を取得します。
     * @return 変更後の要素数
     */
    public long getAfterCount() {
        return addedList.size() + modifiedList.size() + invariantCount;
    }

    /**
     * 追加された要素のリストを取得します。
     * @return 追加された要素のリスト
     */
    public List<Map<T1, T2>> getAddedList() {
        return addedList;
    }
    /**
     * 追加された要素のリストに追加します。
     * @param addedElem 追加された要素
     */
    public void addAddedList(Map<T1, T2> addedElem) {
        this.addedList.add(addedElem);
    }
    /**
     * 追加された要素のリストに追加します。
     * @param addedCollections 追加された要素
     */
    public void addAddedList(Collection<Map<T1, T2>> addedCollections) {
        this.addedList.addAll(addedCollections);
    }
    /**
     * 変更された要素のリストを取得します。
     * @return 変更された要素のリスト
     */
    public List<ModifiedPair<T1, T2>> getModifiedList() {
        return modifiedList;
    }
    /**
     * 変更された要素のリストに追加します。
     * @param beforeElem 変更前の要素
     * @param diffMap 変更が入った要素
     */
    public void addModifiedList(Map<T1, T2> beforeElem, Map<T1, T2> diffMap) {
        this.modifiedList.add(new ModifiedPair<T1, T2>(beforeElem, diffMap));
    }
    /**
     * 変更された要素のリストに追加します。
     * @param modifiedCollections 変更された要素
     */
    public void addModifiedList(Collection<ModifiedPair<T1, T2>> modifiedCollections) {
        this.modifiedList.addAll(modifiedCollections);
    }
    /**
     * 削除された要素のリストを取得します。
     * @return 削除された要素のリスト
     */
    public List<Map<T1, T2>> getDeletedList() {
        return deletedList;
    }
    /**
     * 削除された要素のリストに追加します。
     * @param deletedElem 削除された要素
     */
    public void addDeletedList(Map<T1, T2> deletedElem) {
        this.deletedList.add(deletedElem);
    }
    /**
     * 削除された要素のリストに追加します。
     * @param deletedCollections 削除された要素
     */
    public void addDeletedList(Collection<Map<T1, T2>> deletedCollections) {
        this.deletedList.addAll(deletedCollections);
    }
    /**
     * 変更のなかった要素のリストを取得します。
     * @return 変更のなかった要素のリスト
     */
    public List<Map<T1, T2>> getInvariantList() {
        return invariantList;
    }
    /**
     * 変更のなかった要素のリストに追加します。
     * @param invariantElem 変更のなかった要素
     */
    public void addInvariantList(Map<T1, T2> invariantElem) {
        this.invariantList.add(invariantElem);
        incrementInvariantCount();
    }
    /**
     * 変更のなかった要素のリストに追加します。
     * @param invariantCollections 変更のなかった要素
     */
    public void addInvariantList(Collection<Map<T1, T2>> invariantCollections) {
        this.deletedList.addAll(invariantCollections);
        invariantCount = invariantCount + invariantCollections.size();
    }

    /**
     * 変更のなかった要素数を増加させます。
     */
    public void incrementInvariantCount() {
        invariantCount = invariantCount + 1;
    }

    /**
     * 変更のなかった要素数を設定します。
     * @param invariantCount 変更のなかった要素数
     */
    public void setInvariantCount(long invariantCount) {
        this.invariantCount = invariantCount;
    }

    /**
     * 変更のなかった要素数を取得します。
     * @return 変更のなかった要素数
     */
    public long getInvariantCount() {
        return invariantCount;
    }
}
