package jp.ac.fun.db.data;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * 一意な一行分のDBデータを格納するクラスです。
 * @param <T1> カラムを一意に識別する値の型
 * @param <T2> カラム値の型
 */
public class UniqueEntityImpl<T1, T2> implements UniqueEntity<T1, T2> {
    /** セパレータ */
    private static final String SEPARATOR = ",";

    /** テーブル名 */
    private String tableName;
    /** 一行分のデータ カラム名とカラム値のMap */
    private Map<T1, T2> entity;
    /** 一意性情報 */
    private List<T2> pkDataList;

    /**
     * コンストラクタ
     * @param tableName テーブル名
     * @param pkList PKリスト
     * @param entity 一行分のデータ
     */
    public <T> UniqueEntityImpl(String tableName, List<T1> pkList, Map<T1, T2> entity) {
        this.tableName = tableName;
        this.entity = entity;
        if (pkList == null) {
            pkDataList = new ArrayList<>(entity.values());
        } else {
            pkDataList = createPkDataList(pkList);
        }
    }

    /**
     * 一行分のデータからPKに指定されているカラムのみのデータを抽出します。
     * @param pkList PKリスト
     * @return PKカラムのみのデータリスト
     */
    private List<T2> createPkDataList(List<T1> pkList) {
        List<T2> result = new ArrayList<>();
        for (Entry<T1, T2> entry : entity.entrySet()) {
            if (pkList.contains(entry.getKey())) {
                result.add(entry.getValue());
            }
        }
        return result;
    }

    /*
     * (非 Javadoc)
     * @see jp.co.ctc_g.dmkpf.test.data.UniqueEntity#getKey()
     */
    @Override
    public List<T2> getKey() {
        return pkDataList;
    }

    /*
     * (非 Javadoc)
     * @see jp.co.ctc_g.dmkpf.test.data.UniqueEntity#same(jp.co.ctc_g.dmkpf.test.data.UniqueEntity)
     */
    @Override
    public boolean same(UniqueEntity<T1, T2> entity) {
        return this.getKey().equals(entity.getKey());
    }

    /*
     * (非 Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean equals(Object obj) {
        if (entity == null) {
            return false;
        }

        if (obj instanceof UniqueEntity) {
            return this.same((UniqueEntity<T1, T2>) obj);
        }
        // else case: call super method
        return super.equals(obj);
    }

    /*
     * (非 Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        if (pkDataList.isEmpty()) {
            return 0;
        }
        return pkDataList.hashCode();
    }

    /*
     * (非 Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        for (Object data : getKey()) {
            sb.append(data.toString() + SEPARATOR);
        }
        return sb.substring(0, sb.length() - 1);
    }

    /*
     * (非 Javadoc)
     * @see jp.co.ctc_g.dmkpf.test.data.UniqueEntity#getEntity()
     */
    @Override
    public Map<T1, T2> getEntity() {
        return entity;
    }

    /*
     * (非 Javadoc)
     * @see jp.co.ctc_g.dmkpf.test.data.UniqueEntity#getTableName()
     */
    @Override
    public String getTableName() {
        return tableName;
    }
}
