package jp.ac.fun.db.diff;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import jp.ac.fun.db.data.DiffResult;
import jp.ac.fun.db.data.ModifiedPair;
import jp.ac.fun.db.data.UniqueEntity;
import jp.ac.fun.db.util.DbDiffUtil;

/**
 * DbDifferの基底実装クラスです。<br>
 * 必要に応じて各メソッドをOverrideしてください。<br>
 * init(初期化) → setReadyToProcess(試験実行前DB状態取得)<br>
 *  → getDiff(試験実行後DB状態取得及び差分計算) → output(差分出力) の順で実行します
 *
 * @param <T1> カラムを一意に識別する値の型
 * @param <T2> カラム値の型
 */
public class DefaultDbDiffer<T1, T2> implements DbDiffer<T1, T2> {

	/** 改行コード */
	protected static final String LINE_SEPARATOR = "\n";

	/** テーブル名一覧 */
	protected List<String> tableNameList = new ArrayList<>();
	/**
	 * 変更前の全テーブル行データ.<br>
	 * テーブル名(1):一意性情報の文字列(N):一意性情報(N)
	 */
	protected Map<String, Map<String, UniqueEntity<T1, T2>>> beforeAllTablesRow = new HashMap<>();
	/**
	 * 全テーブルのPK情報.<br>
	 * テーブル名(1):列名(N)
	 */
	protected Map<String, List<T1>> allTablesPk = new HashMap<>();

	/** 変更の入っていないレコードも出力するか */
	private boolean outputInvariantRows = false;
	/** 初期化済みフラグ */
	private boolean initialized = false;
	/** 実行中フラグ */
	private boolean processing = false;
	/** RowsGetter */
	private RowsGetter<T1, T2> rowsGetter;

	/**
	 * 初期化処理を行います。取得対象となるテーブル名を設定します。
	 * @param rowsGetter RowsGetter
	 * @param tableNames 取得対象となるテーブル名
	 */
	public void init(RowsGetter<T1, T2> rowsGetter, String... tableNames) {
		init(rowsGetter, Arrays.asList(tableNames));
	}

	/**
	 * 初期化処理を行います。取得対象となるテーブル名を設定します。
	 * @param rowsGetter RowsGetter
	 * @param tableNameList 取得対象となるテーブル名一覧
	 */
	public void init(RowsGetter<T1, T2> rowsGetter, List<String> tableNameList) {
		if (processing) {
			throw new IllegalStateException("事前DB取得が行われた状態でテーブル一覧設定が変更されました。");
		}
		this.rowsGetter = rowsGetter;
		this.tableNameList.addAll(tableNameList);
		initialized = true;
	}

	/*
	 * (非 Javadoc)
	 * @see jp.co.ctc_g.dmkpf.test.util.DbDiffer#setReadyToProcess()
	 */
	@Override
	public void setReadyToProcess() {
		if (!initialized) {
			throw new IllegalStateException("初期化が行われていません。事前にinitメソッドを実行してください。");
		}

		// pk info get
		allTablesPk = getAllTablesPk();
		// before db get
		beforeAllTablesRow = getBeforeRows();
		processing = true;
	}

	/**
	 * 試験実行前のDBデータを取得します。<br>
	 * 特別な処理が必要な場合はこのメソッドをオーバーライドします。
	 * @return 試験実行前のDBデータ
	 */
	protected Map<String, Map<String, UniqueEntity<T1, T2>>> getBeforeRows() {
		return getAllTablesRow(allTablesPk);
	}

	/*
	 * (非 Javadoc)
	 * @see jp.co.ctc_g.dmkpf.test.util.DbDiffer#getDiff()
	 */
	@Override
	public Map<String, DiffResult<T1, T2>> getDiff() {
		// diff
		Map<String, DiffResult<T1, T2>> resultMap = getDiffResult(getAfterRows());
		processing = false;
		return resultMap;
	}

	/*
	 * (非 Javadoc)
	 * @see jp.co.ctc_g.dmkpf.test.util.AbstractDbDiffer#output(java.util.Map)
	 */
	@Override
	public void output(Map<String, DiffResult<T1, T2>> diffResult) {
		StringBuilder sb = new StringBuilder();
		for (Entry<String, DiffResult<T1, T2>> entry : diffResult.entrySet()) {
			DiffResult<T1, T2> result = entry.getValue();
			sb.append("【" + entry.getKey() + "】 " + result.getBeforeCount() + "件 → " + result.getAfterCount() + "件 " + LINE_SEPARATOR);
			if (result.isDiff()) {
				// created record dump
				dbDump(sb, result.getAddedList(), "◆CREATED RECORDS◆");
				// modified record dump
				dbDumpPairs(sb, result.getModifiedList(), "◆MODIFIED RECORDS◆");
				// deleted record dump
				dbDump(sb, result.getDeletedList(), "◆DELETED RECORDS◆");
				if (isOutputInvariantRows()) {
					// invariant record dump
					dbDump(sb, result.getInvariantList(), "◆INVARIANT RECORDS◆");
				}
			} else {
				if (isOutputInvariantRows() && !result.getInvariantList().isEmpty()) {
					// created record dump
					dbDump(sb, null, "◆CREATED RECORDS◆");
					// modified record dump
					dbDumpPairs(sb, null, "◆MODIFIED RECORDS◆");
					// deleted record dump
					dbDump(sb, null, "◆DELETED RECORDS◆");
					// invariant record dump
					dbDump(sb, result.getInvariantList(), "◆INVARIANT RECORDS◆");
				} else {
					sb.append("nothing modified table." + LINE_SEPARATOR);
				}
			}
			sb.append(LINE_SEPARATOR);
		}
		System.err.println(sb.toString());
	}

	/**
	 * DB Diff結果をログに出力します。
	 * @parma sb StringBuilder
	 * @param dumpList target list
	 * @param startLog
	 */
	protected void dbDumpPairs(StringBuilder sb,
			List<ModifiedPair<T1, T2>> modifiedList, String startLog) {
		int size = (modifiedList == null || modifiedList.isEmpty()) ? 0 : modifiedList.size();
		sb.append(startLog + " " + size + "件"+ LINE_SEPARATOR);
		if (modifiedList == null || modifiedList.isEmpty()) {
			sb.append("nothing row." + LINE_SEPARATOR + LINE_SEPARATOR);
			return;
		}

		for (ModifiedPair<T1, T2> record : modifiedList) {
			// before elem output
			Map<T1, T2> beforeElem = record.getBeforeElem();
			for (Entry<T1, T2> entry : beforeElem.entrySet()) {
				sb.append(entry.getKey() + ":" + DbDiffUtil.nullToEmpty(entry.getValue()) + LINE_SEPARATOR);
			}
			sb.append("    ↓↓↓" + LINE_SEPARATOR);
			// moddified column output
			Map<T1, T2> modiffiedElem = record.getModifiedElem();
			for (Entry<T1, T2> entry : modiffiedElem.entrySet()) {
				sb.append(entry.getKey() + ":" + DbDiffUtil.nullToEmpty(entry.getValue()) + LINE_SEPARATOR);
			}
			sb.append(LINE_SEPARATOR);
		}
	}

	/**
	 * DB Diff結果をログに出力します。
	 * @parma sb StringBuilder
	 * @param dumpList target list
	 * @param startLog
	 */
	protected void dbDump(StringBuilder sb, List<Map<T1, T2>> dumpList, String startLog) {
		int size = (dumpList == null || dumpList.isEmpty()) ? 0 : dumpList.size();
		sb.append(startLog + " " + size + "件" + LINE_SEPARATOR);
		if (dumpList == null || dumpList.isEmpty()) {
			sb.append("nothing row." + LINE_SEPARATOR + LINE_SEPARATOR);
			return;
		}

		for (Map<T1, T2> record : dumpList) {
			for (Entry<T1, T2> entry : record.entrySet()) {
				sb.append(entry.getKey() + ":" + DbDiffUtil.nullToEmpty(entry.getValue()) + LINE_SEPARATOR);
			}
			sb.append(LINE_SEPARATOR);
		}
		sb.append(LINE_SEPARATOR);
	}

	/**
	 * 試験実行後のDBデータを取得します。<br>
	 * 特別な処理が必要な場合はこのメソッドをオーバーライドします。
	 * @return 試験実行前のDBデータ
	 */
	protected Map<String, Map<String, UniqueEntity<T1, T2>>> getAfterRows() {
		return getAllTablesRow(allTablesPk);
	}

	/**
	 * 全テーブルのPKを取得します。
	 * @return 全テーブルのPK
	 */
	protected Map<String, List<T1>> getAllTablesPk() {
		Map<String, List<T1>> allTablesPk = new HashMap<>();
		for (String tableName : this.tableNameList) {
			List<T1> pkList = rowsGetter.getPkRows(tableName);
			allTablesPk.put(tableName, pkList);
		}
		return allTablesPk;
	}

	/**
	 * 全テーブルの全行を取得します。
	 * @param allTablesPk 全テーブルのPK
	 * @return 全テーブルの全行データ
	 */
	protected Map<String, Map<String, UniqueEntity<T1, T2>>> getAllTablesRow(
			Map<String, List<T1>> allTablesPk) {
		Map<String, Map<String, UniqueEntity<T1, T2>>> allTablesRow = new HashMap<>();
		for (String tableName : this.tableNameList) {
			allTablesRow.put(tableName, rowsGetter.getAllRows(allTablesPk.get(tableName), tableName));
		}
		return allTablesRow;
	}

	/**
	 * Diff結果を取得します。
	 * @param map 試験実行後のDBデータ
	 * @return Diff結果
	 */
	protected Map<String, DiffResult<T1, T2>> getDiffResult(
			Map<String, Map<String, UniqueEntity<T1, T2>>> afterAllTablesRow) {
		// diff
		Map<String, DiffResult<T1, T2>> resultMap = new HashMap<>();
		for (String tableName : tableNameList) {
			resultMap.put(tableName, DbDiffUtil.diff(
					beforeAllTablesRow.get(tableName),
					afterAllTablesRow.get(tableName),
					isOutputInvariantRows()));
		}
		return resultMap;
	}

	/**
	 * 変更の入っていないレコードも出力するかを取得します。
	 * @return 変更の入っていないレコードも出力するか
	 */
	public boolean isOutputInvariantRows() {
		return outputInvariantRows;
	}

	/**
	 * 変更の入っていないレコードも出力するかを設定します。<br>
	 * {@link DbDifferImpl#setReadyToProcess()}を呼び出した後にこのメソッドを呼び出した場合、<br>
	 * {@link IllegalStateException}がスローされます。<br>
	 * 連続して呼び出す場合には{@link DbDifferImpl#clear()}を呼び出して状態をリセットします。
	 * @param outputInvariantRows 変更の入っていないレコードも出力するか
	 */
	public void setOutputInvariantRows(boolean outputInvariantRows) {
		if (processing) {
			throw new IllegalStateException("事前DB取得が行われた状態でフラグが変更されました。");
		} else {
			this.outputInvariantRows = outputInvariantRows;
		}
	}

	/**
	 * 実行中フラグをリセットします。
	 */
	public void clear() {
		processing = false;
	}
}
