package jp.ac.fun.db.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import jp.ac.fun.db.data.ColumnConverter;
import jp.ac.fun.db.data.DiffResult;
import jp.ac.fun.db.data.NullToEmptyConverter;
import jp.ac.fun.db.data.UniqueEntity;
import jp.ac.fun.db.data.UniqueEntityImpl;

/**
 * DB差分比較ユーティリティ
 */
public class DbDiffUtil {

	/** コンストラクタ */
	private DbDiffUtil() {};

	/**
	 * {@code map1}と{@code map2}のdiffを取り、結果を返します。<br>
	 * 変更の無かった行データは結果に格納されません。<br>
	 * 変更の無かった行データを結果に格納するには{@link DbDiffUtil#diff(Map, Map, boolean, ColumnConverter)}を使用して下さい。<br>
	 * デフォルトで{@link NullToEmptyConverter}が使用されます。
	 * @param <T1> カラムを一意に識別する値の型
	 * @param <T2> カラム値の型
	 * @param map1 比較対象マップ1
	 * @param map2 比較対象マップ2
	 * @return diff結果
	 */
	public static <T1, T2> DiffResult<T1, T2> diff(Map<String, UniqueEntity<T1, T2>> map1,
			Map<String, UniqueEntity<T1, T2>> map2) {
		return diff(map1, map2, false, new NullToEmptyConverter());
	}

	/**
	 * {@code map1}と{@code map2}のdiffを取り、結果を返します。<br>
	 * 変更の無かった行データは結果に格納されません。<br>
	 * 変更の無かった行データを結果に格納するには{@link DbDiffUtil#diff(Map, Map, boolean, ColumnConverter)}を使用して下さい。
	 * @param <R> コンバーターの返す値の型
	 * @param <T1> カラムを一意に識別する値の型
	 * @param <T2> カラム値の型
	 * @param map1 比較対象マップ1
	 * @param map2 比較対象マップ2
	 * @param converteres カラムコンバーター
	 * @return diff結果
	 */
	@SafeVarargs
	public static <R, T1, T2> DiffResult<T1, T2> diff(Map<String, UniqueEntity<T1, T2>> map1,
			Map<String, UniqueEntity<T1, T2>> map2,
			ColumnConverter<R>... converteres) {
		return diff(map1, map2, false, converteres);
	}

	/**
	 * {@code map1}と{@code map2}のdiffを取り、結果を返します。<br>
	 * 変更の無かった行データは結果に格納されません。<br>
	 * 変更の無かった行データを結果に格納するには{@code cacheInvariantData}を{@code true}に設定します。
	 * @param <R> コンバーターの返す値の型
	 * @param <T1> カラムを一意に識別する値の型
	 * @param <T2> カラム値の型
	 * @param map1 比較対象マップ1
	 * @param map2 比較対象マップ2
	 * @param cacheInvariantData 変更のなかった行データを結果に格納するかどうか
	 * @param converter カラムコンバーター
	 * @return diff結果
	 */
	@SafeVarargs
	public static <R, T1, T2> DiffResult<T1, T2> diff(Map<String, UniqueEntity<T1, T2>> map1,
			Map<String, UniqueEntity<T1, T2>> map2,
			boolean cacheInvariantData, ColumnConverter<R>... converter) {
		DiffResult<T1, T2> result = new DiffResult<>();

		Map<String, UniqueEntity<T1, T2>> innerMap1 = convert(map1, converter);
		Map<String, UniqueEntity<T1, T2>> innerMap2 = convert(map2, converter);
		if (innerMap1 == null || innerMap2 == null) {
			return result;
		}
		for (Entry<String, UniqueEntity<T1, T2>> entry : innerMap1.entrySet()) {
			if (innerMap2.containsKey(entry.getKey())) {
				// exists key

				// delete for innerMap2
				UniqueEntity<T1, T2> map2Entity = innerMap2.remove(entry.getKey());
				Map<T1, T2> diffMap = getChangeColumn(entry.getValue(), map2Entity);
				if (diffMap != null && !diffMap.isEmpty()) {
					// modified
					result.addModifiedList(entry.getValue().getEntity(), diffMap);
				} else if (cacheInvariantData) {
					// not changed and cache
					result.addInvariantList(entry.getValue().getEntity());
				} else {
					// not changed and not cache
					result.incrementInvariantCount();
				}
			} else {
				// deleted
				result.addDeletedList(entry.getValue().getEntity());
			}
		}
		// map1 serach finished
		for (Entry<String, UniqueEntity<T1, T2>> entry : innerMap2.entrySet()) {
			// added
			result.addAddedList(entry.getValue().getEntity());
		}
		return result;
	}

	/**
	 * {@code converter}を使用して新しいMapを作成します。<br>
	 * {@code converter}が{@code null}の場合には何も変換されず、元の値を含むMapが返却されます。
	 * @param <R> コンバーターの返す値の型
	 * @param <T1> カラムを一意に識別する値の型
	 * @param <T2> カラム値の型
	 * @param map 元のMap
	 * @param converteres コンバーター
	 * @return 変換後のMap
	 */
	@SafeVarargs
	private static <R, T1, T2> Map<String, UniqueEntity<T1, T2>> convert(
			Map<String, UniqueEntity<T1, T2>> map, ColumnConverter<R>... converteres) {
		if (converteres == null) {
			return new HashMap<>(map);
		} else if (map == null) {
			return null;
		}

		Map<String, UniqueEntity<T1, T2>> result = new HashMap<>(map);
		for (Entry<String, UniqueEntity<T1, T2>> entry1 : result.entrySet()) {
			String tableName = entry1.getValue().getTableName();
			Map<T1, T2> entities = entry1.getValue().getEntity();
			for (Entry<T1, T2> entry2 : entities.entrySet()) {
				T1 id = entry2.getKey();
				T2 convertedValue = CastUtil.cast(applyConvert(converteres, tableName, id, entry2.getValue()));
				if (entry2.getValue() != convertedValue) {
					// over write
					result.get(entry1.getKey()).getEntity().put(id, convertedValue);
				}
			}

		}
		return result;
	}

	/**
	 * 複数のコンバーターにより{@code value}を変換します。
	 * @param converteres コンバーター
	 * @param tableName テーブル名
	 * @param key カラム名を識別する値
	 * @param value カラム値
	 * @return 変換後の値
	 */
	private static <R, T1, T2> R applyConvert(ColumnConverter<R>[] converteres,
			String tableName, T1 key, T2 value) {
		T2 result = value;
		for (ColumnConverter<R> converter : converteres) {
			R convertedValue = converter.convert(tableName, key, result);
			if (convertedValue != null) {
				result = CastUtil.cast(convertedValue);
			}
		}
		return CastUtil.cast(result);
	}

	/**
	 * 変更が入ったかどうか比較します。<br>
	 * 変更が入っていなかった場合には{@code null}を返します。
	 * @param <T1> カラムを一意に識別する値の型
	 * @param <T2> カラム値の型
	 * @param entity1 対象データ1
	 * @param entity2 対象データ2
	 * @return 変更が入ったかどうか
	 */
	private static <T1, T2> Map<T1, T2> getChangeColumn(UniqueEntity<T1, T2> entity1,
			UniqueEntity<T1, T2> entity2) {
		boolean equal = entity1.getEntity().equals(entity2.getEntity());
		if (equal) {
			return null;
		}
		Map<T1, T2> modiffiedDiff = new HashMap<>();
		Map<T1, T2> beforeEntity = entity1.getEntity();
		Map<T1, T2> afterEntity = entity2.getEntity();
		Set<T1> checkedKey = new HashSet<>();
		for (Entry<T1, T2> entry : beforeEntity.entrySet()) {
			T2 value = afterEntity.get(entry.getKey());
			if (!equals(entry.getValue(), value)) {
				modiffiedDiff.put(entry.getKey(), value);
			}
			checkedKey.add(entry.getKey());
		}
		// 項目が前後で違っている場合に対応
		for (Entry<T1, T2> entry : afterEntity.entrySet()) {
			if (checkedKey.contains(entry.getKey())) {
				continue;
			}
			T2 value = beforeEntity.get(entry.getKey());
			if (!equals(entry.getValue(), value)) {
				modiffiedDiff.put(entry.getKey(), entry.getValue());
			}
		}
		return modiffiedDiff;
	}

	/**
	 * オブジェクト同士の比較を行います。<br>
	 * 比較は{@link T2#equals(Object)}を使用し、等しい場合に{@code true}を返します。
	 * @param obj1 比較オブジェクト1
	 * @param obj2 比較オブジェクト2
	 * @return 比較結果
	 */
	private static <T2> boolean equals(T2 obj1, T2 obj2) {
		if (obj1 == null && obj2 == null) {
			return true;
		}
		if (obj1 == null || obj2 == null) {
			return false;
		}
		return obj1.equals(obj2);
	}

	/**
	 * 対象{@code obj}が{@code null}の場合、空文字を返します。<br>
	 * {@code null}ではない場合は{@link Object#toString()}した結果を返します。
	 * @param obj 対象オブジェクト
	 * @return 変換値
	 */
	public static String nullToEmpty(Object obj) {
		if (obj == null) {
			return "";
		}
		return obj.toString();
	}

	/**
	 * DBからの全行戻り値：{@code List<Map<T1, T2>>}から{@code Map<String, UniqueEntity<T1, T2>}に変換します。
	 * @param <T1> カラムを一意に識別する値の型
	 * @param <T2> カラム値の型
	 * @param tableName テーブル名
	 * @param pkList PKリスト
	 * @param allRowList 全行戻り値
	 * @return 変換後データリスト
	 */
	public static <T1, T2> Map<String, UniqueEntity<T1, T2>> convertUniqueEntities(String tableName, List<T1> pkList,
			List<Map<T1, T2>> allRowList) {

		Map<String, UniqueEntity<T1, T2>> result = new HashMap<>();
		for (Map<T1, T2> rowMap : allRowList) {
			UniqueEntityImpl<T1, T2> entity = new UniqueEntityImpl<>(tableName, pkList, rowMap);
			result.put(entity.toString(), entity);
		}
		return result;
	}
}
