package jp.ac.fun.db.util;

import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import jp.ac.fun.db.data.DiffResult;
import jp.ac.fun.db.data.ModifiedPair;

/**
 * Diff結果をXML出力します。
 */
public class SimpleDiffOutputXml {

    /**
     * DB Diff結果：{@code diffResult}をXMLとして出力します。
     * @param diffResult DB Diff結果
     * @return XML文字列
     * @throws XMLStreamException XML生成例外
     */
    public static <T1, T2> StringWriter outputXML(Map<String, DiffResult<T1, T2>> diffResult) throws XMLStreamException {
        return outputXML(diffResult, null);
    }

    /**
     * DB Diff結果：{@code diffResult}をXMLとして出力します。
     * @param diffResult DB Diff結果
     * @param xslPath xslのパス
     * @return XML文字列
     * @throws XMLStreamException XML生成例外
     */
    public static <T1, T2> StringWriter outputXML(Map<String, DiffResult<T1, T2>> diffResult, String xslPath) throws XMLStreamException {
        XMLOutputFactory factory = XMLOutputFactory.newInstance();

        StringWriter stringWriter = new StringWriter();
        XMLStreamWriter writer = factory.createXMLStreamWriter(stringWriter);
        try {
            writer.writeStartDocument("UTF-8", "1.0");
            if (xslPath != null) {
                writer.writeProcessingInstruction("xml-stylesheet", "type=\"text/xsl\" href=\"" + xslPath + "\"");
            }
            writer.writeDTD("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" "
                            + "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">");

            writer.writeStartElement("DIFF_DATA");
            writer.writeStartElement("TABLES");
            for (Entry<String, DiffResult<T1, T2>> entry : diffResult.entrySet()) {
                writer.writeStartElement("TABLE");
                writer.writeStartElement("NAME");
                writer.writeCharacters(entry.getKey());
                writer.writeEndElement();

                DiffResult<T1, T2> diff = entry.getValue();
                writer.writeStartElement("BEFORE_RECORD_COUNT");
                writer.writeCharacters(String.valueOf(diff.getBeforeCount()));
                writer.writeEndElement();

                writer.writeStartElement("AFTER_RECORD_COUNT");
                writer.writeCharacters(String.valueOf(diff.getAfterCount()));
                writer.writeEndElement();

                writer.writeStartElement("IS_DIFF");
                writer.writeCharacters(String.valueOf(diff.isDiff()));
                writer.writeEndElement();

                writer.writeStartElement("DIFFES");

                writeDiff(writer, diff.getAddedList(), "CREATED");
                writeModifiedDiff(writer, diff.getModifiedList(), "MODIFIED");
                writeDiff(writer, diff.getDeletedList(), "DELETED");
                writeDiff(writer, diff.getInvariantList(), "INVARIANT", diff.getInvariantCount());

                writer.writeEndElement();
                writer.writeEndElement();
            }
            writer.writeEndElement();
            writer.writeEndElement();

            writer.writeEndDocument();

            return stringWriter;
        } finally {
            writer.close();
        }
    }

    /**
     * 差分部分を出力します。
     * @param writer XMLStreamWriter
     * @param list 差分リスト
     * @param type 差分タイプ
     * @throws XMLStreamException XML生成例外
     */
    private static <T1, T2> void writeDiff(XMLStreamWriter writer, List<Map<T1, T2>> list, String type) throws XMLStreamException {
        writeDiff(writer, list, type, list.size());
    }

    /**
     * 差分部分を出力します。
     * @param writer XMLStreamWriter
     * @param list 差分リスト
     * @param type 差分タイプ
     * @parma boolean レコード数
     * @throws XMLStreamException XML生成例外
     */
    private static <T1, T2> void writeDiff(XMLStreamWriter writer, List<Map<T1, T2>> list,
            String type, long listSize) throws XMLStreamException {
        writer.writeStartElement("DIFF");
        writer.writeStartElement("TYPE");
        writer.writeCharacters(type);
        writer.writeEndElement();

        writer.writeStartElement("RECORD_COUNT");
        writer.writeCharacters(String.valueOf(listSize));
        writer.writeEndElement();

        for (Map<T1, T2> elem : list) {
            writer.writeStartElement("RECORD");

            for (Entry<T1, T2> entry : elem.entrySet()) {
                writer.writeStartElement(entry.getKey().toString());
                writer.writeCharacters(DbDiffUtil.nullToEmpty(entry.getValue()));
                writer.writeEndElement();
            }
            writer.writeEndElement();
        }
        writer.writeEndElement();
    }

    /**
     * 差分部分を出力します。
     * @param writer XMLStreamWriter
     * @param list 差分リスト
     * @param type 差分タイプ
     * @throws XMLStreamException XML生成例外
     */
    private static <T1, T2> void writeModifiedDiff(XMLStreamWriter writer, List<ModifiedPair<T1, T2>> list, String type) throws XMLStreamException {
        writer.writeStartElement("DIFF");
        writer.writeStartElement("TYPE");
        writer.writeCharacters(type);
        writer.writeEndElement();

        writer.writeStartElement("RECORD_COUNT");
        writer.writeCharacters(String.valueOf(list.size()));
        writer.writeEndElement();

        for (ModifiedPair<T1, T2> elem : list) {
            writer.writeStartElement("RECORD");

            Map<T1, T2> beforeElem = elem.getBeforeElem();
            Map<T1, T2> modifiedElem = elem.getModifiedElem();
            for (Entry<T1, T2> entry : beforeElem.entrySet()) {
                writer.writeStartElement(entry.getKey().toString());
                if (modifiedElem.containsKey(entry.getKey())) {
                    writer.writeStartElement("BEFORE");
                    writer.writeCharacters(DbDiffUtil.nullToEmpty(entry.getValue()));
                    writer.writeEndElement();

                    writer.writeStartElement("AFTER");
                    writer.writeCharacters(modifiedElem.get(entry.getKey()).toString());
                    writer.writeEndElement();
                } else {
                    writer.writeCharacters(DbDiffUtil.nullToEmpty(entry.getValue()));
                }
                writer.writeEndElement();
            }
            writer.writeEndElement();
        }
        writer.writeEndElement();
    }
}
