package org.dbunitng.assertion;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Comparator;

import junit.framework.Assert;

import org.dbunit.Assertion;
import org.dbunit.DatabaseUnitException;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.excel.XlsDataSet;
import org.dbunit.dataset.filter.DefaultColumnFilter;
import org.dbunit.dataset.xml.FlatXmlDataSet;
import org.dbunitng.annotations.FileType;
import org.dbunitng.exception.DbUnitNGRuntimeException;
import org.dbunitng.util.ResourceUtil;

/**
 * アサーションヘルパークラス。
 * 
 * @author jyukutyo
 * 
 */
public class AssertionHelper {

	/**
	 * hide constructor.
	 */
	protected AssertionHelper() {
	// empty
	}

	/**
	 * <p>
	 * ファイルに記述したカラムのみ比較する。
	 * </p>
	 * 期待値を記述したファイルは、引数のクラスがあるパッケージの以下のサブディレクトリに格納したファイルとなる。
	 * 
	 * <pre>
	 * assertEqualsOnlyColumnsInFile(actual, org.dbunitng.test.Sample.class, &quot;sub1/sub2&quot;, &quot;test.xml&quot;) =&gt; org/dbunitng/test/sub1/sub2/test.xml
	 * </pre>
	 * 
	 * @param actual
	 *            実際の結果
	 * @param clazz
	 *            テストクラス
	 * @param subDirectoryPath
	 *            サブディレクトリのパス
	 * @param fileName
	 *            ファイル名
	 * @throws DatabaseUnitException
	 */
	public static void assertEqualsOnlyColumnsInFile(IDataSet actual,
			Class<?> clazz, String subDirectoryPath, String fileName)
			throws DatabaseUnitException {
		String pathName =
			ResourceUtil
				.replacePackageToDirectory(clazz.getPackage().getName())
				+ addDelimiter(subDirectoryPath) + addDelimiter(fileName);

		assertEqualsOnlyColumnsInFile(actual, pathName);

	}

	/**
	 * デリミタがなければデリミタを付与して返す。
	 * 
	 * @param str
	 * @return デリミタを付与した文字列
	 */
	protected static String addDelimiter(String str) {
		return str.startsWith("/") ? str : "/" + str;
	}

	/**
	 * <p>
	 * ファイルに記述したカラムのみ比較する。
	 * </p>
	 * 期待値を記述したファイルは、引数のクラスがあるパッケージに格納したファイルとなる。
	 * 
	 * <pre>
	 * assertEqualsOnlyColumnsInFile(actual, org.dbunitng.test.Sample.class, &quot;test.xml&quot;) =&gt; org/dbunitng/test/test.xml
	 * </pre>
	 * 
	 * @param actual
	 *            実際の結果
	 * @param clazz
	 *            テストクラス
	 * @param fileName
	 *            ファイル名
	 * @throws DatabaseUnitException
	 */
	public static void assertEqualsOnlyColumnsInFile(IDataSet actual,
			Class<?> clazz, String fileName) throws DatabaseUnitException {

		String pathName =
			ResourceUtil
				.replacePackageToDirectory(clazz.getPackage().getName())
				+ addDelimiter(fileName);

		assertEqualsOnlyColumnsInFile(actual, pathName);
	}

	/**
	 * <p>
	 * ファイルに記述したカラムのみ比較する。
	 * </p>
	 * 
	 * @param actual
	 *            実際の結果
	 * @param pathName
	 *            ファイルのパス
	 * @throws DatabaseUnitException
	 */
	public static void assertEqualsOnlyColumnsInFile(IDataSet actual,
			String pathName) throws DatabaseUnitException {

		InputStream stream = ResourceUtil.getResourceAsStream(pathName);

		FileType type =
			ResourceUtil.toFileType(ResourceUtil.getExtension(pathName));

		IDataSet expected;
		try {
			if (FileType.EXCEL == type) {
				expected = new XlsDataSet(stream);
			} else {
				expected = new FlatXmlDataSet(stream);
			}
		} catch (IOException e) {
			throw new DbUnitNGRuntimeException(e);
		}

		assertEquals(expected, actual);
	}

	/**
	 * Asserts that the two specified dataset are equals. This method ignore the
	 * tables order.
	 */
	protected static void assertEquals(IDataSet expectedDataSet,
			IDataSet actualDataSet) throws DatabaseUnitException {

		// do not continue if same instance
		if (expectedDataSet == actualDataSet) {
			return;
		}

		String[] expectedNames = getSortedUpperTableNames(expectedDataSet);
		String[] actualNames = getSortedUpperTableNames(actualDataSet);

		// tables count
		Assert.assertEquals(
			"table count",
			expectedNames.length,
			actualNames.length);

		// table names in no specific order
		for (int i = 0; i < expectedNames.length; i++) {
			if (!actualNames[i].equals(expectedNames[i])) {
				Assert.fail("expected tables " + Arrays.asList(expectedNames)
					+ " but was " + Arrays.asList(actualNames));
			}

		}

		// tables
		for (int i = 0; i < expectedNames.length; i++) {
			String name = expectedNames[i];

			// change to assert only columns in expected table
			ITable expectedTable = expectedDataSet.getTable(name);
			ITable actualTable =
				DefaultColumnFilter.includedColumnsTable(actualDataSet
					.getTable(name), expectedTable
					.getTableMetaData()
					.getColumns());
			Assertion.assertEquals(expectedTable, actualTable);
		}
	}

	////////////////////////////////////////////////////////////////////////////
	// below methods and class are copies of DbUnit's Assertion class
	// because of their private modifier.

	protected static final ColumnComparator COLUMN_COMPARATOR =
		new ColumnComparator();

	protected static Column[] getSortedColumns(ITableMetaData metaData)
			throws DataSetException {

		Column[] columns = metaData.getColumns();
		Column[] sortColumns = new Column[columns.length];
		System.arraycopy(columns, 0, sortColumns, 0, columns.length);
		Arrays.sort(sortColumns, COLUMN_COMPARATOR);
		return sortColumns;
	}

	protected static class ColumnComparator implements Comparator<Column> {

		public int compare(Column column1, Column column2) {

			String columnName1 = column1.getColumnName();
			String columnName2 = column2.getColumnName();
			return columnName1.compareToIgnoreCase(columnName2);
		}
	}

	protected static String[] getSortedUpperTableNames(IDataSet dataSet)
			throws DataSetException {

		String[] names = dataSet.getTableNames();
		for (int i = 0; i < names.length; i++) {
			names[i] = names[i].toUpperCase();
		}
		Arrays.sort(names);
		return names;
	}
}
