package org.dbunitng.util;

import java.io.InputStream;

import org.dbunitng.annotations.FileType;
import org.dbunitng.exception.TestDataFileNotFoundException;

/**
 * リソース関連のユーティリティ.
 * 
 * @author jyukutyo
 * 
 */
public class ResourceUtil {

	/**
	 * 拡張子からFileTypeを判断する.
	 * 
	 * @param extension
	 *            .なしの拡張子
	 * @return FileType
	 */
	public static FileType toFileType(String extension) {

		if (extension == null) {
			throw new IllegalArgumentException("Argument is null.");
		}

		FileType[] types = FileType.values();
		for (FileType type : types) {
			if (extension.toLowerCase().startsWith(type.toString())) {
				return type;
			}
		}
		return FileType.valueOf(extension);
	}

	/**
	 * 拡張子を返す.
	 * 
	 * @param pathName
	 *            パス
	 * @return 拡張子
	 */
	public static String getExtension(String pathName) {

		if (pathName == null) {
			throw new IllegalArgumentException("Argument is null.");
		}

		return pathName.substring(pathName.lastIndexOf('.') + 1);
	}

	/**
	 * リソースをストリームで返す.
	 * 
	 * @param pathName
	 *            パス
	 * @return リソース
	 */
	public static InputStream getResourceAsStream(String pathName) {
		InputStream stream =
			Thread.currentThread().getContextClassLoader().getResourceAsStream(
				pathName);
		if (stream == null) {
			throw new TestDataFileNotFoundException(pathName + " is not found.");
		}
		return stream;
	}

	/**
	 * パッケージ記述をディレクトリ階層に変換する.
	 * 
	 * <pre>
	 * replacePackageToDirectory(&quot;org.dbunitng.util.ResourceUtil&quot;) =
	 * 	&quot;org/dbunitng/util/ResourceUtil&quot;
	 * </pre>
	 * 
	 * @param fqcn
	 *            「.」区切りの文字列(完全修飾名など)
	 * @return 「/」区切りの文字列
	 */
	public static String replacePackageToDirectory(String fqcn) {
		return fqcn.replace('.', '/');
	}

	/**
	 * パッケージ記述と拡張子からファイルパスを返す.
	 * 
	 * <pre>
	 * toFilePath(&quot;aa.bbb.Ccc&quot;, &quot;xml&quot;) = &quot;aa/bbb/Ccc.xml&quot;
	 * </pre>
	 * 
	 * @param fqcn
	 *            「.」区切りの文字列(完全修飾名など)
	 * @param extension
	 *            拡張子
	 * @return ファイルパス
	 */
	public static String toFilePath(String fqcn, String extension) {
		String suffix =
			extension.indexOf('.') == -1 ? "." + extension : extension;
		return replacePackageToDirectory(fqcn) + suffix;
	}

	/**
	 * リソースをストリームで返す。
	 * 
	 * <p>
	 * getResourceAsStreamFromPackage(&quot;aa.bbb.Ccc&quot;,
	 * &quot;xml&quot;)であればaa/bbb/Ccc.xmlをストリームで返す。
	 * </p>
	 * 
	 * @param fqcn
	 *            「.」区切りの文字列(完全修飾名など)
	 * @param extension
	 *            拡張子
	 * @return リソース
	 */
	public static InputStream getResourceAsStreamFromPackage(String fqcn,
			String extension) {
		return getResourceAsStream(toFilePath(fqcn, extension));
	}
}
