package org.dbunitng.listeners;

import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.sql.SQLException;

import org.apache.commons.dbcp.BasicDataSource;
import org.apache.commons.lang.StringUtils;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.excel.XlsDataSet;
import org.dbunit.dataset.xml.FlatXmlDataSet;
import org.dbunitng.annotations.DatabaseOperationType;
import org.dbunitng.annotations.FileType;
import org.dbunitng.annotations.SetUpOperation;
import org.dbunitng.annotations.TearDownOperation;
import org.dbunitng.exception.DbUnitNGRuntimeException;
import org.dbunitng.exception.TestDataFileNotFoundException;
import org.dbunitng.listeners.internal.DbUnitNGConfig;
import org.dbunitng.listeners.internal.DbUnitNGDatabaseOperation;
import org.dbunitng.util.ResourceUtil;
import org.testng.ITestContext;
import org.testng.ITestListener;
import org.testng.ITestNGMethod;
import org.testng.ITestResult;
import org.testng.log4testng.Logger;

/**
 * TestNGとDbUnitを連携するリスナー。
 * 
 * @author jyukutyo
 * 
 */
public class DbUnitNGTestListener implements ITestListener {

	/** This class' log4testng Logger. */
	private static final Logger LOGGER =
		Logger.getLogger(DbUnitNGTestListener.class);

	/** DBCPを利用した場合のデータソース */
	private BasicDataSource dataSource;

	/** DB接続情報 */
	private DbUnitNGConfig config;

	/**
	 * テストスイート開始時のコールバックメソッド。コンテキストからDB接続情報を取得する.
	 */
	public void onStart(ITestContext context) {

		config = DbUnitNGConfig.create(context);
		config.verifyParamsNotNull();

		if (config.isDbcp()) {
			BasicDataSource source = new BasicDataSource();
			source.setDriverClassName(config.getDriver());
			source.setUrl(config.getUrl());
			source.setUsername(config.getUserName());
			source.setPassword(config.getPassword());
			dataSource = source;
			LOGGER.debug("DBCP DataSource is created.");
		}

	}

	/**
	 * テストメソッドのアノテーションを返す。
	 * 
	 * @param <T>
	 *            アノテーション
	 * @param result
	 * @param annotationClass
	 *            アノテーションクラス
	 * @return アノテーション
	 */
	protected <T extends Annotation> T getAnnotation(ITestResult result,
			Class<T> annotationClass) {

		ITestNGMethod testNGMethod = result.getMethod();
		Method method = testNGMethod.getMethod();

		return method.getAnnotation(annotationClass);

	}

	/**
	 * テストメソッド実行前のコールバックメソッド。SetUp処理を実行する。
	 */
	public void onTestStart(ITestResult result) {

		SetUpOperation setUpOperation =
			this.getAnnotation(result, SetUpOperation.class);

		if (setUpOperation == null) {
			return;
		}
		DatabaseOperationType type = setUpOperation.value();
		if (type == DatabaseOperationType.NONE) {
			return;
		}
		String pathName = setUpOperation.pathname();
		readFileForDatabase(result, type, pathName);
	}

	/**
	 * ファイルを読み込み、データベースに反映する。
	 * 
	 * @param result
	 *            ITestResult
	 * @param type
	 *            データベースへの処理方法
	 * @param pathName
	 *            ファイルパス
	 */
	protected void readFileForDatabase(ITestResult result,
			DatabaseOperationType type, String pathName) {
		String extension;
		if (StringUtils.isBlank(pathName)) {
			// case file path is not specified
			throw new TestDataFileNotFoundException();
		} else if (pathName.indexOf('/') < 0) {
			// case only file name is specified

			extension = ResourceUtil.getExtension(pathName);
			pathName =
				ResourceUtil.replacePackageToDirectory(result
					.getTestClass()
					.getRealClass()
					.getPackage()
					.getName())
					+ "/" + pathName;

		} else {
			// case file path is fully specified
			extension = ResourceUtil.getExtension(pathName);
		}

		InputStream stream = ResourceUtil.getResourceAsStream(pathName);
		operateDatabase(type, stream, ResourceUtil.toFileType(extension));
	}

	/**
	 * データベースへストリームの内容を反映する。
	 * 
	 * @param type
	 *            データベースへの処理方法
	 * @param stream
	 *            ストリーム
	 * @param fileType
	 *            ファイルタイプ(Xls,XML)
	 */
	protected void operateDatabase(DatabaseOperationType type,
			InputStream stream, FileType fileType) {

		if (type == DatabaseOperationType.NONE) {
			return;
		}

		try {

			IDataSet dataSet;
			if (FileType.EXCEL == fileType) {
				dataSet = new XlsDataSet(stream);
			} else if (FileType.XML == fileType) {
				dataSet = new FlatXmlDataSet(stream);
			} else {
				throw new DbUnitNGRuntimeException("FileType is unknown. "
					+ fileType);
			}

			DbUnitNGDatabaseOperation operation;
			if (config.isDbcp()) {
				LOGGER.debug("use DBCP.");
				operation = new DbUnitNGDatabaseOperation(dataSource);
			} else {
				LOGGER.debug("use DriverManager.");
				operation = new DbUnitNGDatabaseOperation(config);
			}
			operation.execute(type, dataSet);

		} catch (DataSetException e) {
			throw new DbUnitNGRuntimeException(e);
		} catch (IOException e) {
			throw new DbUnitNGRuntimeException(e);
		}
	}

	/**
	 * テストスイート終了時のコールバックメソッド。データソース利用時にクローズする。
	 */
	public void onFinish(ITestContext context) {
		if (dataSource != null) {
			try {
				dataSource.close();
			} catch (SQLException e) {
				// empty
			}
		}
	}

	/**
	 * TearDown処理を実行する。
	 * 
	 * @param result
	 */
	protected void onTestFinishWhateverHappens(ITestResult result) {

		TearDownOperation tearDownOperation =
			this.getAnnotation(result, TearDownOperation.class);

		if (tearDownOperation == null) {
			return;
		}
		DatabaseOperationType type = tearDownOperation.value();
		String pathName = tearDownOperation.pathname();
		readFileForDatabase(result, type, pathName);

	}

	public void onTestFailedButWithinSuccessPercentage(ITestResult result) {
		onTestFinishWhateverHappens(result);
	}

	public void onTestFailure(ITestResult result) {
		onTestFinishWhateverHappens(result);
	}

	public void onTestSuccess(ITestResult result) {
		onTestFinishWhateverHappens(result);
	}

	public void onTestSkipped(ITestResult result) {
	// empty
	}
}
