/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

import net.morilib.sh.misc.XtraceStream;

public class ShTreeCase implements ShTree {

	private List<ShPattern> patterns;
	private List<ShTree> block;
	private ShToken variable;

	public ShTreeCase(ShToken variable, List<ShPattern> patterns,
			List<ShTree> block) {
		this.variable = variable;
		this.patterns = new ArrayList<ShPattern>(patterns);
		this.block    = new ArrayList<ShTree>(block);
	}

	public int eval(ShEnvironment env,
			ShFileSystem fs,
			ShBuiltInCommands cmds,
			ShRuntime run,
			InputStream in,
			PrintStream out,
			PrintStream err,
			XtraceStream p) throws IOException, ShSyntaxException {
		String s = ShTrees.substitute(env, run, fs, err, p, variable);

		for(int i = 0; i < patterns.size(); i++) {
			if(patterns.get(i).matches(env, fs, run, err, p, s)) {
				return block.get(i).eval(env, fs, cmds, run, in, out,
						err, p);
			}
		}
		return 0;
	}

	public void compileInternally(ShTreeExpressionMachine.Builder b,
			Object brk, Object cnt) {
		Object l1 = null, l2 = new Object();

		for(int i = 0; i < patterns.size(); i++) {
			if(i > 0)  b.addLabel(l1);
			b.addPtn(variable, patterns.get(i));
			b.addJmpZ(l1 = new Object());
			block.get(i).compileInternally(b, brk, cnt);
			b.addJmp(l2);
		}
		b.addLabel(l1);
		b.addLabel(l2);
		b.addRes();
	}

	public String toString() {
		StringBuffer b = new StringBuffer("case ");

		b.append(variable).append(" in\n");
		for(int i = 0; i < block.size(); i++) {
			b.append(patterns.get(i));
			b.append(")\n").append(block.get(i)).append(";;\n");
		}
		return b.append("esac").toString();
	}

}
