/*
 * Copyright (C) 2007 uguu at users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.deployer;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * <p>
 * ファイルの作成、更新、削除を監視し、イベントを通知します。
 * </p>
 * <p>
 * {@link #monitor()}メソッドの呼び出しでファイルを一回監視することが出来ます。ファイルを定期的に監視する場合、定期的に{@link #monitor()}メソッドを呼び出してください。
 * </p>
 * 
 * @author $Author$
 * @version $Rev$
 */
public final class FileMonitor {

    /**
     * <p>
     * リスナーのリスト。
     * </p>
     */
    private List<FileMonitorListener>       _listenerList      = new ArrayList<FileMonitorListener>();

    /**
     * <p>
     * 監視する基底ディレクトリ。
     * </p>
     */
    private File                            _baseDirectory;

    /**
     * <p>
     * 監視するファイルの正規表現パターン。
     * </p>
     */
    private Pattern                         _filePattern;

    /**
     * <p>
     * 監視中ファイルの情報。
     * </p>
     */
    private Map<String, MonitoringFileInfo> _monitoringFileMap = new HashMap<String, MonitoringFileInfo>();

    /**
     * <p>
     * ファイルの長さを比較するか。
     * </p>
     */
    private boolean                         _checkLength       = true;

    /**
     * <p>
     * ファイルの最終更新日時を比較するか。
     * </p>
     */
    private boolean                         _checkLastModified = true;

    /**
     * <p>
     * ファイルの内容を確認するか。
     * </p>
     */
    private boolean                         _checkContent;

    /**
     * <p>
     * ハッシュ・アルゴリズム。
     * </p>
     */
    private String                          _hashAlgorithm     = "MD5";

    /**
     * <p>
     * 指定したディレクトリの指定したパターンに合致するファイルを監視する{@link FileMonitor}インスタンスを初期化します。
     * </p>
     * 
     * @param baseDirectory
     *            監視するディレクトリ。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。<br>
     *            ディレクトリを表していない場合、{@link IllegalArgumentException}例外をスローします。
     * @param filePattern
     *            監視するファイルのパターンを表す正規表現。絶対パスと比較されます。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。
     */
    public FileMonitor(File baseDirectory, Pattern filePattern) {
        if (baseDirectory == null) {
            throw new IllegalArgumentException("baseDirectory is null.");
        }
        if (!baseDirectory.isDirectory()) {
            throw new IllegalArgumentException("baseDirectory is not directory.");
        }
        if (filePattern == null) {
            throw new IllegalArgumentException("filePattern is null.");
        }
        this._baseDirectory = baseDirectory;
        this._filePattern = filePattern;
    }

    /**
     * <p>
     * 指定したディレクトリの全てのファイルを監視する{@link FileMonitor}インスタンスを初期化します。
     * </p>
     * <p>
     * このコンストラクタ呼び出しは、以下のコードと等価です。
     * </p>
     * 
     * <pre>
     * new FileMonitor(baseDirectory, Pattern.compile(&quot;.*&quot;))
     * </pre>
     * 
     * @param baseDirectory
     *            監視するディレクトリ。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。<br>
     *            ディレクトリを表していない場合、{@link IllegalArgumentException}例外をスローします。
     */
    public FileMonitor(File baseDirectory) {
        this(baseDirectory, Pattern.compile(".*"));
    }

    /**
     * <p>
     * ファイル・サイズを監視するかどうかを取得します。初期値はtrueです。
     * </p>
     * 
     * @return ファイル・サイズを監視する場合はtrue、監視しない場合はfalse。
     */
    public boolean isCheckLength() {
        return this._checkLength;
    }

    /**
     * <p>
     * ファイル・サイズを監視するかどうかを設定します。
     * </p>
     * 
     * @param checkLength
     *            ファイル・サイズを監視する場合はtrue、監視しない場合はfalse。
     */
    public void setCheckLength(boolean checkLength) {
        this._checkLength = checkLength;
    }

    /**
     * <p>
     * 最終更新日時を監視するかどうかを取得します。初期値はtrueです。
     * </p>
     * 
     * @return 最終更新日時を監視する場合はtrue、監視しない場合はfalse。
     */
    public boolean isCheckLastModified() {
        return this._checkLastModified;
    }

    /**
     * <p>
     * 最終更新日時を監視するかどうかを設定します。
     * </p>
     * 
     * @param checkLastModified
     *            最終更新日時を監視する場合はtrue、監視しない場合はfalse。
     */
    public void setCheckLastModified(boolean checkLastModified) {
        this._checkLastModified = checkLastModified;
    }

    /**
     * <p>
     * ファイルの内容(ハッシュ値)を監視するかどうかを取得します。初期値はfalseです。
     * </p>
     * 
     * @return ファイルの内容を監視する場合はtrue、監視しない場合はfalse。
     */
    public boolean isCheckContent() {
        return this._checkContent;
    }

    /**
     * <p>
     * ファイルの内容(ハッシュ値)を監視するかどうかを設定します。
     * </p>
     * <p>
     * ファイルの内容を監視すると、監視のパフォーマンスが低下する場合があります。
     * </p>
     * 
     * @param checkContent
     *            ファイルの内容を監視する場合はtrue、監視しない場合はfalse。
     */
    public void setCheckContent(boolean checkContent) {
        this._checkContent = checkContent;
    }

    /**
     * <p>
     * ファイルの内容を比較するときに使用するハッシュ・アルゴリズムを取得します。初期値は"MD5"です。
     * </p>
     * 
     * @return ハッシュ・アルゴリズム。
     */
    public String getHashAlgorithm() {
        return this._hashAlgorithm;
    }

    /**
     * <p>
     * ファイルの内容を比較するときに使用するハッシュ・アルゴリズムを設定します。
     * </p>
     * <p>
     * ハッシュ値の計算は{@link MessageDigest}クラスを使用します。従って、このメソッドで指定するハッシュ・アルゴリズムは{@link MessageDigest}クラスがサポートするアルゴリズムである必要があります。{@link MessageDigest}クラスがサポートしないハッシュ・アルゴリズムを設定した場合、{@link #monitor()}メソッドの呼び出しで{@link FileMonitorFailException}例外がスローされる可能性があります。
     * </p>
     * 
     * @param hashAlgorithm
     *            ハッシュ・アルゴリズム。
     */
    public void setHashAlgorithm(String hashAlgorithm) {
        this._hashAlgorithm = hashAlgorithm;
    }

    /**
     * <p>
     * ファイルの作成、更新、削除を認識したときにイベントが通知されるリスナーを追加します。
     * </p>
     * <p>
     * このメソッドはスレッドセーフです。
     * </p>
     * 
     * @param listener
     *            イベントが通知されるリスナー。<br>
     *            nullの場合、{@link IllegalArgumentException}例外をスローします。
     */
    public void addListener(FileMonitorListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException("listener is null.");
        }
        synchronized (this._listenerList) {
            this._listenerList.add(listener);
        }
    }

    /**
     * <p>
     * 登録されているリスナーを削除します。
     * </p>
     * <p>
     * このメソッドはスレッドセーフです。
     * </p>
     * 
     * @param listener
     *            削除するリスナー。
     */
    public void removeListener(FileMonitorListener listener) {
        synchronized (this._listenerList) {
            this._listenerList.remove(listener);
        }
    }

    /**
     * <p>
     * リスナーの配列を返します。
     * </p>
     * <p>
     * このメソッドはスレッドセーフです。
     * </p>
     * 
     * @return リスナーの配列。
     */
    private FileMonitorListener[] getListeners() {
        synchronized (this._listenerList) {
            return this._listenerList.toArray(new FileMonitorListener[0]);
        }
    }

    /**
     * <p>
     * ファイルの作成、更新、削除を監視し、リスナーにイベントを通知します。
     * </p>
     * 
     * @throws FileMonitorFailException
     *             ファイルの監視に失敗した場合。
     */
    public void monitor() throws FileMonitorFailException {
        try {
            this.checkCreateOrUpdate(this._baseDirectory);
        } catch (NoSuchAlgorithmException e) {
            throw new FileMonitorFailException(e);
        } catch (IOException e) {
            throw new FileMonitorFailException(e);
        }
        this.checkDelete();
        this.checkClear();
    }

    /**
     * <p>
     * ディレクトリ内のファイルの作成、更新を確認します。
     * </p>
     * 
     * @param dir
     *            確認するディレクトリ。
     * @throws IOException
     *             入出力に失敗した場合。
     * @throws NoSuchAlgorithmException
     *             ハッシュ・アルゴリズムが見つからなかった場合。
     */
    private void checkCreateOrUpdate(File dir) throws IOException, NoSuchAlgorithmException {
        if (dir == null) {
            return;
        }

        File[] files = dir.listFiles();
        if (files == null) {
            return;
        }

        for (File f : files) {
            if (f.isFile()) {
                this.checkFileCreateOrUpdate(f);
            } else {
                this.checkCreateOrUpdate(f);
            }
        }
    }

    /**
     * <p>
     * ファイルの作成、更新を確認します。
     * </p>
     * 
     * @param file
     *            確認するファイル。
     * @throws IOException
     *             入出力に失敗した場合。
     * @throws NoSuchAlgorithmException
     *             ハッシュ・アルゴリズムが見つからなかった場合。
     */
    private void checkFileCreateOrUpdate(File file) throws NoSuchAlgorithmException, IOException {
        if (this._filePattern.matcher(file.getAbsolutePath()).matches()) {
            synchronized (this._monitoringFileMap) {
                if (!this._monitoringFileMap.containsKey(file.getAbsolutePath())) {
                    // ファイルがマップに存在しない場合、新規のモニター対象。
                    MonitoringFileInfo info = new MonitoringFileInfo();
                    info.path = file.getAbsolutePath();
                    this.setMonitoringFileInfo(file, info);
                    info.checked = true;
                    this._monitoringFileMap.put(file.getAbsolutePath(), info);
                    // イベントを通知する。
                    FileMonitorListener[] listeners = this.getListeners();
                    for (FileMonitorListener l : listeners) {
                        l.create(this, file);
                    }
                } else {
                    // ファイルがマップに存在する場合、変更のモニター対象。
                    MonitoringFileInfo info = this._monitoringFileMap.get(file.getAbsolutePath());
                    if (this.isModified(file, info)) {
                        this.setMonitoringFileInfo(file, info);
                        // イベントを通知する。
                        FileMonitorListener[] listeners = this.getListeners();
                        for (FileMonitorListener l : listeners) {
                            l.update(this, file);
                        }
                    }
                    info.checked = true;
                }
            }
        }
    }

    /**
     * <p>
     * 更新されているかどうかを調べます。
     * </p>
     * 
     * @param file
     *            元のファイル。
     * @param info
     *            監視中ファイルの情報。
     * @return 更新されている場合はtrue、更新されていない場合はfalse。
     * @throws IOException
     *             入出力に失敗した場合。
     * @throws NoSuchAlgorithmException
     *             ハッシュ・アルゴリズムが見つからなかった場合。
     */
    private boolean isModified(File file, MonitoringFileInfo info) throws NoSuchAlgorithmException, IOException {
        if (this._checkLength) {
            if (file.length() != info.length) {
                return true;
            }
        }
        if (this._checkLastModified) {
            if (file.lastModified() != info.lastModified) {
                return true;
            }
        }
        if (this._checkContent) {
            if (!Arrays.equals(this.getHashData(file), info.hashData)) {
                return true;
            }
        }
        return false;
    }

    /**
     * <p>
     * 監視中ファイル情報に新しい情報を設定します。
     * </p>
     * 
     * @param file
     *            元のファイル。
     * @param info
     *            監視中ファイルの情報。
     * @throws IOException
     *             入出力に失敗した場合。
     * @throws NoSuchAlgorithmException
     *             ハッシュ・アルゴリズムが見つからなかった場合。
     */
    private void setMonitoringFileInfo(File file, MonitoringFileInfo info) throws NoSuchAlgorithmException, IOException {
        if (this._checkLength) {
            info.length = file.length();
        }
        if (this._checkLastModified) {
            info.lastModified = file.lastModified();
        }
        if (this._checkContent) {
            info.hashData = this.getHashData(file);
        }
    }

    /**
     * <p>
     * ハッシュ値を算出します。
     * </p>
     * 
     * @param file
     *            算出元のファイル。
     * @return ハッシュ値。
     * @throws IOException
     *             入出力に失敗した場合。
     * @throws NoSuchAlgorithmException
     *             ハッシュ・アルゴリズムが見つからなかった場合。
     */
    private byte[] getHashData(File file) throws IOException, NoSuchAlgorithmException {
        MessageDigest md = MessageDigest.getInstance(this._hashAlgorithm);

        FileInputStream fileIn = new FileInputStream(file);
        try {
            byte[] buf = new byte[Consts.BUFFER_LENGTH];
            int len;
            while ((len = fileIn.read(buf)) != -1) {
                md.update(buf, 0, len);
            }
        } finally {
            fileIn.close();
        }

        return md.digest();
    }

    /**
     * <p>
     * 削除されているかどうかを調べます。
     * </p>
     */
    private void checkDelete() {
        synchronized (this._monitoringFileMap) {
            List<String> removeFileList = new ArrayList<String>();
            for (MonitoringFileInfo info : this._monitoringFileMap.values()) {
                if (!info.checked) {
                    File f = new File(info.path);
                    if (!f.exists()) {
                        removeFileList.add(info.path);
                        // イベントを通知する。
                        FileMonitorListener[] listeners = this.getListeners();
                        for (FileMonitorListener l : listeners) {
                            l.delete(this, f);
                        }
                    }
                }
            }
            for (String path : removeFileList) {
                this._monitoringFileMap.remove(path);
            }
        }
    }

    /**
     * <p>
     * 監視中ファイルの情報のチェック済みマークをクリアします。
     * </p>
     */
    private void checkClear() {
        synchronized (this._monitoringFileMap) {
            for (MonitoringFileInfo info : this._monitoringFileMap.values()) {
                info.checked = false;
            }
        }
    }

    /**
     * <p>
     * 監視中ファイルの情報です。
     * </p>
     */
    private class MonitoringFileInfo {

        /**
         * <p>
         * パス。
         * </p>
         */
        private String  path;

        /**
         * <p>
         * ファイルの長さ。
         * </p>
         */
        private long    length;

        /**
         * <p>
         * 最終更新日時。
         * </p>
         */
        private long    lastModified;

        /**
         * <p>
         * ハッシュ値。
         * </p>
         */
        private byte[]  hashData;

        /**
         * <p>
         * チェック済みマーク。
         * </p>
         */
        private boolean checked;

        /**
         * <p>
         * インスタンスを初期化します。
         * </p>
         */
        MonitoringFileInfo() {
        }

    }

}
