/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.storage;

import java.io.*;
import java.util.*;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;

import org.w3c.dom.*;
import org.xml.sax.*;

import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.model.*;
import com.aibonware.viewnaver.xml.*;

public class MessageStorage {
	TreeMap<Integer, Message> sentMessages = new TreeMap<Integer, Message>();
	TreeMap<Integer, Message> receivedMessages = new TreeMap<Integer, Message>();
	
	public MessageStorage() {}

	private TreeMap<Integer, Message> getMessages(String dir) throws StorageException {
		if(!ViewNaver.makeSureDir(dir)) {
			throw new StorageException("fBNg'" + dir + "'쐬ł܂B");
		}

		File[] files = new File(dir).listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return name.startsWith("message") && name.endsWith(".xml");
			}
		});

		TreeMap<Integer, Message> messages = new TreeMap<Integer, Message>();
		
		for(int i=0; i<files.length; i++) {
			File file = files[i];

			Message message = readMessage(file);
			if(message != null) messages.put(message.messageId, message);
		}
		
		return messages;
	}
	
	private Message readMessage(File file) {
		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			Document doc = builder.parse(new BufferedInputStream(new FileInputStream(file)));
	
			Element root = doc.getDocumentElement();
			Element messageElem = BaseXMLReader.getAttrElem(root, "Message");
	
			MessageReader reader = new MessageReader();
			Message message = reader.parse(messageElem);
		
			return message;
		} catch(IOException e) {
			ViewNaver.err(e);
			return null;
		} catch(XMLReaderException e) {
			ViewNaver.err(e);
			return null;
		} catch(SAXException e) {
			ViewNaver.err(e);
			return null;
		} catch(ParserConfigurationException e) {
			ViewNaver.err(e);
			return null;
		}
	}

	public TreeMap<Integer, Message> getSentMessages(String loginUserName) throws StorageException {
		sentMessages = getMessages("message" + File.separator + loginUserName + File.separator + "sent" + File.separator);
		return sentMessages;
	}
	
	public Message getSentMessage(int messageId) {
		return sentMessages.get(messageId);
	}

	public TreeMap<Integer, Message> getReceivedMessages(String loginUserName) throws StorageException {
		receivedMessages = getMessages("message" + File.separator + loginUserName + File.separator + "received" + File.separator);
		return receivedMessages;
	}

	public Message getReceivedMessage(int messageId) {
		return receivedMessages.get(messageId);
	}

	private void saveMessage(String dir, Message message) throws StorageException {
		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
	
			Document doc = builder.newDocument();
	
			Element root = doc.createElement("NDX");
			doc.appendChild(root);
	
			MessageSerializer serializer = new MessageSerializer(doc);
			root.appendChild(serializer.serialize(message));
			
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();
			
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
	
			transformer.transform(new DOMSource(root.getOwnerDocument()), new StreamResult(dir + "message" + message.messageId + ".xml"));

		} catch(ParserConfigurationException e) {
			throw new StorageException(e);
		} catch(TransformerConfigurationException e) {
			throw new StorageException(e);
		} catch(TransformerException e) {
			throw new StorageException(e);
		}
	}
	
	public void saveSentMessage(String loginUserName, Message message) throws StorageException {
		sentMessages.put(message.messageId, message);
		saveMessage("message" + File.separator + loginUserName + File.separator + "sent" + File.separator, message);
	}

	public void saveReceivedMessage(String loginUserName, Message message) throws StorageException {
		receivedMessages.put(message.messageId, message);
		saveMessage("message" + File.separator + loginUserName + File.separator + "received" + File.separator, message);
	}
}
