/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.browser.win32;

import java.util.*;
import java.util.concurrent.*;
import java.awt.*;
import com.aibonware.viewnaver.*;

public class Win32BrowserThread extends Thread {
	private volatile boolean mustExit = false;
	
	private static final int MSG_EXEC_COMMAND = 0x7ff0;

	private static final int CMD_EXIT_LOOP = 0;

	private native static boolean messageLoop();
	private native static void postMessage(int msg, int wparam, int lparam);
	private native static int createBrowser(Canvas canvas, int[] policies) throws Exception;
	private native static void resizeBrowser(int hBrowser, int width, int height) throws Exception;
	private native static void setBodyHtml(int hBrowser, String html) throws Exception;
	private native static void insertBodyHtml(int hBrowser, String tagid, String html) throws Exception;
	private native static void replaceBodyHtml(int hBrowser, String tagid, String html) throws Exception;
	private native static String getBodyHtml(int hBrowser) throws Exception;
	private native static void setEditable(int hBrowser, boolean editable) throws Exception;
	private native static void destroyBrowser(int hBrowser) throws Exception;
	private native static void writeDocHtml(int hBrowser, String html) throws Exception;
	private native static void clearDocHtml(int hBrowser) throws Exception;
	private native static void closeDocHtml(int hBrowser) throws Exception;
	
	private native static void toggleBold(int hBrowser) throws Exception;
	private native static void setColor(int hBrowser, String color) throws Exception;
	private native static void insertHR(int hBrowser) throws Exception;
	private native static void toggleItalic(int hBrowser) throws Exception;
	private native static void toggleUnderLine(int hBrowser) throws Exception;
	private native static void setFontName(int hBrowser, String fontName) throws Exception;
	private native static void setFontSize(int hBrowser, String fontSize) throws Exception;
	private native static void scrollTop(int hBrowser, String tagid) throws Exception;
	private native static void refreshBrowser(int hBrowser) throws Exception;
	private native static void resetBrowser(int hBrowser) throws Exception;

	private static Vector<Win32BrowserImpl> browsers = new Vector<Win32BrowserImpl>();
	
	static void addBrowser(Win32BrowserImpl browser) {
		browsers.addElement(browser);
	}
	
	static void removeBrowser(Win32BrowserImpl browser) {
		browsers.removeElement(browser);
	}
	
	private static Win32BrowserImpl findBrowser(int hBrowser) {
		for(int i=0; i<browsers.size(); i++) {
			Win32BrowserImpl browser = browsers.elementAt(i);
			if(browser.hBrowser == hBrowser) return browser;
		}

		return null;
	}

	private static boolean beforeNavigate2(int hBrowser, String url) {
		Win32BrowserImpl browser = findBrowser(hBrowser);
		if(browser == null) return true;

		return browser.notifyBeforeNavigate(url);
	}
	
	private static void receiveLog(String log) {
		ViewNaver.println(log);
	}

	public Win32BrowserThread() {
		super("Win32BrowserThread");
	}

	private Vector<Task> tasks = new Vector<Task>();
	
	private void addTasks(Task task) {
		synchronized(tasks) {
			tasks.addElement(task);
		}
	}
	
	@Override public void run() {
		while(!mustExit) {
			Vector<Task> executingTasks = new Vector<Task>();

			synchronized(tasks) {
				executingTasks.addAll(tasks);
				tasks.clear();
			}

			for(Task task: executingTasks) {
				try {
					task.result = task.run();
				} catch(Exception e) {
					e.printStackTrace();
					ViewNaver.err(e);
				} finally {
					task.lock.release();
				}
			}

			messageLoop();
		}
	}

	public void exit() {
		mustExit = true;
		wake();
	}

	private void wake() {
		postMessage(MSG_EXEC_COMMAND, CMD_EXIT_LOOP, 0);
	}
	
	private static abstract class Task {
		public Semaphore lock = new Semaphore(1);
		public volatile Object result = null;

		public Task() {
			try {
				lock.acquire();
			} catch(InterruptedException e) {
			}
		}

		public abstract Object run() throws Exception;
	}

	private Object executeTask(Task task) {
		addTasks(task);
		wake();
		try {
			task.lock.acquire();
		} catch(InterruptedException e) {
		}
		Object result = task.result;
		return result;
	}

	public int createWin32Browser(final Canvas canvas) {
		return (Integer)executeTask(new Task() {
			public Object run() throws Exception {
				return Win32BrowserThread.createBrowser(canvas, Win32BrowserFactory.browserConfig.actions.createRawDump());
			}
		});
	}

	public void resizeWin32Browser(final int hBrowser, final int width, final int height) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.resizeBrowser(hBrowser, width, height);
				return null;
			}
		});
	}

	public void setWin32BodyHtml(final int hBrowser, final String html) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.setBodyHtml(hBrowser, html);
				return null;
			}
		});
	}

	public void insertWin32BodyHtml(final int hBrowser, final String tagid, final String html) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.insertBodyHtml(hBrowser, tagid, html);
				return null;
			}
		});
	}

	public void replaceWin32BodyHtml(final int hBrowser, final String tagid, final String html) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.replaceBodyHtml(hBrowser, tagid, html);
				return null;
			}
		});		
	}

	public String getWin32BodyHtml(final int hBrowser) {
		return (String)executeTask(new Task() {
			public Object run() throws Exception {
				return Win32BrowserThread.getBodyHtml(hBrowser);
			}
		});
	}

	public void setWin32Editable(final int hBrowser, final boolean editable) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.setEditable(hBrowser, editable);
				return null;
			}
		});
	}

	public void disposeWin32Browser(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.destroyBrowser(hBrowser);
				return null;
			}
		});
	}

	public void toggleWin32Bold(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.toggleBold(hBrowser);
				return null;
			}
		});
	}
	
	public void setWin32Color(final int hBrowser, final String color) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.setColor(hBrowser, color);
				return null;
			}
		});
	}
	
	public void insertWin32HR(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.insertHR(hBrowser);
				return null;
			}
		});
	}
	
	public void toggleWin32Italic(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.toggleItalic(hBrowser);
				return null;
			}
		});
	}
	
	public void toggleWin32UnderLine(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.toggleUnderLine(hBrowser);
				return null;
			}
		});
	}
	
	public void setWin32FontName(final int hBrowser, final String fontName) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.setFontName(hBrowser, fontName);
				return null;
			}
		});
	}
	
	public void setWin32FontSize(final int hBrowser, final String fontSize) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.setFontSize(hBrowser, fontSize);
				return null;
			}
		});
	}

	public void win32scrollTop(final int hBrowser, final String tagid) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.scrollTop(hBrowser, tagid);
				return null;
			}
		});
	}
	
	public void win32Refresh(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.refreshBrowser(hBrowser);
				return null;
			}
		});
	}
	
	public void win32Reset(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.resetBrowser(hBrowser);
				return null;
			}
		});
	}

	public void writeWin32DocHtml(final int hBrowser, final String html) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.writeDocHtml(hBrowser, html);
				return null;
			}
		});
	}

	public void clearWin32DocHtml(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.clearDocHtml(hBrowser);
				return null;
			}
		});
	}

	public void closeWin32DocHtml(final int hBrowser) {
		executeTask(new Task() {
			public Object run() throws Exception {
				Win32BrowserThread.closeDocHtml(hBrowser);
				return null;
			}
		});
	}

}
