/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.config;

import java.util.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

import com.aibonware.viewnaver.component.*;

@SuppressWarnings("serial")
public class ConfigView extends JPanel {
	private static class DispPath {
		public final String path;
		public final String name;
		
		public DispPath(String path) {
			this.path = path;
			int pos = path.lastIndexOf('.');

			if(pos == -1) name = path;
			else name = path.substring(pos+1);
		}

		public String toString() {
			return name;
		}
	}
	
	private HashMap<String, Vector<ConfigItemView>> viewMap;
	
	private JTree tree;
	DefaultMutableTreeNode root;
	private CardLayout cardLayout;
	private JPanel panel;
	private JPanel emptyView = new JPanel(new BorderLayout()) {{ setOpaque(true); }};
	
	public ConfigView(ConfigItem config) {
		setLayout(new BorderLayout());
		
		cardLayout = new CardLayout();
		panel = new JPanel(cardLayout);
		
		viewMap = new HashMap<String, Vector<ConfigItemView>>();
		root = new DefaultMutableTreeNode("ݒ");
		root.setUserObject(new DispPath("ݒ"));
		setConfigTree(config);
		
		Set<String> paths = viewMap.keySet();

		for(String path: paths) {
			DefaultMutableTreeNode node = makeSure(path);
			node.setUserObject(new DispPath(path));
		}

		for(String path: paths) {
			PagePanel page = new PagePanel();
//			page.setAlignmentY(JComponent.TOP_ALIGNMENT);

			for(ConfigItemView view: viewMap.get(path)) {
				page.addLine(PagePanel.glue(view));
//				cardLayout.addLayoutComponent(view, path);
			}

			panel.add(new FastScrollPane(page), path);
		}
		
		panel.add(emptyView, "EmptyView");
		cardLayout.show(panel, "EmptyView");

		tree = new JTree(root);
		
		DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
		renderer.setLeafIcon(null);
		renderer.setOpenIcon(null);
		renderer.setClosedIcon(null);
		tree.setCellRenderer(renderer);
		
		tree.addTreeSelectionListener(new TreeSelectionListener() {
			public void valueChanged(TreeSelectionEvent e) {
				TreePath treePath = tree.getSelectionPath();
				if(treePath == null) return;
				
				String configPath = ((DispPath)((DefaultMutableTreeNode)treePath.getLastPathComponent()).getUserObject()).path;

				if(viewMap.get(configPath) != null) {
					cardLayout.show(panel, configPath);
				} else {
					cardLayout.show(panel, "EmptyView");
				}
			}
		});

		JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		split.setLeftComponent(tree);
		split.setRightComponent(panel);

		split.setDividerSize(4);
		split.setResizeWeight(0);
		split.setDividerLocation(100);

		add(split, BorderLayout.CENTER);
	}

	private DefaultMutableTreeNode makeSure(String path) {
		StringTokenizer tokens = new StringTokenizer(path, ".");
		DefaultMutableTreeNode cur = root;
		String curPath = "";
		String delimitor = "";

		while(tokens.hasMoreTokens()) {
			String token = tokens.nextToken();
			curPath = curPath + delimitor + token;
			delimitor = ".";
			DefaultMutableTreeNode next = getChildNode(cur, token);
			
			if(next == null) {
				next = new DefaultMutableTreeNode(token);
				next.setUserObject(new DispPath(curPath));
				cur.add(next);
			}

			cur = next;
		}
		
		return cur;
	}

	private DefaultMutableTreeNode getChildNode(DefaultMutableTreeNode parent, String childName) {
		for(int i=0; i<parent.getChildCount(); i++) {
			if(parent.getChildAt(i).toString().equals(childName)) return (DefaultMutableTreeNode)parent.getChildAt(i);
		}
		
		return null;
	}

	public void setConfigTree(ConfigItem config) {
		ConfigItemView view = config.getView();

		if(view != null) {
			String path = config.getDispPath();
			view.setConfig(config);

			if(viewMap.get(path) == null) {
				viewMap.put(path, new Vector<ConfigItemView>());
			}

			Vector<ConfigItemView> views = viewMap.get(path); 
			views.addElement(view);
		}

		for(int i=0; i<config.getChildCount(); i++) {
			setConfigTree(config.getChild(i));
		}
	}

	public void applyConfig() throws ConfigException {
		for(Vector<ConfigItemView> views : viewMap.values()) {
			for(ConfigItemView view: views) {
				view.applyConfig();
			}
		}
	}
}
