/*
 * Copyright (c) 2007, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.server;

import java.util.*;
import java.awt.*;
import java.security.*;

import Acme.Serve.*;
import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.net.*;
import com.aibonware.viewnaver.server.conv.*;
import com.aibonware.viewnaver.server.image.*;
import com.aibonware.viewnaver.server.movie.*;
import com.aibonware.viewnaver.server.servlet.*;

public final class MiniServer extends Thread {
	private Serve serv;
	private String password;
	private final int port;
	public SecureRandom rand;
	public final int articlesPerPage;
	public ContentsConverter conv;
	public final boolean useSmallImage;
	public final int maxImageWidth;
	public final int maxImageHeight;
	public final int maxImageSize;
	public final HashMap<RenderingHints.Key, Object> renderingHints;
	
	public final String movieShrinkerCommandLine;
	public final String movieContentType;
	
	public LoginSession outerSession;
	
	public MiniServer(int port, String password, int articlesPerPage, boolean useSmallImage, int maxImageWidth, int maxImageHeight, int maxImageSize, HashMap<RenderingHints.Key, Object> renderingHints, String movieShrinkerCommandLine, String movieContentType) throws NoSuchAlgorithmException {
		this.port = port;
		this.password = password;
		this.articlesPerPage = articlesPerPage;
		this.useSmallImage = useSmallImage;
		this.maxImageWidth = maxImageWidth;
		this.maxImageHeight = maxImageHeight;
		this.maxImageSize = maxImageSize;
		this.renderingHints = renderingHints;
		this.movieShrinkerCommandLine = movieShrinkerCommandLine;
		this.movieContentType = movieContentType;

		rand = SecureRandom.getInstance("SHA1PRNG");
		rand.nextInt();
		
		conv = new ContentsConverter();
		conv.init(this);

		if(useSmallImage) {
			conv.filterPrototypes.addElement(new ImageConverterFilter());
		}
		
		conv.filterPrototypes.addElement(new MovieConverterFilter());

		outerSession = new LoginSession("");
	}

	private void addServlet(BaseServlet servlet) {
		serv.addServlet("/" + servlet.getName(), servlet);
		ViewNaver.println("servlet " + servlet.getName() + " added.");
	}

	@Override public void run() {
		serv = new Serve(port);
		addServlet(new MainServlet());
		addServlet(new LoginServlet());
		addServlet(new LogoutServlet());
		addServlet(new BoardServlet());
		addServlet(new ThreadServlet());
//		addServlet(new ReceivedMessageServlet());
//		addServlet(new SentMessageServlet());
		addServlet(new PostArticleServlet());

		addServlet(new WriteNewThreadServlet());
		addServlet(new PostNewThreadServlet());
		addServlet(new WriteReplyThreadServlet());
		addServlet(new PostReplyThreadServlet());

		addServlet(new ConfirmDelServlet());
		addServlet(new DeleteArticleServlet());
		addServlet(new DeleteThreadServlet());
		
		addServlet(new ImageServlet());

		addServlet(new MovieServlet());
		addServlet(new MoviePlayServlet());
		
		serv.serve();
	}

	private static final String VALID_SID_CHARS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789._";

	public TreeMap<String /*sid*/, UserSession> sessions = new TreeMap<String, UserSession>();

	public boolean isValidSidForm(String sid) {
		if(sid == null || sid.length() != 20) return false;
		
		for(int i=0; i<sid.length(); i++) {
			if(VALID_SID_CHARS.indexOf(sid.charAt(i)) == -1) return false;
		}
		
		return true;
	}
	
	public boolean isValidSid(String sid) {
		if(!isValidSidForm(sid)) return false;
		
		synchronized(sessions) {
			return sessions.containsKey(sid);
		}
	}

	public synchronized String createSid() {
		StringBuffer sid = new StringBuffer(20);
		
		for(int i=0; i<20; i++) {
			int r = Math.abs(rand.nextInt()) % VALID_SID_CHARS.length();
			sid.append(VALID_SID_CHARS.charAt(r));
		}
		
		return sid.toString();
	}

	public static final int MAX_SESSION = 10;
	
	public void removeSid(String sid) {
		// NbNꂽꍇɃZbV1Ɍ肵Ă܂
		// {̃[UT[o𐧌łȂȂĂ܂̂ŁAdZbVB
		// AMAX_SESSION𒴂狭IɃT[o𗎂ƂB
		synchronized(sessions) {
			sessions.remove(sid);
		}
	}
	
	public UserSession getSession(String sid) {
		synchronized(sessions) {
			if(sid == null) return null;
			if(!sessions.containsKey(sid)) return null;
			return sessions.get(sid);
		}
	}
	
	public String registerSid(String sid) {
		// NbNꂽꍇɃZbV1Ɍ肵Ă܂
		// {̃[UT[o𐧌łȂȂĂ܂̂ŁAdZbVB
		// AMAX_SESSION𒴂狭IɃT[o𗎂ƂB
		synchronized(sessions) {
			UserSession oldSession = sessions.get(sid);
			
			if(oldSession != null) {
				oldSession.warning = "dOCo܂B񃍃OAEgAēxOCĂB";
			} else {
				sessions.put(sid, new UserSession(sid, maxImageWidth, maxImageHeight, maxImageSize, renderingHints));
			}
	
			if(sessions.size() > MAX_SESSION) {
				BaseServlet.mustReject = true;
				return "ZbV" + MAX_SESSION + "𒴂܂BÕANZXً}Ւf܂";
			}

			if(sessions.size() > 1) {
				return "݂̃ZbV" + sessions.size() + "łB<br>" 
					+ "gpIۂɂ͕KOAEgsĂ<br>"
					+ "gɊoꍇAsȃANZX\܂B<br>"
					+ "ɊÕANZXً}ՒfĂ";
			}

			return null;
		}
	}
	
	public boolean isValidPassword(String password) {
		return this.password.equals(password);
	}
}
