// Delphi 5/6/7 and Kylix Implementation
// November 2003
//
//
// LICENSE
//
// The contents of this file are subject to the Mozilla Public License Version
// 1.1 (the "License"); you may not use this file except in compliance with
// the License. You may obtain a copy of the License at
// "http://www.mozilla.org/MPL/"
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
// the specific language governing rights and limitations under the License.
//
// The Original Code is "XDOM_error_reporting.pas".
//
// The Initial Developer of the Original Code is Dieter Khler (Heidelberg,
// Germany, "http://www.philo.de/"). Portions created by the Initial Developer
// are Copyright (C) 1999-2003 Dieter Khler. All Rights Reserved.
//
// Alternatively, the contents of this file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"), in which case the
// provisions of the GPL are applicable instead of those above. If you wish to
// allow use of your version of this file only under the terms of the GPL, and
// not to allow others to use your version of this file under the terms of the
// MPL, indicate your decision by deleting the provisions above and replace them
// with the notice and other provisions required by the GPL. If you do not delete
// the provisions above, a recipient may use your version of this file under the
// terms of any one of the MPL or the GPL.

unit XDOM_error_reporting;

interface

uses
  XDOM_3_1, LangUtils, cUnicodeCodecs, AutoListUtils, SysUtils, Classes;

type
  TDomCustomErrorTranslator = class(TComponent)
  private
    FLanguage: TIso639LanguageCode;
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; virtual; abstract;
    procedure SetLanguage(const Value: TIso639LanguageCode); virtual;
  public
    function Translate(const Error: TdomError): WideString; virtual; abstract;
    property Language: TIso639LanguageCode read FLanguage write SetLanguage;
    property SupportedLanguages: TIso639LanguageCodeSet read
      GetSupportedLanguages;
  end;

  TDomDutchErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomEnglishErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomFrenchErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomGermanErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomItalianErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomPolishErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomPortugueseErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomSpanishErrorTranslator = class(TDomCustomErrorTranslator)
  protected
    function GetSupportedLanguages: TIso639LanguageCodeSet; override;
  public
    constructor Create(AOwner: TComponent); override;
    function Translate(const Error: TdomError): WideString; override;
  end;

  TDomErrorAutoStrings = class(TUtilsCustomAliasedStrings)
  private
    FErrorTranslator: TDomCustomErrorTranslator;
  protected
    function Get(Index: Integer): string; override;
    function GetError(Index: Integer): TDomError; virtual;
    procedure SetErrorTranslator(const AErrorTranslator:
      TDomCustomErrorTranslator); virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
  public
    // List mutation:
    function AddError(AError: TDomError): Integer; overload; virtual;
    function AddError(const ErrorType: TXmlErrorType;
      const StartByteNumber, StartCharNumber, StartColumnNumber,
      StartLineNumber, EndByteNumber, EndCharNumber, EndColumnNumber,
      EndLineNumber: Int64; const UriStr: WideString;
      const RASObject: TdomASObject; const RNode: TdomNode;
      const Code: WideString): Integer; overload; virtual;
    procedure Assign(Source: TPersistent); override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    function InsertError(Index: Integer; AError: TDomError): Boolean; overload;
      virtual;
    function InsertError(Index: Integer; const ErrorType: TXmlErrorType;
      const StartByteNumber, StartCharNumber, StartColumnNumber,
      StartLineNumber, EndByteNumber, EndCharNumber, EndColumnNumber,
      EndLineNumber: Int64; const UriStr: WideString;
      const RASObject: TdomASObject; const RNode: TdomNode;
      const Code: WideString): Boolean; overload; virtual;
    // View Control:
    procedure DisableControls; virtual;
    procedure EnableControls; virtual;

    property Capacity;
    property Count;
    property Errors[Index: Integer]: TdomError read GetError;
    property Strings;
  published
    property Enabled;
    property ErrorTranslator: TDomCustomErrorTranslator read FErrorTranslator
      write SetErrorTranslator;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('XDOM Add-Ons', [TDomErrorAutoStrings,
    TDomDutchErrorTranslator,
      TDomEnglishErrorTranslator,
      TDomFrenchErrorTranslator,
      TDomGermanErrorTranslator,
      TDomItalianErrorTranslator,
      TDomPolishErrorTranslator,
      TDomPortugueseErrorTranslator,
      TDomSpanishErrorTranslator]);
end;

{Local methods}

function ExtractFileNameFromUri(const Uri: WideString): WideString;
var
  I: Integer;
begin
  if Uri = '' then
    Result := ''
  else
  begin
    I := LastDelimiter('/', Uri);
    Result := Copy(Uri, I + 1, MaxInt);
  end;
end;

{ TDomCustomErrorTranslator }

procedure TDomCustomErrorTranslator.SetLanguage(
  const Value: TIso639LanguageCode);
begin
  if not (Value in SupportedLanguages) then
    raise ENot_Supported_Err.create('Not supported error.');
  FLanguage := Value;
end;

{ TDomDutchErrorTranslator }

constructor TDomDutchErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_nl;
end;

function TDomDutchErrorTranslator.GetSupportedLanguages: TIso639LanguageCodeSet;
begin
  Result := [iso639_nl];
end;

function TDomDutchErrorTranslator.Translate(
  const Error: TdomError): WideString;
// This function was provided by Erik van der Poll,
// Locus Warehouse Management Systems b.v., Almere
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'Fatal Fout';
      DOM_SEVERITY_ERROR: SeverityStr := 'Fout';
      DOM_SEVERITY_WARNING: SeverityStr := 'Waarschuwing';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'ET_ATTLIST_DECL_START_EXPECTED';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'ET_CDATA_START_EXPECTED';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'ET_COMMENT_START_EXPECTED';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr := 'ET_CONDITIONAL_SECTION_NOT_ALLOWED';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'ET_DOCTYPE_START_EXPECTED';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'ET_ELEMENT_DECL_START_EXPECTED';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'ET_ENTITY_DECL_START_EXPECTED';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Ongeldige element naam';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'ET_DOUBLE_HYPHEN_IN_COMMENT';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Dubbel begin element';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Dubbele document type declaratie (DTD)';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'ET_HYPHEN_AT_COMMENT_END';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Ongeldige attribuut naam';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'Ongeldige attribuut waarde';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Dubbele attribuut naam in een element';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Ongeldige entiteit naam';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Ongeldige verwerkings instruktie';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Ongeldige XML declaratie';
      ET_INVALID_CHARREF:
        ErrorStr := 'Ongeldige karakter referentie';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'Aanhalingstekens ontbreken';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'Is-teken ontbreekt';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Dubbel is-teken';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'Ontbrekende witruimte';
      ET_MISSING_START_TAG:
        ErrorStr := 'Eind-tag zonder begin-tag';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'ET_MISSING_ELEMENT_NAME';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'ET_MISSING_ENTITY_NAME';
      ET_MISSING_END_TAG:
        ErrorStr := 'Eind-tag ontbreekt';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Ongeldig karakter';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Tekst buiten het begin element';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'ET_NOTATION_DECL_START_EXPECTED';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Ongeldige documenttype declaratie';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'ET_UNCLOSED_ATTLIST_DECL';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := 'ET_SYSTEM_KEYWORD_EXPECTED';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'ET_UNCLOSED_CDATA_SECTION';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'ET_UNCLOSED_CHARREF';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'ET_UNCLOSED_COMMENT';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'ET_UNCLOSED_CONDITIONAL_SECTION';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'ET_UNCLOSED_DOCTYPE';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'ET_UNCLOSED_ELEMENT';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'ET_UNCLOSED_ELEMENT_DECL';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'ET_UNCLOSED_ENTITY_DECL';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_ENTITY_REF';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'ET_UNCLOSED_NOTATION_DECL';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_PARAMETER_ENTITY_REF';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'ET_UNCLOSED_PROCESSING_INSTRUCTION';
      ET_WRONG_ORDER:
        ErrorStr := 'Volgorde fout';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Onbekend declaratie type';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Ongeldige attribuut declaratie';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'ET_INVALID_ATTLIST_DECL_NAME';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Ongeldige conditionele sectie';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Ongeldige element declaratie';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Ongeldige entiteits declaratie';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'Ongeldige markup declaratie';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Ongeldige notatie declaratie';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Ongeldige parameter entiteit declaratie';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Ongeldige tekst declaratie';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_LEFT_SQUARE_BRACKET_EXPECTED';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr := '"<" in de vervangingstekst van een attribuut waarde';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr := 'Attribuut waarde verwijst naar externe entiteit';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := 'ET_PUBLIC_KEYWORD_EXPECTED';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'ET_QUOTATION_MARK_EXPECTED';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'Recursieve referentie gevonden';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Referentie naar een niet verwerkte entiteit';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_RIGHT_SQUARE_BRACKET_EXPECTED';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr := 'Referentie naar een entiteit zonder juiste opmaak';
      ET_INVALID_COMMENT:
        ErrorStr := 'Ongeldig commentaar';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'Ongeldige CDATA sectie';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'Ongeldige systeem tekstconstante';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'Ongeldige publieke tekstconstante';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'Ongeldige namespace naamgeving';
      ET_INVALID_PREFIX:
        ErrorStr := 'Ongeldig namespace voorvoegsel';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'Ongeldig namespace adres';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'Geen namespace adres gevonden';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'Foute nesting van namespace voorvoegsels';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'Codering niet ondersteund';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'Dubbele attribuut declaratie';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Dubbele attribuut definitie';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Dubbele entiteit declaratie';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Dubbele parameter entiteit declaratie';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Onbruikbare entiteit gedefinieerd';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Ontbrekende entiteit declaratie';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'ET_DOCTYPE_NOT_FOUND';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Dubbele element type declaratie';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Naam komt meer keren voor in gemengde inhoud';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr := 'Dubbel ID attribuut binnen hetzelfde element type';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Niet gedefinieerde notatie naam';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr := 'Dubbel notatie attribuut binnen hetzelfde element type';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := 'Dubbel notatie kenmerk';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr := 'Notatie attribuut definitie voor een leeg element type';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := 'Dubbele enumeratie waarde';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'Attribuut type en attribuut waarde komen niet overeen';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Tokens meermalen gedefinieerd';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr :=
          'ID attribuut is niet gedefinieerd als #IMPLIED of #REQUIRED';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'Type van begin element komt niet overeen met de naam van de document type declaratie';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Element type declaratie niet gevonden';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'Element met definitie EMPTY is niet leeg';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'Inhoud van een element met gemengde inhoud komt niet overeen met de declaratie';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'Inhoud van een element met element inhoud komt niet overeen met de declaratie';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr := 'Inhoud element model is niet deterministisch';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Dubbele notatie declaratie';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Attribuut definitie niet gevonden';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Verplicht attribuut niet gevonden';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'Gedefinieerde waarde en werkelijke waarde van een vast attribuut komen niet overeen';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr :=
          'Attribuut type en attribuut default waarde komen niet overeen';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'Dubbele ID waarde';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'IDREF of IDREFS waarde verwijst naar een niet bestaande ID waarde';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'ENTITY of ENTITIES waarde verwijst naar een niet bestaande entiteit';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Foute declaratie voor een voorgedefinieerde entiteit';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Niet oplosbare entiteit referentie';
      ET_UNRESOLVABLE_PARAMETER_ENTITY_REFERENCE:
        ErrorStr := 'Niet oplosbare parameter entiteit referentie';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := 'Externe subset van de DTD niet gevonden';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Parameter entiteit declaratie niet gevonden';
    else
      ErrorStr := 'Fout in broncode';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_1Codec, S);
end;

{ TDomEnglishErrorTranslator }

constructor TDomEnglishErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_en;
end;

function TDomEnglishErrorTranslator.GetSupportedLanguages:
  TIso639LanguageCodeSet;
begin
  Result := [iso639_en];
end;

function TDomEnglishErrorTranslator.Translate(
  const Error: TdomError): WideString;
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'Fatal Error';
      DOM_SEVERITY_ERROR: SeverityStr := 'Error';
      DOM_SEVERITY_WARNING: SeverityStr := 'Warning';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'Start of attribute list declaration expected';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'Start of CDATA section expected';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'Start of comment expected';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr := 'Conditional section not allowed in internal subset';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'Start of document type declaration (DTD) expected';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'Start of element declaration expected';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'Start of entity declaration expected';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Invalid element name';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'Double hyphen in comment';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Double root element';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Double document type declaration (DTD)';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'Comment text ends with a hyphen';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Invalid attribute name';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'Invalid attribute value';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Double attribute name in one element';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Invalid entity name';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Invalid processing instruction';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Invalid XML declaration';
      ET_INVALID_CHARREF:
        ErrorStr := 'Invalid character reference';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'Missing quotation marks';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'Missing equality sign';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Double equality sign';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'Missing white-space';
      ET_MISSING_START_TAG:
        ErrorStr := 'End-tag without start-tag';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'Missing element name';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'Missing entity name';
      ET_MISSING_END_TAG:
        ErrorStr := 'Missing end-tag';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Invalid character';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Character(s) outside the root element';
      ET_ROOT_NOT_FOUND:
        ErrorStr := 'Root element not found';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'Start of notation declaration expected';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Invalid doctype declaration';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'Unclosed attribute-list declaration';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := '"SYSTEM" keyword expected';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'Unclosed CDATA section';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'Unclosed character reference';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'Unclosed comment';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'Unclosed conditional section';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'Unclosed document type declaration (DTD)';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'Unclosed element';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'Unclosed element declaration';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'Unclosed entity declaration';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'Unclosed entity reference';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'Unclosed notation declaration';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'Unclosed parameter entity reference';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'Unclosed processsing instruction';
      ET_WRONG_ORDER:
        ErrorStr := 'Wrong order';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Unknown declaration type';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Invalid attribute declaration';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'Invalid name of attribute declaration';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Invalid conditional section';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Invalid element declaration';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Invalid entity declaration';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'Invalid markup declaration';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Invalid notation declaration';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Invalid parameter entity declaration';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Invalid text declaration';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := '"[" expected';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr :=
          '"<" in the replacement text of an entity referred to directly or indirectly in an attribute value';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr := 'Attribute value refers to external entity';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := '"PUBLIC" keyword expected';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'Quotation mark expected';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'Recursive reference found';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Reference to an unparsed entity';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := '"]" expected';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr := 'Reference to an entity containing no proper markup';
      ET_INVALID_COMMENT:
        ErrorStr := 'Invalid comment';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'Invalid CDATA section';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'Invalid system literal';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'Invalid pubid literal';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'Invalid qualified name';
      ET_INVALID_PREFIX:
        ErrorStr := 'Invalid prefix';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'Invalid namespace URI';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'Namespace URI not found';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'Wrong prefix mapping nesting';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'Encoding not supported';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'Double attribute-list declaration';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Double attribute defintion';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Double entity declaration';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Double parameter entity declaration';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Unusable entity declared';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Entity declaration not found';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'Missing document type declaration (DTD)';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Double element type declaration';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Duplicate name in mixed content';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr := 'Duplicate ID attribute on the same element type declared';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Undeclared notation name';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Duplicate notation attribute on the same element type declared';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := 'Duplicate notation token';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr := 'Notation attribute on an empty element type declared';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := 'Duplicate enumeration token';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'Attribute type and attribute value do not match';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Duplicate tokens declared';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr :=
          'ID attribute is neither declared as #IMPLIED nor as #REQUIRED';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'Type of root element does not match name of document type declaration';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Element type declaration not found';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'Element declared EMPTY is not empty';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'Content of an element with mixed content does not match the declaration';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'Content of an element with element content does not match the declaration';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr := 'Content model of an element is not deterministic';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Duplicate notation declaration';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Attribute definition not found';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Required attribute not found';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'Declared value and actual value of a fixed attribute do not match';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr := 'Attribute type and attribute default do not match';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'Duplicate ID value';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'IDREF or IDREFS value refers to a non-existent target ID value';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'ENTITY or ENTITIES value refers to a non-existent target unparsed entity';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Wrong declaration of a predefined entity';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Unresolvable entity reference';
      ET_UNRESOLVABLE_PARAMETER_ENTITY_REFERENCE:
        ErrorStr := 'Unresolvable parameter entity reference';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := 'External subset of the DTD not found';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Parameter entity declaration not found';
    else
      ErrorStr := 'Invalid source-code';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_1Codec, S);
end;

{ TDomFrenchErrorTranslator }

constructor TDomFrenchErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_fr;
end;

function TDomFrenchErrorTranslator.GetSupportedLanguages:
  TIso639LanguageCodeSet;
begin
  Result := [iso639_fr];
end;

function TDomFrenchErrorTranslator.Translate(
  const Error: TdomError): WideString;
// - This function was provided by Bernard Eblin -
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'Erreur Fatale';
      DOM_SEVERITY_ERROR: SeverityStr := 'Erreur';
      DOM_SEVERITY_WARNING: SeverityStr := 'Attention';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'ET_ATTLIST_DECL_START_EXPECTED';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'ET_CDATA_START_EXPECTED';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'ET_COMMENT_START_EXPECTED';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr := 'ET_CONDITIONAL_SECTION_NOT_ALLOWED';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'ET_DOCTYPE_START_EXPECTED';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'ET_ELEMENT_DECL_START_EXPECTED';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'ET_ENTITY_DECL_START_EXPECTED';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Nom d'';lment invalide';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'ET_DOUBLE_HYPHEN_IN_COMMENT';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Double lment racine';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Double dclaration de type de document (DTD)';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'ET_HYPHEN_AT_COMMENT_END';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Nom d'';attribut invalide';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'Valeur d'';attribut invalide';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Double nom d'';attribut dans un lment';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Nom d'';entit invalide';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Instruction de commande invalide';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Dclaration XML invalide';
      ET_INVALID_CHARREF:
        ErrorStr := 'Rfrence de caractre invalide';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'Apostrophe (''): manquant';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'Signe gale manquant';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Double signe gale';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'Espace manquant';
      ET_MISSING_START_TAG:
        ErrorStr := 'Tag de fin sans tag de dbut';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'ET_MISSING_ELEMENT_NAME';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'ET_MISSING_ENTITY_NAME';
      ET_MISSING_END_TAG:
        ErrorStr := 'Tag de fin manquant';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Caractre invalide';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Caractre(s) en dehors de l'';lment racine';
      ET_ROOT_NOT_FOUND:
        ErrorStr := 'Elment racine non trouv';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'ET_NOTATION_DECL_START_EXPECTED';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Dclaration de type de document invalide';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'ET_UNCLOSED_ATTLIST_DECL';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := 'ET_SYSTEM_KEYWORD_EXPECTED';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'ET_UNCLOSED_CDATA_SECTION';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'ET_UNCLOSED_CHARREF';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'ET_UNCLOSED_COMMENT';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'ET_UNCLOSED_CONDITIONAL_SECTION';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'ET_UNCLOSED_DOCTYPE';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'ET_UNCLOSED_ELEMENT';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'ET_UNCLOSED_ELEMENT_DECL';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'ET_UNCLOSED_ENTITY_DECL';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_ENTITY_REF';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'ET_UNCLOSED_NOTATION_DECL';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_PARAMETER_ENTITY_REF';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'ET_UNCLOSED_PROCESSING_INSTRUCTION';
      ET_WRONG_ORDER:
        ErrorStr := 'Mauvais ordre';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Dclaration de type inconnue';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Dclaration d'';attribut invalide';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'ET_INVALID_ATTLIST_DECL_NAME';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Section conditionnelle invalide';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Dclaration d'';lment invalide';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Dclaration d'';entit invalide';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'ET_INVALID_MARKUP_DECL';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Dclaration de notation invalide';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Dclaration d'';entit paramtre invalide';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Dclaration de texte invalide';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_LEFT_SQUARE_BRACKET_EXPECTED';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr :=
          '"<" dans le texte de remplacement d'';une entit se rfre directement ou indirectement  la valeur d'';un attribut';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr := 'La valeur d'';attribut se rfre  une entit externe';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := 'ET_PUBLIC_KEYWORD_EXPECTED';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'ET_QUOTATION_MARK_EXPECTED';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'Rfrence rcursive trouve';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Rfrence  une entite non parse';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_RIGHT_SQUARE_BRACKET_EXPECTED';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr :=
          'Rfrence  une entit ne contenant pas de marque correcte';
      ET_INVALID_COMMENT:
        ErrorStr := 'Commentaire invalide';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'Section CDATA invalide';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'Systme litral invalide';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'Chane pubid invalide';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'Non qualifi invalide';
      ET_INVALID_PREFIX:
        ErrorStr := 'Prfixe invalide';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'Espace de nom URI invalide';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'Espace de nom URI non trouv';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'Mauvais emboitement du prfixe mapp';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'Encodage non support';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'ET_DOUBLE_ATTLISTDECL';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Double dfinition d'';attribut';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Double dclaration d'';entit';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Double dclaration d'';entit paramtre';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Entit dclare inutilisable';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Entite dclare non trouve';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'ET_DOCTYPE_NOT_FOUND';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Double dclaration de type lment';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Nom dupliqu dans un contenu mixte';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr :=
          'ID d'';attribut dupliqu dans un lment dclar de mme type';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Notation du nom non dclar';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Notation d'';attribut dupliqu dans un lment dclar de mme type';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := 'Notation de token duplique';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr := 'Notation d'';attribut sur un type lment vide dclar';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := 'Token numr dupliqu';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'La valeur d''atribut ne correspond pas  son type';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Token dclar dupliqu';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr := 'ID attribut jamais dclar ni en #IMPLIED ni en #REQUIRED';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'Le type de l'';lment racine ne correspond pas  la dclaration du nom du document';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Dclaration du type d'';lment non trouve';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'Elment dclar EMPTY non vide';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'Le contenu d'';un lment avec un contenu mixte ne correspond pas  la dclaration';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'Le contenu d'';un lment ne correspond pas  la dclaration';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr := 'Model de contenu d'';un lment non dterministe';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Dclaration de notation duplique';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Dfinition d'';attribut non trouve';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Attribut requis non trouv';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'La valeur dclare et la valeur actuelle d'',un attribut fixe ne correspondent pas';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr :=
          'La valeur par dfaut d'';un attribut ne correspond pas  son type';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'valeur ID duplique';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'La valeur IDREF ou IDREFS pointe sur une valeur ID inexistante';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'La valeur ENTITY ou ENTITIES pointe sur une entite non parse inexistante';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Mauvaise dclaration d'';entite prdfinie';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Rfrence entit non rsolvable';
      ET_UNRESOLVABLE_PARAMETER_ENTITY_REFERENCE:
        ErrorStr := 'Rfrence entit paramtre non rsolvable';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := 'Sous ensemble externe de la DTD non trouve';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Dclaration entit paramtre non trouve';
    else
      ErrorStr := 'Code source invalide';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_1Codec, S);
end;

{ TDomGermanErrorTranslator }

constructor TDomGermanErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_de;
end;

function TDomGermanErrorTranslator.GetSupportedLanguages:
  TIso639LanguageCodeSet;
begin
  Result := [iso639_de];
end;

function TDomGermanErrorTranslator.Translate(
  const Error: TdomError): WideString;
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'Kritischer Fehler';
      DOM_SEVERITY_ERROR: SeverityStr := 'Fehler';
      DOM_SEVERITY_WARNING: SeverityStr := 'Warnung';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'Beginn einer Attribute-List-Deklaration erwartet';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'Beginn eines CDATA-Abschnitts erwartet';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'Beginn eines Kommentars erwarted';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr :=
          'Bedingter Abschnitt nicht erlaubt im internen Teil einer DTD';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'Beginn einer Dokument-Typ-Deklaration (DTD) erwartet';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'Beginn einer Element-Deklaration erwartet';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'Beginn einer Entitt-Deklaration erwartet';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Ungltiger Element-Name';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'Doppelter Bindestrich in einem Kommentar';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Doppeltes Wurzel-Element';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Doppelte Dokument-Typ-Deklaration (DTD)';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'Kommentartext endet mit einem Bindestrich';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Ungltiger Attributname';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'Ungltiger Attributwert';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Doppelter Attributname in einem Element';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Ungltiger Entittname';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Ungltige Verarbeitungsanweisung';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Ungltige XML-Deklaration';
      ET_INVALID_CHARREF:
        ErrorStr := 'Ungltige Zeichen-Referenz';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'Fehlende Anfhrungszeichen';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'Fehlendes Gleichheitszeichen';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Doppeltes Gleichheitszeichen';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'Fehlender Leerraum';
      ET_MISSING_START_TAG:
        ErrorStr := 'End-Tag ohne Start-Tag';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'Fehlender Elementname';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'Fehlender Entittenname';
      ET_MISSING_END_TAG:
        ErrorStr := 'Fehlendes End-Tag';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Ungltiges Zeichen';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Zeichen auerhalb des Wurzel-Elements';
      ET_ROOT_NOT_FOUND:
        ErrorStr := 'Wurzel-Element nicht gefunden';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'Beginn einer Notationsdeklaration erwartet';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Ungltige Dokumenttyp-Deklaration';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'Attributlisten-Deklaration nicht geschlossen';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := 'Schlsselwort "SYSTEM" expected';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'CDATA-Abschnitt nicht geschlossen';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'Zeichen-Referenz nicht geschlossen';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'Kommentar nicht geschlossen';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'Bedingter Abschnitt nicht geschlossen';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'Dokument-Typ-Deklaration (DTD) nicht abgeschlossen';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'Element nicht geschlossen';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'Element-Deklaration nicht geschlossen';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'Entitten-Deklaration nicht geschlossen';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'Entitten-Refernz nicht geschlossen';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'Notationsdeklaration nicht geschlossen';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'Parameter-Entitten-Referenz nicht geschlossen';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'Verarbeitungsanweisung nicht geschlossen';
      ET_WRONG_ORDER:
        ErrorStr := 'Falsche Reihenfolge';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Unbekannter Deklarationstyp';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Ungltige Attribut-Deklaration';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'Name einer Attribut-Deklaration ungltig';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Ungltiger bedingter Abschnitt';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Ungltige Element-Deklaration';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Ungltige Entitt-Deklaration';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'Ungltige Markup-Deklaration';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Ungltige Notations-Deklaration';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Ungltige Parameter-Entitten-Deklaration';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Ungltige Text-Deklaration';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := '"[" erwarted';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr :=
          '"<" im Ersetzungstext einer Entitt, die direkt oder indirekt in einem Attributwertes referenziert wird.';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr := 'Attribut-Wert verweist auf eine externe Entitt';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := 'Schlsselwort "PUBLIC" erwartet';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'Anfhrungszeichen erwartet';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'Rekursive Referenz gefunden';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Referenz auf eine nicht analysierte Entitt';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := '"]" erwartet';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr :=
          'Referenz auf eine Entitt mit ungltigen Markierunszeichen';
      ET_INVALID_COMMENT:
        ErrorStr := 'Ungltiger Kommentar';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'Ungltiger CDATA-Abschnitt';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'Ungltiges System-Literal';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'Ungltiges Pubid-Literal';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'Ungltiger qualifizierter Name';
      ET_INVALID_PREFIX:
        ErrorStr := 'Ungltiges Prefix';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'Ungltiger Namensraum-URI';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'Namensraum-URI nicht gefunden';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'Falsche Prefix-Verschachtelung';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'Nicht-untersttze Zeichencodierung';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'Doppelte Attributlisten-Deklaration';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Doppelte Attribut-Defintion';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Doppelte Entitt-Deklaration';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Doppelte Parameter-Entitt-Deklaration';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Unbrauchbare Entitt deklariert';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Entitt-Declaration nicht gefunden';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'Dokument-Typ-Deklaration (DTD) fehlt';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Doppelte Element-Typ-Deklaration';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Doppelter Name in gemischtem Inhalt';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr := 'ID-Attribut mehrfach fr denselben Element-Typ deklariert';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Notationsname nicht deklariert';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Notationsattribut mehrfach fr denselben Element-Typ deklariert';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := 'Doppelte Notationstokens';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr := 'Notationsattribut fr einen leeren Element-Typ deklariert';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := 'Doppeltes Aufzhlungstoken';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'Attribut-Typ und Attribut-Wert stimmen nicht berein';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Token mehrfach deklariert';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr :=
          'ID-Attribut ist weder als #IMPLIED noch als #REQUIRED deklariert';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'Typ des Wurzelelements entspricht nicht der Angabe in der Dokumenttyp-Deklaration';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Elementtyp-Deklaration nicht gefunden';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'Als EMPTY deklariertes Element ist nicht leer';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'Inhalt eines Elements gemischten Inhalts entspricht nicht der Deklaration';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'Inhalt eines Elements mit Element-Inhalt entspricht nicht der Deklaration';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr := 'Inhaltsmodell eines Elements ist nicht deterministisch';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Notation mehrfach deklariert';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Attribut-Definition nicht gefunden';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Als "#REQUIRED" deklariertes Attribut nicht gefunden';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'Werte eines als "#FIXED" deklarierten Attributs entspricht nicht der Deklaration';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr := 'Attribut-Typ und Attribut-Vorgabe stimmen nicht berein';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'ID-Wert mehrfach verwendet';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'Ein IDREF- bzw. IDREFS-Attribute verweist auf ein nicht vorhandenes Ziel';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'Ein ENTITY- bzw. ENTITIES-Attribute verweist auf eine nicht analysierte Entitt, die nicht deklariert wurde';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Falsche Deklaration einer vordefinierten Entitt';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Nicht auflsbare Entitten-Referenz';
      ET_UNRESOLVABLE_PARAMETER_ENTITY_REFERENCE:
        ErrorStr := 'Nicht auflsbare Parameter-Entitten-Referenz';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := 'Externe Teilmenge der DTD nicht gefunden';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Parameter-Entitt-Deklaration nicht gefunden';
    else
      ErrorStr := 'Ungltiger Quellcode';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_1Codec, S);
end;

{ TDomItalianErrorTranslator }

constructor TDomItalianErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_it;
end;

function TDomItalianErrorTranslator.GetSupportedLanguages:
  TIso639LanguageCodeSet;
begin
  Result := [iso639_it];
end;

function TDomItalianErrorTranslator.Translate(
  const Error: TdomError): WideString;
// - This function was provided by Massimo Maria Ghisalberti -
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'Errore Critico';
      DOM_SEVERITY_ERROR: SeverityStr := 'Errore';
      DOM_SEVERITY_WARNING: SeverityStr := 'Attenzione';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'ET_ATTLIST_DECL_START_EXPECTED';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'ET_CDATA_START_EXPECTED';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'ET_COMMENT_START_EXPECTED';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr := 'ET_CONDITIONAL_SECTION_NOT_ALLOWED';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'ET_DOCTYPE_START_EXPECTED';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'ET_ELEMENT_DECL_START_EXPECTED';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'ET_ENTITY_DECL_START_EXPECTED';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Nome elemento non valido';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'ET_DOUBLE_HYPHEN_IN_COMMENT';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Elemento radice duplicato';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Dichiarazione del tipo di documento (DTD) duplicata';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'ET_HYPHEN_AT_COMMENT_END';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Nome attributo non valido';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'Valore attributo non valido';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Nome dell''attributo duplicato in un elemento';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Nome entit non valido';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Istruzione di processo non valida';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Dichiarazione XML non valida';
      ET_INVALID_CHARREF:
        ErrorStr := 'Riferimento a carattere non valida';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'Virgolette mancanti';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'Segno di uguale mancante';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Segno di uguale duplicato';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'Spazio mancante';
      ET_MISSING_START_TAG:
        ErrorStr := 'Tag finale senza tag iniziale';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'ET_MISSING_ELEMENT_NAME';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'ET_MISSING_ENTITY_NAME';
      ET_MISSING_END_TAG:
        ErrorStr := 'Tag finale mancante';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Carattere non valido';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Carattere(i) oltre l''elemento radice';
      ET_ROOT_NOT_FOUND:
        ErrorStr := 'Elemento radice non trovato';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'ET_NOTATION_DECL_START_EXPECTED';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Dichiarazione del tipo di documento non valida';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'ET_UNCLOSED_ATTLIST_DECL';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := 'ET_SYSTEM_KEYWORD_EXPECTED';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'ET_UNCLOSED_CDATA_SECTION';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'ET_UNCLOSED_CHARREF';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'ET_UNCLOSED_COMMENT';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'ET_UNCLOSED_CONDITIONAL_SECTION';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'ET_UNCLOSED_DOCTYPE';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'ET_UNCLOSED_ELEMENT';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'ET_UNCLOSED_ELEMENT_DECL';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'ET_UNCLOSED_ENTITY_DECL';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_ENTITY_REF';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'ET_UNCLOSED_NOTATION_DECL';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_PARAMETER_ENTITY_REF';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'ET_UNCLOSED_PROCESSING_INSTRUCTION';
      ET_WRONG_ORDER:
        ErrorStr := 'Ordine errato';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Dichiarazione di tipo sconosciuta';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Dichiarazione di attributo non valida';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'ET_INVALID_ATTLIST_DECL_NAME';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Sezione condizionale non valida';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Dichiarazione di elemento non valida';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Dichiarazione di entit non valida';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'Dichiarazione di markup non valida';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Dichiarazione di notazione non valida';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Dichiarazione di entit con parametri non valida';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Dichiarazione di testo non valida';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_LEFT_SQUARE_BRACKET_EXPECTED';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr :=
          '"<" nel testo di sostituzione di una entit riferita direttamente o indirettamente al valore di un attributo';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr :=
          'Il valore dell''attributo si riferisce ad un''entit esterna';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := 'ET_PUBLIC_KEYWORD_EXPECTED';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'ET_QUOTATION_MARK_EXPECTED';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'Trovato riferimento ricorsivo';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Riferimento ad una entit non analizzata';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_RIGHT_SQUARE_BRACKET_EXPECTED';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr := 'Riferimento ad una entit senza un marcatore adeguato';
      ET_INVALID_COMMENT:
        ErrorStr := 'Commento non valido';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'Sezione CDATA non valida';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'Stringa di sistema non valida';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'Stringa PUBID non valida';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'Nome qualificano non valido';
      ET_INVALID_PREFIX:
        ErrorStr := 'Prefisso non valido';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'namespace URI non valido';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'namespace URI non trovato';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'Incapsulamento errato nel mapping del presisso';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'Codifica non supportata';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'ET_DOUBLE_ATTLISTDECL';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Definizione di attibuto duplicata';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Dichiarazione di entit duplicata';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Dichiarazione di entit con parametri duplicati';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Entit dichiarata non utilizzabile';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Dichiarazione di entit non trovata';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'ET_DOCTYPE_NOT_FOUND';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Dichiarazione di tipo dell''elemento duplicata';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Nome duplicato in un contenuto misto';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Attributo ID duplicato nella stessa dichiarazione di tipo di elemento';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Nome di notazione non dichiarato';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Attributo di notazione duplicato nello stessa dichiarazione di tipo di elemento';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := 'Notazione del token duplicata';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr :=
          'Attributo di notazione nella dichiarazione di tipo di elemento vuoto';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := 'Enumerazione del token duplicata';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'L''attributo di tipo e il valore non coincidono';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Dichiarazione duplicata di token';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr :=
          'L''attributo ID non  stato dichiarato come #IMPLIED n come #REQUIRED';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'Il tipo dell''elemento radice non coincide con il nome della dichiarazione di tipo del documento';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Dichiarazione del tipo dell''elemento non trovata';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'L''elemento dichiarato vuoto non lo ';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'Il contenuto di un elemento con contenuto misto non coincide con la dichiarazione';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'Il contenuto di un elemento che contiene elementi non coincide con la dichiarazione';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr :=
          'Il modello di contenuto di un elemento non  deterministico';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Dichiarazione di notazione duplicata';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Definizione di attributo non trovata';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Attributo richiesto non trovato';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'Il valore dichiarato e quello attuale di un attributo fisso non coincidono';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr :=
          'Il tipo di un attributo ed il suo predefinito non coincidono';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'Valore ID duplicato';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'Il valore IDREF o IDREFS si riferisce ad un valore ID inesistente';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'Il valore ENTITY o ENTITIES si riferisce ad una entit non analizzata non esistente';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Dichiarazione di entit predefinita errata';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Riferimento non risolvibile ad una entit';
      ET_UNRESOLVABLE_PARAMETER_ENTITY_REFERENCE:
        ErrorStr := 'Riferimento non risolvibile ad una entit con parametri';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := 'Sottoinsieme esterno del DTD non trovato';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Dichiarazione del parametro di entit non trovata';
    else
      ErrorStr := 'Codice sorgente non valido';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_1Codec, S);
end;

{ TDomPolishErrorTranslator }

constructor TDomPolishErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_pl;
end;

function TDomPolishErrorTranslator.GetSupportedLanguages:
  TIso639LanguageCodeSet;
begin
  Result := [iso639_pl];
end;

function TDomPolishErrorTranslator.Translate(
  const Error: TdomError): WideString;
// - This function was provided by Piotr Kuczynski -
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'FATALNY BD';
      DOM_SEVERITY_ERROR: SeverityStr := 'BD';
      DOM_SEVERITY_WARNING: SeverityStr := 'OSTRZEENIE';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'ET_ATTLIST_DECL_START_EXPECTED';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'ET_CDATA_START_EXPECTED';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'ET_COMMENT_START_EXPECTED';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr := 'ET_CONDITIONAL_SECTION_NOT_ALLOWED';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'ET_DOCTYPE_START_EXPECTED';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'ET_ELEMENT_DECL_START_EXPECTED';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'ET_ENTITY_DECL_START_EXPECTED';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Niewaciwa nazwa elementu';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'ET_DOUBLE_HYPHEN_IN_COMMENT';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Podwjnie zdefiniowany korze';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Podwjna deklaracja typu dokumentu (DTD)';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'ET_HYPHEN_AT_COMMENT_END';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Niewaciwa nazwa atrybutu';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'Niewaciwa warto atrybutu';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Powtarzajca si nazwa atrybutu';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Niewaciwa nazwa encji';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Niewaciwa instrukcja przetwarzania';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Niewaciwa deklaracja XML';
      ET_INVALID_CHARREF:
        ErrorStr := 'Niewaciwy znak w odwoaniu';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'Brakujcy cudzysw';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'Brakujcy znak rwnoci';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Podwjny znak rwnoci';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'Brakujcy znak spacji';
      ET_MISSING_START_TAG:
        ErrorStr := 'Znacznik zamykajcy bez znacznika otwierajcego';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'ET_MISSING_ELEMENT_NAME';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'ET_MISSING_ENTITY_NAME';
      ET_MISSING_END_TAG:
        ErrorStr := 'Brakujcy znacznik zamykajcy';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Niewaciwy znak';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Znak(i) umieszczone poza zasigiem korzenia';
      ET_ROOT_NOT_FOUND:
        ErrorStr := 'ET_ROOT_NOT_FOUND';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'ET_NOTATION_DECL_START_EXPECTED';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Niewaciwa deklaracja typu dokumentu';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'ET_UNCLOSED_ATTLIST_DECL';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := 'ET_SYSTEM_KEYWORD_EXPECTED';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'ET_UNCLOSED_CDATA_SECTION';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'ET_UNCLOSED_CHARREF';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'ET_UNCLOSED_COMMENT';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'ET_UNCLOSED_CONDITIONAL_SECTION';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'ET_UNCLOSED_DOCTYPE';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'ET_UNCLOSED_ELEMENT';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'ET_UNCLOSED_ELEMENT_DECL';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'ET_UNCLOSED_ENTITY_DECL';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_ENTITY_REF';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'ET_UNCLOSED_NOTATION_DECL';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_PARAMETER_ENTITY_REF';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'ET_UNCLOSED_PROCESSING_INSTRUCTION';
      ET_WRONG_ORDER:
        ErrorStr := 'Niewaciwa kolejno';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Nieznany typ deklaracji';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Niewaciwa deklaracja atrybutu';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'ET_INVALID_ATTLIST_DECL_NAME';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Niewaciwa sekcja warunkowa';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Niewaciwa deklaracja elementu';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Niewaciwa deklaracja encji';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'ET_INVALID_MARKUP_DECL';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Niewaciwa deklaracja notacji';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Niewaciwa deklaracja encji parametrycznej';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Niewaciwa deklaracja tekstu';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_LEFT_SQUARE_BRACKET_EXPECTED';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr := 'ET_LT_IN_ATTRIBUTE_VALUE';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr := 'Warto atrybutu odwouje si do zewntrznej encji';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := 'ET_PUBLIC_KEYWORD_EXPECTED';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'ET_QUOTATION_MARK_EXPECTED';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'ET_RECURSIVE_REFERENCE';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Odwoanie do encji, ktra nie bya jeszcze parsowana';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_RIGHT_SQUARE_BRACKET_EXPECTED';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr := 'Odwoanie do encji zawierajcej niewaciwy znacznik';
      ET_INVALID_COMMENT:
        ErrorStr := 'ET_INVALID_COMMENT';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'ET_INVALID_CDATA_SECTION';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'ET_INVALID_SYSTEM_LITERAL';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'ET_INVALID_PUBID_LITERAL';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'ET_INVALID_QUALIFIED_NAME';
      ET_INVALID_PREFIX:
        ErrorStr := 'ET_INVALID_PREFIX';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'ET_INVALID_NAMESPACE_URI';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'ET_NAMESPACE_URI_NOT_FOUND';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'ET_WRONG_PREFIX_MAPPING_NESTING';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'ET_ENCODING_NOT_SUPPORTED';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'ET_DOUBLE_ATTLISTDECL';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Podwjna definicja atrybutu';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Podwjna deklaracja encji';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Podwjna deklaracja encji parametrycznej';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Nieuywana deklaracja encji';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Brakujca deklaracja encji';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'ET_DOCTYPE_NOT_FOUND';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Podwjna deklaracja typu elementu';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Podwjna nazwa w mieszanej zawartoci';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Zadeklarowany podwjny atrybut ID dla elementw tego samego typu';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Nie zadeklarowana nazwa notacji';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Zadeklarowany podwjny atrybut notacji dla elementw tego samego typu';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := 'ET_DUPLICATE_NOTATION_TOKEN';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr := 'Atrybut notacji zadeklarowany dla elementu typu pusty';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := 'ET_DUPLICATE_ENUMERATION_TOKEN';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'Typ atrybutu i warto atrybutu nie pasuj do siebie';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Zadeklarowane dwa identyczne tokeny';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr :=
          'Atrybut ID nie jest zadeklarowany ani jako #IMPLIED ani #REQUIRED';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'Typ korzenia nie pasuje do nazwy z deklaracji typu dokumentu';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Nie znaleziona deklaracja typu elementu';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'Element zadeklarowany jako EMPTY nie jest pusty';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'Zawarto elementu o mieszanej zawartoci nie pasuje do deklaracji';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'Zawarto elementu, zawierajcego element, nie pasuje do deklaracji';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr := 'Model zawartoci elementu jest niedeterministyczny';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Podwjna deklaracja notacji';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Definicja atrybutu nie zostaa znaleziona';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Wymagany atrybut nie zosta znaleziony';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'Zadeklarowana i bieca warto staego atrybutu nie pasuja do siebie';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr := 'Typ atrybutu i jego warto domylna nie pasuj do siebie';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'Podwjna warto ID';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'Warto IDREF lub IDREFS odwouje si do wartoci ID nieistniejcego elementu';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'Warto ENTITY lub ENTITIES odwouje si do niesparsowanej encji, ktra nie istnieje';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Niewaciwa deklaracja predefiniowanej encji';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Nierozwizywalna referencja do encji';
      ET_UNRESOLVABLE_PARAMETER_ENTITY_REFERENCE:
        ErrorStr := 'Nierozwizywalna referencja encji parametrycznej';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := 'Zewntrzny podzbir DTD nie zosta znaleziony';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'ET_PARAMETER_ENTITY_DECL_NOT_FOUND';
    else
      ErrorStr := 'Niewaciwy kod rdowy';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_2Codec, S);
end;

{ TDomPortugueseErrorTranslator }

constructor TDomPortugueseErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_pt;
end;

function TDomPortugueseErrorTranslator.GetSupportedLanguages:
  TIso639LanguageCodeSet;
begin
  Result := [iso639_pt];
end;

function TDomPortugueseErrorTranslator.Translate(
  const Error: TdomError): WideString;
// - This function was provided by Ricardo Albuquerque (Recife, Brazil) -
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'Erro Fatal';
      DOM_SEVERITY_ERROR: SeverityStr := 'Erro';
      DOM_SEVERITY_WARNING: SeverityStr := 'Ateno';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'ET_ATTLIST_DECL_START_EXPECTED';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'ET_CDATA_START_EXPECTED';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'ET_COMMENT_START_EXPECTED';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr := 'ET_CONDITIONAL_SECTION_NOT_ALLOWED';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'ET_DOCTYPE_START_EXPECTED';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'ET_ELEMENT_DECL_START_EXPECTED';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'ET_ENTITY_DECL_START_EXPECTED';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Nome de elemento invlido';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'ET_DOUBLE_HYPHEN_IN_COMMENT';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Elemento raiz duplicado';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Definio de tipo de documento(DTD) duplicado';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'ET_HYPHEN_AT_COMMENT_END';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Nome de atributo invlido';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'valor de atributo invlido';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Nome de atributo duplicado em um elemento';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Nome de entidade invlida';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Instruo de processamento invlida';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Declarao XML invlida';
      ET_INVALID_CHARREF:
        ErrorStr := 'Referncia a caracter invalida';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'Aspas faltando';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'Sinal de igualdade faltando';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Sinal de igualdade duplicado';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'Espao em branco faltando';
      ET_MISSING_START_TAG:
        ErrorStr := 'Marca de fim sem marca de incio ';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'ET_MISSING_ELEMENT_NAME';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'ET_MISSING_ENTITY_NAME';
      ET_MISSING_END_TAG:
        ErrorStr := 'Marca de fim faltando';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Caracter invlido';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Caracter(es) fora do elemento raiz';
      ET_ROOT_NOT_FOUND:
        ErrorStr := 'Elemento raiz no encontrado';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'ET_NOTATION_DECL_START_EXPECTED';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Declarao de tipo de documento invlida';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'ET_UNCLOSED_ATTLIST_DECL';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := 'ET_SYSTEM_KEYWORD_EXPECTED';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'ET_UNCLOSED_CDATA_SECTION';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'ET_UNCLOSED_CHARREF';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'ET_UNCLOSED_COMMENT';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'ET_UNCLOSED_CONDITIONAL_SECTION';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'ET_UNCLOSED_DOCTYPE';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'ET_UNCLOSED_ELEMENT';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'ET_UNCLOSED_ELEMENT_DECL';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'ET_UNCLOSED_ENTITY_DECL';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_ENTITY_REF';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'ET_UNCLOSED_NOTATION_DECL';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_PARAMETER_ENTITY_REF';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'ET_UNCLOSED_PROCESSING_INSTRUCTION';
      ET_WRONG_ORDER:
        ErrorStr := 'Ordem errada';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Tipo de declarao invlida';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Declarao de atributo invlido';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'ET_INVALID_ATTLIST_DECL_NAME';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Seo condicional invlida';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Declarao de elemento invlido';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Declarao de entidade invlida';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'Declarao de markup invlido';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Declarao de notao invlida';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Declarao de entidade paramtrica invlida';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Declarao de texto invlido';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_LEFT_SQUARE_BRACKET_EXPECTED';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr :=
          '"<" no texto de substituio de uma entidade refere-se diretamente ou indiretamente a um  valor de atributo';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr := 'O valor do atributo se refere a uma entidade externa';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := 'ET_PUBLIC_KEYWORD_EXPECTED';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'ET_QUOTATION_MARK_EXPECTED';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'Encontrada Referncia recursiva';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Referncia a uma entidade no analisada';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_RIGHT_SQUARE_BRACKET_EXPECTED';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr := 'Referncia a uma entidade contendo marcao imprpria';
      ET_INVALID_COMMENT:
        ErrorStr := 'Comentrio Invlido';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'Sesso CDATA invlida';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'Literal de sistema invlido';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'Literal pubid invlido';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'Nome qualificado invlido';
      ET_INVALID_PREFIX:
        ErrorStr := 'Prefixo invlido ';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'NameSpace URI invlido';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'NameSpace URI no encontrado';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'Aninhamento de mapeamento de prefixo invlido';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'Codificao no suportada';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'ET_DOUBLE_ATTLISTDECL';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Definio de atributo duplicada';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Declarao de entidade duplicada';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Declarao de entidade paramtrica duplicada';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Entidade inutilizvel declarada';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Faltando declarao de entidade';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'ET_DOCTYPE_NOT_FOUND';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Declarao do tipo do elemento duplicada';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Nome duplicado no contedo misto';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Atributo ID duplicado na mesma declarao de tipo de elemento';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Nome de anotao no declarado';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Atributo de anotao duplicado na mesma declarao de tipo de elemento';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := ' "Token"  de notao duplicada';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr :=
          'Atributo de anotao em uma declarao de tipo de elemento vazia';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := ' "Token"  de enumerao duplicada';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'Tipo de atributo e valor de atributo no coincidem';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Declarao de "tokens" duplicada';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr :=
          'Atributo ID no foi declarado como #IMPLIED nem como #REQUIRED';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'O tipo de elemento raiz no tem o mesmo nome que a declarao do tipo de documento';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Declarao de tipo de elemento no encontrada';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'Elemento declarado como "EMPTY" no est vazio';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'O contedo de um elemento com contedo misto no coincide com sua declarao';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'O contedo de um elemento com contedo de elemento no coincide com sua declarao';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr := 'O modelo de contedo de um elemento no  determinante';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Declarao de anotao duplicada';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Definio de atributo no encontrada';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Atributo requerido no encontrado';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'O valor declarado e o valor atual de um atributo fixo no coincidem';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr :=
          'O tipo de atributo e o atributo por definio no coincidem';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'Valor de ID duplicado';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'O valor de IDREF ou IDREFS se refere a um valor de ID no existente';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'O valor de ENTITY ou ENTITIES se refere a uma entidade no existente';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Declarao errnea de uma entidade pr-definida';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Referncia a entidade impossvel de resolver';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := 'Subgrupo externo do DTD no encontrado';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Declarao de entidade de parmetro no encontrada';
    else
      ErrorStr := 'Cdigo fonte invlido';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_1Codec, S);
end;

{ TDomSpanishErrorTranslator }

constructor TDomSpanishErrorTranslator.Create(AOwner: TComponent);
begin
  inherited;
  FLanguage := iso639_es;
end;

function TDomSpanishErrorTranslator.GetSupportedLanguages:
  TIso639LanguageCodeSet;
begin
  Result := [iso639_es];
end;

function TDomSpanishErrorTranslator.Translate(
  const Error: TdomError): WideString;
// - This function was provided by Pedro de Paz -
var
  ErrorStr, FileNameStr, PosStr, SeverityStr, S: string;
begin
  with Error do
  begin
    case Severity of
      DOM_SEVERITY_FATAL_ERROR: SeverityStr := 'Error Fatal';
      DOM_SEVERITY_ERROR: SeverityStr := 'Error';
      DOM_SEVERITY_WARNING: SeverityStr := 'Aviso';
    end;

    FileNameStr := ExtractFileNameFromUri(Uri);
    if EndLineNumber < 1 then
      PosStr := ''
    else if StartLineNumber = EndLineNumber then
    begin
      if StartColumnNumber = EndColumnNumber then
        FmtStr(PosStr, '%d:%d', [EndLineNumber, EndColumnNumber])
      else
        FmtStr(PosStr, '%d:%d-%d', [EndLineNumber, StartColumnNumber,
          EndColumnNumber]);
    end
    else
    begin
      FmtStr(PosStr, '%d:%d-%d:%d', [StartLineNumber, StartColumnNumber,
        EndLineNumber, EndColumnNumber]);
    end;

    case RelatedException of
      ET_ATTLIST_DECL_START_EXPECTED:
        ErrorStr := 'ET_ATTLIST_DECL_START_EXPECTED';
      ET_CDATA_START_EXPECTED:
        ErrorStr := 'ET_CDATA_START_EXPECTED';
      ET_COMMENT_START_EXPECTED:
        ErrorStr := 'ET_COMMENT_START_EXPECTED';
      ET_CONDITIONAL_SECTION_NOT_ALLOWED:
        ErrorStr := 'ET_CONDITIONAL_SECTION_NOT_ALLOWED';
      ET_DOCTYPE_START_EXPECTED:
        ErrorStr := 'ET_DOCTYPE_START_EXPECTED';
      ET_ELEMENT_DECL_START_EXPECTED:
        ErrorStr := 'ET_ELEMENT_DECL_START_EXPECTED';
      ET_ENTITY_DECL_START_EXPECTED:
        ErrorStr := 'ET_ENTITY_DECL_START_EXPECTED';
      ET_INVALID_ELEMENT_NAME:
        ErrorStr := 'Nombre de elemento no vlido';
      ET_DOUBLE_HYPHEN_IN_COMMENT:
        ErrorStr := 'ET_DOUBLE_HYPHEN_IN_COMMENT';
      ET_DOUBLE_ROOT_ELEMENT:
        ErrorStr := 'Elemento raiz duplicado';
      ET_DOUBLE_DOCTYPE:
        ErrorStr := 'Declaracin de tipo de documento (DTD) duplicada';
      ET_HYPHEN_AT_COMMENT_END:
        ErrorStr := 'ET_HYPHEN_AT_COMMENT_END';
      ET_INVALID_ATTRIBUTE_NAME:
        ErrorStr := 'Nombre de atributo no vlido';
      ET_INVALID_ATTRIBUTE_VALUE:
        ErrorStr := 'Valor de atributo no vlido';
      ET_DOUBLE_ATTRIBUTE_NAME:
        ErrorStr := 'Atributo duplicado en un elemento';
      ET_INVALID_ENTITY_NAME:
        ErrorStr := 'Nombre de entidad no vlida';
      ET_INVALID_PROCESSING_INSTRUCTION:
        ErrorStr := 'Instruccin de proceso no vlida';
      ET_INVALID_XML_DECL:
        ErrorStr := 'Declaracin XML no vlida';
      ET_INVALID_CHARREF:
        ErrorStr := 'Referencia a caracter no vlida';
      ET_MISSING_QUOTATION_MARK:
        ErrorStr := 'No se encontraron comillas';
      ET_MISSING_EQUALITY_SIGN:
        ErrorStr := 'No se encontr simbolo de igualdad';
      ET_DOUBLE_EQUALITY_SIGN:
        ErrorStr := 'Signo de igualdad duplicado';
      ET_MISSING_WHITE_SPACE:
        ErrorStr := 'No se encontr espacio en blanco';
      ET_MISSING_START_TAG:
        ErrorStr := 'Tag final sin tag de inicio';
      ET_MISSING_ELEMENT_NAME:
        ErrorStr := 'ET_MISSING_ELEMENT_NAME';
      ET_MISSING_ENTITY_NAME:
        ErrorStr := 'ET_MISSING_ENTITY_NAME';
      ET_MISSING_END_TAG:
        ErrorStr := 'Tag de inicio sin tag final';
      ET_INVALID_CHARACTER:
        ErrorStr := 'Caracter no vlido';
      ET_NOT_IN_ROOT:
        ErrorStr := 'Caracter(es) fuera del elemento raiz';
      ET_ROOT_NOT_FOUND:
        ErrorStr := 'Elemento raiz no encontrado';
      ET_NOTATION_DECL_START_EXPECTED:
        ErrorStr := 'ET_NOTATION_DECL_START_EXPECTED';
      ET_INVALID_DOCTYPE:
        ErrorStr := 'Declaracin de tipo de documento no vlida';
      ET_UNCLOSED_ATTLIST_DECL:
        ErrorStr := 'ET_UNCLOSED_ATTLIST_DECL';
      ET_SYSTEM_KEYWORD_EXPECTED:
        ErrorStr := 'ET_SYSTEM_KEYWORD_EXPECTED';
      ET_UNCLOSED_CDATA_SECTION:
        ErrorStr := 'ET_UNCLOSED_CDATA_SECTION';
      ET_UNCLOSED_CHARREF:
        ErrorStr := 'ET_UNCLOSED_CHARREF';
      ET_UNCLOSED_COMMENT:
        ErrorStr := 'ET_UNCLOSED_COMMENT';
      ET_UNCLOSED_CONDITIONAL_SECTION:
        ErrorStr := 'ET_UNCLOSED_CONDITIONAL_SECTION';
      ET_UNCLOSED_DOCTYPE:
        ErrorStr := 'ET_UNCLOSED_DOCTYPE';
      ET_UNCLOSED_ELEMENT:
        ErrorStr := 'ET_UNCLOSED_ELEMENT';
      ET_UNCLOSED_ELEMENT_DECL:
        ErrorStr := 'ET_UNCLOSED_ELEMENT_DECL';
      ET_UNCLOSED_ENTITY_DECL:
        ErrorStr := 'ET_UNCLOSED_ENTITY_DECL';
      ET_UNCLOSED_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_ENTITY_REF';
      ET_UNCLOSED_NOTATION_DECL:
        ErrorStr := 'ET_UNCLOSED_NOTATION_DECL';
      ET_UNCLOSED_PARAMETER_ENTITY_REF:
        ErrorStr := 'ET_UNCLOSED_PARAMETER_ENTITY_REF';
      ET_UNCLOSED_PROCESSING_INSTRUCTION:
        ErrorStr := 'ET_UNCLOSED_PROCESSING_INSTRUCTION';
      ET_WRONG_ORDER:
        ErrorStr := 'Orden equivocado';
      ET_UNKNOWN_DECL_TYPE:
        ErrorStr := 'Tipo de declaracin desconocida';
      ET_INVALID_ATTRIBUTE_DECL:
        ErrorStr := 'Declaracin de atributo no vlida';
      ET_INVALID_ATTLIST_DECL_NAME:
        ErrorStr := 'ET_INVALID_ATTLIST_DECL_NAME';
      ET_INVALID_CONDITIONAL_SECTION:
        ErrorStr := 'Seccin condicional no vlida';
      ET_INVALID_ELEMENT_DECL:
        ErrorStr := 'Declaracin de elemento no vlida';
      ET_INVALID_ENTITY_DECL:
        ErrorStr := 'Declaracin de entidad no vlida';
      ET_INVALID_MARKUP_DECL:
        ErrorStr := 'Declaracin de markup no vlida';
      ET_INVALID_NOTATION_DECL:
        ErrorStr := 'Declaracin de anotacin no vlida';
      ET_INVALID_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Declaracin de entidad con parametros no vlida';
      ET_INVALID_TEXT_DECL:
        ErrorStr := 'Declaracin de texto no vlida';
      ET_LEFT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_LEFT_SQUARE_BRACKET_EXPECTED';
      ET_LT_IN_ATTRIBUTE_VALUE:
        ErrorStr :=
          'Encontrado el caracter "<" en el texto de sustitucin de una entidad que refiere directa o indirectamente a un valor de atributo';
      ET_ATTRIBUTE_VALUE_REFERS_TO_EXTERNAL_ENTITY:
        ErrorStr := 'Valor de atributo apuntando a una entidad externa';
      ET_PUBLIC_KEYWORD_EXPECTED:
        ErrorStr := 'ET_PUBLIC_KEYWORD_EXPECTED';
      ET_QUOTATION_MARK_EXPECTED:
        ErrorStr := 'ET_QUOTATION_MARK_EXPECTED';
      ET_RECURSIVE_REFERENCE:
        ErrorStr := 'Encontrada referencia recursiva';
      ET_REFERS_TO_UNPARSED_ENTITY:
        ErrorStr := 'Referencia a una entidad no analizada';
      ET_RIGHT_SQUARE_BRACKET_EXPECTED:
        ErrorStr := 'ET_RIGHT_SQUARE_BRACKET_EXPECTED';
      ET_NO_PROPER_MARKUP_REFERENCED:
        ErrorStr :=
          'Referencia a una entidad que contiene un "markup" no adecuado';
      ET_INVALID_COMMENT:
        ErrorStr := 'Comentario no vlido';
      ET_INVALID_CDATA_SECTION:
        ErrorStr := 'Seccin CDATA no vlida';
      ET_INVALID_SYSTEM_LITERAL:
        ErrorStr := 'Literal SYSTEM no vlido';
      ET_INVALID_PUBID_LITERAL:
        ErrorStr := 'Literal PUBID no vlido';
      ET_INVALID_QUALIFIED_NAME:
        ErrorStr := 'Nombre cualificado no vlido';
      ET_INVALID_PREFIX:
        ErrorStr := 'Prefijo no vlido';
      ET_INVALID_NAMESPACE_URI:
        ErrorStr := 'NAMESPACE URI no vlido';
      ET_NAMESPACE_URI_NOT_FOUND:
        ErrorStr := 'NAMESPACE URI no encontrado';
      ET_WRONG_PREFIX_MAPPING_NESTING:
        ErrorStr := 'Anidamiento de mapeo de prefijos incorrecto';
      ET_ENCODING_NOT_SUPPORTED:
        ErrorStr := 'Tipo de ENCODING no soportado';
      ET_DOUBLE_ATTLISTDECL:
        ErrorStr := 'ET_DOUBLE_ATTLISTDECL';
      ET_DOUBLE_ATTDEF:
        ErrorStr := 'Definicin de atributo duplicada';
      ET_DOUBLE_ENTITY_DECL:
        ErrorStr := 'Declaracin de entidad duplicada';
      ET_DOUBLE_PARAMETER_ENTITY_DECL:
        ErrorStr := 'Declaracin de entidad con parametros duplicados';
      ET_UNUSABLE_ENTITY_DECL:
        ErrorStr := 'Declaracin de entidad no utilizable';
      ET_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'No existe declaracin de entidad';
      ET_DOCTYPE_NOT_FOUND:
        ErrorStr := 'ET_DOCTYPE_NOT_FOUND';
      ET_DUPLICATE_ELEMENT_TYPE_DECL:
        ErrorStr := 'Duplicada declaracin de tipo de elemento';
      ET_DUPLICATE_NAME_IN_MIXED_CONTENT:
        ErrorStr := 'Nombre duplicado en contenido mezclado';
      ET_DUPLICATE_ID_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Atributo ID duplicado en la misma declaracin de tipo de elemento';
      ET_UNDECLARED_NOTATION_NAME:
        ErrorStr := 'Nombre de anotacin no declarado';
      ET_DUPLICATE_NOTATION_ON_ELEMENT_TYPE:
        ErrorStr :=
          'Atributo de anotacin duplicado en la misma declaracin de tipo de elemento';
      ET_DUPLICATE_NOTATION_TOKEN:
        ErrorStr := '"token" de notacin duplicado';
      ET_NOTATION_ON_EMPTY_ELEMENT:
        ErrorStr :=
          'Atributo de anotacin en una declaracin de tipo de elemento vacia';
      ET_DUPLICATE_ENUMERATION_TOKEN:
        ErrorStr := '"token" de enumeracin duplicado';
      ET_ATTRIBUTE_TYPE_MISMATCH:
        ErrorStr := 'Tipo de atributo y valor de atributo no coinciden';
      ET_DUPLICATE_TOKENS:
        ErrorStr := 'Declaracin de "tokens" duplicada';
      ET_ID_NEITHER_IMPLIED_NOR_REQUIRED:
        ErrorStr :=
          'Atributo ID no ha sido declarado como #IMPLIED ni como #REQUIRED';
      ET_WRONG_ROOT_ELEMENT_TYPE:
        ErrorStr :=
          'El tipo del elemento raiz no tiene el mismo nombre que la declaracin del tipo de documento';
      ET_ELEMENT_TYPE_DECL_NOT_FOUND:
        ErrorStr := 'Declaracin de tipo de elemento no encontrada';
      ET_ELEMENT_DECLARED_EMPTY_HAS_CONTENT:
        ErrorStr := 'Elemento declarado como "EMPTY" no esta vacio';
      ET_ELEMENT_WITH_ILLEGAL_MIXED_CONTENT:
        ErrorStr :=
          'El contenido de un elemento con contenido mezclado no coincide con su declaracin';
      ET_ELEMENT_WITH_ILLEGAL_ELEMENT_CONTENT:
        ErrorStr :=
          'El contenido de un elemento con contenido de elemento no coincide con su declaracin';
      ET_NONDETERMINISTIC_ELEMENT_CONTENT_MODEL:
        ErrorStr := 'El modelo de contenido de un elemento no es determinante';
      ET_DUPLICATE_NOTATION_DECL:
        ErrorStr := 'Declaracin de anotacin duplicada';
      ET_ATTRIBUTE_DEFINITION_NOT_FOUND:
        ErrorStr := 'Definicin de atributo no encontrada';
      ET_REQUIRED_ATTRIBUTE_NOT_FOUND:
        ErrorStr := 'Atributo requerido no encontrado';
      ET_FIXED_ATTRIBUTE_MISMATCH:
        ErrorStr :=
          'El valor declarado y el valor actual de un atributo fijo no coincide';
      ET_ATTRIBUTE_DEFAULT_TYPE_MISMATCH:
        ErrorStr := 'El tipo de atributo y el atributo por defecto no coincide';
      ET_DUPLICATE_ID_VALUE:
        ErrorStr := 'Valor de ID duplicado';
      ET_TARGET_ID_VALUE_NOT_FOUND:
        ErrorStr :=
          'El valor de IDREF o IDREFS apunta a un valor de ID no existente';
      ET_TARGET_UNPARSED_ENTITY_NOT_FOUND:
        ErrorStr :=
          'El valor de ENTITY o ENTITIES apunta a una entidad no existente';
      ET_WRONG_DECL_OF_PREDEFINED_ENTITY:
        ErrorStr := 'Declaracin erronea de una entidad predefinida';
      ET_UNRESOLVABLE_ENTITY_REFERENCE:
        ErrorStr := 'Referencia a entidad imposible de resolver';
      ET_EXTERNAL_SUBSET_NOT_FOUND:
        ErrorStr := '"Subset" externo del DTD no encontrado';
      ET_PARAMETER_ENTITY_DECL_NOT_FOUND:
        ErrorStr := 'Declaracin de entidad de parmetro no encontrada';
    else
      ErrorStr := 'Codigo fuente no vlido';
    end; {case ...}
  end; {with ...}

  if PosStr <> '' then
    S := Format('[%s] %s(%s): %s', [SeverityStr, FileNameStr, PosStr, ErrorStr])
  else
    S := Format('[%s] %s: %s', [SeverityStr, FileNameStr, ErrorStr]);

  Result := EncodingToUTF16(TISO8859_1Codec, S);
end;

{ TDomErrorAutoStrings }

function TDomErrorAutoStrings.AddError(AError: TDomError): Integer;
begin
  Result := AddObject('', AError.cloneError);
end;

function TDomErrorAutoStrings.AddError(const ErrorType: TXmlErrorType;
  const StartByteNumber, StartCharNumber, StartColumnNumber,
  StartLineNumber, EndByteNumber, EndCharNumber, EndColumnNumber,
  EndLineNumber: Int64; const UriStr: WideString;
  const RASObject: TdomASObject; const RNode: TdomNode;
  const Code: WideString): Integer;
begin
  Result := AddObject('', TDomError.Create(ErrorType, StartByteNumber,
  StartCharNumber, StartColumnNumber, StartLineNumber, EndByteNumber,
  EndCharNumber, EndColumnNumber, EndLineNumber, UriStr, RASObject, RNode,
  Code));
end;

procedure TDomErrorAutoStrings.Assign(Source: TPersistent);
var
  I: Integer;
begin
  if Source is TDomErrorAutoStrings then
  begin
    if Source = Self then
      Exit;
    DisableControls;
    try
      Clear;
      for I := 0 to Pred(TDomErrorAutoStrings(Source).Count) do
        AddError(TDomErrorAutoStrings(Source).Errors[I]);
    finally
      EnableControls;
    end;
    Exit;
  end;
  inherited Assign(Source);
end;

procedure TDomErrorAutoStrings.Clear;
var
  I: Integer;
begin
  for I := 0 to Pred(Count) do
    Errors[I].Free;
  inherited;
end;

procedure TDomErrorAutoStrings.Delete(Index: Integer);
begin
  Errors[Index].Free;
  inherited;
end;

procedure TDomErrorAutoStrings.DisableControls;
begin
  DoNotifyDisableControls;
end;

procedure TDomErrorAutoStrings.EnableControls;
begin
  DoNotifyEnableControls;
end;

function TDomErrorAutoStrings.Get(Index: Integer): string;
begin
  if Assigned(FErrorTranslator) then
  begin
{$IFDEF LINUX}
    Result := UTF16BEDecode(TISO8859_1Codec,
      FErrorTranslator.Translate(Errors[Index]));
{$ELSE}
    try
      Result := UTF16ToEncoding(GetSystemEncodingCodecClass,
        FErrorTranslator.Translate(Errors[Index]));
    except
      Result := Errors[Index].ClassName;
    end;
{$ENDIF}
  end
  else
    Result := Errors[Index].ClassName;
end;

function TDomErrorAutoStrings.GetError(Index: Integer): TDomError;
begin
  Result := Objects[Index] as TDomError;
end;

function TDomErrorAutoStrings.InsertError(Index: Integer;
  AError: TDomError): Boolean;
begin
  Result := InsertObject(Index, '', TDomError.createFromError(AError));
end;

function TDomErrorAutoStrings.InsertError(Index: Integer;
  const ErrorType: TXmlErrorType; const StartByteNumber, StartCharNumber,
  StartColumnNumber, StartLineNumber, EndByteNumber, EndCharNumber,
  EndColumnNumber, EndLineNumber: Int64;
  const UriStr: WideString; const RASObject: TdomASObject;
  const RNode: TdomNode; const Code: WideString): Boolean;
begin
  Result := InsertObject(Index, '', TDomError.Create(ErrorType, StartByteNumber,
  StartCharNumber, StartColumnNumber, StartLineNumber, EndByteNumber,
  EndCharNumber, EndColumnNumber, EndLineNumber, UriStr, RASObject, RNode, Code));
end;

procedure TDomErrorAutoStrings.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) and (AComponent = FErrorTranslator) then
    FErrorTranslator := nil;
end;

procedure TDomErrorAutoStrings.SetErrorTranslator(
  const AErrorTranslator: TDomCustomErrorTranslator);
begin
  if FErrorTranslator = AErrorTranslator then
    exit;
{$IFDEF VER140+}
  if assigned(FErrorTranslator) then
    FErrorTranslator.RemoveFreeNotification(Self);
{$ENDIF}
{$IFDEF LINUX}
  if assigned(FErrorTranslator) then
    FErrorTranslator.RemoveFreeNotification(Self);
{$ENDIF}
  FErrorTranslator := AErrorTranslator;
  if assigned(AErrorTranslator) then
    AErrorTranslator.FreeNotification(Self);
end;

end.

