//
// +----------------------------------------------------------------------+
// | Dev-PHP - Integrated Development Environment                         |
// +----------------------------------------------------------------------+
// | Copyright (C) 2002-2003, http://devphp.sourceforge.net               |
// +----------------------------------------------------------------------+
// | This program is free software; you can redistribute it and/or modify |
// | it under the terms of the GNU General Public License as published by |
// | the Free Software Foundation; either version 2 of the License, or    |
// | (at your option) any later version.                                  |
// | This program is distributed in the hope that it will be useful       |
// | but WITHOUT ANY WARRANTY; without even the implied warranty of       |
// | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         |
// | GNU General Public License for more details.                         |
// +----------------------------------------------------------------------+
// | Author: Leonardo Garca <simbiotik@users.sourceforge.net>            |
// | Author: Urs Mder <ursmaeder@users.sourceforge.net>                  |
// +----------------------------------------------------------------------+
//

unit uParser;

interface

uses
	SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
	Forms, Dialogs,SynUnicode;

type
  TtkTokenKind = (tkComment, tkIdentifier, tkKey, tkNull,
    tkNumber, tkSpace, tkString, tkSymbol, tkUnknown, tkVariable);
  TRangeState = (rsUnKnown, rsString39, rsString34, rsComment, rsVarExpansion,
    rsHeredoc);

  TRangePointer = packed record
    case Boolean of
      True: (Ptr: Pointer);
      False: (Range: Byte; Length: Byte; Checksum: Word);
    end;
  tokstat=(prev,tok1,middle,tok2);
  findstat=(none,found1,found2);
  phpstateChanged=(phpstart,phpstop,nothing);
  TState = (Tstatic, Tfinal, Tconst,Tpublic,Tprivate,Tprotected,TVar,TVariable,TImplements,TExtends,
          TFunction,TClass,TClassStart,TAbstract,TInterface,TInclude,TPhpStart,TPhpStop);
  TStateSet = SET OF TState;

TToken = class(TObject)
  str:string;
  Filename:string;
  include:string;
  pos:integer;
  last:integer;
  state:TStateSet;
  private
    constructor create;
  public
    function writeToken: string;
end;

OnTokenProc = procedure (var token:TToken) of object;

TParse = class(TObject)
  private
    StartList:TList;
    FName1:WideString;
    FName2:WideString;
    FState:TStateSet;
    FcurrToken:WideString;
    FcurrTokEnd:integer;
    fLineStr: WideString;
    FLine: PWideChar;
    Run:Integer;
//    FTokenID: TtkTokenKind;
    fTokenPos: integer;
    fRange: TRangeState;
    fHeredocLength: Byte;
    fHeredocChecksum: Word;
    fToIdent:PWideChar;
    fStringLen:integer;
    fhideStrings:boolean;
    charpos:integer;
    ffertig:boolean;
    Ffilename:string;
    FS:WideString;
    FString:TWideStrings;
    blanks:Set of char;
    lineends:Set of char;
    ends:Set of char;
    valid:Set of char;
    FOnToken: OnTokenProc;
    simpleS:integer;
    doubleS:integer;
    procedure writeFString(s:TWideStrings);
    function getNextChar():Widechar;
    function readUntil(c: widechar;var last:integer): widestring;
    procedure checkInPHP(s: widestring);
    function checkPreviewInPHP(s: widestring): phpstateChanged;
    procedure NextProcedure;
    procedure AndSymbolProc;
    procedure String39Proc;
    procedure StringProc;
    function IsLineEnd(AChar: WideChar): Boolean;
    function IsIdentChar(AChar: WideChar): Boolean;
    procedure BraceCloseProc;
    procedure BraceOpenProc;
    procedure CRProc;
    procedure EqualProc;
    procedure GreaterProc;
    procedure IdentProc;
    function IdentKind(MayBe: PWideChar): TtkTokenKind;
    function getTokenKinds(MayBe: PWideChar): TtkTokenKind;
    function IsCurrentToken(const Token: WideString): Boolean;
    function HashKey(Str: PWideChar): Cardinal;
    procedure LFProc;
    procedure LowerProc;
    procedure MinusProc;
    procedure MultiplyProc;
    procedure NotSymbolProc;
    procedure NullProc;
    procedure NumberProc;
    procedure OrSymbolProc;
    procedure PlusProc;
    procedure PoundProc;
    procedure QuestionProc;
    procedure RemainderSymbolProc;
    procedure RoundCloseProc;
    procedure RoundOpenProc;
    procedure SemiColonProc;
    procedure SlashProc;
    procedure SpaceProc;
    procedure String34Proc;
    procedure VariableProc;
    procedure XOrSymbolProc;
    procedure ShowCurrentToken;
    function getCurrentToken: wideString;
    procedure CallOnToken;
    function getNextWord(withblanks:boolean):string;
    function getPreviewChar(add: integer;withblanks:boolean): widechar;
    function getPreviewWord(var last:integer): string;
    property hideStrings:boolean read fhideStrings write fhideStrings;
    function getStartPoint: integer;
  protected
  public
    bracketcount:integer;
    inphp:boolean;
    constructor create;
    destructor destroy; override;
    function Capitalize(str:string): string;
    Procedure Parse();
  published
    property parsedStr:TWideStrings read FString write writeFString;
    property Filename:string read FFilename write FFilename;
    property OnToken:OnTokenProc read FOnToken write FOnToken;
end;


implementation

uses frmMain;
{ TParse }

constructor TParse.create();
begin
  inherited create;
  ffertig:=false;
  blanks:=[' ',chr(9)];
  lineends:=[chr(10),chr(13)];
  ends:=[';',',','(',')','=','{','}',chr(9),chr(10),chr(13),chr(0)];
  simpleS:=0;
  doubleS:=0;
  FString:=TWideStringList.Create;
  bracketcount:=0;
  valid:=['a'..'z','A'..'Z','0'..'9','_'];
  charpos:=0;
  inphp:=false;
  hideStrings:=true;
  FState:=[];
  StartList:=TList.Create;
end;

destructor TParse.destroy;
begin
  FString.Free;
  StartList.Free;
  inherited;
end;

function TParse.readUntil(c:widechar;var last:integer):widestring;
var oldpos,l:integer;
    phpend:boolean;
    s,rs:widestring;
begin
  oldpos:=charpos;
  repeat
    s:=getNextWord(true);
    phpend:=checkPreviewInPHP(GetPreviewWord(l))=phpstop;
    rs:=rs+s;
  until (pos(c,rs)>0) or (phpend or ffertig);
  result:=rs;
  last:=charpos;
  charpos:=oldpos;
end;

function TParse.getNextChar():Widechar;
begin
  if charpos>length(FS) then begin
    result:=chr(0);charpos:=high(integer);ffertig:=true;exit;
  end;
  result:=FS[charpos];
  inc(charpos);
end;

function TParse.getPreviewChar(add:integer;withblanks:boolean):widechar;
var i:integer;
begin
  if charpos>length(FS) then begin result:=chr(0);exit; end;
  i:=charpos;
  if not withblanks then while char(FS[i]) in blanks do inc(i);
  result:=FS[i];
end;

function TParse.getPreviewWord(var last:integer):string;
var oldpos:integer;
begin
  oldpos:=charpos;   result:=getNextWord(false);  last:=charpos; charpos:=oldpos;
end;

function TParse.getNextWord(withblanks:boolean): string;
var c:widechar;
    s:string;
    backslash:integer;
begin
  s:='';
  repeat
    simpleS:=0;
    doubleS:=0;
    c:=getNextChar();
    if not withblanks then while (not ffertig) and (char(c) in blanks) do c:=getNextChar();
    if (char(c) in ends) then begin s:=c; break; end;
    case c of
    '?': begin
      if (getPreviewChar(1,true)='>') then begin
        s:=s+c;c:=getNextChar();s:=s+c; break;
      end;
    end;
    '<': begin
      if (getPreviewChar(1,true)='?') then begin
        s:=s+c;c:=getNextChar();s:=s+c; break;
      end;
    end;
    '''':  // Strings with single
      begin
        backslash:=0;
        if simpleS=0 then simpleS:=1 else simpleS:=0;
        if hideStrings and (simpleS=1) then begin
          s:=s+c;c:=getNextChar();s:=s+c;  
          while (not ffertig and (c<>'''')) or (backslash=1)do begin
            if c='\' then inc(backslash) else backslash:=0;
            if backslash=2 then backslash:=0;
            c:=getNextChar();s:=s+c;
          end;
          simpleS:=0;
          break;
        end;
      end;
    '"':  // Strings with double
      begin
        backslash:=0;
        if doubleS=0 then doubleS:=1 else doubleS:=0;
        if hideStrings and (doubleS=1) then begin
          s:=s+c;c:=getNextChar();s:=s+c;
          while (not ffertig and (c<>'"')) or (backslash=1) do begin
           if c='\' then inc(backslash) else backslash:=0;
           if backslash=2 then backslash:=0;
            c:=getNextChar();s:=s+c;
          end;
          doubleS:=0;
        end;
        break;
      end;
    ' ':
      begin
        if withblanks then begin
          s:=s+c; break;
        end else begin
          while (char(getPreviewChar(1,true)) in blanks) and not ffertig do
            getNextChar();
          //s:='';
          break;
        end;
      end;
    chr(13),chr(10):
      begin
        s:=s+c; break;
      end;
    '/': //Comments
      begin
        if getPreviewChar(1,true)='*' then begin
          repeat c:=getNextChar();
          until (((c='*') and (getPreviewChar(1,false)='/'))) or ffertig;
          getNextChar();
          s:='';
        end;
        if getPreviewChar(1,true)='/' then begin
          repeat c:=getNextChar(); until(char(c) in lineends) or ffertig;
          s:='';
        end;
      end;
    else
      s:=s+c;
    end;
    if char(getPreviewChar(1,true)) in ends then  break;
    if char(getPreviewChar(1,true)) in blanks then break;
    if c=#0 then break;
    if ffertig then break;
  until false;
  result:=s;
end;

procedure TParse.writeFString(s: TWideStrings);
begin
  FString.Text:=s.Text;
  FS:=s.Text;
  fLineStr := s.Text;
  fLine := PWideChar(fLineStr);
  Run:=0;
  charpos:=1;
  ffertig:=false;
end;

function TParse.Capitalize(str:string):string;
var res:string;
    s:PWideChar;
    i:integer;
begin
  res:=str;
  for i:=1 to length(res) do begin
    s:=Pointer(res[i]);
    if (s^ >= 'a') and (s^ <= 'z') then begin
      Dec(s^, 32);
    end;
  end;
  result:= res;
end;

procedure TParse.checkInPHP(s:widestring);
begin
  if (s='<?php') or (s='<?') then
    inphp:=true;
  if (s='?>') then
    inphp:=false;
end;

function TParse.checkPreviewInPHP(s:widestring):phpstateChanged;
begin
  result:=nothing;
  if (s='<?php') or (s='<?') then
    result:=phpstart;
  if (s='?>') then
    result:=phpstop;
end;

procedure TParse.Parse();
var len:integer;
begin
  len:=length(fLineStr);
  repeat
    self.NextProcedure;
  until Run>=len;
end;

constructor TToken.create;
begin
  state:=[];
  self.str:='';
  self.pos:=0;
  self.last:=0;
  self.Filename:='';
  inherited create;
end;

function TToken.writeToken():string;
var s:string;
begin
  s:=str;
  if TPublic in state then s:=s+' TPublic';
  if TStatic in state then s:=s+' TStatic';
  if TConst in state then s:=s+' TConst';
  if TFinal in state then s:=s+' TFinal';
  if TProtected in state then s:=s+' TProtected';
  if TPrivate in state then s:=s+' TPrivate';
  if TVar in state then s:=s+' TVar';
  if TFunction in state then s:=s+' TFunction';
  if TClass in state then s:=s+' TClass';
  if TAbstract in state then s:=s+' TAbstract';
  if TInterface in state then s:=s+' TInterface';
  if TInclude in state then s:=s+' TInclude';
  result:=s;
end;


procedure TParse.AndSymbolProc;
begin
  case Fline[Run + 1] of
    '=':                               {and assign}
      begin
        inc(Run, 2);
      end;
    '&':                               {conditional and}
      begin
        inc(Run, 2);
      end;
  else                                 {and}
    begin
      inc(Run);
    end;
  end;
end;

function TParse.IsLineEnd(AChar: WideChar): Boolean;
begin
  Result := (AChar = #0) or (AChar = #10) or (AChar = #13);
end;

function TParse.IsIdentChar(AChar: WideChar): Boolean;
begin
  case AChar of
    '_', '0'..'9', 'A'..'Z', 'a'..'z':
      Result := True;
    else
      Result := False;
  end;
end;

procedure TParse.StringProc;

  function IsEscaped: Boolean;
  var
    iFirstSlashPos: Integer;
  begin
    iFirstSlashPos := Run -1;
    while (iFirstSlashPos > 0) and (Fline[iFirstSlashPos] = '\') do
      Dec(iFirstSlashPos);
    Result := (Run - iFirstSlashPos + 1) mod 2 <> 0;
  end;

var
  iCloseChar: WideChar;
  currtok:WideString;
begin
  if IsLineEnd(Fline[Run]) and (fTokenPos = Run) then
  begin
    NextProcedure;
    Exit;
  end;
  if fRange = rsString39 then
    iCloseChar := #39
  else
    iCloseChar := #34;
//  while not IsLineEnd(Fline[Run]) do
  while FLine[Run]<>#0 do
  begin
    if (Fline[Run] = iCloseChar) and not IsEscaped then
      break;
    if (Fline[Run] = '$') and (iCloseChar = '"') and
      ((Fline[Run + 1] = '{') or IsIdentChar(Fline[Run + 1])) then
    begin
      if (Run > 1) and (Fline[Run -1] = '{') then { complex syntax }
        Dec(Run);
      if not IsEscaped then
      begin
        { break the token to process the variable }
//        fRange := rsVarExpansion;
//        Exit;
        inc(run);
      end
      else if Fline[Run] = '{' then
        Inc(Run); { restore Run if we previously deincremented it }
    end;
    if inphp then currtok:=currtok+Fline[Run];
    Inc(Run);
  end;
  if (Fline[Run] = iCloseChar) then
    fRange := rsUnKnown;
  if Fline[Run] <> #0 then inc(Run);
  If TInclude in FState then FCurrToken:=FCurrToken+currtok;
end;

procedure TParse.String39Proc;
begin
  fRange := rsString39;
  Inc( Run );
  StringProc;
end;

procedure TParse.CRProc;
begin
  Case Fline[Run + 1] of
    #10: inc(Run, 2);
  else inc(Run);
  end;
end;

procedure TParse.GreaterProc;
begin
  case Fline[Run + 1] of
    '=':                               {greater than or equal to}
      begin
        inc(Run, 2);
      end;
    '>':
      begin
        inc(Run, 2);
      end;
  else                                 {greater than}
    begin
      inc(Run);
    end;
  end;
end;

function TParse.IsCurrentToken(const Token: WideString): Boolean;
var
  I: Integer;
  Temp: PWideChar;
begin
  Temp := fToIdent;
  if Length(Token) = FStringLen then
  begin
    Result := True;
    for i := 1 to FStringLen do
    begin
      if lowercase(Temp^) <> lowercase(Token[i]) then
      begin
        Result := False;
        break;
      end;
      inc(Temp);
    end;
  end
  else
    Result := False;
end;

Procedure TParse.ShowCurrentToken();
var
  Temp: WideString;
begin
  Temp:=Copy(fToIdent,0,FStringLen);
//  showmessage(temp);
end;

function TParse.getCurrentToken():wideString;
begin
  result:=lowercase(Copy(fToIdent,0,FStringLen));
end;

procedure TParse.CallOnToken();
var t:TToken;
begin
  t:=TToken.create;
  t.str:=FCurrToken;
  t.Filename:=Ffilename;
  t.pos:=getStartPoint();
  t.last:=FCurrTokEnd;
  t.state:=FState;
  if assigned(OnToken) then OnToken(t);
  FCurrToken:='';
  StartList.Clear;
end;

function TParse.getStartPoint():integer;
var i:integer;
begin
  result:=0;
  if StartList.Count>0  then begin
    result:=HIGH(integer);
    for i:=0 to StartList.count-1 do begin
      if Integer(StartList.Items[i])<result then
        result:=(Integer(StartList.Items[i]));
    end;
  end;
end;

function TParse.getTokenKinds(MayBe:PWideChar):TtkTokenKind;
begin
  result:=tkIdentifier;
  if not inphp then exit;
  if frange=rsComment then exit;
  if TConst in FState then begin FName1:=getCurrentToken(); end;
  if TClassStart in FState then begin FName1:=getCurrentToken();Exclude(FState,TClassStart); end;
  if TImplements in FState then begin FName2:=getCurrentToken();  end;
  if TExtends in FState then begin FName2:=getCurrentToken(); end;
  if TFunction in FState then begin FName1:=getCurrentToken(); end;
  case MayBe^ of
  'a','A':  begin
          if IsCurrentToken('array') then Result:=tkKey;
          if IsCurrentToken('abstract') then begin
            Include(FState,TAbstract);
            StartList.Add(pointer(Run));
          end;
        end;
  'c','C':  begin
          if IsCurrentToken('class') then begin
            Include(FState,TClass);
            include(FState,TClassStart);
            StartList.Add(pointer(Run));
          end;
          if IsCurrentToken('const') then begin
            Include(FState,TConst);
            StartList.Add(pointer(Run));
          end;
        end;
  'e','E':  begin
          if IsCurrentToken('extends') then begin
            Include(FState,TExtends);
          end;
        end;
  'f','F':  begin
          if IsCurrentToken('function') then begin
            Include(FState,TFunction);
            StartList.Add(pointer(Run));
          end;
          if IsCurrentToken('final') then begin
            Include(FState,TFinal);
            StartList.Add(pointer(Run));
          end;
        end;
  'g','G':  begin
          if IsCurrentToken('global') then Result:=tkKey;
        end;
  'i','I':  begin
          if IsCurrentToken('include') then begin
            Include(FState,TInclude);
            StartList.Add(pointer(Run));
            FCurrToken:='include ';
          end;
          if IsCurrentToken('include_once') then begin
            Include(FState,TInclude);
            StartList.Add(pointer(Run));
            FCurrToken:='include_once ';
          end;
          if IsCurrentToken('interface') then begin
            Include(FState,TInterface);
            include(FState,TClassStart);
            StartList.Add(pointer(Run));
          end;
          if IsCurrentToken('instanceof') then Result:=tkKey;
          if IsCurrentToken('implements') then begin
            Include(FState,TImplements);
          end;
        end;
  'o','O':  begin
          if IsCurrentToken('object') then Result:=tkKey;
        end;
  'p','P':  begin
          if IsCurrentToken('private') then begin
            Include(FState,TPrivate);
            StartList.Add(pointer(Run));
          end;
          if IsCurrentToken('public') then begin
            Include(FState,TPublic);
            StartList.Add(pointer(Run));
          end;
          if IsCurrentToken('protected') then begin
            Include(FState,TProtected);
            StartList.Add(pointer(Run));
          end;
        end;
  'r','R':  begin
          if IsCurrentToken('require') then begin
            Include(FState,TInclude);
            StartList.Add(pointer(Run));
            FCurrToken:='require ';
          end;
          if IsCurrentToken('require_once') then begin
            Include(FState,TInclude);
            StartList.Add(pointer(Run));
            FCurrToken:='require_once ';
          end;
        end;
  's','S':  begin
          if IsCurrentToken('static') then begin
            Include(FState,TStatic);
            StartList.Add(pointer(Run));
          end;
        end;
  'v','V':  begin
          if IsCurrentToken('var') then begin
            Include(FState,TVar);
            StartList.Add(pointer(Run));
          end;
        end;
  end;
end;

procedure TParse.EqualProc;
begin
  case Fline[Run + 1] of
    '=':                               {logical equal}
      begin
        inc(Run, 2);
      end;
    '>':                               {Hash operator}
      begin
        inc(Run, 2);
      end;
  else                                 {assign}
    begin
      inc(Run);
      if inphp then begin
        FCurrToken:='';
        if (TConst in FState) then FCurrToken:='const '+Fname1;
        if FCurrToken<>'' then begin
          FCurrTokEnd:=Run-1;
          callOnToken();
          Exclude(FState,TConst);
        end;
      end;
    end;
  end;
end;

procedure TParse.BraceCloseProc;
begin
  inc(Run);
  if inphp then begin
    dec(bracketcount);
    if bracketcount<=0 then begin
      Exclude(FState,TClass);
      Exclude(FState,TInterface);
    end;
    exclude(FState,TInclude);
  end;
end;

procedure TParse.BraceOpenProc;
begin
  inc(Run);
  if inphp then begin
    FCurrToken:='';
    if (TFinal in FState) then FCurrToken:='final ';
    if (TAbstract in FState) then FCurrToken:='abstract ';
    if (TClass in FState) and (bracketcount=0) then FCurrToken:=FCurrToken+'class '+fname1;
    if (TInterface in FState) and (bracketcount=0) then FCurrToken:='interface '+fname1;
    if (TImplements in FState) then FCurrToken:=FCurrToken+' implements '+Fname2;
    if (TExtends in FState) then FCurrToken:=FCurrToken+' extends '+Fname2;
    if FCurrToken<>'' then begin
      FCurrTokEnd:=Run-1;
      callOnToken();
      Exclude(FState,TFinal);
      Exclude(FState,TAbstract);
      Exclude(FState,TImplements);
      Exclude(FState,TExtends);
    end;
    inc(bracketcount);
  end;
end;

procedure TParse.RoundOpenProc;
begin
  inc(Run);
  if TInclude in FState then exit;
  if inphp then begin
    FCurrToken:='';
    if TAbstract in FState then FCurrToken:='abstract ';
    if TStatic in FState then FCurrToken:='static ';
    if TFinal in FState then FCurrToken:='final ';
    if TPrivate in FState then FCurrToken:=FCurrToken+'private ';
    if TPublic in FState then FCurrToken:=FCurrToken+'public ';
    if TProtected in FState then FCurrToken:=FCurrToken+'protected ';
    if TFunction in FState then FCurrToken:=FCurrToken+'function ';
    If FCurrToken<>'' then begin
      FCurrToken:=FCurrToken+FName1;
      FCurrTokEnd:=Run-1;
      CallOnToken();
      Exclude(FState,TFunction);
      Exclude(FState,TPrivate);
      Exclude(FState,TProtected);
      Exclude(FState,TPublic);
      Exclude(FState,TStatic);
      Exclude(FState,TFinal);
      Exclude(FState,TAbstract);
    end;
  end;
end;

procedure TParse.VariableProc;
var s:widestring;
begin
  inc(Run);
  if fRange=rsComment then exit;
  if TInclude in FState then exit;
  if not(TVar in FState) then exit;
  while IsIdentChar(fLine[Run]) do begin
    s:=s+fline[run];inc(Run);
  end;
  Include(FState,TVariable);
  FCurrToken:='';
  if TPrivate in FState then FCurrToken:='private ';
  if TPublic in FState then FCurrToken:='public ';
  if TProtected in FState then FCurrToken:='protected ';
  if TStatic in FState then FCurrToken:=FCurrToken+'static ';
  if TVar in FState then FCurrToken:=FCurrToken+'var ';
  if FCurrToken<>'' then begin
    FCurrToken:=FCurrToken+'$'+s;
    FCurrTokEnd:=Run;
    CallOnToken();
    Exclude(FState,TVar);
    Exclude(FState,TPrivate);
    Exclude(FState,TProtected);
    Exclude(FState,TPublic);
    Exclude(FState,TStatic);
  end;
  Exclude(FState,TVariable);
end;

procedure TParse.SemiColonProc;
begin
  inc(Run);                            {semicolon}
  if fRange=rsComment then exit;
  if inphp then begin
    if TInclude in FState then begin
      FCurrTokEnd:=Run-1;
      CallOnToken();
      Exclude(FState,TInclude);
    end;
  end;
end;

{$Q-}
function TParse.HashKey(Str: PWideChar): Cardinal;
begin
  Result := Ord(Str^);
  while IsIdentChar(Str^) do
  begin
    inc(Str);
  end;
  FStringLen := Str - fToIdent;
end;
{$Q+}

function TParse.IdentKind(MayBe: PWideChar): TtkTokenKind;
var str:PWideChar;
begin
  fToIdent := MayBe;  str:=MayBe;
  while IsIdentChar(Str^) do
  begin
    inc(Str);
  end;
  FStringLen := Str - fToIdent;
  result:=getTokenKinds(MayBe);
  HashKey(MayBe);
end;

procedure TParse.IdentProc;
begin
  IdentKind((fLine + Run));
  inc(Run, FStringLen);
  while IsIdentChar(Fline[Run]) do inc(Run);
end;

procedure TParse.LFProc;
begin
  inc(Run);
end;

procedure TParse.LowerProc;
var
  i, Len : Integer;
begin
  case FLine[Run + 1] of
    '=':                               {less than or equal to}
      begin
        inc(Run, 2);
      end;
    '<':
      begin
        if (FLine[Run + 2] = '<') and IsIdentChar(FLine[Run + 3]) then
        begin
          inc(Run, 3);

          i := Run;
          while IsIdentChar(FLine[i]) do Inc(i);
          Len := i - Run;

          if Len > 255 then
          begin
            Exit;
          end;

          fRange := rsHeredoc;
          fHeredocLength := Len;
//TODO          fHeredocChecksum := CalcFCS(FLine[Run], Len);

          Inc(Run, Len);
        end
        else
        if FLine[Run + 2] = '=' then   {shift left assign}
        begin
          inc(Run, 3)
        end
        else                           {shift left}
        begin
          inc(Run, 2);
        end;
      end;
    '?':
      begin
        Inc(Run,2);
        inphp:=true;
      end;
  else                                 {less than}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.MinusProc;
begin
  case FLine[Run + 1] of
    '=':                               {subtract assign}
      begin
        inc(Run, 2);
      end;
    '-':                               {decrement}
      begin
        inc(Run, 2);
      end;
    '>':                               {Class operator}
      begin
        inc(Run, 2);
      end;
  else                                 {subtract}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.MultiplyProc;
begin
  case FLine[Run + 1] of
    '=':                               {multiply assign}
      begin
        inc(Run, 2);
      end;
  else                                 {multiply}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.NotSymbolProc;
begin
  case FLine[Run + 1] of
    '=':                               {not equal}
      begin
        inc(Run, 2);
      end;
  else                                 {logical complement}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.NullProc;
begin
end;

procedure TParse.NumberProc;

  function IsNumberChar: Boolean;
  begin
    case fLine[Run] of
      '0'..'9', '.', '-', 'l', 'L', 'x', 'X', 'A'..'F', 'a'..'f':
        Result := True;
      else
        Result := False;
    end;
  end;

begin
  inc(Run);
  while IsNumberChar do
  begin
    case FLine[Run] of
      '.':
        if FLine[Run + 1] = '.' then break;
    end;
    inc(Run);
  end;
end;

procedure TParse.OrSymbolProc;
begin
  case FLine[Run + 1] of
    '=':                               {inclusive or assign}
      begin
        inc(Run, 2);
      end;
    '|':                               {conditional or}
      begin
        inc(Run, 2);
      end;
  else                                 {inclusive or}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.PlusProc;
begin
  case FLine[Run + 1] of
    '=':                               {add assign}
      begin
        inc(Run, 2);
      end;
    '+':                               {increment}
      begin
        inc(Run, 2);
      end;
  else                                 {add}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.PoundProc;
begin
  repeat
    inc(Run);
  until IsLineEnd(FLine[Run]);
end;

procedure TParse.QuestionProc;
begin
  case FLine[Run + 1] of
    '>':
      begin
        inc(Run,2);
        inphp:=false;
      end;
    else begin
      inc(Run);
    end;
  end;
end;

procedure TParse.RemainderSymbolProc;
begin
  case FLine[Run + 1] of
    '=':                               {remainder assign}
      begin
        inc(Run, 2);
      end;
  else                                 {remainder}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.RoundCloseProc;
begin
  inc(Run);
end;

procedure TParse.SlashProc;
begin
  case FLine[Run + 1] of
    '/':                               {c++ style comments}
      begin
        inc(Run, 2);
        while FLine[Run] <> #0 do
        begin
          case FLine[Run] of
            #10, #13: break;
          end;
          inc(Run);
        end;
      end;
    '*':
      begin
        fRange := rsComment;
        inc(Run);

        inc(Run);
        while fLine[Run] <> #0 do
          case fLine[Run] of
            '*':
              if fLine[Run + 1] = '/' then
              begin
                fRange := rsUnKnown;
                inc(Run, 2);
                break;
              end else inc(Run);
//            #10: break;
//            #13: break;
          else inc(Run);
          end;
      end;
    '=':                               {division assign}
      begin
        inc(Run, 2);
      end;
  else                                 {division}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.SpaceProc;
begin
  inc(Run);
  while (FLine[Run] <= #32) and not IsLineEnd(FLine[Run]) do inc(Run);
end;

procedure TParse.String34Proc;
begin
  fRange := rsString34;
  Inc( Run );
  StringProc;
end;

procedure TParse.XOrSymbolProc;
begin
  Case FLine[Run + 1] of
    '=':                               {xor assign}
      begin
        inc(Run, 2);
      end;
  else                                 {xor}
    begin
      inc(Run);
    end;
  end;
end;

procedure TParse.NextProcedure;
begin
  case Fline[Run] of
    '&': AndSymbolProc;
    #39: String39Proc; // single quote
    '@': inc(run);
    '}': BraceCloseProc;
    '{': BraceOpenProc;
    #13: CRProc;
    ':': inc(run);
    ',': inc(run);
    '=': EqualProc;
    '>': GreaterProc;
    'A'..'Z', 'a'..'z', '_': IdentProc;
    #10: LFProc;
    '<': LowerProc;
    '-': MinusProc;
    '*': MultiplyProc;
    '!': NotSymbolProc;
    #0: NullProc;
    '0'..'9': NumberProc;
    '|': OrSymbolProc;
    '+': PlusProc;
    '.': inc(run);
    '#': PoundProc;
    '?': QuestionProc;
    '%': RemainderSymbolProc;
    ')': RoundCloseProc;
    '(': RoundOpenProc;
    ';': SemiColonProc;
    '/': SlashProc;
    #1..#9, #11, #12, #14..#32: SpaceProc;
    ']': inc(run);
    '[': inc(run);
    #34: String34Proc; // double quote
    '~': inc(Run);
    '$': VariableProc;
    '^': XOrSymbolProc;
    else Inc(Run);
  end;
end;


end.