unit DebugClient;

interface

uses
  Classes,SyncObjs,
  IdExceptionCore,XDOM_3_1,
  IdGlobal,DCPbase64,
  IdStack,dialogs,extCtrls,
  IdTCPServer, idContext,idUri;

type
  TDebugClient = class;
  TOnBreak=procedure (sender:TObject;filename:string;lineno:integer) of object;
  TOnDataAvailable=procedure(sender:TObject;s:string) of object;
  TOnMessage=procedure(sender:TObject;s:string) of object;
  TOnXMLData=procedure(sender:TObject;doc:TdomDocument) of object;
  TOnStackGet=procedure(sender:TObject;filename:string;lineno:integer) of object;
  TOnStepInto=procedure(sender:TObject) of object;
  TOnRun=procedure(sender:TObject) of object;
  TOnStop=procedure(sender:TObject) of object;
  TOnStatus=procedure(sender:TObject) of object;
  TOnBPSet=procedure(sender:TObject) of object;
  TOnEval=procedure(sender:TObject;transid:integer; expr:string; value:string) of object;

  TBreakpointType= (bpLine, bpCall, bpReturn, bpException, bpConditional, bpWatch);
  TBreakpoint = class (TObject)
    bpType: TBreakpointType;
    bpFilename:string;
    bpLine:integer;
    bpExpression:string;
    Ftransactionid:integer;
    Fbreakpointid:string;
    deleted:boolean;
    constructor create;
  end;

  TBreakpointList = class (TList)
    function toggleBP(Filename:string;Line:integer):TBreakpoint; overload;
    procedure addBP(t:TBreakpointType;expr:string); overload;
    procedure clearBP;
  private
    function GetItem(index: integer): TBreakPoint;
    procedure SetItem(index: integer; const Value: TBreakPoint);
  Public
    property BreakPoints[i:integer]:TBreakPoint read GetItem write SetItem; default;
  end;

  TBreakpointPtr = ^TBreakPoint;

  TStackEntry = class (TObject)
    filename:string;
    lineno:integer;
  end;

  TStackEntryList = class (TList)
  private
    function GetItem(index: integer): TStackEntry;
    procedure SetItem(index: integer; const Value: TStackEntry);
  Public
    procedure AddEntry(filename:string;lineno:integer);
    procedure clearList;
    function findLineno(filename:string):integer;
    property StackEntrys[i:integer]:TStackEntry read GetItem write SetItem; default;
  end;

  TDebugMsg = class (TObject)
    Fsend:string;
    Ftransactionid:integer;
    Fbpid:string;
    FCommand:string;
    Fresult:string;
    worked:boolean;
  public
    constructor create(sendstr:string;transID:integer);
    destructor destroy; override;
    property Result:string read Fresult write Fresult;
  end;

  TEvalExpr=class (TObject)
  private
    ftransid: integer;
    fresult: string;
    fexpr: string;
  public
    property transid:integer read ftransid write ftransid;
    property expr:string read fexpr write fexpr;
    property result:string read fresult write fresult;
  end;
  TEvalExprPtr=^TEvalExpr;

  TDebugClient = class ( TIdTCPServer )
  private
    FContext:TIdContext;
    Fterminate: boolean;
    FEvalList:TList;
    FOnCommand: TOnDataAvailable;
    FLastResult: TStringList;
    FMsgList:TList;
    FResult:String;
    FLine: integer;
    transactionid:integer;
    DomImplementation:TDomImplementation;
    xmlToDomParser:TXmlToDomParser;
    factivated: boolean;
    received:boolean;
    FOnStackGet: TOnStackGet;
    FCurrLine: integer;
    FCurrFilename: string;
    FCurrStackDepth: integer;
    FOnStepInto: TOnStepInto;
    FOnRun: TOnRun;
    FOnEval: TOnEval;
    FOnBPset: TOnBPset;
    Fdocs: TDomNodeList;
    FOnStepOver: TOnStepInto;
    FOnStepOut: TOnStepInto;
    FOnStop: TOnStop;
    Fstatus: string;
    FOnStatus: TOnStatus;
    FlogFile: string;
    FLogStrings: TStringList;
    FCurrWhere: string;
    UILock: TCriticalSection;
    logLock:TCriticalSection;
    FOnMessage: TOnMessage;
    FOnBreak: TOnBreak;
    procedure Setterminate(const Value: boolean);
    function GetLastResult: TStringList;
    function findMsg(transid: integer): TDebugMsg;
    procedure wactivated(const Value: boolean);
    function findEvalExpr(transid: integer): TEvalExpr;
    function dorun(s:string):TDebugMsg;
    function findBreakPoint(transid: integer): integer;
    procedure clearBreakpoints;
    procedure dolog(s: string);
    procedure send(AContext: TIdContext; s: string);
    procedure send_get_stack(AContext: TIdContext);
    procedure getCommandAndTransactionID(s: string; var command: string;
      var transactionid: integer);
    procedure getStatusAndID(s: string; var status, ID: string);
    procedure getFilenameAndLineno(s: string; var filename:string;
      var lineno: integer);
  protected
    procedure InitComponent; override;
    function DoExecute(Context: TIdContext): boolean; override;
  public
    breakpoints:TBreakPointList;
    stackentrys:TStackEntryList;
    destructor Destroy;override;
    procedure run();
    procedure setbreak(); overload;
    procedure setbreak(func:string); overload;
    procedure setbreak(filename:string;line:integer); overload;
    procedure get_stack(depth:integer);
    procedure step_into();
    procedure step_over();
    procedure step_out();
    procedure eval(expr:string);
    procedure stop();
    procedure context_get;
    procedure getStatus();
    procedure setBreakpoints();
    procedure evaluate();
    function getLog: string;
  published
    property activated:boolean read factivated write wactivated;
    property DefaultPort default 9000;
    property EvalList:TList read FEvalList;
    property OnCommand:TOnDataAvailable read FOnCommand write FOnCommand;
    property OnBreak:TOnBreak read FOnBreak write FOnBreak;
    property OnStackGet:TOnStackGet read FOnStackGet write FOnStackGet;
    property OnStepInto:TOnStepInto read FOnStepInto write FOnStepInto;
    property OnStepOver:TOnStepInto read FOnStepOver write FOnStepOver;
    property OnStepOut:TOnStepInto read FOnStepOut write FOnStepOut;
    property OnStop:TOnStop read FOnStop write FOnStop;
    property OnStatus:TOnStatus read FOnStatus write FOnStatus;
    property OnRun:TOnRun read FOnRun write FOnRun;
    property OnEval:TOnEval read FOnEval write FOnEval;
    property OnBPset:TOnBPset read FOnBPset write FOnBPset;
    property OnMessage:TOnMessage read FOnMessage write FOnMessage;
    property LastResult:TStringList read GetLastResult;
    property StrResult:string read FResult;
    property line:integer read FLine write FLine;
    property terminate:boolean read Fterminate write Setterminate;
    property currFilename:string read FCurrFilename;
    property currLine:integer read FCurrLine;
    property currStackDepth:integer read FCurrStackDepth;
    property currWhere:string read FCurrWhere;
    property docs:TDomNodeList read Fdocs write Fdocs;
    property status:string read Fstatus write Fstatus;
    property logFile:string read FlogFile write FlogFile;
  end;

  procedure Register;

implementation

uses
  SysUtils;

procedure Register;
begin
  RegisterComponents('Maeder', [TDebugClient]);
end;

procedure TDebugClient.InitComponent;
begin
  inherited;
  UILock := TCriticalSection.Create;
  LogLock:=TCriticalSection.Create;
  FEvalList:=TList.Create;
  FLastResult:=TStringList.Create;
  FMsgList:=TList.Create;
  Breakpoints:=TBreakpointList.Create;
  StackEntrys:=TStackEntryList.Create;
  breakpoints.clearBP;
  transactionid:=1;
  received:=false;
  if not (csDesigning in componentState) then begin
    DomImplementation:=TDomImplementation.create(self);
    XmlToDomParser:=TXmlToDomParser.create(self);
    xmlToDomparser.DOMImpl:=DomImplementation;
    xmlToDomParser.KeepCDATASections:=true;
  end;
  DefaultPort := 9000;
//  FLogFile:='';
  FLogStrings:=TStringList.Create;
end;

destructor TDebugClient.Destroy;
begin
  if assigned(FEvalList) then begin
    FEvalList.Clear;
    FreeAndNil(FEvalList);
  end;
  if assigned(FLastResult) then begin
    FLastResult.Clear;
    FreeAndNil(FLastResult);
  end;
  if assigned(FMsgList) then begin
    FMsgList.Clear;
    FreeAndNil(FMsgList);
  end;
  if assigned(breakpoints) then begin
    breakpoints.clearBP;
    breakpoints.Free;
  end;
  if assigned(stackentrys) then stackentrys.Free;
  if assigned(FLogStrings) then FLogStrings.Free;
  if assigned(UiLock) then freeandnil(Uilock);
  if assigned(loglock) then freeandnil(loglock);  
  inherited Destroy;
end;

//procedure TDebugClient.OnTimer(Sender: TObject);
//
//var i:integer;
//    m:TDebugMsg;
//    idoc:TDomNode;
//    node,node2:TDomNode;
//    uri:TIdUri;
//    value:string;
//    ex:TEvalExpr;
//    index:integer;
//    bp1:TBreakPoint;
//
//  procedure setStatus(d:TDomNode);
//  begin
//    node2:=d.getFirstChildElement('response');
//    status:=node2.attributes.getNamedItem('status').nodeValue;
//    if status='stopped' then begin
//      FCurrFilename:='';
//      FCurrLine:=-1;
//      FCurrStackDepth:=-1;
//      self.stackentrys.clearList;
//    end;
//    if assigned(FOnStatus) then FOnStatus(sender);
//  end;
//
//begin
//  FTimer.Enabled:=false;
//  Uilock.Acquire;
//  if received then begin
//    docs:=xmlToDomParser.DOMImpl.documents;
//    for i:=0 to FMsgList.Count-1 do begin
//      m:=TDebugMsg(FMsgList.Items[i]);
//      if not m.worked then begin
//        try
//          idoc:=docs.item(i+1) as TDomDocument;
//          if (m.FCommand='stack_get') then begin
//            node2:=idoc.getFirstChildElement('response');
//            if assigned(node2) then begin
//              node:=node2.getFirstChildElement('stack');
//              self.stackentrys.clearList;
//              while assigned(node) do begin
//                 uri:=TIdUri.Create(node.attributes.getNamedItem('filename').nodeValue);
//                 FCurrFilename:=TIdUri.URLDecode(uri.Document);
//                 FCurrLine:=strtointdef(node.attributes.getNamedItem('lineno').nodeValue,-1);
//                 self.stackentrys.AddEntry(FCurrFilename,FcurrLine);
//                 FCurrStackDepth:=strtointdef(Node.attributes.getNamedItem('level').nodeValue,-1);
//                 FCurrWhere:=node.attributes.getNamedItem('where').nodeValue;
//                 if (assigned(OnStackGet)) then OnStackGet(sender);
//                 node:=node.getNextSiblingElement('stack');
//              end;
//            end;
//          end;
//          if (m.FCommand='eval') and (assigned(OnEval)) then begin
//           node:=idoc.firstChild;
//           value:=Base64DecodeStr(node.XPathStringValue);
//           ex:=findEvalExpr(m.Ftransactionid);
//           if assigned(ex) then
//           OnEval(sender,m.Ftransactionid,ex.fexpr,value);
//          end;
//          if (m.FCommand='step_into') and (assigned(OnStepInto)) then begin
//            setStatus(idoc);
//            OnStepInto(sender);
//          end;
//          if (m.FCommand='step_over') and (assigned(OnStepOver)) then begin
//            setStatus(idoc);
//            OnStepOver(sender);
//          end;
//          if (m.FCommand='step_out') and (assigned(OnStepOut)) then begin
//            setStatus(idoc);
//            OnStepOut(sender);
//          end;
//          if (m.FCommand='run') and (assigned(OnRun)) then begin
//            setStatus(idoc);
//            OnRun(sender);
//          end;
//          if (m.FCommand='status') and (assigned(OnStatus)) then begin
//            setStatus(idoc);
//            OnStatus(sender);
//          end;
//          if (m.FCommand='stop') and (assigned(OnStop)) then begin
//            setStatus(idoc);
//            FCurrFilename:='';
//            FCurrLine:=-1;
//            FCurrStackDepth:=-1;
//            OnStop(sender);
//          end;
//          if (m.FCommand='breakpoint_set') then begin
//            index:=FindBreakPoint(m.Ftransactionid);
//            if index>=0 then begin
//              bp1:=breakpoints[index];
//              if assigned(bp1) then bp1.Fbreakpointid:=m.Fbpid;
//              if assigned(OnBPset) then OnBPset(sender);
//            end;
//          end;
//        finally
//          m.worked:=true;
//        end;
//      end;
//    end;
//    received:=false;
//  end;
//  UILock.Release;
//  FTimer.Enabled:=true;
//end;

procedure TDebugClient.getStatusAndID(s:string;var status:string;var ID:string);
var
  elem:TDomElement;
  attr:TDomAttr;
  doc:TDomDocument;
  len:integer;
begin
  status:='';ID:='';
  if s='' then exit;
  doc := TDomDocument(xmlToDomParser.parseString(s,'','',nil));
  if assigned(doc) then begin
    elem:=doc.findFirstChildElement;
    while assigned(elem) do begin
      attr:=elem.getAttributeNode('id');
      if assigned(attr) then ID:=attr.value;
      attr:=elem.getAttributeNode('status');
      if assigned(attr) then status:=attr.value;
      elem:=Doc.findNextSiblingElement;
    end;
  end;
end;

procedure TDebugClient.getFilenameAndLineno(s:string;var filename:string;var lineno:integer);
var
  elem,elem2:TDomElement;
  attr:TDomAttr;
  doc:TDomDocument;
  len:integer;
  uri:TIdUri;
begin
  filename:='';lineno:=-1;
  if s='' then exit;
  doc := TDomDocument(xmlToDomParser.parseString(s,'','',nil));
  if assigned(doc) then begin
    elem:=doc.findFirstChildElement;
    if assigned(elem) then begin
      elem2:=elem.getFirstChildElement('stack');
      self.stackentrys.clearList;
      if assigned(elem2) then begin
         uri:=TIdUri.Create(elem2.attributes.getNamedItem('filename').nodeValue);
         filename:=TIdUri.URLDecode(uri.Document);
         lineno:=strtointdef(elem2.attributes.getNamedItem('lineno').nodeValue,-1);
         self.stackentrys.AddEntry(FCurrFilename,FcurrLine);
         FCurrStackDepth:=strtointdef(elem2.attributes.getNamedItem('level').nodeValue,-1);
         FCurrWhere:=elem2.attributes.getNamedItem('where').nodeValue;
         elem2:=elem2.getNextSiblingElement('stack');
      end;
      elem:=Doc.findNextSiblingElement;
    end;
    doc.clear;
  end;
end;

procedure TDebugClient.getCommandAndTransactionID(s:string;var command:string;var transactionid:integer);
var
  elem:TDomElement;
  attr:TDomAttr;
  doc:TDomDocument;
  len:integer;
begin
  command:='';transactionid:=-1;
  if s='' then exit;
  doc := TDomDocument(xmlToDomParser.parseString(s,'','',nil));
  if assigned(doc) then begin
    elem:=doc.findFirstChildElement;
    while assigned(elem) do begin
      attr:=elem.getAttributeNode('command');
      if assigned(attr) then command:=attr.value;
      attr:=elem.getAttributeNode('transaction_id');
      if assigned(attr) then begin
        transactionID:=strtoint(attr.value);
        exit;
      end;
      elem:=Doc.findNextSiblingElement;
    end;
  end;
end;

function TDebugClient.findEvalExpr(transid:integer):TEvalExpr;
var i:integer;
    p:TEvalExpr;
begin
  result:=nil;
  for i:=0 to EvalList.Count-1 do begin
    p:=EvalList[i];
    if p.ftransid=transid then begin
      result:=p; exit;
    end;
  end;
end;

function TDebugClient.findBreakPoint(transid:integer):integer;
var i:integer;
begin
  result:=-1;
  for i:=0 to breakpoints.Count-1 do begin
    if Breakpoints[i].Ftransactionid=transid then begin
      result:=i; exit;
    end;
  end;
end;

function TDebugClient.findMsg(transid:integer):TDebugMsg;
var i:integer;
    m:TDebugMsg;
begin
  for i:=0 to FMsgList.Count-1 do begin
     m:=TDebugMsg(FMsgList.Items[i]);
     if m.Ftransactionid=transid then begin
       result:=m;
       exit;
     end;
  end;
  result:=nil;
end;

procedure TDebugClient.dolog(s:string);
begin
  if flogFile='' then exit;
//  loglock.Acquire;
  try
    FLogStrings.Add(s);
  finally
//    loglock.Release;
  end;
end;

function TDebugClient.DoExecute (Context: TIdContext): boolean;
var s:string;
    command,id,status,filename:string;
    transid,lineno:integer;
    m:TDebugMsg;
begin
  result := true;
  FLastResult.Clear;
  FContext:=Context;
  while FMsgList.Count>0 do begin
    m:=FMsgList.items[0];
    FreeAndNil(m);
    FMsgList.Delete(0);
  end;
//  clearBreakpoints;
  with Context.Connection do begin
    while connected do begin
      try
        s:=''; filename:='';lineno:=0;
        s:=IOHandler.ReadLn(#0);
        if strtointdef(s,-1)=-1 then begin
          loglock.Acquire;
          try
            id:='';
            getCommandAndTransactionID(s,command,transid);
            getStatusAndID(s,status,id);
  //                m:=findMsg(transid);
  //                if assigned(m) then begin
  //                  m.FCommand:=command;
  //                  m.Fbpid:=bpid;
  //                  m.Result:=sm;
  //                  received:=true;
  //                end;
            dolog(s);
            if assigned(FOnMessage) then FonMessage(self,s);
            if (command<>'stack_get')and (status='break') then send_get_stack(Context);
            if (command='stack_get') and (assigned(FOnStackGet)) then begin
              getFilenameAndLineno(s,filename,lineno);
              self.FCurrFilename:=filename;
              self.fcurrLine:=lineno;
              FOnStackGet(self,filename,lineno);
            end;
            if (status='stopped') and (assigned(FOnStop)) then begin
              self.fcurrLine:=-1;
              self.fCurrfilename:='';
              FOnStop(self);
            end;
          finally
            loglock.Release;
          end;
        end;
      except
        break;
      end;
    end;
  end;
  try
//    FcurrLine:=-1;
//    FcurrFilename:='';
//    FContext:=nil;
//      if (logFile<>'') and (FLogStrings.Count>0) then FLogStrings.SaveToFile(logFile);
    Context.Connection.Disconnect;
  except
  end;
end;

function TDebugClient.getLog():string;
var i:integer;
    m:TDebugMsg;
    t:TStringList;
begin
  t:=TStringList.Create;
  for i:=0 to FMsgList.Count-1 do begin
     m:=TDebugMsg(FMsgList.Items[i]);
     t.Add('IDE   : '+m.Fsend);
     t.Add('DEBUG : '+m.Result);
  end;
  result:=t.text;
  t.free;
end;

procedure TDebugClient.send(AContext:TIdContext;s:string);
var msg:TDebugMsg;
begin
  msg:=nil;
  msg:=TDebugMsg.create(s,transactionID);
  FMsgList.Add(msg);
  dolog(s);
  AContext.Connection.IOHandler.Write(s+#0);
  inc(transactionID);
end;

procedure TDebugClient.send_get_stack(AContext:TIdContext);
begin
  send(AContext,'stack_get -i '+inttostr(transactionID));
end;

function TDebugClient.dorun(s:string):TDebugMsg;
var msg:TDebugMsg;
begin
  msg:=nil;
  if assigned(FContext) then begin
    if assigned(FContext.Connection) then begin
      if FContext.Connection.Connected then begin
        msg:=TDebugMsg.create(s,transactionID);
        FMsgList.Add(msg);
        dolog(s);
        FContext.Connection.IOHandler.Write(s+#0);
        inc(transactionID);
      end;
    end;
  end;
  result:=msg;
end;

procedure TDebugClient.evaluate();
var i:integer;
    p:TEvalExpr;
begin
  if not activated then exit;
  for i:=0 to FEvalList.Count-1 do begin
    p:=FEvallist[i];
    p.transid:=transactionID;
    dorun('eval -i '+inttostr(transactionID)+' -- '+Base64EncodeStr(p.expr));
    while not received do if not activated then exit;
  end;
end;

procedure TDebugClient.eval(expr: string);
var index:integer;
    e:TEvalExpr;
begin
  for index:=0 to FEvalList.Count-1 do begin
    e:=FEvalList[index];
    if e.fexpr=expr then exit;
  end;
  e:=TEvalExpr.Create;
  e.fexpr:=expr;
  FEvalList.Add(e);
end;

procedure TDebugClient.run();
begin
  dorun('run -i '+inttostr(transactionID));
end;

procedure TDebugClient.Setterminate(const Value: boolean);
begin
  Fterminate := Value;
end;

procedure TDebugClient.get_Stack(depth:integer);
begin
//  dorun('stack_get -d '+inttostr(depth)+' -i '+inttostr(transactionID));
  dorun('stack_get -i '+inttostr(transactionID));
end;

procedure TDebugClient.step_into();
begin
  dorun('step_into -i '+inttostr(transactionID));
end;

procedure TDebugClient.step_over();
begin
  dorun('step_over -i '+inttostr(transactionID));
end;

procedure TDebugClient.step_out();
begin
  dorun('step_out -i '+inttostr(transactionID));
end;

procedure TDebugClient.stop();
begin
  dorun('stop -i '+inttostr(transactionID));
end;

procedure TDebugClient.context_get();
begin
  dorun('context_get -i '+inttostr(transactionID));
end;

procedure TDebugClient.setbreak();
begin
  breakpoints.addBP(bpCall,'break {main} ');
end;

procedure TDebugClient.setbreak(func: string);
begin
  breakpoints.addBP(bpCall,func);
end;

procedure TDebugClient.setbreak(filename: string;line: integer);
var bp:TBreakPoint;
begin
  bp:=breakpoints.toggleBP(filename,line);
  if status='break' then begin
    if bp.deleted then begin
      dorun('breakpoint_remove -i '+inttostr(transactionID)+ ' -d '+bp.Fbreakpointid);
    end else begin
      dorun('breakpoint_set -i '+inttostr(transactionID)+
          ' -t line -f file://'+bp.bpFilename+' -n '+inttostr(bp.bpLine));
    end;
  end;
end;

procedure TDebugClient.clearBreakpoints();
var bp:TBreakPoint;
    i:integer;
begin
  for i:=breakpoints.Count-1 downto 0 do begin
    bp:=breakpoints[i];
    if bp.deleted then begin
      bp.Free;
      breakpoints.Delete(i);
    end;
  end;
end;

procedure TDebugClient.setBreakpoints();
var i:integer;
begin
  if breakpoints.Count=0 then exit;
  for i:=0 to breakpoints.Count-1 do begin
    breakpoints[i].Ftransactionid:=transactionID;
    case breakpoints[i].bpType of
      bpCall: dorun(breakpoints[i].bpExpression);
      bpLine: dorun('breakpoint_set -i '+inttostr(transactionID)+
                    ' -t line -f file://'+breakpoints[i].bpFilename+' -n '+inttostr(breakpoints[i].bpLine));
    end;
  end;
end;

function TDebugClient.GetLastResult: TStringList;
begin
  result:=FLastResult;
end;

{ TBreakpointList }

function TBreakpointList.toggleBP(Filename: string; Line: integer):TBreakPoint;
var bp:TBreakPoint;
    i:integer;
    found:boolean;
begin
  found:=false;
  bp:=nil;
  for i:=count-1 downto 0 do begin
    bp:=Items[i];
    if (bp.bpType=bpLine) and (bp.bpFilename=Filename)
        and (bp.bpLine=Line) then  begin
      bp.deleted:=not bp.deleted;
      found:=true;break;
    end;
  end;
  if not found then begin
    bp:=TBreakPoint.Create;
    bp.bpType:=bpLine;
    bp.bpFilename:=Filename;
    bp.bpLine:=Line;
    add(bp);
  end;
  result:=bp;
end;

procedure TBreakpointList.addBP(t: TBreakpointType; expr: string);
var bp:TBreakPoint;
begin
  bp:=TBreakpoint.Create;
  bp.bpType:=t;
  bp.bpExpression:=expr;
  add(bp);
end;

procedure TBreakpointList.clearBP;
var bp1:TBreakPoint;
begin
  while Count>0 do begin
    bp1:=First;
    bp1.Free;
    delete(0);
  end;
  clear;
  capacity:=count;
end;

function TBreakpointList.GetItem(index: integer): TBreakPoint;
var p:TBreakPoint;
begin
  p:=Get(index);
  result:=p;
end;

procedure TBreakpointList.SetItem(index: integer;
  const Value: TBreakPoint);
begin
  Put(index,value);
end;

{ TMsg }

constructor TDebugMsg.create(sendstr: string; transID: integer);
begin
  inherited create;
  Fsend:=sendstr;
  FtransactionID:=transid;
  Fbpid:='';
end;

destructor TDebugMsg.destroy;
begin
  inherited;
end;

procedure TDebugClient.wactivated(const Value: boolean);
begin
  factivated := Value;
  if Value then begin
    FMsgList.Clear;
    xmlToDomParser.DOMImpl.clear;
  end;
  active:=Value;
end;

{ TBreakpoint }

constructor TBreakpoint.create;
begin
  deleted:=false;
end;

procedure TDebugClient.getStatus;
begin
  dorun('status -i '+inttostr(transactionID));
end;

{ TStackEntryList }

procedure TStackEntryList.AddEntry(filename: string; lineno: integer);
var p:TStackEntry;
begin
  p:=TStackEntry.Create;
  p.filename:=filename;
  p.lineno:=lineno;
  self.Add(p)
end;

procedure TStackEntryList.clearList;
var p:TStackEntry;
    i:integer;
begin
  for i:=self.Count-1 downto 0 do begin
    p:=Get(i);
    p.Free;
    self.Delete(i);
  end;
end;

function TStackEntryList.findLineno(filename: string): integer;
var i:integer;
begin
  result:=-1;
  for i:=0 to self.Count-1 do begin
    if (filename=getItem(i).filename) then begin
      result:=getItem(i).lineno;
      exit;
    end;
  end;
end;

function TStackEntryList.GetItem(index: integer): TStackEntry;
var p:TStackEntry;
begin
  p:=Get(index);
  result:=p;
end;

procedure TStackEntryList.SetItem(index: integer;
  const Value: TStackEntry);
begin
  Put(index,value);
end;

END.
