/*
 * File: dpi.c
 *
 * Copyright (C) 2002 Jorge Arellano Cid <jcid@inf.utfsm.cl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/*
 * (Prototype code)
 *
 * Dillo plugins (small programs that interact with dillo)
 * This should be able to handle:
 *   bookmarks, FTP, downloads, preferences, https and
 *   a lot of any-to-html filters.
 */

#ifdef _WIN32
#define USE_INET
#define	USE_GLIB_SLEEP
#define DILLO_DIP_PORT	12345
#endif

#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>           /* for errno */
#include <time.h>            /* for nanosleep */

#include <stdio.h>
#include <sys/socket.h>
#ifndef USE_INET
#include <sys/un.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "../intl.h"
#include "Url.h"
#include "IO.h"
#include "../misc.h"

#define DEBUG_LEVEL 4
#include "../debug.h"

/* This one os tricky, some sources state it should include the byte
 * for the terminating NULL, and others say it shouldn't. */
# define D_SUN_LEN(ptr) ((size_t) (((struct sockaddr_un *) 0)->sun_path) \
                        + strlen ((ptr)->sun_path))

/* Solaris may not have this one... */
#ifndef AF_LOCAL
#define AF_LOCAL AF_UNIX
#endif


typedef struct {
   gint FreeBuf;
   gint InTag, InData;
   gint PipeActive;
   gint Send2EOF;

   gint DataTotalSize;
   gint DataRecvSize;

   void *Buf;
   gint BufIdx;
   gint BufSize;

   gchar *Tok;
   gint TokSize;
   gint TokIsTag;

   ChainLink *InfoRecv;
} conn_data_t;


/*
 * Local data
 */

/*
 * Forward references
 */


/*
 * Task: given a tag and an attribute name, return its value.
 *       (character stuffing is removed here)
 * Return value: the attribute value, or NULL if not present or malformed.
 */
char *Get_attr_value(char *tag, int tagsize, char *attrname)
{
   char *p, *q, *ltag, quote, *start, *val = NULL;

   ltag = g_strndup(tag, tagsize);
   if ((p = strstr(ltag, attrname)) &&
       (p = strchr(p, '=')) &&
       (p = strpbrk(p, "'\"")) ) {
      quote = *p;
      start = ++p;
      while ((q = strchr(p, quote)) && q[1] == quote)
         p = q + 2;
      if (q) {
         val = g_strndup(start, q - start);
         for (p = q = val; (*q = *p); ++p, ++q)
            if ((*p == '"' || *p == '\'') && p[1] == p[0])
               ++p;
      }
   }
   g_free(ltag);

   return val;
}

/*
 * Create a new connection data structure
 */
static conn_data_t *Dpi_conn_data_new(ChainLink *Info)
{
   conn_data_t *conn = g_new0(conn_data_t, 1);

   conn->Buf = NULL;
   conn->Tok = NULL;
   conn->InfoRecv = Info;
   return conn;
}

/*
 * Free a connection data structure
 */
static void Dpi_conn_data_free(conn_data_t *conn)
{
   if (conn->FreeBuf)
      g_free(conn->Buf);
   g_free(conn);
}

/*
 * Append the new buffer in 'io' to Buf in 'conn'
 */
static void Dpi_append_io_buf(conn_data_t *conn, IOData_t *io)
{
   if (io->Status > 0) {
      conn->Buf = g_realloc(conn->Buf, io->Status);
      memcpy(conn->Buf + conn->BufSize, io->Buf, io->Status);
      conn->BufSize += io->Status;
      conn->FreeBuf = 0;
   }
}

/*
 * Split the data stream into tokens.
 * Here, a token is either:
 *    a) a dpi tag
 *    b) a raw data chunk
 */
static gint Dpi_get_token(conn_data_t *conn)
{
   gint resp = -1;
   gchar *buf = conn->Buf;

   if (conn->FreeBuf || conn->BufIdx == conn->BufSize) {
      g_free(conn->Buf);
      conn->Buf = NULL;
      conn->BufIdx = conn->BufSize = 0;
      conn->FreeBuf = 0;
      return resp;
   }

   if (conn->Send2EOF) {
      conn->Tok = buf + conn->BufIdx;
      conn->TokSize = conn->BufSize - conn->BufIdx;
      conn->BufIdx = conn->BufSize;
      return 0;
   }

   if (!conn->InTag && !conn->InData) {
      /* search for start of tag */
      while (conn->BufIdx < conn->BufSize && buf[conn->BufIdx] != '<')
         ++conn->BufIdx;
      if (conn->BufIdx < conn->BufSize) {
         /* found */
         conn->InTag = 1;
         conn->Tok = buf + conn->BufIdx;
      } else {
         DEBUG_MSG(4, _("ERROR: [Dpi_get_token] Can't find token start\n"));
         conn->FreeBuf = 1;
         return Dpi_get_token(conn);
      }
   }

   if (conn->InTag) {
      /* search for end of tag */
      while (conn->BufIdx < conn->BufSize && buf[conn->BufIdx] != '>')
         ++conn->BufIdx;
      if (conn->BufIdx < conn->BufSize) {
         /* found EOT */
         conn->TokIsTag = 1;
         conn->TokSize = buf + conn->BufIdx - conn->Tok + 1;
         ++conn->BufIdx;
         conn->InTag = 0;
         resp = 0;
      }
   }

   if (conn->InData) {
      conn->TokIsTag = 0;
      if (conn->DataRecvSize + conn->BufSize - conn->BufIdx <
          conn-> DataTotalSize) {
         conn->TokSize += conn->BufSize - conn->BufIdx;
         conn->DataRecvSize += conn->BufSize - conn->BufIdx;
         conn->FreeBuf = 1;
         resp = 0;
      } else {
         /* srch end of data */
         DEBUG_MSG(4, _("ERROR: [Dpi_get_token] *** NULL code here ***\n"));
         while (conn->BufIdx < conn->BufSize)
            ++conn->BufIdx;
         resp = -1;
      }
   }

   return resp;
}

/*
 * Parse a dpi tag and take the appropriate actions
 */
static void Dpi_parse_token(conn_data_t *conn)
{
   gchar *tag, *cmd, *msg, *urlstr;
   DataBuf *dbuf;

   if (conn->Send2EOF) {
      /* we're receiving data chunks from a HTML page */
      dbuf = a_Chain_dbuf_new(conn->Tok, conn->TokSize, 0);
      a_Chain_fcb(OpSend, conn->InfoRecv, dbuf, "send_page_2eof");
      g_free(dbuf);
      return;
   }

   tag = g_strndup(conn->Tok, conn->TokSize);
   DEBUG_MSG(3, _("Dpi_parse_token: [%s]\n"), tag);
   g_free(tag);

   cmd = Get_attr_value(conn->Tok, conn->TokSize, "cmd");
   if (strcmp(cmd, "send_status_message") == 0) {
      msg = Get_attr_value(conn->Tok, conn->TokSize, "msg");
      a_Chain_fcb(OpSend, conn->InfoRecv, msg, cmd);
      g_free(msg);

   } else if (strcmp(cmd, "chat") == 0) {
      msg = Get_attr_value(conn->Tok, conn->TokSize, "msg");
      a_Chain_fcb(OpSend, conn->InfoRecv, msg, cmd);
      g_free(msg);

   } else if (strcmp(cmd, "start_send_page") == 0) {
      urlstr = Get_attr_value(conn->Tok, conn->TokSize, "url");
      a_Chain_fcb(OpSend, conn->InfoRecv, urlstr, cmd);
      g_free(urlstr);
      /* todo:  Get_attr_value(conn->Tok, conn->TokSize, "send_mode")  */
      conn->Send2EOF = 1;

   } else if (strcmp(cmd, "reload_request") == 0) {
      urlstr = Get_attr_value(conn->Tok, conn->TokSize, "url");
      a_Chain_fcb(OpSend, conn->InfoRecv, urlstr, cmd);
      g_free(urlstr);
   }
   g_free(cmd);
}



/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

/*
 * Get a new data buffer (within an 'io'), save it into local data,
 * split in tokens and parse the contents.
 */
static void Dpi_process_io(int Op, void *Data1, conn_data_t *conn)
{
   IOData_t *io = Data1;

   /* show data as received */
   //fwrite(io->Buf, io->Status, 1, stdout);

   if (Op == IORead) {
      Dpi_append_io_buf(conn, io);
      while (Dpi_get_token(conn) != -1) {
         Dpi_parse_token(conn);
      }

   } else if (Op == IOClose) {
      DEBUG_MSG(3, _("Dpi: [Dpi_process_io] IOClose\n"));
   }
}

/*
 * Start the bookmarks server
 */
static gint Dpi_launch_server()
{
#ifdef _WIN32
      gchar *path1 = a_Misc_prepend_user_home(".dillo/bm_srv12");
      if (spawnl(P_NOWAIT,path1, "bm_srv12", NULL) == -1) {
         g_free(path1);
         if (spawnlp(P_NOWAIT,"bm_srv12", "bm_srv12", NULL) == -1)
            return -1;
      }
#else
   pid_t pid;

   pid = fork();
   if (pid == 0) {
      /* This is the child process.  Execute the command. */
      gchar *path1 = a_Misc_prepend_user_home(".dillo/bm_srv12");
      if (execl(path1, "bm_srv12", NULL) == -1) {
         g_free(path1);
         if (execlp("bm_srv12", "bm_srv12", NULL) == -1)
            exit (EXIT_FAILURE);
      }
   } else if (pid < 0) {
      /* The fork failed.  Report failure.  */
      return -1;
   } else {
      /* This is the parent process. */
   }
#endif
   return 0;
}

/*
 * Connect a socket to a dpi server and return the socket's FD.
 * (-1 on error)
 * todo: design an algorithm that handles generic dpi programs.
 *       Currently it only handles the bookmarks server.
 */
static gint Dpi_connect_socket(gint retry)
{
   char *filename;
#ifdef USE_INET
   struct sockaddr_in pin;
#else
   struct sockaddr_un pun;
#endif
   gint SockFD, i, err;
#ifndef USE_GLIB_SLEEP
   struct timespec delay;
#endif

   /* create a filename for the server's socket */
   filename = g_strconcat("/tmp/bm.", g_get_user_name(), NULL);

   if (access(filename, F_OK) != 0) {
      /* start server */
      DEBUG_MSG(3, _("Server NOT running\n"));
      if (Dpi_launch_server() == 0) {
         /* wait a bit until the server is ready */
#ifndef USE_GLIB_SLEEP
         delay.tv_sec = 0;
         delay.tv_nsec = 25000;
#endif
         for (i = 0; i < 256; ++i) {
#ifdef USE_GLIB_SLEEP
            g_usleep(25000/1000);
#else
            nanosleep(&delay, NULL);
#endif
            if (access(filename, F_OK) == 0)
               break;
         }
         DEBUG_MSG(3, _("<waited %d cycles for server to be on line>\n"), i);
      }
   }

#ifdef USE_INET
   memset(&pin, 0, sizeof(struct sockaddr_in));
   pin.sin_family = AF_INET;
   pin.sin_port = htons(DILLO_DIP_PORT);
   pin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
   g_free(filename);

   if ((SockFD = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1)
      perror("[dpi::socket]");
   else if (connect(SockFD, (struct sockaddr*)&pin, sizeof(struct sockaddr_in)) == -1) {
#else
   memset(&pun, 0, sizeof(struct sockaddr_un));
   pun.sun_family = AF_LOCAL;
   strncpy(pun.sun_path, filename, sizeof (pun.sun_path));
   g_free(filename);

   if ((SockFD = socket(AF_LOCAL, SOCK_STREAM, 0)) == -1)
      perror("[dpi::socket]");
   else if (connect(SockFD, (void*)&pun, D_SUN_LEN(&pun)) == -1) {
#endif
      err = errno;
      SockFD = -1;
      DEBUG_MSG(5, _("[dpi::connect] errno:%d %s\n"), errno, g_strerror(errno));
      if (retry) {
#ifndef _WIN32
         switch (err) {
            case ECONNREFUSED: case EBADF: case ENOTSOCK: case EADDRNOTAVAIL:
               /* the server may crash and its socket name survive */
               unlink(pun.sun_path);
               SockFD = Dpi_connect_socket(FALSE);
               break;
         }
#endif
      }
   }

   return SockFD;
}


/*
 * CCC function for the Dpi module
 */
void a_Dpi_ccc(int Op, int Branch, int Dir, ChainLink *Info,
               void *Data1, void *Data2)
{
   static gint PipeReadFD = -1;
   gint SockFD;

   a_Chain_debug_msg("a_Dpi_ccc", Op, Branch, Dir);

   if (Branch == 1) {
      if (Dir == BCK) {
         /* Cache request */
         /* ( Data1 = URL;  Data2 = Web ) */
         switch (Op) {
         case OpStart:
            /* Tell the cache to start the receiving CCC */
            a_Chain_fcb(OpSend, Info, (void *)PipeReadFD, NULL);
            /* End the requesting CCC */
            a_Chain_fcb(OpEnd, Info, NULL, NULL);
            break;
         }
      }

   } else if (Branch == 2) {
      if (Dir == BCK) {
         /* Send commands to dpi-server */
         switch (Op) {
         case OpStart:
            SockFD = Dpi_connect_socket(TRUE);
            if (SockFD != -1) {
               Info->LocalKey = (void *)SockFD;
               a_Chain_link_new(Info, a_Dpi_ccc, BCK, a_IO_ccc, 3, 2);
               a_Chain_bcb(OpStart, Info, (void *)SockFD, NULL);
               /* tell the capi to start the receiving branch */
               a_Chain_fcb(OpSend, Info, (void *)SockFD, "SockFD");
            } else {
               a_Dpi_ccc(OpAbort, 2, FWD, Info, NULL, NULL);
            }
            break;
         case OpSend:
            a_Chain_bcb(OpSend, Info, Data1, NULL);
            break;
         case OpEnd:
            a_Chain_bcb(OpEnd, Info, NULL, NULL);
            g_free(Info);
            break;
         case OpAbort:
            DEBUG_MSG(4, _(" Not implemented\n"));
            break;
         }
      } else {  /* FWD */
         /* Send commands to dpi-server (status) */
         switch (Op) {
         case OpEnd:
            a_Chain_del_link(Info, BCK);
            g_free(Info);
            break;
         case OpSend:
         case OpAbort:
            DEBUG_MSG(4, _(" Not implemented\n"));
            break;
         }
      }

   } else if (Branch == 3) {
      if (Dir == FWD) {
         /* Receiving from server */
         switch (Op) {
         case OpSend:
            Dpi_process_io(IORead, Data1, Info->LocalKey);
            break;
         case OpEnd:
            a_Chain_del_link(Info, BCK);
            Dpi_process_io(IOClose, Data1, Info->LocalKey);
            Dpi_conn_data_free(Info->LocalKey);
            a_Chain_fcb(OpEnd, Info, NULL, NULL);
            break;
         case OpAbort:
            g_print(" Not implemented\n");
            break;
         }
      } else {  /* BCK */
         switch (Op) {
         case OpStart:
            {
            IOData_t *io2;
            Info->LocalKey = Dpi_conn_data_new(Info);
            io2 = a_IO_new(IORead, (gint)Data1); /* SockFD */
            a_IO_set_buf(io2, NULL, IOBufLen);
            a_Chain_link_new(Info, a_Dpi_ccc, BCK, a_IO_ccc, 2, 3);
            a_Chain_bcb(OpStart, Info, io2, NULL);
            break;
            }
         case OpSend:
            PipeReadFD = (gint) Data1;
            break;
         case OpAbort:
            g_print(" Not implemented\n");
            break;
         }
      }

   } else if (Branch == 4) {
      /* Send content, from DPI, through a pipe */
      if (Dir == BCK) {
         switch (Op) {
         case OpStart:
            Info->LocalKey = Data1;
            a_Chain_link_new(Info, a_Dpi_ccc, BCK, a_IO_ccc, 1, 4);
            a_Chain_bcb(OpStart, Info, Data1, NULL);
            break;
         case OpSend:
            a_Chain_bcb(OpSend, Info, Data1, NULL);
            break;
         case OpAbort:
            a_Chain_bcb(OpAbort, Info, NULL, NULL);
            g_free(Info);
            break;
         }
      } else {  /* FWD */
         switch (Op) {
         case OpEnd:
            /* unlink IO_Info */
            a_Chain_del_link(Info, BCK);
            g_free(Info);
            break;
         case OpAbort:
            g_print(" Not implemented\n");
            break;
         }
      }
   }
}

