/*
 * File: gtkframeset.c
 * Copyright (C) 2003 Frank de Lange <frank@unternet.org>
 *
 *   a frameset widget for GTK - The GIMP Toolkit
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* this ifdef is Dillo-specific... */
#ifdef XHTML_DTD_FRAMESET

#include "gtkframeset.h"
#include <stdlib.h>              /* for strtod() */
#include <ctype.h>               /* for isspace() */
#include <string.h>              /* for strpbrk() */
//#include <stdlib.h>              /* for abs() */


#define DEBUG_ALLOC 10
#define DEBUG_EVENT 10
/* #define DEBUG_LEVEL 10 */
#include "debug.h"

/* child args */
enum
{
  CHILD_ARG_0,
  CHILD_ARG_ROW_ATTACH,
  CHILD_ARG_COL_ATTACH,
  CHILD_ARG_X_PADDING,
  CHILD_ARG_Y_PADDING,
  CHILD_ARG_BORDER,
  CHILD_ARG_NORESIZE
};
  
/* declarations */
static void gtk_frameset_class_init          (GtkFramesetClass  *klass);
static void gtk_frameset_init	             (GtkFrameset       *frameset);
static void gtk_frameset_finalize	     (GtkObject	        *object);
static void gtk_frameset_size_allocate       (GtkWidget	        *widget,
					      GtkAllocation     *allocation);
static void gtk_frameset_map	             (GtkWidget	        *widget);
static void gtk_frameset_unmap	             (GtkWidget	        *widget);
static void gtk_frameset_draw	             (GtkWidget	        *widget,
					      GdkRectangle      *area);
static gint gtk_frameset_expose	             (GtkWidget	        *widget,
					      GdkEventExpose    *event);
static void gtk_frameset_set_child_arg       (GtkContainer      *container,
					      GtkWidget         *child,
					      GtkArg            *arg,
					      guint              arg_id);
static void gtk_frameset_get_child_arg       (GtkContainer      *container,
					      GtkWidget         *child,
					      GtkArg            *arg,
					      guint              arg_id);
static GtkType gtk_frameset_child_type       (GtkContainer      *container);
static void gtk_frameset_add	             (GtkContainer      *container,
					      GtkWidget	        *widget);
static void gtk_frameset_remove	             (GtkContainer      *container,
					      GtkWidget	        *widget);
static void gtk_frameset_forall	             (GtkContainer      *container,
					      gboolean	         include_internals,
					      GtkCallback        callback,
					      gpointer	         callback_data);
static void gtk_frameset_realize             (GtkWidget         *widget);
static void gtk_frameset_unrealize           (GtkWidget         *widget);
static gint gtk_frameset_button_press        (GtkWidget         *widget,
					      GdkEventButton    *event);
static gint gtk_frameset_button_release      (GtkWidget         *widget,
					      GdkEventButton    *event);
static gint gtk_frameset_motion              (GtkWidget         *widget,
					      GdkEventMotion    *event);
static gint gtk_frameset_enter               (GtkWidget         *widget,
					      GdkEventCrossing  *event);
static gint gtk_frameset_leave               (GtkWidget         *widget,
					      GdkEventCrossing  *event);

/* private functions */
static void gtk_frameset_size_allocate_init (GtkFrameset        *frameset);
static void gtk_frameset_size_allocate_pass1(GtkFrameset        *frameset);
static GSList *gtk_frameset_get_multi_length(const gchar        *attr);
static void gtk_frameset_calculate_lengths  (GtkFrameset        *frameset);
static gint gtk_frameset_set_resize_rowcol  (GtkWidget          *widget,
					     guint               x,
					     guint               y);

static GtkContainerClass *parent_class = NULL;

/* standard GTK function */
GtkType
gtk_frameset_get_type (void)
{
  static GtkType frameset_type = 0;
  
  if (!frameset_type)
    {
      static const GtkTypeInfo frameset_info =
      {
	"GtkFrameset",
	sizeof (GtkFrameset),
	sizeof (GtkFramesetClass),
	(GtkClassInitFunc) gtk_frameset_class_init,
	(GtkObjectInitFunc) gtk_frameset_init,
        /* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) NULL,
      };
      
      frameset_type = gtk_type_unique (gtk_container_get_type (), &frameset_info);
    }
  
  return frameset_type;
}

/* standard GTK function */
static void
gtk_frameset_class_init (GtkFramesetClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  
  parent_class = gtk_type_class (gtk_container_get_type ());
  
  gtk_container_add_child_arg_type ("GtkFrameset::row_attach", GTK_TYPE_UINT, GTK_ARG_READWRITE, CHILD_ARG_ROW_ATTACH);
  gtk_container_add_child_arg_type ("GtkFrameset::col_attach", GTK_TYPE_UINT, GTK_ARG_READWRITE, CHILD_ARG_COL_ATTACH);
  gtk_container_add_child_arg_type ("GtkFrameset::x_padding", GTK_TYPE_UINT, GTK_ARG_READWRITE, CHILD_ARG_X_PADDING);
  gtk_container_add_child_arg_type ("GtkFrameset::y_padding", GTK_TYPE_UINT, GTK_ARG_READWRITE, CHILD_ARG_Y_PADDING);
  gtk_container_add_child_arg_type ("GtkFrameset::border", GTK_TYPE_BOOL, GTK_ARG_READWRITE, CHILD_ARG_BORDER);
  gtk_container_add_child_arg_type ("GtkFrameset::noresize", GTK_TYPE_BOOL, GTK_ARG_READWRITE, CHILD_ARG_NORESIZE);

  object_class->finalize = gtk_frameset_finalize;
  
  widget_class->size_allocate = gtk_frameset_size_allocate;
  widget_class->map = gtk_frameset_map;
  widget_class->unmap = gtk_frameset_unmap;
  widget_class->draw = gtk_frameset_draw;
  widget_class->expose_event = gtk_frameset_expose;
  widget_class->realize = gtk_frameset_realize;
  widget_class->unrealize = gtk_frameset_unrealize;
  widget_class->button_press_event = gtk_frameset_button_press;
  widget_class->button_release_event = gtk_frameset_button_release;
  widget_class->motion_notify_event = gtk_frameset_motion;
  widget_class->enter_notify_event = gtk_frameset_enter;
  widget_class->leave_notify_event = gtk_frameset_leave;
  
  container_class->add = gtk_frameset_add;
  container_class->remove = gtk_frameset_remove;
  container_class->forall = gtk_frameset_forall;
  container_class->child_type = gtk_frameset_child_type;
  container_class->set_child_arg = gtk_frameset_set_child_arg;
  container_class->get_child_arg = gtk_frameset_get_child_arg;
}

/* standard GTK function */
static GtkType
gtk_frameset_child_type (GtkContainer   *container)
{
  return GTK_TYPE_WIDGET;
}

/* standard GTK function */
static void
gtk_frameset_set_child_arg (GtkContainer   *container,
			    GtkWidget      *child,
			    GtkArg         *arg,
			    guint           arg_id)
{
  GtkFrameset *frameset;
  GtkFramesetChild *frameset_child;
  GList *list;

  frameset = GTK_FRAMESET (container);
  frameset_child = NULL;
  for (list = frameset->children; list; list = list->next)
    {
      frameset_child = list->data;

      if (frameset_child->widget == child)
	break;
    }
  if (!list)
    return;

  switch (arg_id)
    {
    case CHILD_ARG_ROW_ATTACH:
      frameset_child->row_attach = GTK_VALUE_UINT (*arg);
      break;
    case CHILD_ARG_COL_ATTACH:
      frameset_child->col_attach = GTK_VALUE_UINT (*arg);
      break;
    case CHILD_ARG_X_PADDING:
      frameset_child->xpadding = GTK_VALUE_UINT (*arg);
      break;
    case CHILD_ARG_Y_PADDING:
      frameset_child->ypadding = GTK_VALUE_UINT (*arg);
      break;
    case CHILD_ARG_BORDER:
      frameset_child->border = GTK_VALUE_BOOL (*arg);
      /* only SET this value (it is set to FALSE by default), otherwise
       * border frames followed by noborder frames will lose their border
       * 
       * This feature can also be provided by manipulating the child widget's
       * border directly. I have tested this and it works. I might use it in a later
       * version of GtkFrameset */
      if(GTK_VALUE_BOOL(*arg)) {
	if(frameset_child->row_attach > 0)
	  /* set top border */
	  frameset->rows[frameset_child->row_attach - 1].border = GTK_VALUE_BOOL (*arg);
	frameset->rows[frameset_child->row_attach].border = GTK_VALUE_BOOL (*arg);
	if(frameset_child->col_attach > 0)
	  /* set left border */
	  frameset->cols[frameset_child->col_attach - 1].border = GTK_VALUE_BOOL (*arg);
	frameset->cols[frameset_child->col_attach].border = GTK_VALUE_BOOL (*arg);
      }
      break;
    case CHILD_ARG_NORESIZE:
      frameset_child->noresize = GTK_VALUE_BOOL (*arg);
      /* only SET this value (it is set to FALSE by default), otherwise
       * noresize frames followed by resizable frames will be resizable as
       * well through the resizable frames' left/top edge */
      if(GTK_VALUE_BOOL(*arg)) {
	if(frameset_child->row_attach + 1 < frameset->nrows)
	  /* make lower edge unresizable */
	  frameset->rows[frameset_child->row_attach + 1].noresize = GTK_VALUE_BOOL (*arg);
	frameset->rows[frameset_child->row_attach].noresize = GTK_VALUE_BOOL (*arg);
	if(frameset_child->col_attach + 1 < frameset->ncols)
	  /* make right edge unresizable */
	  frameset->cols[frameset_child->col_attach + 1].noresize = GTK_VALUE_BOOL (*arg);
	frameset->cols[frameset_child->col_attach].noresize = GTK_VALUE_BOOL (*arg);
      }
      break;
    default:
      break;
    }
  if (GTK_WIDGET_VISIBLE (child) && GTK_WIDGET_VISIBLE (frameset))
    gtk_widget_queue_resize (child);
}

/* standard GTK function */
static void
gtk_frameset_get_child_arg (GtkContainer   *container,
			 GtkWidget      *child,
			 GtkArg         *arg,
			 guint           arg_id)
{
  GtkFrameset *frameset;
  GtkFramesetChild *frameset_child;
  GList *list;

  frameset = GTK_FRAMESET (container);
  frameset_child = NULL;
  for (list = frameset->children; list; list = list->next)
    {
      frameset_child = list->data;

      if (frameset_child->widget == child)
	break;
    }
  if (!list)
    return;

  switch (arg_id)
    {
    case CHILD_ARG_ROW_ATTACH:
      GTK_VALUE_UINT (*arg) = frameset_child->row_attach;
      break;
    case CHILD_ARG_COL_ATTACH:
      GTK_VALUE_UINT (*arg) = frameset_child->col_attach;
      break;
    case CHILD_ARG_X_PADDING:
      GTK_VALUE_UINT (*arg) = frameset_child->xpadding;
      break;
    case CHILD_ARG_Y_PADDING:
      GTK_VALUE_UINT (*arg) = frameset_child->ypadding;
      break;
    case CHILD_ARG_BORDER:
      GTK_VALUE_BOOL (*arg) = frameset_child->border;
      break;
    case CHILD_ARG_NORESIZE:
      GTK_VALUE_BOOL (*arg) = frameset_child->noresize;
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

/*
 * Standard GTK function
 */
static void
gtk_frameset_realize (GtkWidget *widget)
{
  GtkFrameset *frameset = GTK_FRAMESET(widget);
  GdkWindowAttr attributes;
  gint attributes_mask;

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;
  attributes_mask = (GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP);
  attributes.event_mask |= (GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_POINTER_MOTION_MASK |
			    GDK_POINTER_MOTION_HINT_MASK |
			    GDK_ENTER_NOTIFY_MASK |
			    GDK_LEAVE_NOTIFY_MASK);
  
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
				   &attributes, attributes_mask);

  frameset->cursor_rowcol = gdk_cursor_new(GDK_FLEUR);
  frameset->cursor_row = gdk_cursor_new(GDK_SB_V_DOUBLE_ARROW);
  frameset->cursor_col = gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);

  gdk_window_set_user_data (widget->window, frameset);
  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
  gdk_window_set_back_pixmap (widget->window, NULL, TRUE);
}

/*
 * standard GTK function
 */
static void
gtk_frameset_unrealize (GtkWidget *widget)
{
  GtkFrameset *frameset = GTK_FRAMESET(widget);

  gdk_cursor_destroy(frameset->cursor_rowcol);
  gdk_cursor_destroy(frameset->cursor_row);
  gdk_cursor_destroy(frameset->cursor_col);

  GTK_WIDGET_CLASS(parent_class)->unrealize (widget);
}
  
/* standard GTK function */
static void
gtk_frameset_init (GtkFrameset *frameset)
{
  GTK_WIDGET_UNSET_FLAGS (frameset, GTK_NO_WINDOW);

  frameset->children = NULL;
  frameset->rows = NULL;
  frameset->cols = NULL;
  frameset->nrows = 0;
  frameset->ncols = 0;
  frameset->resize_row = RESIZE_NONE;
  frameset->resize_col = RESIZE_NONE;
  frameset->in_drag = FALSE;
  frameset->cursor_rowcol = NULL;
  frameset->cursor_row = NULL;
  frameset->cursor_col = NULL;

  gtk_frameset_resize (frameset, "*", "*");
}

/*
 * standard GTK function
 *
 * resize frameset. It is only possible to enlarge the
 * frameset (or change dimensions while keeping the same
 * size) as reducing the size could orphan children.
 */
void
gtk_frameset_resize (GtkFrameset *frameset,
		     gchar *row_multilengths,
		     gchar *col_multilengths)
{
  gint n_rows, n_cols, n;
  GSList *rows, *cols;
  gfloat total_percentage, total_relative;
  gfloat row_per_relative, row_per_percent;
  gfloat col_per_relative, col_per_percent;

  g_return_if_fail (frameset != NULL);
  g_return_if_fail (GTK_IS_FRAMESET(frameset));
  g_return_if_fail (row_multilengths || col_multilengths);

  DEBUG_MSG(DEBUG_EVENT, "gtk_frameset_resize(%d, %s, %s)\n", (gint) frameset, row_multilengths, col_multilengths);
  
  rows = gtk_frameset_get_multi_length(row_multilengths);
  cols = gtk_frameset_get_multi_length(col_multilengths);

  /* (older versions of) compiler happiness */
  row_per_relative = 0;
  row_per_percent = 0;
  col_per_relative = 0;
  col_per_percent = 0;

  n_rows = g_slist_length(rows);
  n_cols = g_slist_length(cols);

   /* calculate row and column dimensions. These calculations are slightly hairy...
    * The theory goes as follows:
    * 
    * Starting point is the frameset width and height. 
    * First come frames with absolute dimensions. No frame can be bigger than the frameset,
    * so checks are performed to make sure that absolute frame dimensions do not
    * exceed frameset dimensions. When the total number of frames with absolute dimensions
    * exceeds the frameset dimensions, the available space is divided by ratio to those frames.
    * In case there are only absolute dimensioned frames, available space is divided by ratio to
    * these frames and the absolute dimensions are transformed into percentual dimensions.
    *
    * Next come frames with percentual dimensions. They get to divide the remaining space after
    * all absolute dimensions have been allocated. If the total of all percentual dimensions adds
    * up to more than 100, the calculation is normalised to 100%.
    *
    * Last come frames with relative dimensions. They get to divide the remaining space after
    * all absolute and percentual dimensions have been allocated. All available space is divided
    * by ratio to relative weights and allocated to frames. In the second stage, relative dimensions
    * are transformed into percentual dimensions so the frameset ends up having only absolute (pixel)
    * and percentual dimensions.
    *
    * The calculations are performed in two stages. In the first (aggregation) stage, the total weight of
    * absolute, percentual and relative dimensions is calculated. In the second (normalisation) stage, relative
    * dimensions are transformed into percentual dimensions. The actual calculation of pixel dimensions
    * is performed in the callback function as it depends on information about the current size of the
    * frameset widget.
    *
    * Can this be optimised? Sure, but... later...
    */

  if(n_rows >= frameset->nrows)
    {
      frameset->nrows = n_rows;
      frameset->rows = g_realloc (frameset->rows, frameset->nrows * sizeof (GtkFramesetRowCol));

      frameset->row_total_absolute = 0;
      total_percentage = 0;
      total_relative = 0;
      
      for(n = 0; n < frameset->nrows; n++) {
	frameset->rows[n].length = GPOINTER_TO_INT(g_slist_nth_data(rows, n));
	frameset->rows[n].noresize = FALSE;
	frameset->rows[n].border = FALSE;
	if(LENGTH_IS_RELATIVE(frameset->rows[n].length))
	  (LENGTH_GET_RELATIVE(frameset->rows[n].length) ?
	   total_relative += LENGTH_GET_RELATIVE(frameset->rows[n].length) :
	   total_relative++);
	else if(LENGTH_IS_PERCENTAGE(frameset->rows[n].length))
	  total_percentage += LENGTH_GET_PERCENTAGE(frameset->rows[n].length);
	else if(LENGTH_IS_ABSOLUTE(frameset->rows[n].length)) {
	  frameset->row_total_absolute += LENGTH_GET_ABSOLUTE(frameset->rows[n].length);
	}
      }
      
      if((total_percentage == 0) && (total_relative == 0)) {
	for(n = 0; n < frameset->nrows; n++)
	  if(LENGTH_IS_ABSOLUTE(frameset->rows[n].length))
	    frameset->rows[n].length =
	      LENGTH_CREATE_PERCENTAGE((gfloat) LENGTH_GET_ABSOLUTE(frameset->rows[n].length) /
				       frameset->row_total_absolute);
	frameset->row_total_absolute = 0;
	row_per_percent = 1;
      } else if(total_percentage < 1) {
	row_per_relative = (gfloat) (1 - total_percentage) / total_relative;
	row_per_percent = (gfloat) (total_relative > 0 ? 1 : 1.0 / total_percentage);
      } else {
	row_per_percent = (gfloat) (1 / total_percentage);
	row_per_relative = 0;
      }
    }

  if(n_cols >= frameset->ncols)
    {
      frameset->ncols = n_cols;
      frameset->cols = g_realloc (frameset->cols, frameset->ncols * sizeof (GtkFramesetRowCol));
      frameset->col_total_absolute = 0;
      total_percentage = 0;
      total_relative = 0;
      
      for(n = 0; n < frameset->ncols; n++) {
	frameset->cols[n].length = GPOINTER_TO_INT(g_slist_nth_data(cols, n));
	frameset->cols[n].noresize = FALSE;
	frameset->cols[n].border = FALSE;
	if(LENGTH_IS_RELATIVE(frameset->cols[n].length))
	  (LENGTH_GET_RELATIVE(frameset->cols[n].length) ?
	   total_relative += LENGTH_GET_RELATIVE(frameset->cols[n].length) :
	   total_relative++);
	else if(LENGTH_IS_PERCENTAGE(frameset->cols[n].length))
	  total_percentage += LENGTH_GET_PERCENTAGE(frameset->cols[n].length);
	else if(LENGTH_IS_ABSOLUTE(frameset->cols[n].length)) {
	  frameset->col_total_absolute += LENGTH_GET_ABSOLUTE(frameset->cols[n].length);
	}
      }

      /* normalize values */
      if((total_percentage == 0) && (total_relative == 0)) {
	for(n = 0; n < frameset->ncols; n++)
	  if(LENGTH_IS_ABSOLUTE(frameset->cols[n].length))
	    frameset->cols[n].length =
	      LENGTH_CREATE_PERCENTAGE((gfloat) LENGTH_GET_ABSOLUTE(frameset->cols[n].length) /
				       frameset->col_total_absolute);
	frameset->col_total_absolute = 0;
	col_per_percent = 1;
	col_per_relative = 0;
      } else if(total_percentage < 1) {
	col_per_relative = (gfloat) (1 - total_percentage) / total_relative;
	col_per_percent = (gfloat) (total_relative > 0 ? 1 : 1.0 / total_percentage);
      } else {
	col_per_percent = (gfloat) (1 / total_percentage);
	col_per_relative = 0;
      }
    }

  /* now, calculate actual width/height distribution */
  if(frameset->nrows > 1) {
     for(n_rows = 0; n_rows < frameset->nrows; n_rows++) {
       if(LENGTH_IS_RELATIVE(frameset->rows[n_rows].length))
	 frameset->rows[n_rows].length = 
	   LENGTH_CREATE_PERCENTAGE((LENGTH_GET_RELATIVE(frameset->rows[n_rows].length) ?
				     (LENGTH_GET_RELATIVE(frameset->rows[n_rows].length) * row_per_relative) :
				     row_per_relative));
       else if(LENGTH_IS_PERCENTAGE(frameset->rows[n_rows].length))
	 frameset->rows[n_rows].length =
	   LENGTH_CREATE_PERCENTAGE((LENGTH_GET_PERCENTAGE(frameset->rows[n_rows].length) * row_per_percent));
     }
  } else
    frameset->rows[0].length = LENGTH_CREATE_PERCENTAGE(1);

  if(frameset->ncols > 1) {
     for(n_cols = 0; n_cols < frameset->ncols; n_cols++) {
       if(LENGTH_IS_RELATIVE(frameset->cols[n_cols].length))
	 frameset->cols[n_cols].length = 
	   LENGTH_CREATE_PERCENTAGE((LENGTH_GET_RELATIVE(frameset->cols[n_cols].length) ?
				     (LENGTH_GET_RELATIVE(frameset->cols[n_cols].length) * col_per_relative) :
				     col_per_relative));
       else if(LENGTH_IS_PERCENTAGE(frameset->cols[n_cols].length))
	 frameset->cols[n_cols].length =
	   LENGTH_CREATE_PERCENTAGE(LENGTH_GET_PERCENTAGE(frameset->cols[n_cols].length) * col_per_percent);
     }
  } else
    frameset->cols[0].length = LENGTH_CREATE_PERCENTAGE(1);

  g_slist_free(rows);
  g_slist_free(cols);
}

/* standard GTK function */
GtkWidget*
gtk_frameset_new (gchar	*row_multilengths,
		  gchar	*col_multilengths)
{
  GtkFrameset *frameset;

  if (!row_multilengths)
    row_multilengths = "*";
  if (!col_multilengths)
    col_multilengths = "*";
  
  frameset = gtk_type_new (gtk_frameset_get_type ());

  DEBUG_MSG(DEBUG_EVENT, "/*\n");

  gtk_frameset_resize(frameset, row_multilengths, col_multilengths);

  DEBUG_MSG(DEBUG_EVENT, " * NEW frameset: %d\n", (gint) frameset);
  DEBUG_MSG(DEBUG_EVENT, " *               %d rows, %d cols\n", frameset->nrows, frameset->ncols);
  DEBUG_MSG(DEBUG_EVENT, " */\n");
  
  return GTK_WIDGET (frameset);
}

/* standard GTK function */
void
gtk_frameset_attach (GtkFrameset	  *frameset,
		     GtkWidget	          *child,
		     guint		   row_attach,
		     guint		   col_attach,
		     guint                 xpadding,
		     guint                 ypadding,
		     gboolean              border,
		     gboolean              noresize)
{
  GtkFramesetChild *frameset_child;
  
  g_return_if_fail (frameset != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (frameset));
  g_return_if_fail (child != NULL);
  g_return_if_fail (GTK_IS_WIDGET (child));
  g_return_if_fail (child->parent == NULL);
  g_return_if_fail (col_attach < frameset->ncols);
  g_return_if_fail (row_attach < frameset->nrows);

  DEBUG_MSG(DEBUG_EVENT, "gtk_frameset_attach(%d, %d, %d, %d, %d, %d, %d, %d)\n",
	    (gint) frameset, (gint) child, row_attach, col_attach,
	    xpadding, ypadding, border, noresize);
  
  frameset_child = g_new (GtkFramesetChild, 1);
  frameset_child->widget = child;
  frameset_child->row_attach = row_attach;
  frameset_child->col_attach = col_attach;
  frameset_child->xpadding = xpadding;
  frameset_child->ypadding = ypadding;
  frameset_child->border = border;
  frameset_child->noresize = noresize;
  
  frameset->children = g_list_prepend (frameset->children, frameset_child);
  
  gtk_widget_set_parent (child, GTK_WIDGET (frameset));
  
  if (GTK_WIDGET_REALIZED (child->parent))
    gtk_widget_realize (child);

  if (GTK_WIDGET_VISIBLE (child->parent) && GTK_WIDGET_VISIBLE (child))
    {
      if (GTK_WIDGET_MAPPED (child->parent))
	gtk_widget_map (child);
      
      gtk_widget_queue_resize (child);
    }
}

/*
 * Standard GTK function
 *
 * add a widget (frame) to the frameset. The widget
 * will be put in the current_frame (which will
 * be increased in the process)
 */
static void
gtk_frameset_add (GtkContainer *frameset,
		  GtkWidget    *widget)
{
  guint row, col;

  /* is there space left in the frameset for this frame? */
  if (GTK_FRAMESET(frameset)->current_frame >= 
      (GTK_FRAMESET(frameset)->nrows * GTK_FRAMESET(frameset)->ncols)) {
    DEBUG_MSG(DEBUG_EVENT, "No space in frameset for frame\n");
    return;
  }
  
  /* calculate row and column for frame */
  col = GTK_FRAMESET(frameset)->current_frame % GTK_FRAMESET(frameset)->ncols;
  row = GTK_FRAMESET(frameset)->current_frame / GTK_FRAMESET(frameset)->ncols;
  
  DEBUG_MSG(DEBUG_EVENT, "NEW FRAME in frameset %d\n", (gint) frameset);
  DEBUG_MSG(DEBUG_EVENT, "      ROW %d COL %d\n", row, col);
  
  gtk_frameset_attach(GTK_FRAMESET(frameset), /* the frameset widget */
		      widget,                 /* the child widget */
		      row,                    /* row to attach to */
		      col,                    /* column to attach to */
		      0,                      /* marginwidth */
		      0,                      /* marginheight */
		      FALSE,                  /* border */
		      FALSE);                 /* noresize */

  (GTK_FRAMESET(frameset)->current_frame)++;
}

/* standard GTK function */
static void
gtk_frameset_remove (GtkContainer *container,
		     GtkWidget    *widget)
{
  GtkFrameset *frameset;
  GtkFramesetChild *child;
  GList *children;
  
  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (container));
  g_return_if_fail (widget != NULL);
  
  frameset = GTK_FRAMESET (container);
  children = frameset->children;
  
  while (children)
    {
      child = children->data;
      children = children->next;
      
      if (child->widget == widget)
	{
	  gboolean was_visible = GTK_WIDGET_VISIBLE (widget);
	  
	  gtk_widget_unparent (widget);
	  
	  frameset->children = g_list_remove (frameset->children, child);
	  g_free (child);
	  
	  if (was_visible && GTK_WIDGET_VISIBLE (container))
	    gtk_widget_queue_resize (GTK_WIDGET (container));
	  break;
	}
    }
}

/* standard GTK function */
static void
gtk_frameset_forall (GtkContainer *container,
		  gboolean	include_internals,
		  GtkCallback	callback,
		  gpointer	callback_data)
{
  GtkFrameset *frameset;
  GtkFramesetChild *child;
  GList *children;
  
  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (container));
  g_return_if_fail (callback != NULL);
  
  frameset = GTK_FRAMESET (container);
  children = frameset->children;
  
  while (children)
    {
      child = children->data;
      children = children->next;
      
      (* callback) (child->widget, callback_data);
    }
}

/* standard GTK function */
/* standard GTK function */
static void
gtk_frameset_finalize (GtkObject *object)
{
  GtkFrameset *frameset;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (object));
  
  frameset = GTK_FRAMESET (object);
  
  g_free (frameset->rows);
  g_free (frameset->cols);
  
  (* GTK_OBJECT_CLASS (parent_class)->finalize) (object);
}

/* standard GTK function */
static void
gtk_frameset_map (GtkWidget *widget)
{
  GtkFrameset *frameset;
  GtkFramesetChild *child;
  GList *children;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (widget));
  
  frameset = GTK_FRAMESET (widget);
  GTK_WIDGET_SET_FLAGS (frameset, GTK_MAPPED);
  
  children = frameset->children;
  while (children)
    {
      child = children->data;
      children = children->next;
      
      if (GTK_WIDGET_VISIBLE (child->widget) &&
	  !GTK_WIDGET_MAPPED (child->widget))
	gtk_widget_map (child->widget);
    }

  gdk_window_show(widget->window);
}

/* standard GTK function */
static void
gtk_frameset_unmap (GtkWidget *widget)
{
  GtkFrameset *frameset;
  GtkFramesetChild *child;
  GList *children;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (widget));
  
  frameset = GTK_FRAMESET (widget);
  GTK_WIDGET_UNSET_FLAGS (frameset, GTK_MAPPED);
  
  children = frameset->children;
  while (children)
    {
      child = children->data;
      children = children->next;
      
      if (GTK_WIDGET_VISIBLE (child->widget) &&
	  GTK_WIDGET_MAPPED (child->widget))
	gtk_widget_unmap (child->widget);
    }

  gdk_window_hide(widget->window);
}

/* standard GTK function */
static void
gtk_frameset_draw (GtkWidget    *widget,
		  GdkRectangle *area)
{
  GtkFrameset *frameset;
  GtkFramesetChild *child;
  GList *children;
  GdkRectangle child_area;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (widget));
  
  if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget))
    {
      frameset = GTK_FRAMESET (widget);
      
      children = frameset->children;
      while (children)
	{
	  child = children->data;
	  children = children->next;
	  
	  if (gtk_widget_intersect (child->widget, area, &child_area))
	    gtk_widget_draw (child->widget, &child_area);
	}
    }
}

/* standard GTK function */
static gint
gtk_frameset_expose (GtkWidget	    *widget,
		     GdkEventExpose *event)
{
  GtkFrameset *frameset;
  GtkFramesetChild *child;
  GList *children;
  GdkEventExpose child_event;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_FRAMESET (widget), FALSE);
  
  if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget))
    {
      frameset = GTK_FRAMESET (widget);
      
      child_event = *event;
      
      children = frameset->children;
      while (children)
	{
	  child = children->data;
	  children = children->next;
	  
	  if (GTK_WIDGET_NO_WINDOW (child->widget) &&
	      gtk_widget_intersect (child->widget, &event->area, &child_event.area))
	    gtk_widget_event (child->widget, (GdkEvent*) &child_event);
	}
    }
  
  return FALSE;
}

/* standard GTK function */
static void
gtk_frameset_size_allocate (GtkWidget     *widget,
			    GtkAllocation *allocation)
{
  GtkFrameset *frameset;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_FRAMESET (widget));
  g_return_if_fail (allocation != NULL);

  DEBUG_MSG(DEBUG_EVENT, "gtk_frameset_size_allocate(%d, {w:%d, h:%d, x:%d, y:%d})\n",
	    (gint) widget, allocation->width, allocation->height, allocation->x, allocation->y);
  
  widget->allocation = *allocation;
  frameset = GTK_FRAMESET (widget);

  if (GTK_WIDGET_REALIZED (widget))
    gdk_window_move_resize (widget->window,
			    allocation->x, allocation->y,
			    allocation->width, allocation->height);
  
  gtk_frameset_size_allocate_init (frameset);
  gtk_frameset_size_allocate_pass1 (frameset);
}

/*
 * Calculate child dimensions relative to those of frameset,
 * based on row and column length specifications
 */
void
gtk_frameset_size_allocate_init(GtkFrameset *frameset)
{
  gint n_rows, n_cols;
  gfloat per_percent;
  gint real_width, real_height;
  gint row_borderspace, col_borderspace;
  guint location;
  
  /* calculate total amount of border space used */
  row_borderspace = 0;
  col_borderspace = 0;

  for(n_rows = 0; n_rows < frameset->nrows - 1; n_rows++)
    row_borderspace += (frameset->rows[n_rows].border ? GTKFRAMESET_DEFAULT_BORDER_SIZE : 0);
  for(n_cols = 0; n_cols < frameset->ncols - 1; n_cols++)
    col_borderspace += (frameset->cols[n_cols].border ? GTKFRAMESET_DEFAULT_BORDER_SIZE : 0);

  real_width =
    (col_borderspace < GTK_WIDGET(frameset)->allocation.width ?
     GTK_WIDGET(frameset)->allocation.width - col_borderspace : 0);
  real_height =
    (row_borderspace < GTK_WIDGET(frameset)->allocation.height ?
     GTK_WIDGET(frameset)->allocation.height - row_borderspace : 0);
  
  location = 0;
  if(frameset->nrows > 1) {
    /* calculate number of pixels to dole out per percent */
    per_percent = (gfloat) (real_height > frameset->row_total_absolute ?
			    ((real_height - frameset->row_total_absolute) / 100.0) : 0);
    for(n_rows = 0; n_rows < frameset->nrows; n_rows++) {
      if(LENGTH_IS_PERCENTAGE(frameset->rows[n_rows].length))
	frameset->rows[n_rows].allocation =
	  (gint) (per_percent * 100 * LENGTH_GET_PERCENTAGE(frameset->rows[n_rows].length));
      else if(LENGTH_IS_ABSOLUTE(frameset->rows[n_rows].length))
	frameset->rows[n_rows].allocation =
	  (gint) LENGTH_GET_ABSOLUTE(frameset->rows[n_rows].length);
      
      frameset->rows[n_rows].location = location;
      location += frameset->rows[n_rows].allocation +
	(frameset->rows[n_rows].border ? GTKFRAMESET_DEFAULT_BORDER_SIZE : 0);
    }
  } else {
    frameset->rows[0].allocation = real_height;
    frameset->rows[0].location = 0;
  }
  
  location = 0;
  if(frameset->ncols > 1) {
    /* calculate number of pixels to dole out per percent */
    per_percent = (gfloat) (real_width > frameset->col_total_absolute ?
			    ((real_width - frameset->col_total_absolute) / 100.0) : 0);
    for(n_cols = 0; n_cols < frameset->ncols; n_cols++) {
      if(LENGTH_IS_PERCENTAGE(frameset->cols[n_cols].length))
	frameset->cols[n_cols].allocation =
	  (gint) (per_percent * 100 * LENGTH_GET_PERCENTAGE(frameset->cols[n_cols].length));
      else if(LENGTH_IS_ABSOLUTE(frameset->cols[n_cols].length))
	frameset->cols[n_cols].allocation =
	  (gint) LENGTH_GET_ABSOLUTE(frameset->cols[n_cols].length);
      
      frameset->cols[n_cols].location = location;
      location += frameset->cols[n_cols].allocation +
	(frameset->cols[n_cols].border ? GTKFRAMESET_DEFAULT_BORDER_SIZE : 0);
    }
  } else {
    frameset->cols[0].allocation = real_width;
    frameset->cols[0].location = 0;
  }
}

/*
 * allocate children according to their placement in frameset
 */
static void
gtk_frameset_size_allocate_pass1(GtkFrameset *frameset)
{
  GtkFramesetChild *child;
  GList *children;
  GtkAllocation *alloc;

  alloc = g_new0(GtkAllocation, 1);

  children = frameset->children;
  while(children)
    {
      child = children->data;
      children = children->next;
      
      if(GTK_WIDGET_VISIBLE(child->widget)) {
	alloc->width = (frameset->cols[child->col_attach].allocation < (2 * child->xpadding) ?
			0 : frameset->cols[child->col_attach].allocation - (2 * child->xpadding));
	
	alloc->height = (frameset->rows[child->row_attach].allocation < (2 * child->ypadding) ?
			 0 : frameset->rows[child->row_attach].allocation - (2 * child->ypadding));
	alloc->x = frameset->cols[child->col_attach].location + child->xpadding;
	alloc->y = frameset->rows[child->row_attach].location + child->ypadding;
	gtk_widget_size_allocate(GTK_WIDGET(child->widget), alloc);

	DEBUG_MSG(DEBUG_ALLOC, "widget %d (row %d, col %d) allocated (w:%d, h:%d, x:%d, y:%d)\n",
		  (gint) child->widget, child->row_attach, child->col_attach,
		  alloc->width, alloc->height,
		  frameset->cols[child->col_attach].location,
		  frameset->rows[child->row_attach].location);
      }
    }
  g_free(alloc);
}



/*
 * Parse a comma-separated list of %MultiLengths, and returns a GSList
 * of lenghts. The caller has to free the GSList.
 */
static GSList*
gtk_frameset_get_multi_length (const gchar *attr)
{
  GSList *list;
  gdouble value;
  gchar *end;
  Length length;

  g_return_val_if_fail(attr != NULL, NULL);

  list = NULL;

  while(TRUE) {
    value = g_strtod (attr, &end);
    switch (*end) {
    case '%':
      length = LENGTH_CREATE_PERCENTAGE (value / 100);
      break;
      
    case '*':
      length = LENGTH_CREATE_RELATIVE (value);
      break;
      
    default:
      length = LENGTH_CREATE_ABSOLUTE ((gint) value);
      break;
    }

    list = g_slist_append(list, GINT_TO_POINTER(length));
 
    /* there MUST be a comma between values */
    if(!(end = strchr(end, ',')))
      break;
    /* valid %MultiLength characters: 0123456789%* */
    if(!(attr = strpbrk(end, "0123456789%*")))
      break;
  }
 
  return list;
}

/*
 * given the current location and allocation values, calculate
 * row/col length values. The results are used in size_allocate
 */
static void
gtk_frameset_calculate_lengths(GtkFrameset *frameset)
{
  gint n;
  guint total_percentage;

  total_percentage = 0;
  frameset->col_total_absolute = 0;
  for(n = 0; n < frameset->ncols; n++)
    if(LENGTH_IS_ABSOLUTE(frameset->cols[n].length)) {
      frameset->cols[n].length = LENGTH_CREATE_ABSOLUTE(frameset->cols[n].allocation);
      frameset->col_total_absolute += frameset->cols[n].allocation;
    } else
      total_percentage += frameset->cols[n].allocation;

  for(n = 0; n < frameset->ncols; n++) {
    if(!(LENGTH_IS_ABSOLUTE(frameset->cols[n].length)))
      frameset->cols[n].length =
	LENGTH_CREATE_PERCENTAGE(((gfloat) frameset->cols[n].allocation) / total_percentage);
  }

  total_percentage = 0;
  frameset->row_total_absolute = 0;
  for(n = 0; n < frameset->nrows; n++)
    if(LENGTH_IS_ABSOLUTE(frameset->rows[n].length)) {
      frameset->rows[n].length = LENGTH_CREATE_ABSOLUTE(frameset->rows[n].allocation);
      frameset->row_total_absolute += frameset->rows[n].allocation;
    } else
      total_percentage += frameset->rows[n].allocation;

  for(n = 0; n < frameset->nrows; n++) {
    if(!(LENGTH_IS_ABSOLUTE(frameset->rows[n].length)))
      frameset->rows[n].length =
	LENGTH_CREATE_PERCENTAGE(((gfloat) frameset->rows[n].allocation) / total_percentage);
  }
}

/*
 * set the resize_row and resize_col attributes and set resize cursor (if any)
 */
static gint
gtk_frameset_set_resize_rowcol(GtkWidget *widget, guint x, guint y)
{
  GtkFrameset *frameset = GTK_FRAMESET(widget);
  guint n;

  g_return_val_if_fail (widget != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_FRAMESET (widget),FALSE);

  frameset->resize_row = RESIZE_NONE;
  frameset->resize_col = RESIZE_NONE;

  for(n=1; n < frameset->ncols; n++) {
    if(frameset->cols[n - 1].allocation + frameset->cols[n - 1].location - GTKFRAMESET_DEFAULT_BORDER_SIZE <= x &&
       frameset->cols[n].location + GTKFRAMESET_DEFAULT_BORDER_SIZE >= x &&
       !frameset->cols[n].noresize) {
      frameset->resize_col = n - 1;
      break;
    }
  }

  for(n=1; n < frameset->nrows; n++) {
    if(frameset->rows[n - 1].allocation + frameset->rows[n - 1].location - GTKFRAMESET_DEFAULT_BORDER_SIZE <= y &&
       frameset->rows[n].location + GTKFRAMESET_DEFAULT_BORDER_SIZE >= y &&
       !frameset->rows[n].noresize) {
      frameset->resize_row = n - 1;
      break;
    }
  }

  if((frameset->resize_row != RESIZE_NONE) && (frameset->resize_col != RESIZE_NONE))
    gdk_window_set_cursor(widget->window, frameset->cursor_rowcol);
  else if(frameset->resize_row != RESIZE_NONE)
    gdk_window_set_cursor(widget->window, frameset->cursor_row);
  else if(frameset->resize_col != RESIZE_NONE)
    gdk_window_set_cursor(widget->window, frameset->cursor_col);
  else
    gdk_window_set_cursor(widget->window, NULL);

  return TRUE;
}

/*
 * standard GTK function
 *
 * enter drag mode on button_1_press over frame border
 */
static gint
gtk_frameset_button_press (GtkWidget *widget, GdkEventButton *event)
{
  GtkFrameset *frameset = GTK_FRAMESET(widget);

  g_return_val_if_fail (widget != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_FRAMESET (widget),FALSE);

  if(!frameset->in_drag &&
     (event->button == 1) &&
     (frameset->resize_row != RESIZE_NONE || frameset->resize_col != RESIZE_NONE))
    {
      frameset->in_drag = TRUE;
      gdk_pointer_grab (widget->window, FALSE,
			GDK_POINTER_MOTION_HINT_MASK 
			| GDK_BUTTON1_MOTION_MASK 
			| GDK_BUTTON_RELEASE_MASK,
			NULL, NULL, event->time);
    }
  
  return TRUE;
}

/*
 * standard GTK function
 */
static gint
gtk_frameset_button_release (GtkWidget *widget, GdkEventButton *event)
{
  GtkFrameset *frameset = GTK_FRAMESET(widget);

  g_return_val_if_fail (widget != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_FRAMESET (widget),FALSE);

  if(frameset->in_drag && (event->button == 1))
    {
      frameset->in_drag = FALSE;
      gdk_pointer_ungrab (event->time);
      gtk_widget_queue_resize(GTK_WIDGET(frameset));
    }

  return TRUE;
}

/*
 * standard GTK function
 *
 * determine row/col to resize (from x, y and noresize attribute),
 * set appropriate cursor (row resize, col resize or both)
 */
static gint
gtk_frameset_enter (GtkWidget *widget, GdkEventCrossing *event)
{
  g_return_val_if_fail (widget != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_FRAMESET (widget),FALSE);

  return gtk_frameset_set_resize_rowcol(widget, event->x, event->y);
}

/*
 * standard GTK function
 *
 * unset resize cursor and resize_row/resize_col
 */
static gint
gtk_frameset_leave (GtkWidget *widget, GdkEventCrossing *event)
{
  GtkFrameset *frameset = GTK_FRAMESET(widget);

  g_return_val_if_fail (widget != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_FRAMESET (widget),FALSE);

  frameset->resize_row = RESIZE_NONE;
  frameset->resize_col = RESIZE_NONE;
  gdk_window_set_cursor(widget->window, NULL);

  return TRUE;
}

/*
 * standard GTK function
 *
 * resize row and/or column when in_drag is true
 *  otherwise
 * if resize_row or resize_col is set
 * set resize_row/resize_col and cursor
 *
 * This way, motion events without prior enter events will not set the
 * cursor
 */
static gint
gtk_frameset_motion (GtkWidget *widget, GdkEventMotion *event)
{
  GtkFrameset *frameset = GTK_FRAMESET(widget);
  guint x, y;
  gint diff;
  gboolean return_val;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_FRAMESET (widget), FALSE);

  if (event->is_hint)
    gtk_widget_get_pointer(widget, &x, &y);
  else {
    x = event->x;
    y = event->y;
  }
  
  if (frameset->in_drag)
    {
      if(frameset->resize_row >= 0) {
	diff = (y - frameset->rows[frameset->resize_row].location) -
	  frameset->rows[frameset->resize_row].allocation;
	if(frameset->rows[frameset->resize_row].allocation + diff > 0 &&
	   frameset->rows[frameset->resize_row + 1].allocation - diff > 0) {
	  frameset->rows[frameset->resize_row].allocation += diff;
	  frameset->rows[frameset->resize_row + 1].location += diff;
	  frameset->rows[frameset->resize_row + 1].allocation -= diff;
	}
      }

      if(frameset->resize_col >= 0) {
	diff = (x - frameset->cols[frameset->resize_col].location) -
	  frameset->cols[frameset->resize_col].allocation;
	if(frameset->cols[frameset->resize_col].allocation + diff > 0 &&
	   frameset->cols[frameset->resize_col + 1].allocation - diff > 0) {
	  frameset->cols[frameset->resize_col].allocation += diff;
	  frameset->cols[frameset->resize_col + 1].location += diff;
	  frameset->cols[frameset->resize_col + 1].allocation -= diff;
	}
      }

      gtk_frameset_calculate_lengths(frameset);
      gtk_widget_queue_resize(GTK_WIDGET(frameset));
      return_val = TRUE;
    } else
      /* only adjust resize_row/resize_col if either of these is already set,
       * this way the cursor will not be changed by motion events without prior
       * enter event */
      if(frameset->resize_row != RESIZE_NONE ||
	 frameset->resize_col != RESIZE_NONE)
         return_val = gtk_frameset_set_resize_rowcol(widget, x, y);
      else
	return_val = TRUE;

  return return_val;
}

/* this endif is Dillo-specific... */
#endif /* XHTML_DTD_FRAMESET */
