/*
 * Dillo
 *
 * Some code copied from:
 * The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "dw_gtk_scrolled_window.h"
#include "browser.h"
#include "interface.h"
#include "dw_page.h"         /* for a_Dw_page_add_anchor */
#include "encoding.h"
#include "misc.h"

/*
 * Functions to manipulate Pagemarks menu.
 */

/*
 * make the page scroll to the pagemark
 */
static void Pagemark_goto_pagemark(GtkWidget *widget,
                                         gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *) client_data;
   char anchor[32];

   g_snprintf (anchor, 32, "#%ld", (long int)widget);
   a_Dw_gtk_scrolled_window_set_anchor(GTK_DW_SCROLLED_WINDOW(dd->docwin),
                                       anchor);
}

/*
 * Deallocate memory used by a pagemark menu
 */
void a_Pagemark_destroy(DilloDoc *dd)
{
  /* set pagemarks menu insensitive if this is the current tab */
  if(dd->bw->dd == dd)
    gtk_widget_set_sensitive(dd->bw->pagemarks_menuitem, FALSE);
  if (GTK_IS_WIDGET(dd->pagemarks_menu))
    {
      gtk_object_unref(GTK_OBJECT(dd->pagemarks_menu));
      dd->pagemarks_menu = NULL;
      dd->pagemarks_last = NULL;
    }
}

/*
 * Deallocate the memory used by a pagemarks menu and create a new one
 */
void a_Pagemark_new(DilloDoc *dd)
{
   a_Pagemark_destroy(dd);
   dd->pagemarks_menu = gtk_menu_new();
   /* add a reference to the object, otherwise it will be destroyed
    * when the document (tab) is switched */
   gtk_object_ref(GTK_OBJECT(dd->pagemarks_menu));
   dd->pagemarks_last = NULL;

   /* add to bw's menu only if this is the current tab */
   if(dd->bw->dd == dd)
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(dd->bw->pagemarks_menuitem),
                               dd->pagemarks_menu);

   /* todo: add a scroller so one can access  all menu entries,
    * even if there's a lot of pagemarks. */
   /* --EG: I haven't found a way to pass a scroller
    * as widget for the submenu */
}

/*
 * Add a new pagemark (opening H tag).
 * The text can be set only at the closing H tag.
 * level is the level of the heading (1-6).
 */
void a_Pagemark_add(DilloDoc *dd, void *page, void *style,
                          gint level)
{
   char anchor[32], spaces[32], name[32];

   /* set pagemarks menu sensitive if this is the current tab */
   if(dd->bw->dd == dd)
     gtk_widget_set_sensitive(dd->bw->pagemarks_menuitem, TRUE);
   g_snprintf(spaces, 32, "%*s", 3 * (level - 1), "");
   dd->pagemarks_last = gtk_menu_item_new_with_label(spaces);
   gtk_menu_append(GTK_MENU(dd->pagemarks_menu),dd->pagemarks_last);
   gtk_widget_show(dd->pagemarks_last);
   gtk_signal_connect(GTK_OBJECT(dd->pagemarks_last), "activate",
                      (GtkSignalFunc) Pagemark_goto_pagemark, dd);
   /* was:
    * Menu_add(dd->pagemarks_menu, spaces, NULL, dd,
    *          Pagemark_goto_pagemark, dd);
    * but Menu_add is a private function for menu.c so... */
   g_snprintf(anchor, 32, "#%ld", (glong)(dd->pagemarks_last));
   a_Dw_page_add_anchor(page, anchor, style);
   gtk_signal_connect (GTK_OBJECT (dd->pagemarks_last), "select",
                       GTK_SIGNAL_FUNC (a_Interface_scroll_popup), NULL);
   g_snprintf(name, 32, "dilloHeading%d", level);
   gtk_widget_set_name(dd->pagemarks_last, name);
}

/*
 * Set the text for the last created pagemark.
 */
void a_Pagemark_set_text(DilloDoc *dd, const char *str)
{
   GString *text;
   GtkWidget *child;
   char *label_text;
   char *enc_buf = a_Encoding_Convert(DILLO_CHARSET, DW_CHARSET,
	   str, strlen(str));

   /* Avoid problems with lonely closing tags and nested headings */
   if ( dd->pagemarks_last &&
       (child = GTK_BIN (dd->pagemarks_last)->child) &&
       GTK_IS_LABEL (child) ) {
      text = g_string_new(GTK_LABEL(child)->label);
      g_string_append(text, enc_buf);
      if ( text->len > 64 ) {
         g_string_truncate(text, 64);
         g_string_append(text, "...");
      }
	  label_text = a_Misc_string_check(text->str);
      gtk_label_set_text(GTK_LABEL (child), label_text);
	  g_free(label_text);
      g_string_free(text, 1);
      dd->pagemarks_last = NULL;
   }
   g_free(enc_buf);
}
