/*
 * File: capi.c
 *
 * Copyright 2002 Jorge Arellano Cid <jcid@inf.utfsm.cl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/*
 * Cache API
 * This is the module that manages the cache and starts the CCC chains
 * to get the requests served. Kind of a broker.
 */

#include "IO/Url.h"
#include "web.h"
#include "chain.h"
#include "list.h"
#include "interface.h"
#include "history.h"
#include "nav.h"

/* for testing dpi chat */
#include "bookmark.h"

#define DEBUG_LEVEL 5
#include "debug.h"

typedef struct {
   DilloDoc *dd;
   gchar *server;
   gint SockFD;
   gint Flags;
   gint DpiPipe[2];
   ChainLink *InfoSend;
   ChainLink *InfoRecv;
   ChainLink *InfoPipe;

   gint Ref;
} dpi_conn_t;


/*
 * Local data
 */
/* Data list for active dpi connections */
static dpi_conn_t **DpiConn = NULL;
static gint DpiConnSize;
static gint DpiConnMax = 4;


/*
 * Forward declarations
 */
void a_Capi_ccc(int Op, int Branch, int Dir, ChainLink *Info,
                void *Data1, void *Data2);
gint a_Capi_dpi_send_cmd(DilloDoc *dd, char *cmd, char *server, gint flags);


/* ------------------------------------------------------------------------- */

/*
 * Create a new connection data structure
 */
static dpi_conn_t *Capi_dpi_conn_new(DilloDoc *dd, char *server)
{
   dpi_conn_t *conn;

   conn = g_new(dpi_conn_t, 1);
   conn->dd = dd;
   conn->server = server;
   conn->SockFD = -1;
   conn->Flags = 0;         /* not used yet */
   conn->InfoSend = a_Chain_new();
   conn->InfoRecv = NULL;   /* will be set later */
   conn->InfoPipe = NULL;   /* may be set later */
   conn->Ref = 0;           /* Reference count */
   return conn;
}

/*
 * Increment the reference count and add to the list if not present
 */
static void Capi_dpi_conn_ref(dpi_conn_t *conn)
{
   if (++conn->Ref == 1) {
      /* add the connection data to list */
      a_List_add(DpiConn, DpiConnSize, DpiConnMax);
      DpiConn[DpiConnSize] = conn;
      DpiConnSize++;
   }
}

/*
 * Decrement the reference count (and remove from list when zero)
 */
static void Capi_dpi_conn_unref(dpi_conn_t *conn)
{
   gint i;

   --conn->Ref;
   if (conn->Ref == 0) {
      for (i = 0; i < DpiConnSize; ++i)
         if (DpiConn[i] == conn) {
            a_List_remove(DpiConn, i, DpiConnSize);
            g_free(conn);
            break;
         }
   }
}

/*
 * Find connection data by server
 */
static dpi_conn_t *Capi_dpi_conn_find(gchar *server)
{
   gint i;

   for (i = 0; i < DpiConnSize; ++i)
      if (strcmp(server, DpiConn[i]->server) == 0)
         return DpiConn[i];

   return NULL;
}

/* ------------------------------------------------------------------------- */

/*
 * Safety test: only allow dpi-urls from dpi-generated pages.
 */
gint Capi_verify_dpi_url_request(DilloWeb *web)
{
   DilloUrl *referer = NULL;
   gint allow = FALSE;

   /* test POST and GET */
   if (strchr(URL_STR(web->url), '?') || URL_DATA_(web->url)) {
      /* safety measure: only allow dpi requests from dpi-generated urls */
      if (a_Nav_stack_size(web->dd)) {
         referer = a_History_get_url(NAV_TOP(web->dd));
         if (g_strncasecmp(URL_STR(referer), "dpi:/", 5) == 0)
            allow = TRUE;
      }
   } else {
      allow = TRUE;
   }

   if (!allow) {
      g_print("Capi_verify_dpi_url_request: Permission Denied!\n");
      g_print("  URL_STR : %s\n", URL_STR(web->url));
      if (URL_DATA_(web->url))
         g_print("  URL_DATA: %s\n", URL_DATA(web->url));
   }
   return allow;
}

/*
 * Most used function.
 */
gint a_Capi_open_url(DilloWeb *web, CA_Callback_t Call, void *CbData)
{
   char *cmd;
   char *url_str = URL_STR(web->url);

   if (g_strncasecmp(url_str, "dpi:/", 5) == 0 && !Call) {
      /* dpi request! */
      if (Capi_verify_dpi_url_request(web)) {
         cmd = g_strdup_printf("<dpi cmd='open_url' url='%s'>", url_str);
         a_Capi_dpi_send_cmd(web->dd, cmd, "bm", 1);
         g_free(cmd);
      }
   } else {
      return a_Cache_open_url(web, Call, CbData);
   }

   return 0;
}

/*
 * This one is used by DPI (the interface is not well defined yet).
 */
static gint Capi_open_url_raw(DilloDoc *dd, gchar *url_str)
{
   DilloWeb *Web;
   DilloUrl *Url;

   /* todo: URL_E2EReload _always_ forces end to end reload */
   Url = a_Url_new(url_str, NULL, URL_E2EReload, 0);
   Web = a_Web_new(Url);
   Web->dd = dd;
   Web->flags |= WEB_RootUrl;
   a_Cache_open_url(Web, NULL, NULL);
   a_Url_free(Url);

   /* todo: return a meaningful value */
   return 0;
}

/*
 * Retrieve a char* to the cache contents for URL.
 */
char *a_Capi_url_read(const DilloUrl *Url, gint *Size)
{
   return a_Cache_url_read(Url, Size);
}

/*
 * Send a non-URI dpi cmd.
 * (For instance: add_bookmark, send_preferences, get_status_tag)
 */
gint a_Capi_dpi_send_cmd(DilloDoc *dd, char *cmd, char *server, gint flags)
{
   dpi_conn_t *conn;
   DataBuf *dbuf;

   //**/sleep(1);
   //**/a_Interface_msg(dd->bw, cmd);

   if (flags & 1) {
      /* open a new connection to server */

      /* Create a new connection data struct and add it to the list */
      conn = Capi_dpi_conn_new(dd, server);
      Capi_dpi_conn_ref(conn);

      /* start the CCC operations */
      dbuf = a_Chain_dbuf_new(cmd, strlen(cmd), 0);
      a_Capi_ccc(OpStart, 1, BCK, conn->InfoSend, conn, NULL);
      a_Capi_ccc(OpSend, 1, BCK, conn->InfoSend, dbuf, NULL);
      g_free(dbuf);

   } else {
      /* Re-use an open connection */
      conn = Capi_dpi_conn_find(server);
      if (conn) {
         /* found */
         dbuf = a_Chain_dbuf_new(cmd, strlen(cmd), 0);
         a_Capi_ccc(OpSend, 1, BCK, conn->InfoSend, dbuf, NULL);
         g_free(dbuf);
      } else {
         g_print(" ERROR: [a_Capi_dpi_send_cmd] No open connection found\n");
      }
   }

   return 0;
}


/*
 * CCC function for the CAPI module
 */
void a_Capi_ccc(int Op, int Branch, int Dir, ChainLink *Info,
                void *Data1, void *Data2)
{
   dpi_conn_t *conn;

   a_Chain_debug_msg("a_Capi_ccc", Op, Branch, Dir);

   if (Branch == 1) {
      if (Dir == BCK) {
         /* Command sending branch */
         switch (Op) {
         case OpStart:
            Info->LocalKey = Data1;
            a_Chain_link_new(Info, a_Capi_ccc, BCK, a_Dpi_ccc, 2, 1);
            a_Chain_bcb(OpStart, Info, Data1, Data2); /* Data1, Data2 ?? */
            break;
         case OpSend:
            a_Chain_bcb(OpSend, Info, Data1, NULL);
            break;
         case OpEnd:
            a_Chain_bcb(OpEnd, Info, NULL, NULL);
            Capi_dpi_conn_unref(Info->LocalKey);
            g_free(Info);
            break;
         case OpStop:
         case OpAbort:
            g_print(" Not implemented\n");
            break;
         }
      } else {  /* FWD */
         /* Command sending branch (status) */
         switch (Op) {
         case OpSend:
            if (Data2 && strcmp(Data2, "SockFD") == 0) {
               dpi_conn_t *conn = Info->LocalKey;
               conn->SockFD = (gint)Data1;
               a_Capi_ccc(OpStart, 2, BCK, a_Chain_new(),Info->LocalKey, NULL);
            }
            break;
         case OpStop:
         case OpAbort:
            g_print(" Not implemented\n");
            break;
         }
      }

   } else if (Branch == 2) {
      if (Dir == BCK) {
         /* Server listening branch (status) */
         switch (Op) {
         case OpStart:
            {
            dpi_conn_t *conn = Data1;
            Info->LocalKey = Data1;
            Capi_dpi_conn_ref(conn);
            conn->InfoRecv = Info;
            a_Chain_link_new(Info, a_Capi_ccc, BCK, a_Dpi_ccc, 3, 2);
            a_Chain_bcb(OpStart, Info, (void *)conn->SockFD, "SockFD");
            break;
            }
         case OpStop:
         case OpAbort:
            Capi_dpi_conn_unref(Info->LocalKey);
            g_print(" Not implemented\n");
            break;
         }
      } else {  /* FWD */
         /* Server listening branch */
         switch (Op) {
         case OpSend:
            conn = Info->LocalKey;
            if (strcmp(Data2, "send_status_message") == 0) {
               a_Interface_msg(conn->dd->bw, Data1);
            } else if (strcmp(Data2, "chat") == 0) {
               a_Interface_msg(conn->dd->bw, Data1);
               a_Bookmarks_chat_add(NULL, NULL, Data1);
            } else if (strcmp(Data2, "start_send_page") == 0) {
               /* start the pipe-writing chain */
               a_Capi_ccc(OpStart, 3, BCK, a_Chain_new(), Info->LocalKey,NULL);
               /* let the dpi know the reading end of the pipe */
               a_Chain_bcb(OpSend, Info, (void *)conn->DpiPipe[0], NULL);
               /* start the pipe-reading chain */
               Capi_open_url_raw(conn->dd, Data1);
            } else if (strcmp(Data2, "send_page_2eof") == 0) {
               a_Capi_ccc(OpSend, 3, BCK, conn->InfoPipe, Data1, NULL);
            } else if (strcmp(Data2, "reload_request") == 0) {
               a_Nav_reload(conn->dd);
            }
            break;
         case OpEnd:
            {
            dpi_conn_t *conn = Info->LocalKey;
            a_Chain_del_link(Info, BCK);
            conn->InfoRecv = NULL;
            Capi_dpi_conn_unref(conn);
            if (conn->InfoSend) {
               /* Propagate OpEnd to the sending branch too */
               a_Capi_ccc(OpEnd, 1, BCK, conn->InfoSend, NULL, NULL);
            }
            if (conn->InfoPipe) {
               /* Propagate OpEnd to the pipe branch too */
               a_Capi_ccc(OpEnd, 3, BCK, conn->InfoPipe, NULL, NULL);
            }
            g_free(Info);
            break;
            }
         case OpStop:
         case OpAbort:
            g_print(" Not implemented\n");
            break;
         }
      }

   } else if (Branch == 3) {
      if (Dir == BCK) {
         /* Pipe writing branch */
         switch (Op) {
         case OpStart:
            {
            dpi_conn_t *conn = Data1;
            Info->LocalKey = Data1;
            Capi_dpi_conn_ref(conn);
            conn->InfoPipe = Info;
            if (pipe(conn->DpiPipe)) {
               g_print(" Error with pipe\n");
               return;
            }
            a_Chain_link_new(Info, a_Capi_ccc, BCK, a_IO_ccc, 3, 3);
            a_Chain_bcb(OpStart, Info, (void *)conn->DpiPipe[1], "SockFD");
            break;
            }
         case OpSend:
            a_Chain_bcb(OpSend, Info, Data1, NULL);
            break;
         case OpEnd:
            a_Chain_bcb(OpEnd, Info, NULL, NULL);
            Capi_dpi_conn_unref(Info->LocalKey);
            g_free(Info);
            break;
         case OpStop:
         case OpAbort:
            Capi_dpi_conn_unref(Info->LocalKey);
            g_print(" Not implemented\n");
            break;
         }
      } else {  /* FWD */
         /* Pipe branch (status) */
      }
   }
}
