#ifndef __UtilBezier_h
#define __UtilBezier_h

#include "UtilMisc.h"	// For FFCalcFrame.CalcPowDistance.
#include "UtilCoord.h"	// For mm2lg.
#include "PolyJoint.h"	// For FCPolyJoint.

//;----- <EXPORT TO SDK> -----;

/*************************************************************************
 * <_xZkqqHx_iNfMjBkCjBmM_xHqqkZx_>	FCBezierControls
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_iDhIiDfHiDeGiLmIjApMiCmMiCfDiCmCiCmMjAkHiMoEjDfPiBgJiCgAiBeBiCgBiBeBiCgCiBeBiCgDiBgKiCpAiLeMjIfOiClHiCoJiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	05.04.01 Fukushiro M. _xZkqqHx_iNoMjAkM_xHqqkZx_
 *************************************************************************/
struct FCBezierControls
{
	FCBezierControls () {}
	FCBezierControls (	const DBLPoint& mmptA,
						const DBLPoint& mmptB,
						const DBLPoint& mmptC,
						const DBLPoint& mmptD)
		{
			m_mmptControl[0] = mmptA;
			m_mmptControl[1] = mmptB;
			m_mmptControl[2] = mmptC;
			m_mmptControl[3] = mmptD;
		}
	DBLPoint& A () { return m_mmptControl[0]; }
	DBLPoint& B () { return m_mmptControl[1]; }
	DBLPoint& C () { return m_mmptControl[2]; }
	DBLPoint& D () { return m_mmptControl[3]; }

	const DBLPoint& A () const { return m_mmptControl[0]; }
	const DBLPoint& B () const { return m_mmptControl[1]; }
	const DBLPoint& C () const { return m_mmptControl[2]; }
	const DBLPoint& D () const { return m_mmptControl[3]; }

//;----- </EXPORT TO SDK> -----;
	// _xZkqqHx_iImIiJlKiCmN_xHqqkZx_ CalcPowDistance _xZkqqHx_iCkK_xHqqkZx_SDK_xZkqqHx_iCmJiPgPjHmNiClDiCoKiCmIiCkCiClNiCnPjGlDiMpIiBeC_xHqqkZx_
	BOOL IsLine () const
		{
			// A=B,C=D_xZkqqHx_iCmMiPoKiNiHiBeC_xHqqkZx_
			if (almosteq(A(), B()) && almosteq(C(), D()))
				return TRUE;
			// _xZkqqHx_iCfDjDfPiNmAjFfHiCmMjBePiIpIjDfPiBeFiMoDiIpIjDfPiCkKjHlMjCfLiCmMjIeBiMiLjDfPiCmMiKnEiCmJiCkAiCoJiPoKiNiHiBeC_xHqqkZx_
			if (almost0(CalcPowDistance(DBLPoint(), A(), D(), B())) &&
				almost0(CalcPowDistance(DBLPoint(), A(), D(), C())))
				return TRUE;
			return FALSE;
		}

//;----- <EXPORT TO SDK> -----;

	DBLPoint	m_mmptControl[4];
}; // struct FCBezierControls.

typedef vector<FCBezierControls> FCBezierControlsVector;

/*************************************************************************
 * <_xZkqqHx_iDeOiDiJiDfI_xHqqkZx_>	FCPolyBezier
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_iDhMiDiKiDiJiDeDiDjDiBeBiDhIiDfHiDeGiLmIjApMiCmMjHlMjFpLiCpAiIlFiCkEiClNiCnPiCmMjEhKjHpBiDeOiDiJiDfIiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	05.04.04 Fukushiro M. _xZkqqHx_iNoMjAkM_xHqqkZx_
 *************************************************************************/
template<class T> class FCPolyBezier
{
public:
	FCPolyBezier () : m_bIsBezier(FALSE) {}
	FCPolyBezier (BOOL bIsBezier, const T* pPoint, long lSize) : m_bIsBezier(bIsBezier), m_vPoint(pPoint, pPoint + lSize) {}
	void clear ()
		{
			m_bIsBezier = FALSE;
			m_vPoint.clear();
		}
	void push_back (const T& v) { m_vPoint.push_back(v); }
	long size () const { return m_vPoint.size(); }
	void assign (long lCount) { m_vPoint.assign(lCount); }
	T& operator [] (long lC) { return m_vPoint[lC]; }
	const T& operator [] (long lC) const { return m_vPoint[lC]; }
	void erase (T* p) { m_vPoint.erase(p); }
	T& back () { return m_vPoint.back(); }
	const T& back () const { return m_vPoint.back(); }
	T& front () { return m_vPoint.front(); }
	const T& front () const { return m_vPoint.front(); }
	bool empty () const { return m_vPoint.empty(); }

	void SetIsBezier (BOOL bIsBezier) { m_bIsBezier = bIsBezier; }
	BOOL GetIsBezier () const { return m_bIsBezier; }
	void Reverse () { reverse(m_vPoint.begin(), m_vPoint.end()); }
	void Insert (long lPosition, const T* pPoint, long lSize) { m_vPoint.insert(&m_vPoint[lPosition], pPoint, pPoint + lSize); }

	T* GetArray () { return &m_vPoint.front(); }
	const T* GetArray () const { return &m_vPoint.front(); }

protected:
	BOOL		m_bIsBezier;
	vector<T>	m_vPoint;
}; // class FCPolyBezier.

typedef FCPolyBezier<CPoint> FCLgPolyBezier;
typedef FCPolyBezier<DBLPoint> FCMmPolyBezier;

//;----- </EXPORT TO SDK> -----;

inline void mm2lg (FCLgPolyBezier& vlgpt, const DBLPointVector& vmmpt, double dZoom = g_->dZoom)
{
	vlgpt.assign(vmmpt.size());
	for (long lC = 0; lC < vmmpt.size(); lC++)
		vlgpt[lC] = mm2lg(vmmpt[lC], dZoom);
} // mm2lg.

inline void mm2lg (FCLgPolyBezier& vlgpt, FCMmPolyBezier& vmmpt, double dZoom = g_->dZoom)
{
	vlgpt.SetIsBezier(vmmpt.GetIsBezier());
	vlgpt.assign(vmmpt.size());
	for (long lC = 0; lC < vmmpt.size(); lC++)
		vlgpt[lC] = mm2lg(vmmpt[lC], dZoom);
} // mm2lg.

extern void FFControlConv (FCMmPolyBezier& polyBezier, const FCBezierControlsVector& vControls);
extern void FFControlConv (FCBezierControlsVector& vControls, const FCMmPolyBezier& polyBezier);

extern mmetol FFGetWideWidth (int iFromVal, int iUntilVal, double dN, mmetol mmWF, mmetol mmWB);
extern void FFCalcWideWidthArray (	mmetolVector& vmmW,
									DBLPointVector& vmmptPoly,
									BYTEVector& vIsJoint,
									mmetol mmElemLen,
									mmetol mmWF, mmetol mmWB,
									int iFromVal, int iUntilVal);
extern BOOL FFCalcBCFromADLMP2 (DBLPoint& mmptB,
								DBLPoint& mmptC,
								const DBLPoint& mmptA,
								const DBLPoint& mmptD,
								DBLPoint mmptL,
								DBLPoint mmptM,
								const DBLPoint& mmptP1,
								const DBLPoint& mmptP2,
								double dRP1,
								double dRP2,
								double dAnglePlay);

extern BOOL FFCalcUFromVADLMP (	mmetol& mmU,
								double& dT,
								mmetol mmV,
								const DBLPoint& mmptA,
								const DBLPoint& mmptD,
								const DBLPoint& mmptL,
								const DBLPoint& mmptM,
								const DBLPoint& mmptP);

extern BOOL FFCalcVFromUADLMP (	mmetol& mmV,
								double& dT,
								mmetol mmU,
								DBLPoint mmptA,
								DBLPoint mmptD,
								DBLPoint mmptL,
								DBLPoint mmptM,
								DBLPoint mmptP);

extern BOOL FFCalcBFromACDLP (	DBLPoint& mmptB,
								const DBLPoint& mmptA,
								const DBLPoint& mmptC,
								const DBLPoint& mmptD,
								const DBLPoint& mmptL,
								const DBLPoint& mmptP);

extern BOOL FFCalcCFromABDMP (	DBLPoint& mmptC,
								const DBLPoint& mmptA,
								const DBLPoint& mmptB,
								const DBLPoint& mmptD,
								const DBLPoint& mmptM,
								const DBLPoint& mmptP);

extern DBLPoint FFGetBezierVelocity (const FCBezierControls& bc, double dT);

extern BOOL FFChopBezier (	FCBezierControls& bcChopped,
							FCBezierControls bc,
							double dT0,
							double dTn,
							BOOL bIsBeqA = FALSE,
							BOOL bIsCeqD = FALSE);

extern double FFCalcAfterDistanceT (const FCBezierControls& bc, const DBLPoint& mmptQ, double dT0, mmetol mmTargetLen);
extern double FFCalcBeforeDistanceT (const FCBezierControls& bc, double dT0, mmetol mmTargetLen);

extern void FFCalcT (	long& lControlIndex,
						double& dT,
						DBLPoint& mmptQ,
						const FCBezierControlsVector& vControls,
						const DBLPoint& mmptP);

extern void FFCalcBezier (	DBLPointVector& vmmptPoint,
							BYTEVector& vIsJoint,
							const FCMmPolyBezier& polyBezier,
							mmetol mmWeight,
							mmetol mmElemLen,
							const DBLRect* mmrtClip,
							long lClipSize);

extern void FFCalcBezier (	DBLPointVector& vmmptPoint,
							const FCBezierControls& bc,
							mmetol mmWeight,
							mmetol mmElemLen,
							const DBLRect* mmrtClip,
							long lClipSize);

// 06.12.04 Fukushiro M. 1_xZkqqHx_iNhDjCmHiJmB_xHqqkZx_ ()
extern void FFCalcFrame (DBLRect& mmrt, long lCount, const FCBezierControls bcs[]);
// 06.12.03 Fukushiro M. 1_xZkqqHx_iNhDjCmHiJmB_xHqqkZx_ ()
extern mmetol FFGetStdElemLen ();

/*************************************************************************
 * <_xZkqqHx_iKnGjAjE_xHqqkZx_>	FFCalcBezierPoint
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_iDhIiDfHiDeGiLmIjApMiCmMiOjOiNiP_xHqqkZx_t_xZkqqHx_iCmJiCkIiCkPiCoJiImKjChF_xHqqkZx_P_xZkqqHx_iCpAiMhGiOfKiClHiCoJiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iIpIjAjE_xHqqkZx_>	bc	:_xZkqqHx_iDhIiDfHiDeGiCmMjAkHiMoEjDfPiCpAiOhHjCoIiBeC_xHqqkZx_
 *			dT	:_xZkqqHx_iOjOiNiP_xHqqkZx_t_xZkqqHx_iCpAiOhHjCoIiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jFnEjCgM_xHqqkZx_>	_xZkqqHx_iOjOiNiP_xHqqkZx_t_xZkqqHx_iCmJiCkIiCkPiCoJiImKjChF_xHqqkZx_P_xZkqqHx_iBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	05.04.06 Fukushiro M. _xZkqqHx_iNoMjAkM_xHqqkZx_
 *************************************************************************/
inline DBLPoint FFCalcBezierPoint (const FCBezierControls& bc, double dT)
{
	const double dK = 1.0 - dT;
	return bc.A() * POW3(dK) + 3.0 * bc.B() * POW2(dK) * dT + 3.0 * bc.C() * dK * POW2(dT) + bc.D() * POW3(dT);
} // FFCalcBezierPoint.

/*************************************************************************
 * <_xZkqqHx_iKnGjAjE_xHqqkZx_>	FFCalcBezierAngle
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_iDhIiDfHiDeGiLmIjApMiCmMiOjOiNiP_xHqqkZx_t_xZkqqHx_iCmJiCkIiCkPiCoJiKhAjDhIiCpAiMhGiOfKiClHiCoJiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iIpIjAjE_xHqqkZx_>	bc	:_xZkqqHx_iDhIiDfHiDeGiCmMjAkHiMoEjDfPiCpAiOhHjCoIiBeC_xHqqkZx_
 *			dT	:_xZkqqHx_iOjOiNiP_xHqqkZx_t_xZkqqHx_iCpAiOhHjCoIiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jFnEjCgM_xHqqkZx_>	_xZkqqHx_iOjOiNiP_xHqqkZx_t_xZkqqHx_iCmJiCkIiCkPiCoJiKhAjDhIiBeC_xHqqkZx_[-_xZkqqHx_iDmO_xHqqkZx_,_xZkqqHx_iDmO_xHqqkZx_] _xZkqqHx_iKhAjDhIiCkKiMiIiCnMiCoHiCmIiCkCiPoKiNiHiCmNiCePiCpAjFnEiClHiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iJpAjAoA_xHqqkZx_>	P = A * (1-t)^3 + 3 * B * (1-t)^2 * t + 3 * C * (1-t) * t^2 + D * t^3
 *			_xZkqqHx_iClBiCoKiCpA_xHqqkZx_ t _xZkqqHx_iCmJiKnGiClFiCmEjEpHjFkKiClFiCmE_xHqqkZx_
 *			P' = -3*(A - B) + 6*(A - 2*B + C)*t + -3*(A - 3*B + 3*C - D)*t^2
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	05.04.10 Fukushiro M. _xZkqqHx_iNoMjAkM_xHqqkZx_
 *************************************************************************/
inline double FFCalcBezierAngle (const FCBezierControls& bc, double dT)
{
	const DBLPoint mmptL = FFGetBezierVelocity(bc, dT);
	// dT = 0, bc.A = bc.B _xZkqqHx_iCmMiPoKiNiHiCmIiCmHiBeB_xHqqkZx_mmptL = (0,0)_xZkqqHx_iCoAiCkAiCoIjDlOiCoJiBeC_xHqqkZx_
	// atan2_xZkqqHx_iCmNiBeBiIpIjAjEiCkK_xHqqkZx_(0,0)_xZkqqHx_iClOiCmG_xHqqkZx_0_xZkqqHx_iCpAjFnEiClHiBeC_xHqqkZx_
	return atan2(mmptL.y, mmptL.x);
} // FFCalcBezierAngle.

/*************************************************************************
 * <_xZkqqHx_iKnGjAjE_xHqqkZx_>	StdMinRatio
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_jFfHiPiAiCmM_xHqqkZx_CalcBezier2_xZkqqHx_jHhAiPmIjHkKiKoOiPiAjCgM_xHqqkZx_(dMinRatio)_xZkqqHx_iCpAiMhGiOfKiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iJpAjAoA_xHqqkZx_>	lgrt_xZkqqHx_iCmNiBeBjCmKiPoNiCmNjFjEjFgJiCmMjBoFiCkLiClDiCpAiOhHjCoIiBeCiMhGiOfKiOkOiCmNiMgPiMlBjDeJiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	00.07.19 Fukushiro M. _xZkqqHx_iNoMjAkM_xHqqkZx_
 *************************************************************************/
inline double StdMinRatio (const CRect& lgrt)
{
	if (lgrt.Width() == 0 && lgrt.Height() == 0) return 0.0;
	// _xZkqqHx_jBoFiCkLiCkCjFpLiCmMjFnDiCpAiKoOiPiAiCmJiMhGiOfKiClHiCoJiBeC_xHqqkZx_
	if (lgrt.Width() < lgrt.Height())
		return 2.5 / double(lgrt.Height());
	else
		return 2.5 / double(lgrt.Width());
} // StdMinRatio.

//----- 06.10.17 Fukushiro M. _xZkqqHx_jCmHiJmBiOgO_xHqqkZx_ ()-----
// SDK_xZkqqHx_iCmMiDiBiDjDiDgPiBfLjCmHiJmBiCmMiClNiCnPiClBiClBiCmJjChFiCkNiBeC_xHqqkZx_
// _xZkqqHx_iOjOiNiP_xHqqkZx_T_xZkqqHx_iCmJiCkIiCkPiCoJiJmBjBkMjDhIiCpAiMhGiOfKiBeC_xHqqkZx_
extern DBLPoint FFGetBezierAcceleration (const FCBezierControls& bc, double dT);
// _xZkqqHx_iOjOiNiP_xHqqkZx_T_xZkqqHx_iCmJiCkIiCkPiCoJiKhAjDhIiCpAiMhGiOfKiBeC_xHqqkZx_
extern double FFGetBezierAngle (const FCBezierControls& bc, double dT);
//----- 06.10.17 Fukushiro M. _xZkqqHx_jCmHiJmBiPeJ_xHqqkZx_ ()-----

/*************************************************************************
 * <_xZkqqHx_iKnGjAjE_xHqqkZx_>	CalcBezier2
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_iDhIiDfHiDeHiLmIjApMiCpAiNnHiCkJiCkCjClMjApMiCmJjFkKiKiEiClHiCoJiClNiCnPiCmMiMhGiOfKiCpAiClHiCoJiBeC_xHqqkZx_
 *			pcoPoint_xZkqqHx_iCkJiCoHiBeBiNnHiCkJiCkCjApMjFkKiCmMjClIjDfPiCpAiOfKiPgPiClFiBeB_xHqqkZx_lgpPoly_xZkqqHx_iCmJjFnEiClHiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iIpIjAjE_xHqqkZx_>	vlgptPoly	:_xZkqqHx_jClIjDfPiCmMjCgMiCmMjEhKjHpBiCpAjFnEiClHiBeCjIfPjHjNiNmAjFfHiMgOiBeC_xHqqkZx_
 *			pjtpJoint	:_xZkqqHx_jIeBiMiLjDfPiCmMiPfHiNiHiCpAiOhHjCoIiBeCiOmAiNmAjFfHiMgOiBeC_xHqqkZx_
 *			lJointSize	:_xZkqqHx_jIeBiMiLjDfPiCmMjAjEiBeBjBkGiClP_xHqqkZx_pjtpJoint_xZkqqHx_iCmMjEhKjHpBiDfEiDeDiDfJiCpAiOhHjCoIiBeC_xHqqkZx_
 *			mmfWeight	:_xZkqqHx_jApMiCmMjFjNiCpAiOhHjCoIiBeCiOmAiNmAjFfHiMgOiBeCiDiKiBfLiDfHiDiHiDjDiMhGiOfKjHhAiCmIiCmMiCmFiBeBjBlOiCnPiCmJ_xHqqkZx_
 *						 _xZkqqHx_iOhHjCoIiClHiCoJjFkKiCmJiCmNjGoCjBoIjGlDiCkCiBgJiMhGiOfKiOjOiKnEiCkKjBjNiCkGiCoJiPoKiNiHiCmNiCkAiCoJiCkKiBgK_xHqqkZx_
 *			lJointFrom	:_xZkqqHx_iMhGiOfKiKeKiKeKiOgOjDfPiCpAiOhHjCoIiBeC_xHqqkZx_pjtpJoint[lJointFrom]_xZkqqHx_iCpAiOkGiClHiBeC_xHqqkZx_
 *			lJointNum	:_xZkqqHx_iMhGiOfKiKeKiKeKiOgOjDfPiCkJiCoHiPeJjHlJjDfPiCnMiCmFiCmMjApMiCmMjAjEiCpAiOhHjCoIiBeC_xHqqkZx_
 *			lgdElemLen	:tlgpPos_xZkqqHx_iCmFjFnEiClDiCoKiCoJjClIjDfPiKnEiCmMiLjHjHkD_xHqqkZx_._xZkqqHx_jIfPjHjNiNmAjFfHjCgM_xHqqkZx_._xZkqqHx_jGnKiImA_xHqqkZx_.
 *			dZoom		:_xZkqqHx_iMlLiNnNiCmMiDfJiBfLiDiAjCgM_xHqqkZx_
 *			tlgrtaClip	:_xZkqqHx_jFgAiJoGiBgJiMhGiOfKiBgKjHmMiIoGiCmMiPfHiNiH_xHqqkZx_._xZkqqHx_jIfPjHjNiNmAjFfHiMgO_xHqqkZx_.
 *			iClipSize	:tlgrtaClip_xZkqqHx_iCmMiDfEiDeDiDfJ_xHqqkZx_.
 *			dMinRatio	:_xZkqqHx_iPmIjHkKiClHiCoJjDfPiCpAiMiIjCoIiClHiCoJiKoOiPiAjCgM_xHqqkZx_._xZkqqHx_jApMjFkKiCmMjClHiClDiCmJjBmOiClHiCoJ_xHqqkZx_
 *						 _xZkqqHx_jDfPiCmGjApMjFkKiCmMiLjHjHkDiCmMiKiEiNiHiBeCiClBiCmMjCgMiCkKiPkMiClDiCkCjCpG_xHqqkZx_
 *						 _xZkqqHx_iPmIjHkKiClHiCoJjDfPiCpAiMlFjGkHiCmJjElLjCgGiClHiCoJiBeCiCmCiCnMiCoIiPmIjHkKiClDiCoKiCoJ_xHqqkZx_
 *						 _xZkqqHx_jDfPiCkKiPkNiCmIiCkNiCmIiCoJiCmGiCkCiCkEiClBiCmGiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iJpAjAoA_xHqqkZx_>	lgfElemLen_xZkqqHx_iCmFjHfOiCkGiCoHiCoKiClNiLjHjHkDiCpAjGnKiImAiCmJiDhIiDfHiDeHiLmIjApMiCpAiCmHiCoKiCkNiCoHiCkC_xHqqkZx_
 *			_xZkqqHx_iNnHiCkJiCkNiMhGiOfKiClHiCoJiCkJiMiIjCoIiClHiCoJiBeC_xHqqkZx_lgfElemLen_xZkqqHx_iCmJjAnNjCoIiClHiCoJjCgMiCkKiPkMiClDiCkCiCnJiCmH_xHqqkZx_
 *			_xZkqqHx_iNnHiCkJiCkCiLmIjApMiCkKiMhGiOfKiClDiCoKiCoJiCkKiBeBiMhGiOfKiOjOiKnEiCkKjBjNiCkGiBeB_xHqqkZx_tlgpPos_xZkqqHx_iCmMjEhKjHpBiCoA_xHqqkZx_
 *			_xZkqqHx_jBoFiCkLiCkNiCmIiCoJiBeC_xHqqkZx_
 *			tlgrtaClip_xZkqqHx_iCmFjHfOiCkGiCoHiCoKiCoJiDeOiDiKiDgCiDhGjHmMiIoGiCmGiMpAiCoNiCoJiCfDjDfPiNmAjFfHiCmM_xHqqkZx_
 *			_xZkqqHx_iDhIiDfHiDeHiLmIjApMiCmNjAlDiKgNiCmJiMhGiOfKiClDiCoKiCoJiCkKiBeBjHmMiIoGiCmGiMpAiCoNiCoHiCmIiCkCiCfDjDfPiNmAjFfHiCmN_xHqqkZx_
 *			_xZkqqHx_jClMjApMiCmGiClFiCmE_xHqqkZx_tlgpPos_xZkqqHx_iCmJjAnNjCoIiClDiCoKiCoJiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	99.04.27 Fukushiro M. _xZkqqHx_iDhIiDfHiDeHjFkKiKiEjAjEiCpAjCfEiCoJiInHiCmJiBeBjCfEiNlIjHhAiCfDjFkKiKiE_xHqqkZx_
 *								_xZkqqHx_iDhIiDfHiDeHiCmMiNmFiPkMjClHiCmMjFnDiCpAiKoOiPiAiCmJjFkKiKiEjAjEiCpAiMhGiOfK_xHqqkZx_
 *								_xZkqqHx_iClFiCmEiCkCiClNiCkKiBeBiKgFjFnDiClCiCmGiCmJjBoFiCkLiCmIiCmOiCoHiCmCiCkLiCkK_xHqqkZx_
 *								_xZkqqHx_iCkAiCoJiClBiCmGiCkKjFkKiCkJiCmBiClNiBeCiClBiCmMiClNiCnPiBeBjFnDiNiHiMhGiCmM_xHqqkZx_
 *								_xZkqqHx_jClHiClDiCpAiKoOiPiAiCmJiMhGiOfKiClHiCoJiClBiCmGiCmJiClFiClNiBeC_xHqqkZx_
 *			99.06.13 Fukushiro M. UtilMisc.h_xZkqqHx_iCkJiCoHiInKjDkOiBeC_xHqqkZx_POINTVECT_xZkqqHx_jBmOiJjOiBeC_xHqqkZx_
 *			99.06.19 Fukushiro M. OmitPolyPoint2_xZkqqHx_iCmN_xHqqkZx_CalcBezier2_xZkqqHx_iCmMiMoDiCmJiKePjFjEiCmF_xHqqkZx_
 *								_xZkqqHx_iOmAiNhDiClFiCmEiCoAiCoHiCkEiOgEjHgMiClOiCmBiClNiCkKiBeBjIeBiMiLjDfPiCpA_xHqqkZx_
 *								_xZkqqHx_iMnHiCkCiCmF_xHqqkZx_OmitPolyPoint2_xZkqqHx_iCpAiOmAiNhDiClHiCoJiCmGiBeBiPoBiKfBiCkK_xHqqkZx_
 *								_xZkqqHx_jEkNjAlGiClHiCoJiClBiCmGiCkKiCkAiCoJiInHiBeBjCiGiCmJjBgHiCnNiNjOiCpBiClOiBeC_xHqqkZx_
 *								_xZkqqHx_jIeBiMiLjDfPiCpAiCmNiClDiCpBiCmFiIoKjFpLiCkKjClMjApMiBeBjBlMjFpLiCkKiLmIjApMiCmM_xHqqkZx_
 *								_xZkqqHx_iPoKiNiHiCmNiBeBjClMjApMiCkKjEpHjGkNiCmJiLmIjApMjBkEiCmJjAeIiCkCiNjOiCnOiClBiCmG_xHqqkZx_
 *								_xZkqqHx_iCkKiCkAiCoJiBeC_xHqqkZx_
 *************************************************************************/
template<class POINTVECT, class TRECT>
void CalcBezier2 (	POINTVECT& vlgptPoly,
					const FCPolyJoint* pjtpJoint,
					const long lJointSize,
					float mmfWeight,
					long lJointFrom,
					long lJointNum,
					double lgdElemLen,
					double dZoom,
					const TRECT tlgrtaClip[],
					int iClipSize,
					double dMinRatio = 0.0)
{
	vlgptPoly.clear();
	POINTVECT::iterator iP;
	iP = vlgptPoly.insert(vlgptPoly.end());
	RSUB(*iP, pjtpJoint[lJointFrom].m_mmptPivot * dZoom);

	DBLPoint lgdptaStone4[4];	// _xZkqqHx_iDhIiDfHiDeHiMhGiOfKjHhAiCmMiCfDjDfPiNmAjFfH_xHqqkZx_._xZkqqHx_jIfPjHjNiNmAjFfHiMgO_xHqqkZx_.
	DBLPointVector vlgdptPolyTest;	// _xZkqqHx_jClCiNlIjHhAjClIjDfPiNmAjFfHjEhKjHpBiDgPiDgCiDhEiDeA_xHqqkZx_.
	
	DBLRect	lgdrtStone;			// _xZkqqHx_iCfDjDfPiNmAjFfHiCmJiKePjAnKiClHiCoJiLoJiMgA_xHqqkZx_._xZkqqHx_jIfPjHjNiNmAjFfHiMgO_xHqqkZx_.

	const FCPolyJoint* pjtp0;
	const FCPolyJoint* pjtp1;
	for (long lJ = lJointFrom; 0 < lJointNum; lJointNum--, lJ++)
	{
		/*----- _xZkqqHx_jClIjDfPiNmAjFfHjEhKjHpBiCkKiKeKiOgOjDfPiCpAiOpMiCoJiOjOiCmMiPiIjHjN_xHqqkZx_ -----*/
		if (lJointSize <= lJ) lJ = 0;
		pjtp0 = &pjtpJoint[lJ];
		if (lJointSize <= lJ + 1)	pjtp1 = &pjtpJoint[0];
		else						pjtp1 = &pjtpJoint[lJ + 1];

		/*----- _xZkqqHx_iDhIiDfHiDeHiLmIjApMiMhGiOfKiCmJiOgHiCkEiCfDjDfPiCpAiLiBiCnPiCoJ_xHqqkZx_ -----*/
		lgdptaStone4[0] = pjtp0->m_mmptPivot * dZoom;
		lgdptaStone4[1] = (pjtp0->m_mmptPivot + pjtp0->m_mmptNext) * dZoom;
		lgdptaStone4[2] = (pjtp1->m_mmptPivot + pjtp1->m_mmptPrev) * dZoom;
		lgdptaStone4[3] = pjtp1->m_mmptPivot * dZoom;

		// _xZkqqHx_iDhMiDiKiDfDiDjDiMhGiOfKiCkKiPmIjHkKiJmCjEfMiCkJiBeI_xHqqkZx_
		BOOL bIsOmit = FALSE;

		if (0 < iClipSize)
		/*----- _xZkqqHx_iMhGiOfKjHmMiIoGiCkKiOhHjCoIiClDiCoKiCmEiCkCiCoJiPoKiNiH_xHqqkZx_ -----*/
		{
			// lgdptaStone4_xZkqqHx_iCmJiKePjAnKiClHiCoJiLoJiMgAiCpAiMhGiOfK_xHqqkZx_.
			::FFCalcFrame(lgdrtStone, 4L, lgdptaStone4);
			// _xZkqqHx_iKePjAnKiLoJiMgAiCpAjApMjFjNiClOiCkPiNeMiClAiCoJ_xHqqkZx_.
			lgdrtStone.Extend(mmfWeight * dZoom / 2.0 + 1.0);
			/*----- lgdptaStone4_xZkqqHx_iKePjAnKiLoJiMgAiCmGiDiKiBfLiDfHiDiHiDjDiCkKiPgEiCmIiCoJiCkJjClCiCnHiCoJ_xHqqkZx_ -----*/
			bIsOmit = TRUE;
			for (int iClip = 0; iClip < iClipSize; iClip++)
			{
				if (!(lgdrtStone & tlgrtaClip[iClip]).IsRectEmpty())
				/*----- _xZkqqHx_iPgEiCmIiCmBiCmEiCkCiCoJiPoKiNiH_xHqqkZx_ -----*/
				{
					bIsOmit = FALSE;
					break;
				}
			}
		}
		// _xZkqqHx_iCfDjDfPiNmAjFfHiCmMjBePiIpIjDfPiBeFiMoDiIpIjDfPiCkKiCePiCmMiPoKiNiHiCmNiPmIjHkKiBeC_xHqqkZx_
		if (!bIsOmit &&
			almosteq(pjtp0->m_mmptNext.x, 0) && almosteq(pjtp0->m_mmptNext.y, 0) &&
			almosteq(pjtp1->m_mmptPrev.x, 0) && almosteq(pjtp1->m_mmptPrev.y, 0))
			bIsOmit = TRUE;

		// _xZkqqHx_iCfDjDfPiNmAjFfHiCmMjBePiIpIjDfPiBeFiMoDiIpIjDfPiCkKjHlMjCfLiCmMjIeBiMiLjDfPiCmMiKnEiCmJiCkAiCoJiPoKiNiHiCmNiPmIjHkKiBeC_xHqqkZx_
		if (!bIsOmit &&
			almosteq(CalcPowDistance(DBLPoint(), lgdptaStone4[0],
									lgdptaStone4[3], lgdptaStone4[1]), 0) &&
			almosteq(CalcPowDistance(DBLPoint(), lgdptaStone4[0],
									lgdptaStone4[3], lgdptaStone4[2]), 0))
			bIsOmit = TRUE;

		if (bIsOmit)
		//----- _xZkqqHx_iDhIiDfHiDeHiLmIjApMiMhGiOfKiPmIjHkKiCmMiPoKiNiH_xHqqkZx_ -----
		{
			iP = vlgptPoly.insert(vlgptPoly.end());
			RSUB(*iP, pjtp1->m_mmptPivot * dZoom);
		} else
		//----- _xZkqqHx_iDhIiDfHiDeHiLmIjApMiMhGiOfKiCkKjFeLjHhGiCmIiPoKiNiH_xHqqkZx_ -----
		{
			/*----------------------------------------------------
			 * _xZkqqHx_iCnMiClIjCfEiNlIjHhAiCmJiCfHjFkKiKiEiCmMiDhIiDfHiDeHiCpAjFgAiCkLiBeBiCfHjFnDiCmMiNiHiMhGjClHiCpA_xHqqkZx_
			 * _xZkqqHx_iKoOiPiAiCmJjFkKiKiEjAjEiCpAiMiIjCoIiClHiCoJiBeC_xHqqkZx_
			 *----------------------------------------------------*/
			// _xZkqqHx_jCfEiNlIjHhAiCfHjFkKiKiEiDhIiDfHiDeHiCpAiMhGiOfKiBeC_xHqqkZx_
			DrawBezierFast2(vlgdptPolyTest, lgdptaStone4, (SHORT)8);
			double dTestLen = (vlgdptPolyTest[0] - lgdptaStone4[0]).Len();
			for (int m = 0; m < 7; m++)
				dTestLen += (vlgdptPolyTest[m + 1] - vlgdptPolyTest[m]).Len();
//----- 00.09.19 Fukushiro M. _xZkqqHx_jCmHiJmBiOgO_xHqqkZx_ ()-----
			ASSERT(!almost0(lgdElemLen));
//----- 00.09.19 Fukushiro M. _xZkqqHx_jCmHiJmBiPeJ_xHqqkZx_ ()-----
			long lSplit = D2L(dTestLen / lgdElemLen);
			// _xZkqqHx_iNmFjCoBjFkKiKiEjAjEiCpAiCfHiCmJiClHiCoJiBeCiClBiCoKiImIiJlKiCmJiCmIiCoJiCmGiBeB_xHqqkZx_ObjPoly_xZkqqHx_iCmMiDiKiDfEiDeDiDfJiDhEiDiMiBfLiDiAiMhGiOfKiOjOjDjJiCmJjFhDjDhDiNiHiCkKjAlGiClGiCoJiBeC_xHqqkZx_
			if (lSplit < 8) lSplit = 8;
			// _xZkqqHx_iNmFjBoFjFkKiKiEjAjEiCpA_xHqqkZx_200_xZkqqHx_iCmJiClHiCoJiBeCjBoFiCkLiClHiCkMiCoJiCmGjFgAiJoGiCmJiOjOiKnEiCkKiCkJiCkJiCmBiClNiCoIiBeB_xHqqkZx_DrawBezierFast2_xZkqqHx_iCmM_xHqqkZx_short_xZkqqHx_iMfOiCmJjFmPiKlHiCmFiCkLiCmIiCkCiPoKiNiHiCkKiCkAiCoJiBeC_xHqqkZx_
			if (200 < lSplit) lSplit = 200;
			// _xZkqqHx_jAlDiOkOiCmIiDhIiDfHiDeHiMhGiOfKiBeC_xHqqkZx_
			POINTVECT vlgptPolyTmp;
			DrawBezierFast2(vlgptPolyTmp, lgdptaStone4, static_cast<short>(lSplit));
			// _xZkqqHx_iDhIiDfHiDeHjEhKjHpBiCmMiPmIjHkKiPiIjHjNiBeCjAnKjBlBjDfPiCpAiMnHiCkCiCmFiPmIjHkKiClFiCmIiCkCiInHiCmJiClBiClBiCmJjChFiCkNiBeC_xHqqkZx_
			if (dMinRatio != 0.0) OmitPolyPoint2(vlgptPolyTmp, dMinRatio);
			vlgptPoly.insert(vlgptPoly.end(),
							vlgptPolyTmp.begin(), vlgptPolyTmp.end());
		}
	}
} // CalcBezier2.

/*************************************************************************
 * <_xZkqqHx_iKnGjAjE_xHqqkZx_>	OmitPolyPoint2
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_jHfOiCkGiCoHiCoKiClNiDhMiDiKiDfDiDjDjClIjDfPiNmAjFfHjEhKjHpBiCmMjApMjFkKiCpAjFkKjAmNiClFiBeBiLnPiCkCiKhAjDhIiCmM_xHqqkZx_
 *			_xZkqqHx_jApMjFkKiCkKjIeBjBlBiClHiCoJiPoKiNiHiBeBjFhDjHhGiCmIjClIjDfPiNmAjFfHiCpAiPmIjHkKiClHiCoJiClBiCmGiCmF_xHqqkZx_
 *			_xZkqqHx_iClLiCoKiCpAiIoKiCmCiCmMjApMjFkKiCmJiCnMiCmGiCnPiCoJiBeCjAnNiMhGiDgNiBfLiDgH_xHqqkZx_P211_xZkqqHx_iBgA_xHqqkZx_P213,215,P231
 *
 * <_xZkqqHx_iIpIjAjE_xHqqkZx_>	vptPoly		:_xZkqqHx_iDhMiDiKiDfDiDjDjClIjDfPiNmAjFfHjEhKjHpBiCpAjAnNjCoIiClHiCoJiBeCiMhGiOfKiMoDiCmM_xHqqkZx_
 *						  _xZkqqHx_iNmAjFfHjEhKjHpBiCpAjFnEiClH_xHqqkZx_.
 *			dMinRatio	:_xZkqqHx_iPmIjHkKiClHiCoJjDfPiCpAiMiIjCoIiClHiCoJiKoOiPiAjCgM_xHqqkZx_._xZkqqHx_jApMjFkKiCmMjClHiClDiCmJjBmOiClHiCoJ_xHqqkZx_
 *						 _xZkqqHx_jDfPiCmGjApMjFkKiCmMiLjHjHkDiCmMiKiEiNiHiBeCiClBiCmMjCgMiCkKiPkMiClDiCkCjCpG_xHqqkZx_
 *						 _xZkqqHx_iPmIjHkKiClHiCoJjDfPiCpAiMlFjGkHiCmJjElLjCgGiClHiCoJiBeCiCmCiCnMiCoIiPmIjHkKiClDiCoKiCoJ_xHqqkZx_
 *						 _xZkqqHx_jDfPiCkKiPkNiCmIiCkNiCmIiCoJiCmGiCkCiCkEiClBiCmGiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iJpAjAoA_xHqqkZx_>	N_xZkqqHx_jEnEjGnKiCmMjDfP_xHqqkZx_ P(N) _xZkqqHx_iCkJiCoH_xHqqkZx_ N+k_xZkqqHx_jEnEjGnKiCmMjDfP_xHqqkZx_ P(N+k) _xZkqqHx_iCmJjClMjApMiCpAiIpIiCkLiBeB_xHqqkZx_
 *			_xZkqqHx_iClLiCmMiKnEiCmMjDfP_xHqqkZx_ P(N+1)_xZkqqHx_iBgA_xHqqkZx_P(N+k-1) _xZkqqHx_iCmMjBfDiCmEiCmJiCkIiCkCiCmEiBeBjDfPiCmG_xHqqkZx_
 *			_xZkqqHx_jClMjApMiCmMiLjHjHkDiCkK_xHqqkZx_ dMinLen _xZkqqHx_iImIiJlKiCmFiCkAiCoJiCmIiCoHiCmOiBeB_xHqqkZx_P(N+1)_xZkqqHx_iBgA_xHqqkZx_P(N+k-1)_xZkqqHx_iCmN_xHqqkZx_
 *			_xZkqqHx_iPmIjHkKiClDiCoKiCoJiBeCiClBiCmMiPiIjHjNiCpA_xHqqkZx_ k=2 _xZkqqHx_iCkJiCoH_xHqqkZx_1_xZkqqHx_iClIiCmCjBjNiCoCiClFiCmIiCkKiCoHiMhGiOfKiClHiCoJiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	98.09.28 Fukushiro M. CalcBezier_xZkqqHx_iCkJiCoHjFkKjHkD_xHqqkZx_.
 *			98.10.21 Fukushiro M. _xZkqqHx_jDfPiCpAiPmIjHkKiClHiCoJiCkJiCmHiCkEiCkJiCmMjElLjCoIiKoOiPiAiCmN_xHqqkZx_
 *							_xZkqqHx_jApMjFkKiCkJiCoHiCmMjAoCjBmOjDeJiCmIiLjHjHkDiCmFiCkAiCmBiClNiCmMiClOiCkKiBeBiClBiCoKiCpA_xHqqkZx_
 *							_xZkqqHx_jApMjFkKiCmMjClHiClDiCmJjBmOiClHiCoJiKiEiNiHiCmJjFmPiNfIiClFiClNiBeCiClBiCoKiCmJiCoGiCoIiBeB_xHqqkZx_
 *							_xZkqqHx_iNnHiCkJiCkCiLmIjApMiCkKiCkLiCoKiCkCiCmJiPgPiCoJiCoGiCkEiCmJiCmIiCmBiClNiBeC_xHqqkZx_
 *			99.06.13 Fukushiro M. UtilMisc.h_xZkqqHx_iCkJiCoHiInKjDkOiBeC_xHqqkZx_POINTVECT_xZkqqHx_jBmOiJjOiBeC_xHqqkZx_
 *			99.06.14 Fukushiro M. _xZkqqHx_iPfNjHiIiCmMjDfPiCmGjApMiCmMiLjHjHkDiMhGiOfKiCmFiCmNiBeB_xHqqkZx_P231_xZkqqHx_iCmMjHoBiCmF_xHqqkZx_
 *							_xZkqqHx_jDfPiCgCiCkKjDfPiCgDiCmMiMpMiClBiCkEjBkEiCmJiCkAiCoJiPoKiNiHiCoAjDfPiCgCiCkKiPmIjHkK_xHqqkZx_
 *							_xZkqqHx_iClDiCoKiCmEiClFiCnMiCkEiClNiCnPiBeB_xHqqkZx_CalcPowDistance_xZkqqHx_iCpAiOgHiCkEiCoGiCkEiCmJ_xHqqkZx_
 *							_xZkqqHx_jFmPiNfIiClFiClNiBeC_xHqqkZx_
 *************************************************************************/
template<class POINTVECT>
void OmitPolyPoint2 (POINTVECT& vptPoly, double dMinRatio)
{
	// _xZkqqHx_iDhMiDeDiDjDiDgHjAjEiCkKiCfAiCmMiPoKiNiHiCmNjAlDiPoNjDkOiNoMiClFiCmIiCkCiBeCiDhMiDeDiDjDiDgHjAjEiCkKiCfBiImIiJlKiCmMiPoKiNiHiCmNjFeLjHhGiCmIiClFiBeC_xHqqkZx_
	if (vptPoly.size() < 3) return;
	long	lBgn = 0;	// P211_xZkqqHx_iCmJiCkIiCkPiCoJjDfPiCgAiBeC_xHqqkZx_
	long	lEnd;		// P211_xZkqqHx_iCmJiCkIiCkPiCoJjDfPiCgDiBeC_xHqqkZx_
	long	lMid;		// P211_xZkqqHx_iCmJiCkIiCkPiCoJjDfPiCgBiBeDiCgCjDjJiBeCiMlLiNnNjClCiNlIjCiGiCmMjDfPiBeC_xHqqkZx_
	long	lNewPos = 1;// _xZkqqHx_iPmIjHkKiMoDiCmMiDhMiDiKiDiJiDeDiDjDiNoMjAkMjHhAiDeKiDeFiDjDiDfOiBeC_xHqqkZx_
	const double dMinRatioPow = dMinRatio * dMinRatio;
	DBLPoint	dptS;
	DBLPoint	dptBgn;
	DBLPoint	dptEnd;
	DBLPoint	dptMid;

	for (lEnd = 2; lEnd < vptPoly.size(); lEnd++)
	{
		RSUB(dptBgn, vptPoly[lBgn]);
		RSUB(dptEnd, vptPoly[lEnd]);
		const double dLen2 = (dptEnd - dptBgn).PowLen();
		// _xZkqqHx_jApMiCmGjDfPiCmMiLjHjHkDiCmMiKoOiPiAjCgMiBeCiClBiCoKiImIiJlKiCmIiCoHiPmIjHkKiClDiCoKiCoJiBeC_xHqqkZx_
		const double dMinLenPow = dLen2 * dMinRatioPow;
		for (lMid = lBgn + 1; lMid < lEnd; lMid++)
		{
			RSUB(dptMid, vptPoly[lMid]);
			// _xZkqqHx_jApMjFkK_xHqqkZx_(dptBgn,dptEnd)_xZkqqHx_iCmGjDfP_xHqqkZx_dptMid_xZkqqHx_iCmMiLjHjHkDiCkKiKoOiPiAjCgMiCoGiCoIjBoFiCkLiCkCiPoKiNiHiBeC_xHqqkZx_
			if (dMinLenPow < CalcPowDistance(dptS, dptBgn, dptEnd, dLen2, dptMid))
			{
				vptPoly[lNewPos++] = vptPoly[lEnd - 1];
				lBgn = lEnd - 1;
				break;
			}
		}
	}
	vptPoly[lNewPos++] = vptPoly[lEnd - 1];
	vptPoly.resize(lNewPos);
} // OmitPolyPoint2.

/*************************************************************************
 * <_xZkqqHx_iKnGjAjE_xHqqkZx_>	DrawBezierFast2
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_iCfDjDfPiNmAjFfHiCkJiCoH_xHqqkZx_nSplit_xZkqqHx_iMmCiCmJjFkKiKiEiClDiCoKiClNiDhIiDfHiDeHiLmIjApMiCpAiOfKiPgPiClFiBeBiClLiCmM_xHqqkZx_
 *			_xZkqqHx_jClIjDfPjEhKjHpBiCpAjFnEiClHiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iIpIjAjE_xHqqkZx_>	vlgptPoly		:_xZkqqHx_jClIjDfPiCmMjCgMiCmMjEhKjHpBiCpAjFnEiClHiBeC_xHqqkZx_
 *			lgdptaStone4	:_xZkqqHx_iCfDjDfPiNmAjFfHiBeC_xHqqkZx_
 *			nSplit			:_xZkqqHx_iDhIiDfHiDeHjFkKiKiEjAjEiBeCjClIjDfPiNmAjFfHiCmMjAjEiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iJpAjAoA_xHqqkZx_>	_xZkqqHx_jClIjDfPjEhKjHpBiCmJiCmNiBeBiCfDjDfPiNmAjFfHiCmMjBoGiCfAiNmAjFfHiCmNiKnMiCnMiCoKiClIiBeBjBoGiCfDiNmAjFfHiCmN_xHqqkZx_
 *			_xZkqqHx_iKnMiCnMiCoKiCoJiBeCjHoBiCkGiCmOiCfDjDfPiNmAjFfHiCgAiBeBiCgBiBeBiCgCiBeBiCgDiCkJiCoHiCfEjFkKiKiEiCmMiDhIiDfHiDeHiLmIjApM_xHqqkZx_
 *			_xZkqqHx_iCpAiOfKiPgPiClHiCoJiCmGiBeB_xHqqkZx_vlgptPoly[0]_xZkqqHx_iCmNiCgAiCmMiOjPiCmMjClIjDfPiNmAjFfHiCkKjAnNjCoIiClDiCoKiBeB_xHqqkZx_
 *			vlgptPoly[4]_xZkqqHx_iCmJiCmNiCgDiCmMiNmAjFfHiCkKjAnNjCoIiClDiCoKiCoJiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	99.06.13 Fukushiro M. UtilMisc.h_xZkqqHx_iCkJiCoHiInKjDkOiBeC_xHqqkZx_POINTVECT_xZkqqHx_jBmOiJjOiBeC_xHqqkZx_
 *************************************************************************/
template<class POINTVECT>
void DrawBezierFast2 (POINTVECT& vlgptPoly, const DBLPoint lgdptaStone4[], SHORT nSplit)
{
//	TRACE(_T("0,%f,%f\n"), lgdptaStone4[0].x, lgdptaStone4[0].y);

	long		k, _k;
	DBLPoint	dptTmp;
	vlgptPoly.assign(nSplit);
	for (k = 1; k <= nSplit; k++)
	{
		_k = nSplit - k;
		dptTmp = (double)_k * _k * _k * lgdptaStone4[0] +
					(double)3 * k * _k * _k * lgdptaStone4[1] +
					(double)3 * k * k * _k * lgdptaStone4[2] +
					(double)k * k * k * lgdptaStone4[3];
		dptTmp /= double(nSplit) * double(nSplit) * double(nSplit);
		RSUB(vlgptPoly[k - 1], dptTmp);

//		TRACE(_T("%d,%f,%f\n"), k, dptTmp.x, dptTmp.y);

	}
} // DrawBezierFast2.

/*************************************************************************
 * <_xZkqqHx_iKnGjAjE_xHqqkZx_>	CalcBezierMid
 *
 * <_xZkqqHx_iLeAjEfM_xHqqkZx_>	_xZkqqHx_iCfDjDfPiNmAjFfHiCkJiCoHiBeB_xHqqkZx_nSplit_xZkqqHx_iMmCiCmJjFkKiKiEiClDiCoKiClNiDhIiDfHiDeHiLmIjApMiCmJiCkIiCkPiCoJjCiGiKnEjDfPiCmM_xHqqkZx_
 *			_xZkqqHx_iNmAjFfHiCpAjFnEiClHiBeC_xHqqkZx_
 *
 * <_xZkqqHx_iIpIjAjE_xHqqkZx_>	dptaStone4	:_xZkqqHx_iCfDjDfPiNmAjFfHiBeC_xHqqkZx_
 *			ptMid		:_xZkqqHx_jClIjDfPiNmAjFfHiCpAiOpLiCnPiCoJjEhKjHpB_xHqqkZx_.
// *			nSplit		:_xZkqqHx_iDhIiDfHiDeHjFkKiKiEjAjEiBeC_xHqqkZx_
 *
 * <_xZkqqHx_jHjKjHpA_xHqqkZx_>	99.05.18 Fukushiro M. _xZkqqHx_iNoMjAkMiBeC_xHqqkZx_
 *************************************************************************/
template<class T>
void CalcBezierMid (const DBLPoint dptaStone4[], T& ptMid)
{
	DBLPoint pt =
		(dptaStone4[0] + 3 * dptaStone4[1] + 3 * dptaStone4[2] + dptaStone4[3]) / 8.0;
	RSUB(ptMid.x, pt.x);
	RSUB(ptMid.y, pt.y);
} // CalcBezierMid.

#endif//__UtilBezier_h

