/*
 * 쐬: 2004/05/28
 *
 * $Id: HttpRequest.java,v 1.7 2004/06/13 17:00:27 yz125_97 Exp $
 * 

Copyright (c) 2004, Naomine Egawa.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer. 
   Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution. 

2. Neither the name of the EasyHttpd project nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 */
package easyhttpd.http;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.TimeZone;

import easyhttpd.EasyHttpd;

public class HttpRequest {
	private static final String HEADER_CONNECTION = "Connection";
	private static final String HEADER_CONTENT_LENGTH = "Content-Length";
	private static final String HEADER_CONTENT_TYPE = "Content-Type";
	private static final String HEADER_HOST = "Host";
	private static final String HEADER_USER_AGENT = "User-Agent";
	private static final String HEADER_IF_MODIFIED_SINCE = "If-Modified-Since";

	private String method;
	private URL url;
	private int majorVersion;
	private int minorVersion;
	private HashMap header = new HashMap();
	private byte[] data;

	private void initFields() {
		method = null;
		url = null;
		majorVersion = 0;
		minorVersion = 0;
		header.clear();
		data = null;
	}

	private void parseRequestLine(String line) throws HttpProtocolException {
		EasyHttpd.getLogger().finest(line);
		if (line != null
			&& line.length() > 0
			&& line.matches(
				"(?i:GET|POST|HEAD) (http://[^/]+){0,1}/.* HTTP/[0-9].[0-9]")) {
			String[] parts = line.split(" ");
			parts[0] = parts[0].toUpperCase();
			if (parts[0].equals("GET") || parts[0].equals("POST")) {
				method = parts[0];
			} else {
				throw new HttpProtocolException(405, "Method Not Allowed");
			}
			try {
				url =
					new URL(
						"http",
						EasyHttpd.getProperties().getHostName(),
						EasyHttpd.getProperties().getPortNumber(),
						parts[1]);
			} catch (MalformedURLException e) {
				throw new HttpProtocolException(400, "Bad Request");
			}
			if (parts[2].equals("HTTP/1.0")) {
				majorVersion = 1;
				minorVersion = 0;
			} else if (parts[2].equals("HTTP/1.1")) {
				majorVersion = 1;
				minorVersion = 1;
			} else {
				throw new HttpProtocolException(
					505,
					"HTTP Version Not Supported");
			}
		} else {
			throw new HttpProtocolException(400, "Bad Request");
		}
	}

	private void putRequestHeader(String line) throws HttpProtocolException {
		EasyHttpd.getLogger().finest(line);
		if (line != null
			&& line.length() > 0
			&& line.matches("[a-zA-Z0-9-]+: .+")) {
			String[] parts = line.split(":", 2);
			parts[0] = parts[0].toUpperCase();
			parts[1] = parts[1].trim();
			if (parts[0].equals(HEADER_CONNECTION.toUpperCase())) {
				header.put(HEADER_CONNECTION, parts[1]);
			} else if (parts[0].equals(HEADER_CONTENT_LENGTH.toUpperCase())) {
				try {
					Integer length = new Integer(parts[1]);
					header.put(HEADER_CONTENT_LENGTH, length);
				} catch (NumberFormatException e) {
					throw new HttpProtocolException(400, "Bad Request");
				}
			} else if (parts[0].equals(HEADER_CONTENT_TYPE.toUpperCase())) {
				header.put(HEADER_CONTENT_TYPE, parts[1]);
			} else if (parts[0].equals(HEADER_HOST.toUpperCase())) {
				header.put(HEADER_HOST, parts[1]);
			} else if (parts[0].equals(HEADER_USER_AGENT.toUpperCase())) {
				header.put(HEADER_USER_AGENT, parts[1]);
			} else if (
				parts[0].equals(HEADER_IF_MODIFIED_SINCE.toUpperCase())) {
				final SimpleDateFormat sdf =
					new SimpleDateFormat(
						"EEE, d MMM yyyy HH:mm:ss z",
						Locale.US);
				sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
				try {
					final Date date = sdf.parse(parts[1]);
					header.put(HEADER_IF_MODIFIED_SINCE, date);
				} catch (ParseException e) {
					throw new HttpProtocolException(400, "Bad Request");
				}
			}
		} else {
			throw new HttpProtocolException(400, "Bad Request");
		}
	}

	/* 
	 * line separater = LF & CRLF
	 */
	private String readLine(InputStream inputStream) throws IOException {
		int i = inputStream.read();
		if (i == -1) {
			throw new IOException();
		}
		char c = (char) i;
		final StringBuffer s = new StringBuffer();
		while (c != '\n') {
			if (c != '\r') {
				s.append(c);
			}
			i = inputStream.read();
			c = (char) i;
		}
		return s.toString();
	}

	private int getContentLength() {
		return ((Integer) header.get(HEADER_CONTENT_LENGTH)).intValue();
	}

	private String getData() {
		if (data != null) {
			return new String(data);
		} else {
			return null;
		}
	}

	public void init(InputStream inputStream)
		throws IOException, HttpProtocolException {
		EasyHttpd.getLogger().entering(
			this.getClass().getName(),
			"init(InputStream)");
		initFields();
		String line = readLine(inputStream);
		while (line == null || line.length() < 1) {
			line = readLine(inputStream);
		}
		parseRequestLine(line);
		line = readLine(inputStream);
		while (line != null && line.length() > 0) {
			putRequestHeader(line);
			line = readLine(inputStream);
		}
		if (method.equals("POST") && getContentLength() > 0) {
			data = new byte[getContentLength()];
			inputStream.read(data);
		}
		EasyHttpd.getLogger().fine(
			new StringBuffer(method)
				.append(" ")
				.append(url)
				.append(" HTTP/")
				.append(majorVersion)
				.append(".")
				.append(minorVersion)
				.append(" ")
				.append(header)
				.append(" ")
				.append(getData())
				.toString());
		EasyHttpd.getLogger().exiting(
			this.getClass().getName(),
			"init(InputStream)");
	}

	public String getRequestPath() {
		return url.getPath();
	}

	public int getMajorVersion() {
		return majorVersion;
	}

	public int getMinorVersion() {
		return minorVersion;
	}

	public Date getIfModifierdSince() {
		return (Date) header.get(HEADER_IF_MODIFIED_SINCE);
	}

	public boolean isKeepAlive() {
		String connection = (String) header.get(HEADER_CONNECTION);
		if (connection == null || connection.length() < 1) {
			return false;
		}
		connection = connection.toUpperCase();
		if ("KEEP-ALIVE".equals(connection)) {
			return true;
		}
		return false;
	}
}
