/*
 * 쐬: 2004/05/28
 *
 * $Id: HttpResponse.java,v 1.7 2004/06/13 17:00:27 yz125_97 Exp $
 * 

Copyright (c) 2004, Naomine Egawa.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer. 
   Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution. 

2. Neither the name of the EasyHttpd project nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 */
package easyhttpd.http;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.TimeZone;

import easyhttpd.EasyHttpd;

public class HttpResponse {
	private static final String HEADER_CONNECTION = "Connection";
	private static final String HEADER_CONTENT_LENGTH = "Content-Length";
	private static final String HEADER_CONTENT_TYPE = "Content-Type";
	private static final String HEADER_DATE = "Date";
	private static final String HEADER_SERVER = "Server";
	private static final String HEADER_LAST_MODIFIED = "Last-Modified";

	private int statusCode;
	private String statusMessage;
	private int majorVersion;
	private int minorVersion;
	private HashMap header = new HashMap();
	private byte[] data;

	private void initFields() {
		statusCode = 0;
		statusMessage = null;
		majorVersion = 0;
		minorVersion = 0;
		header.clear();
		data = null;
	}

	private void loadFile(File file) throws HttpProtocolException {
		FileInputStream input = null;
		try {
			input = new FileInputStream(file);
			data = new byte[input.available()];
			input.read(data);
			header.put(HEADER_CONTENT_LENGTH, new Integer(data.length));
		} catch (FileNotFoundException e) {
			throw new HttpProtocolException(404, "Not Found");
		} catch (IOException e) {
			throw new HttpProtocolException(500, "Internal Server Error");
		} finally {
			try {
				input.close();
			} catch (Exception e) {
			}
		}
	}

	private void setContentType(File file) {
		String[] parts = file.getName().split("\\.");
		if (parts != null && parts.length > 1) {
			String extension = parts[parts.length - 1].toLowerCase();
			String mediaType = null;
			if (extension.equals("txt")) {
				mediaType = "text/plain";
			} else if (extension.equals("html") || extension.equals("htm")) {
				mediaType = "text/html";
			} else if (extension.equals("xml")) {
				mediaType = "text/xml";
			} else if (extension.equals("js")) {
				mediaType = "text/javascript";
			} else if (extension.equals("vbs")) {
				mediaType = "text/vbscript";
			} else if (extension.equals("css")) {
				mediaType = "text/css";
			} else if (extension.equals("gif")) {
				mediaType = "image/gif";
			} else if (extension.equals("jpg") || extension.equals("jpeg")) {
				mediaType = "image/jpeg";
			} else if (extension.equals("png")) {
				mediaType = "image/png";
			}
			header.put(HEADER_CONTENT_TYPE, mediaType);
		}
	}

	public void init(HttpRequest request) throws HttpProtocolException {
		EasyHttpd.getLogger().entering(
			this.getClass().getName(),
			"init(HttpRequest)");
		initFields();
		majorVersion = request.getMajorVersion();
		minorVersion = request.getMinorVersion();
		setKeepAlive(request.isKeepAlive());
		final String documentRoot = EasyHttpd.getProperties().getDocumentRoot();
		File file = new File(documentRoot, request.getRequestPath());
		if (file.exists()) {
			if (file.isDirectory()) {
				file = new File(file, "index.html");
			}
			if (!file.exists()) {
				throw new HttpProtocolException(403, "Forbidden");
			}
		} else {
			throw new HttpProtocolException(404, "Not Found");
		}
		final SimpleDateFormat sdf =
			new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z", Locale.US);
		sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
		String fileLastModified = sdf.format(new Date(file.lastModified()));
		String requestIfModiredSince = null;
		if (request.getIfModifierdSince() != null) {
			requestIfModiredSince = sdf.format(request.getIfModifierdSince());
		}
		if (fileLastModified.equals(requestIfModiredSince)) {
			statusCode = 304;
			statusMessage = "Not Modified";
		} else {
			header.put(HEADER_LAST_MODIFIED, fileLastModified);
			setContentType(file);
			loadFile(file);
			statusCode = 200;
			statusMessage = "OK";
		}
		EasyHttpd.getLogger().fine(
			new StringBuffer("HTTP/")
				.append(majorVersion)
				.append(".")
				.append(minorVersion)
				.append(" ")
				.append(statusCode)
				.append(" ")
				.append(statusMessage)
				.append(" ")
				.append(header)
				.append(" ")
				.append(file)
				.toString());
		EasyHttpd.getLogger().exiting(
			this.getClass().getName(),
			"init(HttpRequest)");
	}

	public void init(HttpProtocolException e) {
		initFields();
		majorVersion = 1;
		minorVersion = 0;
		statusCode = e.getErrorCode();
		statusMessage = e.getErrorMessage();
		setKeepAlive(false);
	}

	public void commit(OutputStream outputStream) throws IOException {
		EasyHttpd.getLogger().entering(
			this.getClass().getName(),
			"commit(OutputStream)");
		final SimpleDateFormat sdf =
			new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z", Locale.US);
		sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
		header.put(HEADER_DATE, sdf.format(new Date()));
		header.put(
			HEADER_SERVER,
			EasyHttpd.getProperties().getApplicationName());
		final StringBuffer headerBuff = new StringBuffer();
		headerBuff
			.append("HTTP/")
			.append(majorVersion)
			.append(".")
			.append(minorVersion)
			.append(" ")
			.append(statusCode)
			.append(" ")
			.append(statusMessage)
			.append("\r\n");
		for (Iterator i = header.keySet().iterator(); i.hasNext();) {
			String key = (String) i.next();
			headerBuff.append(key).append(": ").append(header.get(key)).append(
				"\r\n");
		}
		headerBuff.append("\r\n");
		int headerLength = headerBuff.toString().getBytes().length;
		byte[] buff;
		if (data != null && data.length > 0) {
			buff = new byte[headerLength + data.length];
			System.arraycopy(
				headerBuff.toString().getBytes(),
				0,
				buff,
				0,
				headerLength);
			System.arraycopy(data, 0, buff, headerLength, data.length);
		} else {
			buff = new byte[headerLength];
			System.arraycopy(
				headerBuff.toString().getBytes(),
				0,
				buff,
				0,
				headerLength);
		}
		outputStream.write(buff);
		outputStream.flush();
		EasyHttpd.getLogger().exiting(
			this.getClass().getName(),
			"commit(OutputStream)");
	}

	private void setKeepAlive(boolean keepAlive) {
		if (keepAlive) {
			header.put(HEADER_CONNECTION, "Keep-Alive");
		} else {
			header.put(HEADER_CONNECTION, "Close");
		}
	}

	public boolean isKeepAlive() {
		String connection = (String) header.get(HEADER_CONNECTION);
		if (connection == null || connection.length() < 1) {
			return false;
		}
		if ("Keep-Alive".equals(connection)) {
			return true;
		}
		return false;
	}
}
