
#include "defs.h"
#include "dump.h"
#include "ebook.h"
#include "history.h"
#include "jcode.h"
#include "textview.h"

static void dump_dlg_close(GtkWidget *widget, gpointer data)
{
    gulong d = (gulong)data;
    gtk_widget_destroy(dump.dlg[d]);
    dump.dlg[d] = NULL;
}

static void dump_clear_textbuf(GtkTextBuffer *buf)
{
    GtkTextIter start, end;
    gtk_text_buffer_get_bounds(buf, &start, &end);
    gtk_text_buffer_delete(buf, &start, &end);
}

static void dump_hex_page(GtkWidget *widget, gpointer data)
{
    gint i, page;
    const gchar *p;
    gchar *text, *tmp, *p_hex = NULL, *p_char = NULL, hex_buff[512], char_buff[512];
    GtkTextIter iter;
    GtkTextTag  *tag;
    RESULT *res = history_last_result();

    if(!res)
	return;

    p = gtk_entry_get_text(GTK_ENTRY(dump.page[DUMP_HEX]));
    if(!strlen(p))
	return;

    page = strtol(p, NULL, 16);

    text = ebook_get_raw_text(res->binfo->book, page, 0);
    if(!text)
	return;

    dump_clear_textbuf(dump.buf[DUMP_HEX]);

    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(dump.view[DUMP_HEX]), GTK_WRAP_NONE);
    tag = gtk_text_buffer_create_tag(dump.buf[DUMP_HEX], NULL, "family", "Monospace", NULL);
    gtk_text_buffer_get_start_iter(dump.buf[DUMP_HEX], &iter);
    for(i = 0;  i < EB_SIZE_PAGE; i = i + 2)
    {
	if((i % 16) == 0)
	{
	    p_hex = hex_buff;
	    p_char = char_buff;
	    memset(p_hex, 0, 512);
	    memset(p_char, 0, 512);
	    sprintf(p_hex, "0x%02x ",(i / 16));
	    p_hex += 5;
	    sprintf(p_hex, "(0x%08x)  ",(page - 1) * EB_SIZE_PAGE + i);
	    p_hex += 14;
	    sprintf(p_char, " ");
	    p_char += 1;
	}
	sprintf(p_hex, "%02x ",(unsigned char)text[i]);
	p_hex += 3;
	sprintf(p_hex, "%02x ",(unsigned char)text[i+1]);
	p_hex += 3;
	if(is_jis_printable((guchar*)&text[i]))
	{
	    *p_char = text[i] + 0x80;
	    p_char++;
	    *p_char = text[i + 1] + 0x80;
	    p_char++;
	}
	else
	{
	    *p_char = 0xA1;
	    p_char++;
	    *p_char = 0xA5;
	    p_char++;
	}
	if((i % 16) == 14)
	{
	    gtk_text_buffer_insert_with_tags(dump.buf[DUMP_HEX], &iter, hex_buff, -1, tag, NULL);
	    tmp = iconv_convert(ENC_EUC_JP, ENC_UTF8, char_buff);
	    if(tmp)
	    {
		gtk_text_buffer_insert_with_tags(dump.buf[DUMP_HEX], &iter, tmp, -1, tag, NULL);
		g_free(tmp);
	    }
	    gtk_text_buffer_insert_with_tags(dump.buf[DUMP_HEX], &iter, "\r\0", -1, tag, NULL);
	}
    }
    g_free(text);
}

static void dump_hex_page_change(gboolean next)
{
    gint page;
    const gchar *p;
    gchar buff[64];

    p = gtk_entry_get_text(GTK_ENTRY(dump.page[DUMP_HEX]));
    page = strtol(p, NULL, 16);
    if(!page)
	return;
    page += next ? 1 : -1;

    sprintf(buff, "%08x",page);
    gtk_entry_set_text(GTK_ENTRY(dump.page[DUMP_HEX]), buff);
    dump_hex_page(NULL, NULL);
}

static void dump_hex_page_prev(GtkWidget *widget, gpointer data)
{
    dump_hex_page_change(FALSE);
}

static void dump_hex_page_next(GtkWidget *widget, gpointer data)
{
    dump_hex_page_change(TRUE);
}

static void dump_text_page(GtkWidget *widget, gpointer data)
{
    gchar *text, *utf_text;
    GtkTextIter iter;
    RESULT *res = history_last_result();

    if(!res)
	return;
    text = ebook_get_text(res);
    if(!text)
	return;

    dump_clear_textbuf(dump.buf[DUMP_TEXT]);

    gtk_text_buffer_get_start_iter(dump.buf[DUMP_TEXT], &iter);
    utf_text = iconv_convert(ENC_EUC_JP, ENC_UTF8, text);
    gtk_text_buffer_insert(dump.buf[DUMP_TEXT], &iter, utf_text, -1);
    g_free(text);
    g_free(utf_text);
}

void dump_do_dlg(gulong d)
{
    GtkWidget *btn, *hbox, *scroll;
    gboolean hex = (d == DUMP_HEX);
    gchar buff[16];
    RESULT *res = history_last_result();

    if(dump.dlg[d])
    {
	dump_dlg_close(NULL, (gpointer)d);
	return;
    }

    dump.dlg[d] = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(dump.dlg[d]), hex ? _("Hex dump") : _("Text dump"));
    g_signal_connect(G_OBJECT(dump.dlg[d]), "hide", G_CALLBACK(dump_dlg_close), (gpointer)d);
    gtk_box_set_homogeneous(GTK_BOX(GTK_DIALOG(dump.dlg[d])->vbox), FALSE);

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dump.dlg[d])->vbox), hbox, FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("page")), FALSE, FALSE, 0);
    dump.page[d] = gtk_entry_new();
    gtk_editable_set_editable(GTK_EDITABLE(dump.page[d]), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), dump.page[d], FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(dump.page[d]), "activate", G_CALLBACK(hex ? dump_hex_page : dump_text_page), NULL);

    if(hex)
    {
	btn = gtk_button_new_with_label("  <<  ");
	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dump.dlg[d])->action_area), btn, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(dump_hex_page_prev), NULL);

	btn = gtk_button_new_with_label("  >>  ");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dump.dlg[d])->action_area), btn, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(dump_hex_page_next), NULL);
    }
    else
    {
	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("offset")), FALSE, FALSE, 0);

	dump.offset = gtk_entry_new();
	gtk_editable_set_editable(GTK_EDITABLE(dump.offset), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), dump.offset, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(dump.offset), "activate", G_CALLBACK(dump_text_page), NULL);
    }
    scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll), GTK_SHADOW_ETCHED_IN);
    gtk_widget_set_size_request(scroll, 600, 300);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dump.dlg[d])->vbox), scroll, TRUE, TRUE, 0);

    dump.buf[d] = gtk_text_buffer_new(NULL);
    dump.view[d] = gtk_text_view_new_with_buffer(dump.buf[d]);

    gtk_text_view_set_editable(GTK_TEXT_VIEW(dump.view[d]), FALSE);
    gtk_text_view_set_left_margin(GTK_TEXT_VIEW(dump.view[d]), 10);
    gtk_text_view_set_right_margin(GTK_TEXT_VIEW(dump.view[d]), 10);
    gtk_text_view_set_pixels_above_lines(GTK_TEXT_VIEW(dump.view[d]), 3);
    gtk_text_view_set_pixels_inside_wrap(GTK_TEXT_VIEW(dump.view[d]), 3);
    gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(dump.view[d]), FALSE);
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(dump.view[d]), GTK_WRAP_WORD);

    gtk_container_add(GTK_CONTAINER(scroll), dump.view[d]);

    btn = gtk_button_new_with_label(_("Close"));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dump.dlg[d])->action_area), btn, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(dump_dlg_close), (gpointer)d);

    gtk_widget_show_all(dump.dlg[d]);

    sprintf(buff, "%08x", res ? res->pos.page : 0);
    gtk_entry_set_text(GTK_ENTRY(dump.page[d]), buff);
    if(!hex)
    {
	sprintf(buff, "%08x", res ? res->pos.offset : 0);
	gtk_entry_set_text(GTK_ENTRY(dump.offset), buff);
    }
    hex ? dump_hex_page(NULL, NULL) : dump_text_page(NULL, NULL);
}

void dump_hex()
{
    dump_do_dlg(DUMP_HEX);
}

void dump_text()
{
    dump_do_dlg(DUMP_TEXT);
}

void dump_update()
{
    if(dump.dlg[DUMP_HEX])
	dump_hex_page(NULL, NULL);
    if(dump.dlg[DUMP_TEXT])
	dump_text_page(NULL, NULL);
}

