
#include "defs.h"

#include "dictbar.h"
#include "eb123.h"
#include "headword.h"
#include "history.h"
#include "hotkeys.h"
#include "mainwnd.h"
#include "preferences.h"
#include "selection.h"
#include "textview.h"

const HOTKEY_COMMAND hotkeys_list[] = {
    { _("Go back"),			history_prev, FALSE},
    { _("Clear search field"),		mainwnd_clear_combo, FALSE},
    { _("Next hit"),			headword_next, FALSE},
    { _("Previous hit"),		headword_prev, FALSE},
    { _("Select all dictionaries"),	dictbar_all_select, FALSE},
    { _("Unselect all dictionaries"),	dictbar_all_unselect, FALSE},
    { _("Quit program"),		app_quit, FALSE},
    { _("Iconify/restore main window"), mainwnd_iconify_restore, TRUE},
    { _("Search in main window"),	selection_search_in_mainwnd, TRUE},
    { _("Search in popup window"),	selection_search_in_popupwnd, TRUE},
    {NULL, NULL, FALSE}
};

void hotkeys_list_init()
{
    gint i;
    HOTKEY_EVENT *events = g_try_new0(HOTKEY_EVENT, sizeof(hotkeys_list));
    if(!events) return;
    hotkeys.list = NULL;
    for(i = 0; hotkeys_list[i].name; i++)
	hotkeys.list = g_list_append(hotkeys.list, (gpointer)&(events[i]));
}

static gboolean hotkeys_local_activated_cb(GtkAccelGroup *accelgroup, GObject *arg1, guint arg2, GdkModifierType arg3, gpointer user_data)
{
    gint i;
    guint cmp;
    HOTKEY_EVENT *evt;

    gulong keyval = (gulong)user_data;
    gulong mask = arg3;
    for(i = 0; (evt = g_list_nth_data(hotkeys.list, i)); i++)
    {
	if(evt->keyval != keyval) continue;
	cmp = (evt->mask ^ mask);
	if(hotkeys.ignore_locks)
	    cmp = cmp & !(GDK_MOD2_MASK | GDK_LOCK_MASK);
	if(!cmp)
	{
	    hotkeys_list[i].func();
	    return FALSE;
	}
    }
    return FALSE;
}

static void hotkeys_local_uninstall()
{
#if 0
    GList *item;
    GtkWidget *widget;
        item = g_list_first(accel_item_list);
        while(item){
            widget = (GtkWidget *)(item->data);
            gtk_widget_destroy(widget);
            item = g_list_next(item);
        }
        g_list_free(accel_item_list);
        accel_item_list = NULL;
        g_object_unref(accel_group);
#endif
}

void hotkeys_local_install()
{
    static GtkAccelGroup *accel_group = NULL;
    GClosure *closure;
    gint i;
    HOTKEY_EVENT *evt;
    hotkeys_local_uninstall();
    if(accel_group)
    {
        gtk_window_remove_accel_group(GTK_WINDOW(mainwnd.wnd), accel_group);
        g_object_unref(accel_group);
    }
    accel_group = gtk_accel_group_new();
    gtk_window_add_accel_group(GTK_WINDOW(mainwnd.wnd), accel_group);
    g_signal_connect(G_OBJECT(accel_group), "accel-activate", G_CALLBACK(hotkeys_local_activated_cb), NULL);

    for(i = 0; (evt = g_list_nth_data(hotkeys.list, i)); i++)
    {
	gulong keyval = evt->keyval;
	if(!evt->enabled || hotkeys_list[i].global) continue;
	if((evt->mask == 0) && (evt->keyval == GDK_Return)) continue;
	closure = g_cclosure_new(G_CALLBACK(hotkeys_local_activated_cb), (gpointer)keyval, NULL);
	gtk_accel_group_connect(accel_group, evt->keyval, evt->mask, GTK_ACCEL_VISIBLE, closure);
    }
}

static GdkFilterReturn hotkeys_global_cb(GdkXEvent *gdk_xevent, GdkEvent *event, gpointer data)
{
    XEvent *xevent = (XEvent *) gdk_xevent;
    Display *d = XOpenDisplay(NULL);
    const guint mods = LockMask | Mod2Mask;
    guint i;
    HOTKEY_EVENT *evt;

    switch(xevent->type)
    {
	case KeyPress:
	{
	    for(i = 0; (evt = g_list_nth_data(hotkeys.list, i)); i++)
	    {
		if(!hotkeys_list[i].global) continue;
		gboolean mask_cmp;
		if(hotkeys.ignore_locks)
		    mask_cmp = ((xevent->xkey.state | mods) == (evt->mask | mods));
		else
		    mask_cmp = (xevent->xkey.state == evt->mask);
		if(mask_cmp && (XKeysymToKeycode(d, evt->keyval) == xevent->xkey.keycode))
		{
		    hotkeys_list[i].func();
		    break;
		}
	    }
	    break;
	}
	default:
	    break;
    }

    return GDK_FILTER_CONTINUE;
}

void hotkeys_global_install()
{
    static gboolean cb_added = FALSE;
    GdkWindow *rootwin = gdk_get_default_root_window();
    Display *d = GDK_WINDOW_XDISPLAY(rootwin);
    gint i, j;
    HOTKEY_EVENT *evt;

    if(!cb_added)
    {
	gdk_window_add_filter(rootwin, hotkeys_global_cb, NULL);
	cb_added = TRUE;
    }
    const guint mod_masks [] = {
	0,
	LockMask, /* Caps Lock */
	Mod2Mask, /* Num Lock */
	LockMask | Mod2Mask
    };
    // remove all hotkeys
    for(i = 0; (evt = g_list_nth_data(hotkeys.list, i)); i++)
    {
	if(!hotkeys_list[i].global) continue;
	for(j = 0; j < G_N_ELEMENTS(mod_masks); j++)
	    XUngrabKey(d, XKeysymToKeycode(d, evt->keyval), evt->mask | mod_masks[j], DefaultRootWindow(d));
    }
    // install currently assigned hotkeys
    for(i = 0; (evt = g_list_nth_data(hotkeys.list, i)); i++)
    {
	if(!hotkeys_list[i].global) continue;
	if(hotkeys.ignore_locks)
	{
	    for(j = 0; j < G_N_ELEMENTS(mod_masks); j++)
		XGrabKey(d, XKeysymToKeycode(d, evt->keyval), evt->mask | mod_masks[j], DefaultRootWindow(d), True, GrabModeAsync, GrabModeAsync);
	}
	else
	    XGrabKey(d, XKeysymToKeycode(d, evt->keyval), evt->mask, DefaultRootWindow(d), True, GrabModeAsync, GrabModeAsync);
    }
}

HOTKEY_EVENT* hotkeys_find(const gchar *name)
{
    gint i;
    for(i = 0; hotkeys_list[i].name; i++)
    {
	if(!g_strcmp0(hotkeys_list[i].name, name))
	    return (HOTKEY_EVENT*)g_list_nth_data(hotkeys.list, i);
    }
    return NULL;
}

void hotkeys_save()
{
    gchar filename[PATH_MAX], buff[16];
    gint i;
    HOTKEY_EVENT *evt;
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"hotkeys", NULL);
    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_HOTKEYS);
    for(i = 0; (evt = g_list_nth_data(hotkeys.list, i)); i++)
    {
	if(!evt->enabled) continue;
	xmlNodePtr node1 = xmlAddChild((xmlNodePtr)doc->children, xmlNewNode(NULL, (xmlChar*)"hotkeys"));
	xmlNewProp(node1, (xmlChar*)"command", (xmlChar*)hotkeys_list[i].name);
	sprintf(buff, "0x%04x", evt->mask);
	xmlNewProp(node1, (xmlChar*)"mask", (xmlChar*)buff);
	sprintf(buff, "0x%04x", evt->keyval);
	xmlNewProp(node1, (xmlChar*)"keyval", (xmlChar*)buff);
    }
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

void hotkeys_load_item(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    if(!atts) return;
    HOTKEY_EVENT *evt = hotkeys_find((gchar*)atts[1]);
    if(!evt) return;
    evt->mask = strtol((gchar*)atts[3], NULL, 16);
    evt->keyval = strtol((gchar*)atts[5], NULL, 16);
    evt->enabled = (evt->keyval != 0);
}

/* Add local "Ctrl + q" hotkey */
void hotkeys_add_defaults()
{
    gint i;
    for(i = 0; hotkeys_list[i].name; i++)
    {
        if(hotkeys_list[i].func == app_quit)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(hotkeys.list, i);
            evt->mask = 0x4;
            evt->keyval = 0x71;
	    evt->enabled = TRUE;
        }
        else if(hotkeys_list[i].func == dictbar_all_select)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(hotkeys.list, i);
            evt->mask = 0x4;
            evt->keyval = 0x65;
	    evt->enabled = TRUE;
        }
        else if(hotkeys_list[i].func == dictbar_all_unselect)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(hotkeys.list, i);
            evt->mask = 0x4;
            evt->keyval = 0x64;
	    evt->enabled = TRUE;
        }
        else if(hotkeys_list[i].func == mainwnd_clear_combo)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(hotkeys.list, i);
            evt->mask = 0x4;
            evt->keyval = 0x77;
	    evt->enabled = TRUE;
        }
    }
}

void hotkeys_load()
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_HOTKEYS);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
    {
        hotkeys_add_defaults();
        return;
    }
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &hotkeys_load_item;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
}

void hotkey_to_string(guint mask, guint keyval, gchar *key)
{
    key[0] = '\0';

    if(mask & GDK_CONTROL_MASK)
        strcat(key, "<Ctrl>");

    if(mask & GDK_SHIFT_MASK)
        strcat(key, "<Shift>");

    if(mask & GDK_LOCK_MASK)
        strcat(key, "<Lock>");

    if(mask & GDK_MOD1_MASK)
        strcat(key, "<Alt>");

    if(mask & GDK_MOD2_MASK)
            strcat(key, "<NumLock>");

    if(mask & GDK_MOD3_MASK)
        strcat(key, "<Mod3>");

    if(mask & GDK_MOD4_MASK)
        strcat(key, "<Mod4>");

    if(mask & GDK_MOD5_MASK)
        strcat(key, "<ScrollLock>");

    if(mask & GDK_BUTTON1_MASK)
        strcat(key, "<Button1>");

    if(mask & GDK_BUTTON2_MASK)
        strcat(key, "<Button2>");

    if(mask & GDK_BUTTON3_MASK)
        strcat(key, "<Button3>");

    if(mask & GDK_BUTTON4_MASK)
        strcat(key, "<Button4>");

    if(mask & GDK_BUTTON5_MASK)
        strcat(key, "<Button5>");

    if(mask & GDK_RELEASE_MASK)
        strcat(key, "<Release>");

    strcat(key, gdk_keyval_name(keyval));
}

