
#include "defs.h"

#include <gdk/gdkkeysyms.h>
#include <sys/types.h>

#include "dictbar.h"
#include "dump.h"
#include "eb123.h"
#include "ebook.h"
#include "icon.xpm"
#include "headword.h"
#include "history.h"
#include "jcode.h"
#include "mainwnd.h"
#include "render.h"
#include "preferences.h"
#include "popupwnd.h"
#include "hotkeys.h"
#include "selection.h"
#include "textview.h"
#include "toolbar.h"

extern struct _search_method methods[];

void mainwnd_search(const gchar *word)
{
    gchar *euc_str = iconv_convert(ENC_UTF8, ENC_EUC_JP, word);
    gint method = ebook_search_method();
    g_strstrip(euc_str);
    if(!strlen(euc_str))
    {
        g_free(euc_str);
        return;
    }
    history_insert_word(word);

    ebook_search(euc_str, method, mainwnd.results, headword.maxhits);
    g_free(euc_str);
    headword_show_results();
    gtk_window_deiconify(GTK_WINDOW(mainwnd.wnd));
    gtk_widget_show_all(mainwnd.wnd);
    gtk_window_present(GTK_WINDOW(mainwnd.wnd));
}

static void mainwnd_search_cb(GtkWidget *combo, gpointer user_data)
{
    const gchar *word = gtk_combo_box_get_active_text(GTK_COMBO_BOX(mainwnd.combo_word));
    mainwnd_search(word);
}

void mainwnd_show_about(GtkWidget *w, gpointer data)
{
    eb_textview_insert_message(paned.view, "\n\teb123 is a very basic electronic book viewer based on eb library and ebview source code.", TRUE);
    eb_textview_insert_message(paned.view, "\n\nInstalled version: ", FALSE);
    eb_textview_insert_message(paned.view, PACKAGE_VERSION, FALSE);
    eb_textview_insert_message(paned.view, "\n\n(C) evgeny, 2009-2010", FALSE);
}

static void mainwnd_prev(GtkWidget *widget, gpointer data)
{
    history_prev();
}

static void mainwnd_method_changed(GtkWidget *combo)
{
    mainwnd.search = gtk_combo_box_get_active(GTK_COMBO_BOX(mainwnd.combo_method));
}

void mainwnd_clear_combo()
{
    gtk_entry_set_text(GTK_ENTRY(GTK_BIN(mainwnd.combo_word)->child), "");
    gtk_widget_grab_focus(mainwnd.combo_word);
}

void mainwnd_clear_combo_cb(GtkWidget *widget, gpointer data)
{
    mainwnd_clear_combo();
}

gboolean mainwnd_key_press(GtkWidget *widget, GdkEventKey *event, gpointer user_data)
{
    if(event->keyval == GDK_BackSpace)
    {
        if(!GTK_WIDGET_HAS_FOCUS(GTK_BIN(mainwnd.combo_word)->child))
        {
            gtk_widget_grab_focus(mainwnd.combo_word);
            gtk_editable_set_position(GTK_EDITABLE(GTK_BIN(mainwnd.combo_word)->child), -1);
            return TRUE;
        }
    }
    return FALSE;
}

GtkWidget *mainwnd_search_method_combo()
{
    gint i;
    GtkWidget *w = gtk_combo_box_new_text();
    for(i = 0; methods[i].search != NULL; i++)
        gtk_combo_box_append_text(GTK_COMBO_BOX(w), methods[i].name);
    return w;
}

static GtkWidget *mainwnd_search_bar()
{
    GtkWidget *hbox, *btn;
    hbox = gtk_hbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);

    btn = toolbar_button(mainwnd_clear_combo_cb, GTK_STOCK_CLEAR, _("Clear word"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 5);

    gtk_box_pack_start(GTK_BOX(hbox), mainwnd.combo_word, TRUE, TRUE, 0);
    g_signal_connect(gtk_bin_get_child(GTK_BIN(mainwnd.combo_word)), "activate", G_CALLBACK(mainwnd_search_cb), NULL);

    btn = toolbar_button(mainwnd_search_cb, GTK_STOCK_FIND, _("Start search"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);

    mainwnd.combo_method = mainwnd_search_method_combo();
    gtk_widget_set_size_request(GTK_WIDGET(mainwnd.combo_method), 200, -1);
    gtk_box_pack_start(GTK_BOX (hbox), mainwnd.combo_method, FALSE, TRUE, 0);
    gtk_widget_set_tooltip_text(mainwnd.combo_method, _("Select search method"));
    g_signal_connect(G_OBJECT(mainwnd.combo_method), "changed", G_CALLBACK(mainwnd_method_changed), NULL);
    gtk_combo_box_set_active(GTK_COMBO_BOX(mainwnd.combo_method), mainwnd.search);

    btn = toolbar_button(mainwnd_prev, GTK_STOCK_UNDO, _("Show previous word"), TRUE);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);

    return hbox;
}

GtkWidget *mainwnd_create_vbox()
{
    GtkWidget *vbox, *widget, *scroll;

    vbox = gtk_vbox_new(FALSE, 0);

    widget = mainwnd_search_bar();
    gtk_box_pack_start(GTK_BOX(vbox), widget, FALSE, FALSE, 0);

    mainwnd.dictbar = dictbar_create();
    gtk_box_pack_start(GTK_BOX(vbox), mainwnd.dictbar, FALSE, FALSE, 0);

    paned.pane = gtk_hpaned_new();
    gtk_container_set_border_width(GTK_CONTAINER(paned.pane), 3);
    gtk_box_pack_start(GTK_BOX(vbox), paned.pane, TRUE, TRUE, 0);

    paned.tree = create_headword_tree();
    gtk_paned_add1(GTK_PANED(paned.pane), paned.tree);

    scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll), GTK_SHADOW_IN);
    gtk_paned_add2(GTK_PANED(paned.pane), scroll);

    paned.view = g_object_new(EB_TYPE_TEXTVIEW, NULL);
    g_signal_connect(G_OBJECT(paned.view), "populate_popup", G_CALLBACK(eb_textview_populate_popup), paned.view);
    EB_TEXTVIEW_GET_CLASS(paned.view)->history_next = history_next;
    gtk_container_add(GTK_CONTAINER(scroll), GTK_WIDGET(paned.view));

    widget = toolbar_create();
    gtk_box_pack_start(GTK_BOX(vbox), widget, FALSE, FALSE, 0);
 
    gtk_widget_show_all(vbox);
    return vbox;
}

void mainwnd_reset_font()
{
    gchar *font = mainwnd.custom_font ? mainwnd.font : "";
    if(mainwnd.custom_font && mainwnd.font)
    {
	gchar **a = g_strsplit(mainwnd.font, " ", 2);
	if(g_strv_length(a) > 1)
	    eb_textview_set_pixels(paned.view, atoi(a[1]));
	g_strfreev(a);
    }
    gchar *s = g_strdup_printf("style \"custom-font\"{font_name=\"%s\"}\nclass \"GtkWidget\" style \"custom-font\"", font);
    gtk_rc_parse_string(s);
    g_free(s);
    GdkScreen *screen = gtk_window_get_screen(GTK_WINDOW(mainwnd.wnd));
    GtkSettings *settings = gtk_settings_get_for_screen(screen);
    gtk_rc_reset_styles(settings);
}

void mainwnd_status_icon_menu(GtkStatusIcon *status_icon, guint button, guint activate_time, gpointer data)
{
    GtkWidget *menu = gtk_menu_new(), *item;
    item = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(app_exit), NULL);
    gtk_widget_show_all(menu);
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
}

void mainwnd_iconify_restore()
{
    if(GTK_WIDGET_VISIBLE(mainwnd.wnd))
    {
	preferences_close_wnd();
	popupwnd_close();
	gtk_widget_hide_all(mainwnd.wnd);
    }
    else
	gtk_widget_show_all(mainwnd.wnd);
}

void mainwnd_status_icon_activate(GtkWidget *widget, gpointer data)
{
    mainwnd_iconify_restore();
}

void mainwnd_status_icon_add()
{
    GdkPixbuf *pixbuf;
    GtkStatusIcon *icon;
    pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)icon_xpm);
    icon = gtk_status_icon_new_from_pixbuf(pixbuf);
    g_signal_connect(G_OBJECT(icon), "activate", G_CALLBACK(mainwnd_status_icon_activate), NULL);
    g_signal_connect(G_OBJECT(icon), "popup_menu", G_CALLBACK(mainwnd_status_icon_menu), NULL);
    gdk_pixbuf_unref(pixbuf);
}

void mainwnd_create()
{
    GtkWidget *vbox, *widget;

    mainwnd.wnd = gtk_widget_new(GTK_TYPE_WINDOW, "type", GTK_WINDOW_TOPLEVEL, "title", PACKAGE_NAME, "allow-shrink", TRUE, "allow-grow", TRUE, NULL);
    g_signal_connect(G_OBJECT(mainwnd.wnd), "delete_event", G_CALLBACK(app_exit), NULL);
    g_signal_connect(G_OBJECT(mainwnd.wnd), "key_press_event", G_CALLBACK(mainwnd_key_press), NULL);

    if(mainwnd.remember_pos)
    {
        gtk_window_set_default_size(GTK_WINDOW(mainwnd.wnd), mainwnd.w, mainwnd.h);
        gtk_window_move(GTK_WINDOW(mainwnd.wnd), mainwnd.x, mainwnd.y);
    }
    gtk_widget_realize(mainwnd.wnd);
    mainwnd_status_icon_add();

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(mainwnd.wnd), vbox);

    widget = mainwnd_create_vbox();
    gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(widget), TRUE, TRUE, 0);
    
    gtk_widget_show(vbox);
    gtk_widget_set_size_request(paned.tree, paned.treew, paned.treeh);

    mainwnd_reset_font();
    gtk_widget_grab_focus(mainwnd.combo_word);
    gtk_widget_show_all(mainwnd.wnd);

    hotkeys_local_install();
    hotkeys_global_install();
}

void mainwnd_open(RESULT *res)
{
    eb_textview_open(paned.view, res, TRUE, 0);
    gtk_widget_show_all(mainwnd.wnd);
}

