
#include "defs.h"

#include "dicts.h"
#include "ebook.h"
#include "history.h"
#include "jcode.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "render.h"
#include "textview.h"
#include "toolbar.h"

void popupwnd_close(GtkWidget *widget, gpointer data)
{
    if(popupwnd.wnd)
    {
	gtk_window_get_position(GTK_WINDOW(popupwnd.wnd), &popupwnd.x, &popupwnd.y);
        gtk_widget_destroy(popupwnd.wnd);
        gtk_widget_destroy(popupwnd.menu);
        popupwnd.wnd = NULL;
        popupwnd.menu = NULL;
    }
}

static gboolean popupwnd_close_(gpointer data)
{
    popupwnd_close(NULL, NULL);
    return FALSE;
}

static void popupwnd_start_stop_timer(gboolean start)
{
    static gint timeout_id = 0;
    if(timeout_id != 0)
        g_source_remove(timeout_id);
    timeout_id = 0;
    if(!start || popupwnd.lock)
        return;
    timeout_id = g_timeout_add(popupwnd.timeout, popupwnd_close_, NULL);
}

void popupwnd_leave_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer user_data)
{
    popupwnd_start_stop_timer((event->detail == GDK_NOTIFY_NONLINEAR) || (event->detail == GDK_NOTIFY_NONLINEAR_VIRTUAL));
}

void popupwnd_enter_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer user_data)
{
    popupwnd_start_stop_timer(FALSE);
}

void popupwnd_lock(GtkWidget *widget, gpointer user_data)
{
    popupwnd.lock = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
}

void popupwnd_query_in_mainwnd(GtkWidget *widget, gpointer data)
{
    const gchar *str = gtk_label_get_text(GTK_LABEL(popupwnd.label));
    mainwnd_search(str);
    gtk_window_deiconify(GTK_WINDOW(mainwnd.wnd));
}

static void popupwnd_dict_changed(GtkWidget *w, gpointer data)
{
    if(w)
    {
        popupwnd.binfo = (BOOK_INFO*)data;
        popupwnd.lock = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(popupwnd.lockbtn));
    }
    if(!popupwnd.binfo)
    {
        g_free(popupwnd.group);
        popupwnd.group = NULL;
        g_free(popupwnd.dict);
        popupwnd.dict = NULL;
        return;
    }
    GtkTreePath *path = dicts_find_path(popupwnd.binfo->book);
    if(path)
    {
        GtkTreeIter iter, iter1;
        gtk_tree_model_get_iter(GTK_TREE_MODEL(dicts.store), &iter1, path);
        gtk_tree_model_iter_parent(GTK_TREE_MODEL(dicts.store), &iter, &iter1);
        gtk_tree_path_free(path);
        g_free(popupwnd.group);
        g_free(popupwnd.dict);
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter, DICT_ALIAS, &popupwnd.group, -1);
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter1, DICT_ALIAS, &popupwnd.dict, -1);
    }
    popupwnd_search(NULL);
}

static gboolean popupwnd_dict_menu_append(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    static gboolean sel_group = FALSE;
    static GSList *group = NULL;
    static GtkWidget *submenu = NULL;
    GtkWidget *item;
    if(!model)
    {
        if(popupwnd.menu)
            gtk_widget_destroy(popupwnd.menu);
        group = NULL;
        popupwnd.menu = gtk_menu_new();
        item = gtk_radio_menu_item_new_with_label(group, _("Main window selection"));
        g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(popupwnd_dict_changed), NULL);
        group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
        gtk_menu_shell_append(GTK_MENU_SHELL(popupwnd.menu), item);
        submenu = NULL;
        return FALSE;
    }
    if(gtk_tree_path_get_depth(path) == 1)
    {
        gchar *name;
        submenu = gtk_menu_new();
        gtk_tree_model_get(model, iter, DICT_ALIAS, &name, -1);
        gtk_menu_shell_append(GTK_MENU_SHELL(popupwnd.menu), item = gtk_menu_item_new_with_label(name));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
        sel_group = !g_strcmp0(popupwnd.group, name);
        g_free(name);
    }
    else
    {
        gchar *name;
        BOOK_INFO *binfo;
        gtk_tree_model_get(model, iter, DICT_ALIAS, &name, DICT_BINFO, &binfo, -1);
        item = gtk_radio_menu_item_new_with_label(group, binfo->title);
        group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
        gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item);
        if(sel_group && !g_strcmp0(popupwnd.dict, name))
        {
            popupwnd.binfo = binfo;
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE); 
        }
        g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(popupwnd_dict_changed), binfo);
        g_free(name);
        
    }
    return FALSE;
}

static void popupwnd_dict_menu_create(gboolean newmenu)
{
    if(newmenu || !popupwnd.menu)
    {
        popupwnd_dict_menu_append(NULL, NULL, NULL, NULL);
        gtk_tree_model_foreach(GTK_TREE_MODEL(dicts.store), popupwnd_dict_menu_append, NULL);
    }
}

static void popupwnd_set_dict(GtkWidget *widget, gpointer data)
{
    popupwnd_dict_menu_create(False);
    popupwnd.lock = TRUE;
    gtk_widget_show_all(popupwnd.menu);
    gtk_menu_popup(GTK_MENU(popupwnd.menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
}

static void popupwnd_title_event(gboolean drag)
{
    static gboolean bdrag = FALSE;
    static gint sx, sy;
    gint x, y; 
    GdkModifierType mask;
    GdkWindow *root_win = gdk_window_foreign_new(GDK_ROOT_WINDOW());
    gdk_window_get_pointer(root_win, &x, &y, &mask);
    if(drag && bdrag)
    {
        gint wx, wy;
        gtk_window_get_position(GTK_WINDOW(popupwnd.wnd), &wx, &wy);
        gtk_window_move(GTK_WINDOW(popupwnd.wnd), wx + x - sx, wy + y - sy);
    }
    sx = x; sy = y;
    bdrag = drag;
}

static gint popupwnd_title_motion_notify_cb(GtkWidget *widget, GdkEventMotion *event)
{
    popupwnd_title_event(TRUE);
    return TRUE;
}

static gint popupwnd_title_button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    popupwnd_title_event(TRUE);
    return FALSE;
}

static gint popupwnd_title_button_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    popupwnd_title_event(FALSE);
    return FALSE;
}

static void popupwnd_scroll_mark(GtkTextBuffer *buffer, gboolean down)
{
    gint i, x, y, n, len, d;;
    gchar buf[16];
    GtkTextIter iter, iter1;
    GtkTextMark *mark;
    gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(popupwnd.view), GTK_TEXT_WINDOW_TEXT, 1, 1, &x, &y);
    gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(popupwnd.view), &iter, x, y);
    len = g_sequence_get_length(popupwnd.results);
    if(down)
    {
        n = 0; d = 1;
    }
    else
    {
        n = len - 1; d = -1;
    }
    for(i = n;; i += d)
    {
        sprintf(buf, "%d", i);
        mark = gtk_text_buffer_get_mark(buffer, buf);
        if(!mark)
            break;
        gtk_text_buffer_get_iter_at_mark(buffer, &iter1, mark);
        if(gtk_text_iter_compare(&iter, &iter1) == -d)
        {
            gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(popupwnd.view), mark, 0.0, TRUE, 0.0, 0.0);
            break;
        }
    }
}

static void popupwnd_scroll_down_cb(GtkWidget *widget, gpointer data)
{
    popupwnd_scroll_mark(gtk_text_view_get_buffer(GTK_TEXT_VIEW(popupwnd.view)), TRUE);
}

static void popupwnd_scroll_up_cb(GtkWidget *widget, gpointer data)
{
    popupwnd_scroll_mark(gtk_text_view_get_buffer(GTK_TEXT_VIEW(popupwnd.view)), FALSE);
}

static void popupwnd_window_create()
{
    GdkModifierType mask;
    gint pos_x, pos_y;
    gint pointer_x, pointer_y;
    gint root_x, root_y;

    GtkWidget *vbox, *hbox, *eventbox, *frame, *btn, *image, *scroll;

    if(popupwnd.remember_pos && (popupwnd.x != -1) && (popupwnd.y != -1))
    {
	pos_x = popupwnd.x;
	pos_y = popupwnd.y;
    }
    else
    {
	GdkWindow *root_win = gdk_window_foreign_new(GDK_ROOT_WINDOW());
	gdk_drawable_get_size(root_win, &root_x, &root_y);
	gdk_window_get_pointer(root_win, &pointer_x, &pointer_y, &mask);
	pos_x = pointer_x + 10;
	pos_y = pointer_y + 10;

	if(pos_x + popupwnd.w > root_x)
	    pos_x = root_x - popupwnd.w;
	
	if(pos_y + popupwnd.h > root_y)
	    pos_y = root_y - popupwnd.h;
    }

    popupwnd.wnd = gtk_widget_new(GTK_TYPE_WINDOW,
	    "type", GTK_WINDOW_POPUP,
	    "default-width", popupwnd.w,
	    "default-height", popupwnd.h,
	    NULL);
    gtk_window_move(GTK_WINDOW(popupwnd.wnd), pos_x, pos_y);
    g_signal_connect(G_OBJECT(popupwnd.wnd), "delete_event", G_CALLBACK(popupwnd_close), NULL);
    g_signal_connect(G_OBJECT(popupwnd.wnd), "leave_notify_event", G_CALLBACK(popupwnd_leave_notify), (gpointer)NULL);
    g_signal_connect(G_OBJECT(popupwnd.wnd), "enter_notify_event", G_CALLBACK(popupwnd_enter_notify), (gpointer)NULL);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(popupwnd.wnd), vbox);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
    gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(frame), hbox);

    eventbox = gtk_event_box_new();
    gtk_box_pack_start(GTK_BOX(hbox), eventbox, TRUE, TRUE, 2);
    g_signal_connect(G_OBJECT(eventbox), "button_press_event", G_CALLBACK(popupwnd_title_button_press_cb), (gpointer)NULL);
    g_signal_connect(G_OBJECT(eventbox), "motion_notify_event", G_CALLBACK(popupwnd_title_motion_notify_cb), (gpointer)NULL);
    g_signal_connect(G_OBJECT(eventbox), "button_release_event", G_CALLBACK(popupwnd_title_button_release_cb), (gpointer)NULL);
    popupwnd.label = gtk_label_new("");
    gtk_container_add(GTK_CONTAINER(eventbox), popupwnd.label);

    btn = toolbar_button(popupwnd_query_in_mainwnd, GTK_STOCK_FIND, _("Query in the main window"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    btn = toolbar_button(popupwnd_set_dict, GTK_STOCK_INDEX, _("Select dictionary"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    btn = toolbar_button(popupwnd_scroll_up_cb, GTK_STOCK_GO_BACK, _("Scroll up to the previous result"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    btn = toolbar_button(popupwnd_scroll_down_cb, GTK_STOCK_GO_FORWARD, _("Scroll down to the next result"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    popupwnd.lockbtn = gtk_toggle_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(popupwnd.lockbtn), popupwnd.lock);
    gtk_button_set_relief(GTK_BUTTON(popupwnd.lockbtn), GTK_RELIEF_NONE);
    g_signal_connect(G_OBJECT(popupwnd.lockbtn), "toggled", G_CALLBACK(popupwnd_lock), (gpointer)NULL);
    gtk_box_pack_start(GTK_BOX(hbox), popupwnd.lockbtn, FALSE, FALSE, 0);
    gtk_widget_set_tooltip_text(popupwnd.lockbtn, _("Lock popup window on the screen."));
    image = gtk_image_new_from_stock(GTK_STOCK_REFRESH, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_container_add(GTK_CONTAINER(popupwnd.lockbtn), image);

    btn = toolbar_button(popupwnd_close, GTK_STOCK_CLOSE, _("Close popup window"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
    gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

    scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(frame), scroll);

    popupwnd.view = g_object_new(EB_TYPE_TEXTVIEW, NULL);
    gtk_container_add(GTK_CONTAINER(scroll), GTK_WIDGET(popupwnd.view));

    gtk_widget_realize(popupwnd.wnd);
    gdk_window_set_decorations(popupwnd.wnd->window, 0);

    gtk_widget_show_all(popupwnd.wnd);
}

static gint popupwnd_scroll_to_top()
{
    GtkTextIter iter;
    GtkTextMark *mark;
    GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(popupwnd.view));
    gtk_text_buffer_get_start_iter(buf, &iter);
    mark =  gtk_text_buffer_create_mark(buf, "start", &iter, TRUE);
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(popupwnd.view), mark, 0.0, TRUE, 0.0, 0.0);
    gtk_text_buffer_delete_mark(buf, mark);
    return 0;
}

void popupwnd_open(RESULT *res)
{
    static int counter = 0;
    if(!res)
    {
        counter = 0;
        return;
    }
    eb_textview_open(popupwnd.view, res, FALSE, counter++);
    popupwnd_scroll_to_top();
}

void popupwnd_render(gpointer data, gpointer user_data)
{
    RESULT *res = (RESULT*)data;
    popupwnd_open(res);
}

void popupwnd_show(const gchar *word)
{
    eb_textview_clear_textbuf(popupwnd.view);
    popupwnd_open(NULL);
    if(g_sequence_get_length(popupwnd.results))
        g_sequence_foreach(popupwnd.results, popupwnd_render, NULL);
    else
        eb_textview_insert_message(popupwnd.view, "No hit.", TRUE);
    gtk_label_set_text(GTK_LABEL(popupwnd.label), word);
    gtk_window_present(GTK_WINDOW(popupwnd.wnd));
}

void popupwnd_search(const gchar *str)
{
    if(!str)
        str = gtk_label_get_text(GTK_LABEL(popupwnd.label));

    gchar *euc_str = iconv_convert(ENC_UTF8, ENC_EUC_JP, str);
    if(!validate_euc(euc_str))
    {
        g_free(euc_str);
        return;
    }
    if(!popupwnd.wnd)
    {
        popupwnd_window_create();
        popupwnd.binfo = NULL;
    }
    popupwnd_dict_menu_create(False);
    if(popupwnd.binfo)
        ebook_search_book(euc_str, popupwnd.search_method, popupwnd.results, popupwnd.maxhits, popupwnd.binfo);
    else
        ebook_search(euc_str, popupwnd.search_method, popupwnd.results, popupwnd.maxhits);
    g_free(euc_str);
    popupwnd_show(str);
}

