
#include "defs.h"

#include "audio.h"
#include "dictbar.h"
#include "dump.h"
#include "eb123.h"
#include "ebook.h"
#include "headword.h"
#include "history.h"
#include "mainwnd.h"
#include "render.h"
#include "textview.h"
#include "video.h"

G_DEFINE_TYPE(EbTextView, eb_textview, GTK_TYPE_TEXT_VIEW);

static void eb_textview_search_selection(GtkWidget *w, gpointer data)
{
    gchar *text;
    GtkTextIter start, end;
    GtkTextBuffer *buf = GTK_TEXT_BUFFER(data);

    gtk_text_buffer_get_selection_bounds(buf, &start, &end);
    text = gtk_text_buffer_get_text(buf, &start, &end, FALSE);
    mainwnd_search(text);
}

static GtkTextTag *eb_textview_get_link_under_cursor(EbTextView *self, gint ex, gint ey)
{
    gint        x, y;
    GSList      *tag_list = NULL;
    GtkTextIter iter;
    GtkTextTag  *res = NULL;

    GtkTextView *view = GTK_TEXT_VIEW(self);
    gtk_text_view_window_to_buffer_coords(view, GTK_TEXT_WINDOW_TEXT, ex, ey, &x, &y);
    gtk_text_view_get_iter_at_location(view, &iter, x, y);
    if((tag_list = gtk_text_iter_get_tags(&iter)))
    {
        GSList *tag_l = tag_list;
        while(tag_l)
        {
            GtkTextTag *tag = GTK_TEXT_TAG(tag_l->data);
            gpointer p1 = g_object_get_data(G_OBJECT(tag), "link");
            gpointer p2 = g_object_get_data(G_OBJECT(tag), "audio");
            gpointer p3 = g_object_get_data(G_OBJECT(tag), "video");
            if(p1 || p2 || p3)
            {
                res = tag;
                break;
            }
            tag_l = g_slist_next(tag_l);
        }
        g_slist_free(tag_list);
    }
    return res;
}

void eb_textview_underline_link(EbTextView *self, GdkEventMotion *event)
{
    GdkCursor   *cursor;
    if(self->tag)
        g_object_set(G_OBJECT(self->tag), "underline", PANGO_UNDERLINE_NONE, NULL);

    if(!event)
        return;

    self->tag = eb_textview_get_link_under_cursor(self, (gint)(event->x), (gint)(event->y));
    cursor = gdk_cursor_new(self->tag ? GDK_HAND2 : GDK_LEFT_PTR);
    gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(self), GTK_TEXT_WINDOW_TEXT), cursor);
    gdk_cursor_unref(cursor);
    if(self->tag)
        g_object_set(G_OBJECT(self->tag), "underline", PANGO_UNDERLINE_SINGLE, NULL);
}

gint eb_textview_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
    eb_textview_underline_link(EB_TEXTVIEW(widget), event);
    return !(event->state & GDK_BUTTON1_MASK);
}

gint eb_textview_leave_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
    GdkCursor *cursor = gdk_cursor_new(GDK_LEFT_PTR);
    EbTextView *view = EB_TEXTVIEW(widget);
    gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(widget), GTK_TEXT_WINDOW_TEXT), cursor);
    gdk_cursor_unref(cursor);
    eb_textview_underline_link(view, NULL);
    view->tag = NULL;
    return FALSE;
}

void eb_textview_populate_popup(GtkTextView *tview, GtkMenu *menu, gpointer user_data)
{
    GtkTextBuffer *buf = gtk_text_view_get_buffer(tview);
    GtkWidget *item = NULL;

    if(gtk_text_buffer_get_has_selection(buf))
    {
	item = gtk_separator_menu_item_new();
	gtk_widget_show(item);
	gtk_menu_shell_prepend(GTK_MENU_SHELL(menu), item);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_FIND, NULL);
	gtk_widget_show(item);
	gtk_menu_shell_prepend(GTK_MENU_SHELL(menu), item);
	g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(eb_textview_search_selection), (gpointer)buf);
    }
    item = gtk_separator_menu_item_new();
    gtk_widget_show(item);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    item = gtk_check_menu_item_new_with_label(_("Hex dump"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), dump.dlg[DUMP_HEX] != NULL);
    gtk_widget_show(item);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    dump.handler[DUMP_HEX] = g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(dump_hex), NULL);

    item = gtk_check_menu_item_new_with_label(_("Text dump"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), dump.dlg[DUMP_TEXT] != NULL);
    gtk_widget_show(item);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    dump.handler[DUMP_TEXT] = g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(dump_text), NULL);
}

gint eb_textview_button_press(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    EbTextView *view = EB_TEXTVIEW(widget);
    if((event->type == GDK_BUTTON_PRESS) && (event->button == 1))
    {
        GtkTextTag *tag = eb_textview_get_link_under_cursor(view, (gint)(event->x), (gint)(event->y));
        if(tag)
        {
            RESULT *res = (RESULT*)g_object_get_data(G_OBJECT(tag), "link");
	    if(res)
		eb_textview_open(view, res, TRUE, 0);
	    return TRUE;
        }
    }
    return FALSE;
}

void eb_textview_open(EbTextView *self, RESULT *res, gboolean clear, gint mark)
{
    if(!res)
        return;
    gchar *text = ebook_get_text(res);
    if(clear)
	eb_textview_clear_textbuf(self);
    else
    {
	gchar str[16];
	GtkTextIter iter;
	GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(self));
	gtk_text_buffer_get_end_iter(buf, &iter);
	if(gtk_text_buffer_get_char_count(buf))
	    gtk_text_buffer_insert(buf, &iter, "\n", -1);
	g_sprintf(str, "%d", mark);
	gtk_text_buffer_create_mark(buf, str, &iter, TRUE);
    }
    render_content(res->binfo, self, text);
    g_free(text);
    EbTextViewClass *klass = EB_TEXTVIEW_GET_CLASS(self);
    if(klass->history_next)
	klass->history_next(res);
}

void eb_textview_set_pixels(EbTextView *self, gint n)
{
    GtkTextView *tview = GTK_TEXT_VIEW(self);
    gtk_text_view_set_pixels_below_lines(tview, n);
    gtk_text_view_set_pixels_inside_wrap(tview, n);
}

static void eb_textview_class_init(EbTextViewClass *klass)
{
    klass->history_next = NULL;
}

static void eb_textview_init(EbTextView *self)
{
    g_signal_connect(G_OBJECT(self), "motion_notify_event", G_CALLBACK(eb_textview_motion_notify), NULL);
    g_signal_connect(G_OBJECT(self), "button_press_event", G_CALLBACK(eb_textview_button_press), NULL);
    g_signal_connect(G_OBJECT(self), "leave_notify_event", G_CALLBACK(eb_textview_leave_notify), NULL);
    gtk_widget_add_events(GTK_WIDGET(self), GDK_LEAVE_NOTIFY_MASK);
    
    gtk_text_view_set_editable(GTK_TEXT_VIEW(self), FALSE);
    gtk_text_view_set_left_margin(GTK_TEXT_VIEW(self), 10);
    gtk_text_view_set_right_margin(GTK_TEXT_VIEW(self), 10);
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(self), GTK_WRAP_WORD);
    gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(self), FALSE);
}

void eb_textview_clear_textbuf(EbTextView *self)
{
    GtkTextIter start, end;
    GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(self));
    gtk_text_buffer_get_bounds(buf, &start, &end);
    gtk_text_buffer_delete(buf, &start, &end);
}

void eb_textview_insert_message(EbTextView *self, gchar *msg, gboolean clear)
{
    GtkTextIter iter;
    if(clear)
	eb_textview_clear_textbuf(self);
    GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(self));
    gtk_text_buffer_get_end_iter(buf, &iter);
    gtk_text_buffer_insert(buf, &iter, msg, -1);
}

