/**************************************************************************
 * Copyright (C) 2008 Cocha                                               *
 * http://sourceforge.jp/projects/ecodecotool/                            *
 *                                                                        *
 *  This Program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2, or (at your option)   *
 *  any later version.                                                    *
 *                                                                        *
 *  This Program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with GNU Make; see the file COPYING.  If not, write to          *
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. *
 *                                                                        *
 **************************************************************************/

#include "EcoDecoVorbis.h"

#define PACKET_HEADER_BIT            0x01
#define PACKET_HEADER_IDENTIFICATION 0x01
#define PACKET_HEADER_COMMENT        0x03
#define PACKET_HEADER_SETUP          0x05
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CaoTuVDec::GetInFormat(void *pFormat)
{  // ̓s̃tH[}bg擾

   CheckPointer(m_pInput, E_POINTER);
   CheckPointer(pFormat, E_POINTER);

   if(m_pInput->IsConnected() == FALSE)
      return VFW_E_NOT_CONNECTED;

   ::CopyMemory(pFormat, &m_inFormat, sizeof(m_inFormat));
   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CaoTuVDec::GetOutFormat(WAVEFORMATEX *pwf)
{  // o̓s̃tH[}bg擾

   CheckPointer(m_pInput, E_POINTER);
   CheckPointer(pwf, E_POINTER);

   ::CopyMemory(pwf, &m_outFormat, sizeof(m_outFormat));
   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CaoTuVDec::SetOutBitsPerSample(WORD wFormatTag, WORD wBitsPerSample)
{  // o͂̃rbg[gݒ肷

   CheckPointer(m_pInput, E_POINTER);

   if(m_pOutput->IsConnected() == TRUE)
      return VFW_E_ALREADY_CONNECTED;

   if(wFormatTag == WAVE_FORMAT_PCM)
   {
      if(wBitsPerSample != 8 && wBitsPerSample != 16 && wBitsPerSample != 24 && wBitsPerSample != 32)
         return E_FAIL;
   }
   else if(wFormatTag == WAVE_FORMAT_IEEE_FLOAT)
   {
      if(wBitsPerSample != 32 && wBitsPerSample != 64)
         return E_FAIL;
   }
   else
      return E_FAIL;

   m_outFormat.wFormatTag = wFormatTag;
   m_outFormat.wBitsPerSample = wBitsPerSample;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
CaoTuVDec::CaoTuVDec(IUnknown *pUnk, HRESULT *phr) :
   CTransformFilter(FILTER_AOTUVDEC_NAME, pUnk, CLSID_aoTuvDec)
{  // RXgN^

   // ϐ̏
   ::ZeroMemory(&m_inFormat, sizeof(m_inFormat));
   ::ZeroMemory(&m_outFormat, sizeof(m_outFormat));

   m_outFormat.wFormatTag = WAVE_FORMAT_PCM;
   m_outFormat.wBitsPerSample = 16;
}
// -----------------------------------------------------------------------------------------------------------------------------------
CaoTuVDec::~CaoTuVDec()
{  // fXgN^

}
// -----------------------------------------------------------------------------------------------------------------------------------
void CaoTuVDec::HeaderInitialize(void)
{
   vorbis_info_init(&m_vi);
   vorbis_comment_init(&m_vc);
}
// -----------------------------------------------------------------------------------------------------------------------------------
void CaoTuVDec::BlockInitialize(void)
{
   vorbis_synthesis_init(&m_vd, &m_vi);
   vorbis_block_init(&m_vd, &m_vb);
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::CheckInputType(const CMediaType *mtIn)
{  // ̓s󂯓邩`FbN

   CAutoLock cObjectLock(m_pLock);

   if(*mtIn->Subtype() == MEDIASUBTYPE_Vorbis && *mtIn->FormatType() == FORMAT_VorbisFormat)
   {
      ::CopyMemory(&m_inFormat, mtIn->pbFormat, sizeof(m_inFormat));

      // ̓sڑ́Ao̓tH[}bg̓tH[}bgƓɂĂ
      m_outFormat.nSamplesPerSec = m_inFormat.nSamplesPerSec;
      m_outFormat.nChannels = m_inFormat.nChannels;
      m_outFormat.cbSize = 0;
      m_outFormat.nBlockAlign = m_outFormat.nChannels * (m_outFormat.wBitsPerSample / 8);
      m_outFormat.nAvgBytesPerSec = m_outFormat.nSamplesPerSec * m_outFormat.nBlockAlign;

      return S_OK;
   }

   return S_FALSE;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::GetMediaType(int iPosition, CMediaType *pOutMediaType)
{  // o̓s̃fBA^Cvݒ肷

   CheckPointer(m_pInput, E_POINTER);

   if(iPosition < 0)
      return E_INVALIDARG;

   if(iPosition > 0)
      return VFW_S_NO_MORE_ITEMS;

   pOutMediaType->SetType(&MEDIATYPE_Audio);
   pOutMediaType->SetSubtype(&MEDIASUBTYPE_PCM);

   if (m_pInput->IsConnected() == FALSE)
   {
      pOutMediaType->SetFormatType(&FORMAT_None);
      return S_OK;
   }

   m_outFormat.nSamplesPerSec = m_inFormat.nSamplesPerSec;
   m_outFormat.nChannels = m_inFormat.nChannels;
   m_outFormat.cbSize = 0;
   m_outFormat.nBlockAlign = m_outFormat.nChannels * (m_outFormat.wBitsPerSample / 8);
   m_outFormat.nAvgBytesPerSec = m_outFormat.nSamplesPerSec * m_outFormat.nBlockAlign;

   pOutMediaType->SetType(&MEDIATYPE_Audio);
   pOutMediaType->SetSubtype(&MEDIASUBTYPE_PCM);
   pOutMediaType->SetTemporalCompression(FALSE);
   pOutMediaType->SetSampleSize(0);
   pOutMediaType->SetFormatType(&FORMAT_WaveFormatEx);
   pOutMediaType->SetFormat((BYTE*)&m_outFormat, sizeof(m_outFormat));

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::CheckTransform(const CMediaType *mtIn, const CMediaType *mtOut)
{  // ̓fBA^Cvo̓fBA^Cvƌ݊邩`FbN

   CheckPointer(mtIn, E_POINTER);
   CheckPointer(mtOut, E_POINTER);

   if(mtIn->majortype == MEDIATYPE_Audio && mtIn->subtype == MEDIASUBTYPE_Vorbis && mtIn->formattype == FORMAT_VorbisFormat)
   {
      if(mtOut->majortype == MEDIATYPE_Audio && mtOut->subtype == MEDIASUBTYPE_PCM && mtOut->formattype == FORMAT_WaveFormatEx)
         return S_OK;
   }

   return VFW_E_TYPE_NOT_ACCEPTED;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::DecideBufferSize(IMemAllocator *pAlloc, ALLOCATOR_PROPERTIES *pProperties)
{  // o̓s̃obt@vݒ肷

   CheckPointer(pAlloc, E_POINTER);
   CheckPointer(pProperties, E_POINTER);
   CheckPointer(m_pInput, E_FAIL);

   if(m_pInput->IsConnected()==FALSE)
      return E_FAIL;

   HRESULT hr;

   // o͉̐ݒ
   // igpeditŃ_Oςgrft@CǂݍނƁAGetMediaTypeĂ΂Ȃj
   m_outFormat.nSamplesPerSec = m_inFormat.nSamplesPerSec;
   m_outFormat.nChannels = m_inFormat.nChannels;
   m_outFormat.cbSize = 0;
   m_outFormat.nBlockAlign = m_outFormat.nChannels * (m_outFormat.wBitsPerSample / 8);
   m_outFormat.nAvgBytesPerSec = m_outFormat.nSamplesPerSec * m_outFormat.nBlockAlign;

   // o͂̐ݒ
   pProperties->cbBuffer = 4096 * m_outFormat.nBlockAlign;
   pProperties->cBuffers = 1;
   pProperties->cbAlign = 1;
   pProperties->cbPrefix = 0;

   // AP[^͗vɑ΂ĐmɈvłƂ͌Ȃ߁AۂɊmۂꂽ`FbN
   ALLOCATOR_PROPERTIES Actual;
   hr = pAlloc->SetProperties(pProperties, &Actual);
   if(FAILED(hr)) return hr;

   if( (pProperties->cBuffers > Actual.cBuffers) || (pProperties->cbBuffer > Actual.cbBuffer) )
       return E_FAIL;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::StartStreaming(void)
{  // Xg[~OJnɌĂ΂

   CAutoLock cObjectLock(m_pLock);

   m_bHeaderInitialized = false;;
   m_bBlockInitialized = false;;

   m_bUseRatio = false;
   m_dGain = 1.0;
   m_nPacketNumber = 0;
   pBitConv = NULL;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
void CaoTuVDec::ReceiveHeader(BYTE *pPacketData, int nPacketDataLength)
{
   ogg_packet op;

   if(pPacketData[0] == PACKET_HEADER_IDENTIFICATION)
   {  // mFwb_̏ꍇ
      op.b_o_s = 1;
      op.e_o_s = 0;
      op.granulepos = 0;
      op.packetno = m_nPacketNumber;
      op.packet = pPacketData;
      op.bytes = nPacketDataLength;
      vorbis_synthesis_headerin(&m_vi, &m_vc, &op);

      m_nPacketNumber++;
   }
   else if(pPacketData[0] == PACKET_HEADER_COMMENT)
   { // Rgwb_̏ꍇ
      op.b_o_s = 0;
      op.e_o_s = 0;
      op.granulepos = 0;
      op.packetno = m_nPacketNumber;
      op.packet = pPacketData;
      op.bytes = nPacketDataLength;
      vorbis_synthesis_headerin(&m_vi, &m_vc, &op);

      m_nPacketNumber++;
   }
   else if(pPacketData[0] == PACKET_HEADER_SETUP)
   { // ݒwb_̏ꍇ
      op.b_o_s = 0;
      op.e_o_s = 0;
      op.granulepos = 0;
      op.packetno = m_nPacketNumber;
      op.packet = pPacketData;
      op.bytes = nPacketDataLength;
      vorbis_synthesis_headerin(&m_vi, &m_vc, &op);

      m_nPacketNumber++;
   }
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::Decode(IMediaSample *pInSample, bool bEos)
{  // ϊ

   int vr, nSampleSize, nTotalSampleSize, nActualDataLength;
   HRESULT hr;
   ogg_packet op;
   float **ppInBuffer;
   BYTE *pbOutBuffer = NULL;
   IMediaSample *pOutSample = NULL;
   REFERENCE_TIME rtStart, rtStop;

   pInSample->GetPointer(&op.packet);
   op.bytes = pInSample->GetActualDataLength();

   if(m_bHeaderInitialized == false)
   {  // vorbis_commentvorbis_infȍ
      HeaderInitialize();
      m_bHeaderInitialized = true;
   }

   if(op.packet[0] & PACKET_HEADER_BIT)
   {  // pPbgwb_̏ꍇ
      ReceiveHeader(op.packet, op.bytes);
      return S_OK;
   }

   if(m_bBlockInitialized == false)
   {
      // QC񂪂邩`FbN
      if(vorbis_comment_query_count(&m_vc, "LWING_GAIN"))
      {
         m_bUseRatio = true;
         m_dGain = ::atof(vorbis_comment_query(&m_vc, "LWING_GAIN", 0));
      }
      else if(vorbis_comment_query_count(&m_vc, "POSTGAIN"))
      {
         m_bUseRatio = true;
         m_dGain = ::atof(vorbis_comment_query(&m_vc, "POSTGAIN", 0));
      }

      // pBitConv̏
      pBitConv = new CBitConv();
      pBitConv->Initialize(&m_outFormat, m_dGain);

      // vorbis_blockvorbis_dsp_statȅ
      BlockInitialize();
      m_bBlockInitialized = true;
   }

   // fBAԂ擾
   hr = pInSample->GetTime(&rtStart, &rtStop);
   if FAILED(hr) return hr;

   // ogg_packet̐ݒ
   op.b_o_s = 0;

   if(bEos == false)
      op.e_o_s = 0;
   else
      op.e_o_s = 1;

   op.packetno = m_nPacketNumber;
   m_nPacketNumber++;

   // t[̈ʒuݒ(t[ʒuɂȂꍇ́Aľܓ)
   op.granulepos = (ogg_int64_t)( ((double)rtStart * m_vi.rate / 10000000.0) + 0.5); // rtStart̒Pʂ100imb(rtStart/10000000ŒPʂbɂȂ)

   vr = vorbis_synthesis(&m_vb, &op);
   if(vr >= 0)
      vorbis_synthesis_blockin(&m_vd, &m_vb);

   if(vorbis_synthesis_pcmout(&m_vd, NULL) == 0)
      return S_OK; // No pending samples

   // o͂̏
   hr = m_pOutput->GetDeliveryBuffer(&pOutSample, NULL, NULL, 0);
   if FAILED(hr)
   {  // o̓obt@̍쐬Ɏsꍇ

      while(true)
      {
         nSampleSize = vorbis_synthesis_pcmout(&m_vd, &ppInBuffer);
         if(nSampleSize <= 0)
            break;

         vorbis_synthesis_read(&m_vd, nSampleSize);
      }

      return hr;
   }

   pOutSample->GetPointer(&pbOutBuffer);

   nTotalSampleSize = 0;
   nActualDataLength = 0;

   while(true)
   {  // fR[h
      nSampleSize = vorbis_synthesis_pcmout(&m_vd, &ppInBuffer);
      if(nSampleSize <= 0)
         break;

      if( (nActualDataLength + nSampleSize * m_outFormat.nBlockAlign) < (4096 * m_outFormat.nBlockAlign))
      {
         if(m_outFormat.wFormatTag == WAVE_FORMAT_PCM)
         {
            if(m_outFormat.wBitsPerSample == 16)
               pBitConv->ConvertTo16I((short *)pbOutBuffer, ppInBuffer, nSampleSize);
            else if(m_outFormat.wBitsPerSample == 8)
               pBitConv->ConvertTo8I(pbOutBuffer, ppInBuffer, nSampleSize);
            else if(m_outFormat.wBitsPerSample == 24)
               pBitConv->ConvertTo24I(pbOutBuffer, ppInBuffer, nSampleSize);
            else // if(m_outFormat.wBitsPerSample == 32)
               pBitConv->ConvertTo32I((int *)pbOutBuffer, ppInBuffer, nSampleSize);
         }
         else // if(m_outFormat.wFormatTag == WAVE_FORMAT_IEEE_FLOAT)
         {
            if(m_outFormat.wBitsPerSample == 32)
               pBitConv->ConvertTo32F((float *)pbOutBuffer, ppInBuffer, nSampleSize);
            else // if(m_outFormat.wBitsPerSample == 64)
               pBitConv->ConvertTo64D((double *)pbOutBuffer, ppInBuffer, nSampleSize);
         }

         nActualDataLength += (nSampleSize * m_outFormat.nBlockAlign);
      }

      nTotalSampleSize += nSampleSize;
      vorbis_synthesis_read(&m_vd, nSampleSize);
   }

   if(nActualDataLength > pOutSample->GetSize())
   {
      SAFE_RELEASE(pOutSample);
      return VFW_E_BUFFER_OVERFLOW;
   }
   
   pOutSample->SetSyncPoint(TRUE);
   pOutSample->SetDiscontinuity(FALSE);
   pOutSample->SetPreroll(FALSE);
   pOutSample->SetActualDataLength(nActualDataLength);

   // t[݈̌ʒu玞Ԃ擾(؂Ȃꍇ́Aľܓ)
   rtStart = (LONGLONG)( (((double)m_vd.granulepos-(ogg_int64_t)nTotalSampleSize) * 10000000.0 / m_vi.rate) + 0.5);  // rtStart̒Pʂ100imb(b100imbɕϊɂ10000000)
   rtStop  = (LONGLONG)( ((double)m_vd.granulepos * 10000000.0 /m_vi.rate) + 0.5);

   // fBAԂ̐ݒ
   pOutSample->SetTime(&rtStart, &rtStop);

   // o̓f[^̒ݒ
   pOutSample->SetActualDataLength(nActualDataLength);

   // f[^o
   hr = m_pOutput->Deliver(pOutSample);

   // ㏈
   SAFE_RELEASE(pOutSample);

   return hr;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::Receive(IMediaSample *pInSample)
{
   CAutoLock lck(&m_myLock);

   // AM_STREAM_MEDIAȊOȂʉ߂
   AM_SAMPLE2_PROPERTIES * const pProps = m_pInput->SampleProps();
   if(pProps->dwStreamId != AM_STREAM_MEDIA)
      return m_pOutput->Deliver(pInSample);

   HRESULT hr = S_OK;

   hr = Decode(pInSample, false);

	return hr;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::StopStreaming()
{  // ~ꍇɌĂ΂

   CAutoLock lck(&m_myLock);

   // ㏈
   if(m_bBlockInitialized == true)
   {
      vorbis_block_clear(&m_vb);
      vorbis_dsp_clear(&m_vd);

      m_bBlockInitialized = false;
   }

   if(m_bHeaderInitialized == true)
   {
      vorbis_comment_clear(&m_vc);
      vorbis_info_clear(&m_vi);

      m_bHeaderInitialized = false;
   }

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::BeginFlush()
{
   HRESULT hr;

   CAutoLock lck(&m_myLock);

   hr = CTransformFilter::BeginFlush();

   // ㏈
   if(m_bBlockInitialized == true)
   {
      vorbis_block_clear(&m_vb);
      vorbis_dsp_clear(&m_vd);

      m_bBlockInitialized = false;
   }

   if(m_bHeaderInitialized == true)
   {
      vorbis_comment_clear(&m_vc);
      vorbis_info_clear(&m_vi);

      m_bHeaderInitialized = false;
   }

	return hr;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CaoTuVDec::EndOfStream(void)
{  // Xg[~OIɌĂ΂

   CAutoLock lck(&m_myLock);

   // ㏈
   if(m_bBlockInitialized == true)
   {
      vorbis_block_clear(&m_vb);
      vorbis_dsp_clear(&m_vd);

      m_bBlockInitialized = false;
   }

   if(m_bHeaderInitialized == true)
   {
      vorbis_comment_clear(&m_vc);
      vorbis_info_clear(&m_vi);

      m_bHeaderInitialized = false;
   }

   return CTransformFilter::EndOfStream();
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CaoTuVDec::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{  // C^[tFCX擾AQƃJEgCNg

   CheckPointer(ppv, E_POINTER);
   CAutoLock cObjectLock(m_pLock);

   if(riid == IID_IaoTuvDecInterface)
   {
      return GetInterface((IaoTuvDecInterface *)this, ppv);
   }

   return CTransformFilter::NonDelegatingQueryInterface(riid, ppv);
}
// -----------------------------------------------------------------------------------------------------------------------------------
CUnknown * WINAPI CaoTuVDec::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{  // IuWFNg쐬

   ASSERT(phr);
    
   CaoTuVDec *pNewObject = new CaoTuVDec(punk, phr);
   if (pNewObject == NULL)
   {
      if(phr != NULL)
         *phr = E_OUTOFMEMORY;
   }

   return dynamic_cast<CUnknown *>(pNewObject);
}
// -----------------------------------------------------------------------------------------------------------------------------------
