///////////////////////////////////////////////////////////////////////////////
// Table: bitrate_tbl
// Used in: parse_header()
// Memonics: bitrate_tbl[ver][layer][bitrate_index]
// Returns: bitrate in kbps
// Desc:
//   'ver' is MPEG version: 0 = MPEG1; 1 = MPEG2

const int bitrate_tbl[2][3][15] =
{
  { // MPEG1
    {0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448 },
    {0, 32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384 },
    {0, 32, 40, 48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320 }
  }, 
  { // MPEG2 LSF
    {0, 32, 48, 56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256 },
    {0,  8, 16, 24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160 },
    {0,  8, 16, 24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160 }
  }
};

///////////////////////////////////////////////////////////////////////////////
// Table: freq_tbl
// Used in: parse_header()
// Memonics: freq_tbl[ver][freq_index]
// Returns: sampling frequency in Hz
// Desc:
//   'ver' is MPEG version: 0 = MPEG1; 1 = MPEG2

const int freq_tbl[2][3] =
{
  {44100, 48000, 32000},  // MPEG1
  {22050, 24000, 16000}   // MPEG2 LSF
};

///////////////////////////////////////////////////////////////////////////////
// Table: slots_tbl
// Used in: parse_header()
// Memonics: slots_tbl[layer]
// Returns: frame size factor
// Desc:
//   Used to calculate frame size (in slots):
//   frame_size = slots_tbl[layer] * bitrate / sample_rate

const int slots_tbl[3]      = {12, 144, 144};

///////////////////////////////////////////////////////////////////////////////
// Table: jsbound_tbl
// Used in: parse_header()
// Memonics: jsbound[layer][mode_ext]
// Returns: joint stereo subband bound
// Desc:
//   only in joint stereo mode (mode = 01)

const int jsbound_tbl[3][4] =
{ 
  { 4, 8, 12, 16 }, 
  { 4, 8, 12, 16 },
  { 0, 4,  8, 16 } 
};

///////////////////////////////////////////////////////////////////////////////
// LayerII tables
///////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////
// Table: II_half_bitrate_tbl
// Used in: parse_header()
// Memonics: II_half_bitrate_tbl[bitrate_index]
// Returns: bitrate index corresponding to half-bitrate
// Desc:
//   Bitrate-per-channel in 2-channel modes. Used to determine
//   allocation table.

const int II_half_bitrate_tbl[15] =
  { 0, 1, 1, 1, 1, 1, 2, 3, 4, 5, 6, 6, 6, 6, 6 };

///////////////////////////////////////////////////////////////////////////////
// Table: II_table_tbl
// Used in: parse_header()
// Memonics: II_table_tbl[freq_index][bitrate_index]
// Returns: allocation table number
// Desc:

const int II_table_tbl[3][15] =
{// v free format             unused vvvvvvvvvv
  { 1, 2, 2, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0 },
  { 0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
  { 1, 3, 3, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0 }
};

///////////////////////////////////////////////////////////////////////////////
// Table: II_sblimit_tbl
// Used in: parse_header()
// Memonics: II_sblimit_tbl[table]
// Returns: number of subbands
// Desc:

const int II_sblimit_tbl[5] = { 27, 30, 8, 12, 30 };  // II_sblimit_tbl[table]

///////////////////////////////////////////////////////////////////////////////
// Table: scale_tbl
// Used in: I_decode_frame()  (decode scalefactors)
//          II_decode_frame() (decode scalefactors)
// Memonics: scale = scale_tbl[scale_index]
// Returns: scale factor value
// Desc:
//   FFT is a linear operation so we can pre-scale frequency coefficients
//   up to output level

const sample_t scale_tbl[64] = 
{                                                                                                                                                      
  2.00000000000000, 1.58740105196820, 1.25992104989487,
  1.00000000000000, 0.79370052598410, 0.62996052494744, 0.50000000000000,
  0.39685026299205, 0.31498026247372, 0.25000000000000, 0.19842513149602,
  0.15749013123686, 0.12500000000000, 0.09921256574801, 0.07874506561843,
  0.06250000000000, 0.04960628287401, 0.03937253280921, 0.03125000000000,
  0.02480314143700, 0.01968626640461, 0.01562500000000, 0.01240157071850,
  0.00984313320230, 0.00781250000000, 0.00620078535925, 0.00492156660115,
  0.00390625000000, 0.00310039267963, 0.00246078330058, 0.00195312500000,
  0.00155019633981, 0.00123039165029, 0.00097656250000, 0.00077509816991,
  0.00061519582514, 0.00048828125000, 0.00038754908495, 0.00030759791257,
  0.00024414062500, 0.00019377454248, 0.00015379895629, 0.00012207031250,
  0.00009688727124, 0.00007689947814, 0.00006103515625, 0.00004844363562,
  0.00003844973907, 0.00003051757813, 0.00002422181781, 0.00001922486954,
  0.00001525878906, 0.00001211090890, 0.00000961243477, 0.00000762939453,
  0.00000605545445, 0.00000480621738, 0.00000381469727, 0.00000302772723,
  0.00000240310869, 0.00000190734863, 0.00000151386361, 0.00000120155435,
  1E-20f
};                  

/*
// pre-scaled
sample_t scale_tbl[64] = 
{                                                                                                                                                      
  (sample_t)(2.00000000000000 * SCALE), (sample_t)(1.58740105196820 * SCALE), (sample_t)(1.25992104989487 * SCALE),
  (sample_t)(1.00000000000000 * SCALE), (sample_t)(0.79370052598410 * SCALE), (sample_t)(0.62996052494744 * SCALE), (sample_t)(0.50000000000000 * SCALE),
  (sample_t)(0.39685026299205 * SCALE), (sample_t)(0.31498026247372 * SCALE), (sample_t)(0.25000000000000 * SCALE), (sample_t)(0.19842513149602 * SCALE),
  (sample_t)(0.15749013123686 * SCALE), (sample_t)(0.12500000000000 * SCALE), (sample_t)(0.09921256574801 * SCALE), (sample_t)(0.07874506561843 * SCALE),
  (sample_t)(0.06250000000000 * SCALE), (sample_t)(0.04960628287401 * SCALE), (sample_t)(0.03937253280921 * SCALE), (sample_t)(0.03125000000000 * SCALE),
  (sample_t)(0.02480314143700 * SCALE), (sample_t)(0.01968626640461 * SCALE), (sample_t)(0.01562500000000 * SCALE), (sample_t)(0.01240157071850 * SCALE),
  (sample_t)(0.00984313320230 * SCALE), (sample_t)(0.00781250000000 * SCALE), (sample_t)(0.00620078535925 * SCALE), (sample_t)(0.00492156660115 * SCALE),
  (sample_t)(0.00390625000000 * SCALE), (sample_t)(0.00310039267963 * SCALE), (sample_t)(0.00246078330058 * SCALE), (sample_t)(0.00195312500000 * SCALE),
  (sample_t)(0.00155019633981 * SCALE), (sample_t)(0.00123039165029 * SCALE), (sample_t)(0.00097656250000 * SCALE), (sample_t)(0.00077509816991 * SCALE),
  (sample_t)(0.00061519582514 * SCALE), (sample_t)(0.00048828125000 * SCALE), (sample_t)(0.00038754908495 * SCALE), (sample_t)(0.00030759791257 * SCALE),
  (sample_t)(0.00024414062500 * SCALE), (sample_t)(0.00019377454248 * SCALE), (sample_t)(0.00015379895629 * SCALE), (sample_t)(0.00012207031250 * SCALE),
  (sample_t)(0.00009688727124 * SCALE), (sample_t)(0.00007689947814 * SCALE), (sample_t)(0.00006103515625 * SCALE), (sample_t)(0.00004844363562 * SCALE),
  (sample_t)(0.00003844973907 * SCALE), (sample_t)(0.00003051757813 * SCALE), (sample_t)(0.00002422181781 * SCALE), (sample_t)(0.00001922486954 * SCALE),
  (sample_t)(0.00001525878906 * SCALE), (sample_t)(0.00001211090890 * SCALE), (sample_t)(0.00000961243477 * SCALE), (sample_t)(0.00000762939453 * SCALE),
  (sample_t)(0.00000605545445 * SCALE), (sample_t)(0.00000480621738 * SCALE), (sample_t)(0.00000381469727 * SCALE), (sample_t)(0.00000302772723 * SCALE),
  (sample_t)(0.00000240310869 * SCALE), (sample_t)(0.00000190734863 * SCALE), (sample_t)(0.00000151386361 * SCALE), (sample_t)(0.00000120155435 * SCALE),
  (sample_t)1E-20f
};                  
*/

///////////////////////////////////////////////////////////////////////////////
// Table: c_tbl; d_tbl
// Used in: II_decode_frame() (decode scalefactors)
//          II_decode_fraction() (dequantize)
// Memonics: c = c_tbl[bits]; d = d_tbl[bits]
// Returns: 'c' & 'd' factors in dequantize equation
// Desc:
//   dequanize equation: s' = (s + d) * c * scale_factor
//   'c' & 'd' factors depend on quantization level that in order depends on 
//   number of bits allocated for sample. This tables give factors for
//   bits >= 3. For bits < 0 (sample triplets) factors are placed in this
//   order:
//     c_tbl[0]; d_tbl[0] - bits = -5   (3 quantization levels)
//     c_tbl[1]; d_tbl[1] - bits = -7   (5 quantization levels)
//     c_tbl[2]; d_tbl[2] - bits = -10  (9 quantization levels)
//
//   'c' factor is scaled to produce floating point value in range (-1; 1)
//   from 16bit left-aligned fixed point value (i.e. 0.1011...)

// pre-scaled
const sample_t c_tbl[17]  = 
{
  (sample_t)(1.33333333333 / 32768.0), (sample_t)(1.60000000000 / 32768.0), (sample_t)(1.77777777777 / 32768.0), 
  (sample_t)(1.14285714286 / 32768.0), (sample_t)(1.06666666666 / 32768.0), (sample_t)(1.03225806452 / 32768.0),
  (sample_t)(1.01587301587 / 32768.0), (sample_t)(1.00787401575 / 32768.0), (sample_t)(1.00392156863 / 32768.0),
  (sample_t)(1.00195694716 / 32768.0), (sample_t)(1.00097751711 / 32768.0), (sample_t)(1.00048851979 / 32768.0),
  (sample_t)(1.00024420024 / 32768.0), (sample_t)(1.00012208522 / 32768.0), (sample_t)(1.00006103888 / 32768.0),
  (sample_t)(1.00003051851 / 32768.0), (sample_t)(1.00001525902 / 32768.0) 
};

const int d_tbl[17]    = 
{
  0x4000, 0x4000, 0x4000, 0x2000, 
  0x1000, 0x0800, 0x0400, 0x0200,
  0x0100, 0x0080, 0x0040, 0x0020,
  0x0010, 0x0008, 0x0004,
  0x0002, 0x0001 
};


///////////////////////////////////////////////////////////////////////////////
// Table: II_ba_bits_tbl
// Used in: II_decode_frame() (bit allocation)
// Memonics: ba_bits = II_ba_bits[table][sb]
// Returns: number of bits for bit allocation
// Desc:
//   Number of bits for bit allocation information depends on allocatin table
//   and subband number

const int16_t II_ba_bits_tbl[5][SBLIMIT] =
{
//  0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31
  { 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 0, 0, 0, 0, 0 }, // table 0
  { 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 0, 0 }, // table 1
  { 4, 4, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, // table 2
  { 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, // table 3
  { 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0 }  // table 5 (MPEG2 LSF)
};

///////////////////////////////////////////////////////////////////////////////
// Table: II_ba_tbl
// Used in: II_decode_frame() (bit allocation)
// Memonics: ba = II_ba[table][sb][ba_index]
// Returns: number of bits allocated for sample
// Desc:
//   Number of bits allocated for sample depends on allocation table, 
//   subband number and bit allocation index read from bitstream at 
//   bit allocation. It also defines quantization level as:
//     qlevels = 2^bits - 1 for bits > 0
//     qlevels = 3 for bits = -5
//     qlevels = 5 for bits = -7
//     qlevels = 9 for bits = -10

const int16_t II_ba_tbl[5][SBLIMIT][16] = 
{
  { //table 0
    { 0,  -5,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16  }, // subband 0
    { 0,  -5,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16  }, // subband 1
    { 0,  -5,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16  }, // subband 2
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 3
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 4
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 5
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 6
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 7
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 8
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 9
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 10
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 11
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 12
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 13
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 14
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 15
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 16
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 17
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 18
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 19
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 20
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 21
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 22
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 23
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 24
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 25
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 26
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 27
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 28
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 29
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 30
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 31
  },
  { //table 1
    { 0,  -5,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16  }, // subband 0
    { 0,  -5,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16  }, // subband 1
    { 0,  -5,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16  }, // subband 2
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 3
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 4
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 5
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 6
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 7
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 8
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 9
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  16  }, // subband 10
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 11
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 12
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 13
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 14
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 15
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 16
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 17
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 18
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 19
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 20
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 21
    { 0,  -5,  -7,   3, -10,   4,   5,  16,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 22
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 23
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 24
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 25
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 26
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 27
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 28
    { 0,  -5,  -7,  16,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 29
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 30
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 31
  },
  { //table 2
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15  }, // subband 0
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15  }, // subband 1
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 2
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 3
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 4
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 5
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 6
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 7
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 8
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 9
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 10
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 11
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 12
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 13
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 14
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 15
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 16
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 17
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 18
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 19
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 20
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 21
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 22
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 23
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 24
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 25
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 26
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 27
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 28
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 29
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 30
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 31
  },
  { //table 3
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15  }, // subband 0
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15  }, // subband 1
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 2
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 3
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 4
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 5
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 6
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 7
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 8
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 9
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 10
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 11
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 12
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 13
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 14
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 15
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 16
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 17
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 18
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 19
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 20
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 21
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 22
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 23
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 24
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 25
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 26
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 27
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 28
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 29
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 30
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 31
  },
  { //table 5 (MPEG2 LSF)
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14  }, // subband 0
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14  }, // subband 1
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14  }, // subband 2
    { 0,  -5,  -7,   3, -10,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14  }, // subband 3
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 4
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 5
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 6
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 7
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 8
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 9
    { 0,  -5,  -7, -10,   4,   5,   6,   7,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 10
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 11
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 12
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 13
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 14
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 15
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 16
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 17
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 18
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 19
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 20
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 21
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 22
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 23
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 24
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 25
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 26
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 27
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 28
    { 0,  -5,  -7, -10,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 29
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 30
    { 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0  }, // subband 31
  }
};
