/**************************************************************************
 * Copyright (C) 2008 Cocha                                               *
 * http://sourceforge.jp/projects/ecodecotool/                            *
 *                                                                        *
 *  This Program is free software; you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation; either version 2, or (at your option)   *
 *  any later version.                                                    *
 *                                                                        *
 *  This Program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with GNU Make; see the file COPYING.  If not, write to          *
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. *
 *                                                                        *
 **************************************************************************/

#include "EcoDecoVorbis.h"

CEncodeVorbis::CEncodeVorbis(IUnknown *pUnk, HRESULT *phr)
   : CSimpleWriter(L"OggVorbis Writer", pUnk, CLSID_EncodeVorbis, phr)
{  

   ::ZeroMemory(&m_inFormat, sizeof(m_inFormat));
   ::ZeroMemory(&m_outFormat, sizeof(m_outFormat));
   m_bInitialize = false;

   m_nWait = 0;
   m_llInputSize = 0;
}
CEncodeVorbis::~CEncodeVorbis()  
{  

   if(m_bInitialize == true)
   {
      ogg_stream_clear(&m_os);
      vorbis_block_clear(&m_vb);
      vorbis_dsp_clear(&m_vd);
      vorbis_comment_clear(&m_vc);
      vorbis_info_clear(&m_vi);

      m_bInitialize = false;
   }
}  
HRESULT CEncodeVorbis::SetWait(int nWait)
{
   CAutoLock lock(&m_Lock);

   m_nWait = nWait;

   return S_OK;
}
HRESULT CEncodeVorbis::GetTransformedBytes(LONGLONG *pllBytes)
{
   CAutoLock lock(&m_Lock);

   *pllBytes = m_llInputSize;

   return S_OK;
}
HRESULT CEncodeVorbis::GetInFormat(WAVEFORMATEX *pFormat)
{  
   CheckPointer(pFormat, E_POINTER);
   ::CopyMemory(pFormat, &m_inFormat, sizeof(m_inFormat));

   return S_OK;
}
HRESULT CEncodeVorbis::GetOutFormat(void *pFormat)
{  
   CheckPointer(pFormat, E_POINTER);
   ::CopyMemory(pFormat, &m_outFormat, sizeof(m_outFormat));

   return S_OK;
}
HRESULT CEncodeVorbis::SetOutFormat(void *pFormat)
{
   ::CopyMemory(&m_outFormat, pFormat, sizeof(m_outFormat));

   m_outFormat.nChannels = m_inFormat.nChannels;
   m_outFormat.nSamplesPerSec = m_inFormat.nSamplesPerSec;

   return S_OK;
}
HRESULT CEncodeVorbis::OnConnectInPin(const CMediaType *pmtIn)
{  

   if(pmtIn->majortype != MEDIATYPE_Audio)
      return S_FALSE;

   if(pmtIn->subtype != MEDIASUBTYPE_PCM && pmtIn->subtype != MEDIASUBTYPE_IEEE_FLOAT)
      return S_FALSE;

   if(pmtIn->formattype != FORMAT_WaveFormatEx)
      return S_FALSE;

   WAVEFORMATEX *pwf = (WAVEFORMATEX *)pmtIn->Format();
   if(pwf->nChannels <= 2 && pwf->nSamplesPerSec <= 48000)
   {  

      ::CopyMemory(&m_inFormat, pwf, sizeof(m_inFormat));

      m_outFormat.nChannels = pwf->nChannels;
      m_outFormat.nSamplesPerSec = pwf->nSamplesPerSec;

      return S_OK;
   }

   return S_FALSE;
}
HRESULT CEncodeVorbis::OnStart(HANDLE hFile)
{
   int eos = 0, ret;

   vorbis_info_init(&m_vi);

   if(-0.11f <= m_outFormat.fQuality && m_outFormat.fQuality <= 1.01f)
   {  
      ret = vorbis_encode_init_vbr(&m_vi, (long)m_outFormat.nChannels, m_outFormat.nSamplesPerSec, m_outFormat.fQuality);
   }
   else
   {  
      ret = vorbis_encode_init(&m_vi, (long)m_outFormat.nChannels, m_outFormat.nSamplesPerSec, m_outFormat.nMaxBitsPerSec, m_outFormat.nAvgBitsPerSec, m_outFormat.nMinBitsPerSec);
   }

   if(ret != 0)
   {  

      vorbis_info_clear(&m_vi);
      return E_FAIL;
   }

   vorbis_comment_init(&m_vc);

   vorbis_analysis_init(&m_vd, &m_vi);
   vorbis_block_init(&m_vd, &m_vb);

   srand((unsigned int)time(NULL));
   ogg_stream_init(&m_os, rand());

   ogg_packet header;
   ogg_packet header_comm;
   ogg_packet header_code;

   vorbis_analysis_headerout(&m_vd, &m_vc, &header, &header_comm, &header_code);
   ogg_stream_packetin(&m_os, &header);
   ogg_stream_packetin(&m_os, &header_comm);
   ogg_stream_packetin(&m_os, &header_code);

   while(!eos)
   {
      int result = ogg_stream_flush(&m_os, &m_og);

      if(result==0)
         break;

      BOOL flag;
      DWORD dwWriteSize;

      flag = ::WriteFile(hFile , (LPVOID)m_og.header, m_og.header_len, &dwWriteSize , NULL);
      if(flag == FALSE)
      {
         vorbis_info_clear(&m_vi);
         return E_FAIL;
      }

      flag = ::WriteFile(hFile , (LPVOID)m_og.body, m_og.body_len, &dwWriteSize , NULL);
      if(flag == FALSE)
      {
         vorbis_info_clear(&m_vi);
         return E_FAIL;
      }
   }

   m_bInitialize = true;

   return S_OK;
}
HRESULT CEncodeVorbis::Encode(HANDLE hFile, int nBufferSize)
{  

   int eos = 0;

   vorbis_analysis_wrote(&m_vd, nBufferSize);

   while(vorbis_analysis_blockout(&m_vd, &m_vb)==1)
   {
      vorbis_analysis(&m_vb, NULL);
      vorbis_bitrate_addblock(&m_vb);

      while(vorbis_bitrate_flushpacket(&m_vd, &m_op))
      {
         ogg_stream_packetin(&m_os, &m_op);
    
         while(!eos)
         {
            int result = ogg_stream_pageout(&m_os, &m_og);
            if(result == 0)
               break;

            BOOL flag;
            DWORD dwWriteSize;

            flag = ::WriteFile(hFile , (LPVOID)m_og.header, m_og.header_len, &dwWriteSize , NULL);
            if(flag == FALSE) return E_FAIL;

            flag = ::WriteFile(hFile , (LPVOID)m_og.body, m_og.body_len, &dwWriteSize , NULL);
            if(flag == FALSE) return E_FAIL;

            if(ogg_page_eos(&m_og))
               eos = 1;
         }
      }
   }

   return S_OK;
}
void CEncodeVorbis::Convert8to32f(BYTE *pbInBuffer, float **ppfOutBuffer, int nEncodeSamples)
{
   int i, j;
   int nChannels = m_outFormat.nChannels;

   for(i=0;i<nEncodeSamples;i++)
   {
      for(j=0;j<nChannels;j++)
      {
         if(*pbInBuffer > 128)
            ppfOutBuffer[j][i] = (float)(*pbInBuffer - 128)/ 127.f;
         else
            ppfOutBuffer[j][i] = (float)(*pbInBuffer - 128)/ 128.f;

         *pbInBuffer++;
      }
   }
}
void CEncodeVorbis::Convert16to32f(short *psInBuffer, float **ppfOutBuffer, int nEncodeSamples)
{
   int i, j;
   int nChannels = m_outFormat.nChannels;

   for(i=0;i<nEncodeSamples;i++)
   {
      for(j=0;j<nChannels;j++)
      {
         if(*psInBuffer > 0)
            ppfOutBuffer[j][i] = (float)*psInBuffer / 32767.f;
         else
            ppfOutBuffer[j][i] = (float)*psInBuffer / 32768.f;

         *psInBuffer++;
      }
   }
}
void CEncodeVorbis::Convert24to32f(_int24 *pbInBuffer, float **ppfOutBuffer, int nEncodeSamples)
{
   int i, j, nData;
   int nChannels = m_outFormat.nChannels;

   for(i=0;i<nEncodeSamples;i++)
   {
      for(j=0;j<nChannels;j++)
      {
         if(pbInBuffer->b[2] & 0x80)
         {
            nData = int(0xff000000 | (pbInBuffer->b[2] << 16) | (pbInBuffer->b[1] << 8) | pbInBuffer->b[0]);
            ppfOutBuffer[j][i] = (float)nData / 8388608.f;
         }
         else
         {
            nData = int((pbInBuffer->b[2] << 16) | (pbInBuffer->b[1] << 8) | pbInBuffer->b[0]);
            ppfOutBuffer[j][i] = (float)nData / 8388607.f;
         }

         pbInBuffer++;
      }
   }
}
void CEncodeVorbis::Convert32to32f(int *pnInBuffer, float **ppfOutBuffer, int nEncodeSamples)
{
   int i, j;
   int nChannels = m_outFormat.nChannels;

   for(i=0;i<nEncodeSamples;i++)
   {
      for(j=0;j<nChannels;j++)
      {
         if(*pnInBuffer > 0)
            ppfOutBuffer[j][i] = *pnInBuffer / 2147483647.f;
         else
            ppfOutBuffer[j][i] = *pnInBuffer / 2147483648.f;

         *pnInBuffer++;
      }
   }
}
void CEncodeVorbis::Convert32fto32f(float *pfInBuffer, float **ppfOutBuffer, int nEncodeSamples)
{
   int i, j;
   int nChannels = m_outFormat.nChannels;

   for(i=0;i<nEncodeSamples;i++)
   {
      for(j=0;j<nChannels;j++)
      {
         ppfOutBuffer[j][i] = *pfInBuffer;
         *pfInBuffer++;
      }
   }
}
void CEncodeVorbis::Convert64to32f(double *pdInBuffer, float **ppfOutBuffer, int nEncodeSamples)
{
   int i, j;
   int nChannels = m_outFormat.nChannels;

   for(i=0;i<nEncodeSamples;i++)
   {
      for(j=0;j<nChannels;j++)
      {
         ppfOutBuffer[j][i] = (float)*pdInBuffer;
         *pdInBuffer++;
      }
   }
}
HRESULT CEncodeVorbis::OnReceive(HANDLE hFile, IMediaSample *pInSample)
{
   HRESULT hr;
   float **ppfBuffer = NULL;

   if(m_nWait > 0)
   {
      int nWait = int((m_llInputSize + pInSample->GetActualDataLength()) * 10 / m_inFormat.nAvgBytesPerSec) - int(m_llInputSize * 10 / m_inFormat.nAvgBytesPerSec);

      if(nWait > 0)
         ::Sleep(nWait * m_nWait * 2);
   }

   m_llInputSize += pInSample->GetActualDataLength();

   int nRestSamples = (int)pInSample->GetActualDataLength() / m_inFormat.nBlockAlign;

   BYTE *pbInBuffer = NULL;
   pInSample->GetPointer(&pbInBuffer);

   ppfBuffer = vorbis_analysis_buffer(&m_vd, nRestSamples);

   if(m_inFormat.wBitsPerSample == 16)
      Convert16to32f((short *)pbInBuffer, ppfBuffer, nRestSamples);
   else if(m_inFormat.wBitsPerSample == 8)
      Convert8to32f(pbInBuffer, ppfBuffer, nRestSamples);
   else if(m_inFormat.wBitsPerSample == 24)
      Convert24to32f((_int24 *)pbInBuffer, ppfBuffer, nRestSamples);
   else if(m_inFormat.wBitsPerSample == 32)
   {
      if(m_inFormat.wFormatTag == WAVE_FORMAT_PCM)
         Convert32to32f((int *)pbInBuffer, ppfBuffer, nRestSamples);
      else
         Convert32fto32f((float *)pbInBuffer, ppfBuffer, nRestSamples);
   }
   else
      Convert64to32f((double *)pbInBuffer, ppfBuffer, nRestSamples);

   hr = Encode(hFile, nRestSamples);
   if(FAILED(hr))
      return hr;

   return S_OK;
}
HRESULT CEncodeVorbis::OnStop(HANDLE hFile, bool bEos)
{
   if(bEos == true)
   {
      Encode(hFile, 0);

      ogg_stream_clear(&m_os);
      vorbis_block_clear(&m_vb);
      vorbis_dsp_clear(&m_vd);
      vorbis_comment_clear(&m_vc);
      vorbis_info_clear(&m_vi);

      m_bInitialize = false;
   }

   return S_OK;
}
STDMETHODIMP CEncodeVorbis::OnQueryInterface(REFIID riid, void ** ppv)
{
   if(riid == IID_IEcoDecoInterface)
      return GetInterface((IEcoDecoInterface *)this, ppv);
   else if(riid == IID_IDecodeInterface)
      return GetInterface((IDecodeInterface *)this, ppv);

   return CBaseFilter::NonDelegatingQueryInterface(riid, ppv);
}
CUnknown * WINAPI CEncodeVorbis::CreateInstance(LPUNKNOWN punk, HRESULT *phr)
{  

   ASSERT(phr);
    
   CEncodeVorbis *pNewObject = new CEncodeVorbis(punk, phr);
   if (pNewObject == NULL)
   {
      if(phr != NULL)
         *phr = E_OUTOFMEMORY;
   }

   return dynamic_cast<CUnknown *>(pNewObject);
}
