/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

#define EH_HASH_SIZE 389

typedef struct {
  FILE *fp;
  int refcount;
} eh_filespool_entry_t;

static void
eh_filespool_hentry_delete (eh_strhash_entry_t *hent)
{
  eh_filespool_entry_t *ee = (eh_filespool_entry_t *)hent->val;
  if (ee) {
    if (ee->fp)
      fclose (ee->fp);
    x_free (ee);
  }
  x_free (hent->key);
}

static eh_filespool_entry_t *
eh_filespool_entry_new (const char *filename)
{
  FILE *fp;
  eh_filespool_entry_t *ee;
  fp = fopen (filename, "a");
  if (fp == NULL) {
    eh_log_perror (EH_LOG_FATAL, filename);
    return NULL;
  }
  ee = (eh_filespool_entry_t *)x_malloc (sizeof (*ee));
  ee->fp = fp;
  ee->refcount = 1;
  return ee;
}

void
eh_filespool_delete (eh_filespool_t *es)
{
  eh_strhash_delete (es->files_ht);
  x_free (es);
}

static void
eh_filespool_hentry_flush (eh_strhash_entry_t *hent, void *data)
{
  eh_filespool_entry_t *ee = (eh_filespool_entry_t *)hent->val;
  if (fflush (ee->fp))
    eh_log_perror (EH_LOG_FATAL, "failed to flush a filespool entry");
}

void
eh_filespool_flush (eh_filespool_t *es)
{
  eh_strhash_foreach (es->files_ht, eh_filespool_hentry_flush, NULL);
}

eh_filespool_t *
eh_filespool_new (void)
{
  eh_filespool_t *es;
  es = (eh_filespool_t *)x_malloc (sizeof (*es));
  memset (es, 0, sizeof (*es));
  es->files_ht = eh_strhash_new (EH_HASH_SIZE, eh_filespool_hentry_delete);
  return es;
}

FILE *
eh_filespool_find (eh_filespool_t *es, const char *filename,
		   int open_if_notfound)
{
  eh_strhash_entry_t *hent;
  char *fname;
  eh_filespool_entry_t *ee;
  
  if (!open_if_notfound) {
    ee = (eh_filespool_entry_t *)eh_strhash_find (es->files_ht,
						  filename, NULL);
    if (ee == NULL)
      return NULL;
    return ee->fp;
  }
  fname = x_strdup (filename);
  hent = eh_strhash_find_create (es->files_ht, fname);
  if (hent->key != fname) {
    x_free (fname);
    ee = (eh_filespool_entry_t *)hent->val;
    ee->refcount++;
  } else {
    ee = eh_filespool_entry_new (fname);
    hent->val = ee;
    if (ee == 0) {
      eh_strhash_remove (es->files_ht, fname);
      return NULL;
    }
  }
  return ee->fp;
}

int
eh_filespool_close (eh_filespool_t *es, const char *filename)
{
  eh_filespool_entry_t *ee;
  int r = 0;

  ee = (eh_filespool_entry_t *)eh_strhash_find (es->files_ht, filename, NULL);
  assert (ee->refcount > 0);
  assert (ee->fp);
  
  ee->refcount--;
  if (ee->refcount == 0) {
    r = fclose (ee->fp);
    ee->fp = 0;
  }
  return r;
}
