/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"
#include <regex.h>

static void
eh_htaccess_entry_free (eh_strhash_entry_t *he)
{
  x_free (he->key);
  x_free (he->val);
}

typedef struct {
  char *htaccess_fname;
  eh_strhash_t *ht;
  int filesmatch_flag;
  int ignore_flag;
} eh_htaccess_context_t;

static void
eh_htaccess_context_init (eh_htaccess_context_t *ctxt,
			  const char *htaccess_fname)
{
  ctxt->htaccess_fname = x_strdup (htaccess_fname);
  ctxt->ht = eh_strhash_new (37, eh_htaccess_entry_free);
  ctxt->filesmatch_flag = 0;
  ctxt->ignore_flag = 0;
}

static void
eh_htaccess_context_discard (eh_htaccess_context_t *ctxt)
{
  x_free (ctxt->htaccess_fname);
  if (ctxt->ht)
    eh_strhash_delete (ctxt->ht);
}

static int
eh_htaccess_parse_filesmatch (eh_htaccess_context_t *ctxt,
			      char *line, eh_request_t *er)
{
  if (strcmp (line, "</FilesMatch>") == 0) {
    if (ctxt->filesmatch_flag) {
      eh_log (EH_LOG_FATAL, "unexpected </FilesMatch> in %s",
	      ctxt->htaccess_fname);
      return 1;
    }
    ctxt->filesmatch_flag = 0;
    ctxt->ignore_flag = 0;
    return 0;
  }
  if (strncmp (line, "<FilesMatch", 11) == 0) {
    char *p, *q, *s;
    const char *bn;
    regex_t re;
    int r;
    if (ctxt->filesmatch_flag) {
      eh_log (EH_LOG_FATAL, "unexpected <FilesMatch> in %s",
	      ctxt->htaccess_fname);
      return 1;
    }
    p = line + 11;
    p += strspn (p, " \t");
    q = p + strcspn (p, " \t>");
    s = q + strspn (q, " \t");
    if (s[0] != '>' || s[1] != '\0') {
      eh_log (EH_LOG_FATAL, "invalid '<FilesMatch>' line in %s",
	      ctxt->htaccess_fname);
      return 1;
    }
    *q = '\0';
    eh_unescape_doublequotes (p);
    r = regcomp (&re, p, REG_EXTENDED | REG_NOSUB);
    if (r) {
      char buf[1025];
      buf[1024] = '\0';
      regerror (r, &re, buf, 1024);
      eh_log (EH_LOG_FATAL, "invalid regular expression '%s': %s",
	      p, buf);
      regfree (&re);
      return 1;
    }
    bn = strrchr (er->filename, '/');
    if (bn) {
      bn++;
    } else {
      bn = er->filename;
    }
    r = regexec (&re, bn, 0, NULL, 0);
    eh_debug ("regexec returns %d", r);
    regfree (&re);
    ctxt->ignore_flag = (r == 0) ? 0 : 1;
    return 0;
  }
  eh_log (EH_LOG_FATAL, "unexpected token %s in %s",
	  line, ctxt->htaccess_fname);
  return 1;
}

static int
eh_htaccess_parse_line (eh_htaccess_context_t *ctxt,
			char *line, eh_request_t *er)
{
  int i;
  char *key, *val;
  eh_strhash_entry_t *en;
  if (line[0] == '<') {
    return eh_htaccess_parse_filesmatch (ctxt, line, er);
  }
  if (ctxt->ignore_flag)
    return 0;
  i = strcspn (line, " \t");
  key = x_strndup (line, i);
  i += strspn (line + i, " \t");
  val = x_strdup (line + i);
  en = eh_strhash_find_create (ctxt->ht, key);
  if (en->key != key) {
    x_free (key);
    return 1;
  }
  en->val = val;
  return 0;
}

static int
eh_htaccess_check_options (eh_request_t *er, eh_strhash_t *h)
{
  const char *options = NULL;
  options = (const char *)eh_strhash_find (h, "Options", NULL);
  eh_debug ("options = %s", options);
  if (options && strcmp (options, "Indexes") == 0)
    return 0;
  return 1;
}

static int
eh_htaccess_auth_passwd (eh_request_t *er, eh_accesslog_t *ea,
			 const char *passwd_fname,
			 const char *req_users, const char *authstr)
{
  char *decoded = NULL;
  char *line, *p;
  const char *user_name, *user_passwd;
  FILE *fp = NULL;
  int i, retval = 1;
  if ((i = strcspn (authstr, " \t")) != 5 ||
      strncasecmp (authstr, "Basic", i) != 0) {
    retval = 1;
    goto finish;
  }
  authstr += 5;
  authstr += strspn (authstr, " \t");
  decoded = x_strdup (authstr);
  if (eh_base64_decode (decoded) < 0) {
    retval = 1;
    goto finish;
  }
  eh_debug ("decoded = %s", decoded);
  user_name = decoded;
  p = strchr (decoded, ':');
  if (p == NULL) {
    retval = 1;
    goto finish;
  }
  *p++ = '\0';
  user_passwd = p;
  eh_request_set_remote_user (er, user_name);
  eh_accesslog_set_remote_user (ea, user_name);
  eh_debug ("user_name = %s, user_passwd = %s", user_name, user_passwd);
  if (req_users &&
      !eh_str_is_member (user_name, req_users, " \t")) {
    eh_debug ("user '%s' is not a member of '%s'", user_name, req_users);
    retval = 1;
    goto finish;
  }
  eh_debug ("passwd_fname = %s", passwd_fname);
  fp = fopen (passwd_fname, "r");
  if (fp == NULL) {
    eh_log_perror (EH_LOG_FATAL, passwd_fname);
    retval = 1;
    goto finish;
  }
  while ((line = eh_file_get_line (fp)) != NULL) {
    char *q;
    char salt[3];
    char *crypt_val;
    p = line + strspn (line, " \t");
    if (p[0] == '#')
      goto cont;
    q = strchr (p, ':');
    if (q == NULL)
      goto cont;
    *q++ = '\0';
    if (strcmp (user_name, p) != 0) {
      eh_debug ("skipping user %s, user_name = %s", p, user_name);
      goto cont;
    }
    salt[0] = q[0], salt[1] = q[1], salt[2] = '\0';
    eh_debug ("user_passwd = %s", user_passwd);
    crypt_val = crypt (user_passwd, salt);
    eh_debug ("pwent passwd = %s", q);
    eh_debug ("crypt_val = %s", crypt_val);
    retval = strcmp (q, crypt_val);
    eh_debug ("result %d", retval);
    x_free (line);
    goto finish;
    
  cont:
    x_free (line);
  }
  
 finish:
  if (fp)
    fclose (fp);
  if (decoded)
    x_free (decoded);
  return retval;
}

static int
eh_htaccess_check_auth (eh_request_t *er, eh_accesslog_t *ea, eh_strhash_t *h,
			const char **authname_r)
{
  const char *authname = NULL, *auth_user_file = NULL;
  const char *req_users = NULL, *auth_type = NULL;
  int i;
  auth_type = (const char *)eh_strhash_find (h, "AuthType", NULL);
  if (auth_type == NULL) {
    return 0;
  }
  if (strcasecmp (auth_type, "Basic") != 0) {
    eh_debug ("AuthType is %s", auth_type);
    return 1;
  }
  authname = (const char *)eh_strhash_find (h, "AuthName", NULL);
  if (authname == NULL) {
    eh_debug ("AuthName is not set");
    return 1;
  }
  *authname_r = authname;
  auth_user_file = (const char *)eh_strhash_find (h, "AuthUserFile", NULL);
  if (auth_user_file == NULL) {
    eh_debug ("AuthUserFile is not set");
    return 1;
  }
  req_users = (const char *)eh_strhash_find (h, "require", NULL);
  if (req_users == NULL) {
    eh_debug ("no 'require' line is found");
    return 1;
  }
  if ((i = strcspn (req_users, " \t")) == 4 &&
      strncasecmp (req_users, "user", i) == 0) {
    req_users += 4;
    req_users += strspn (req_users, " \t");
  } else if (strcasecmp (req_users, "valid-user") == 0) {
    req_users = NULL;
  } else {
    eh_debug ("unknown 'require' format: %s", req_users);
    return 1;
  }
  if (er->headers.predef.authorization == NULL) {
    eh_debug ("no authorization header");
    return 1;
  }
  return eh_htaccess_auth_passwd (er, ea, auth_user_file, req_users,
				  er->headers.predef.authorization);
}

static int
eh_connection_check_htaccess_internal (eh_connection_t *ec, eh_request_t *er,
				       unsigned int allowoverride)
{
  const char *authname = NULL;
  char *fname, *line, *p;
  FILE *fp = NULL;
  int auth_reject = 1;
  int index_reject = 1;
  eh_htaccess_context_t ctxt;

  if (!S_ISDIR (er->statbuf.st_mode))
    index_reject = 0;
  
  fname = x_strdup (er->filename);
  while ((p = strrchr (fname, '/')) != NULL) {
    int len;
    p[1] = '\0';
    len = p - fname;
    fname = (char *)x_realloc (fname, len + 15);
    strcat (fname, ".htaccess");
    eh_debug ("trying fname = %s", fname);
    fp = fopen (fname, "r");
    if (fp) {
      break;
    }
    fname[len] = '\0';
  }
  eh_htaccess_context_init (&ctxt, fname);
  x_free (fname);
  if (fp == NULL) {
    auth_reject = 0;
    goto finish;
  }
  
  while ((line = eh_file_get_line (fp)) != NULL) {
    p = line + strspn (line, " \t");
    if (p[0] == '#') {
      x_free (line);
      continue;
    }
    if (eh_htaccess_parse_line (&ctxt, p, er)) {
      x_free (line);
      goto finish;
    }
    x_free (line);
  }
  if (ctxt.filesmatch_flag) {
    eh_log (EH_LOG_FATAL, "missing </FilesMatch> in %s", ctxt.htaccess_fname);
    goto finish;
  }

  if (index_reject &&
      (allowoverride & EH_ALLOWOVERRIDE_OPTIONS) &&
      eh_htaccess_check_options (er, ctxt.ht) == 0)
    index_reject = 0;
  if (!(allowoverride & EH_ALLOWOVERRIDE_AUTHCONFIG) ||
      eh_htaccess_check_auth (er, &ec->accesslog, ctxt.ht, &authname) == 0)
    auth_reject = 0;
  
 finish:
  if (index_reject) {
    eh_connection_append_wvec_response (ec, er->method, "403", NULL, NULL, 0);
  } else if (auth_reject && authname) {
    char *header;
    header = x_strdup_printf ("WWW-Authenticate: Basic realm=\"%s\"\r\n",
			      authname);
    eh_connection_append_wvec_response (ec, er->method, "401", header,
					NULL, 0);
    x_free (header);
  } else if (auth_reject) {
    eh_connection_append_wvec_response (ec, er->method, "403", NULL, NULL, 0);
  }
  if (fp)
    fclose (fp);
  eh_htaccess_context_discard (&ctxt);
  eh_debug ("check_htaccess_internal: index_reject=%d auth_reject=%d\n",
	    index_reject, auth_reject);
  return index_reject | auth_reject;
}

int
eh_connection_check_htaccess (eh_connection_t *ec, eh_request_t *er,
			      unsigned int allowoverride)
{
  if ((allowoverride & ~EH_ALLOWOVERRIDE_OPTIONS) == 0 &&
      !S_ISDIR (er->statbuf.st_mode)) {
    /* er is a regular file, and no flag other than 'Options' is set */
    return 0;
  }
  if ((allowoverride & EH_ALLOWOVERRIDE_OPTIONS) == 0 &&
      S_ISDIR (er->statbuf.st_mode)) {
    /* er is a directory, and overriding 'Options' is not allowed */
    eh_connection_append_wvec_response (ec, er->method, "403", NULL, NULL, 0);
    return 1;
  }
  return eh_connection_check_htaccess_internal (ec, er, allowoverride);
}
