/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define EH_DONT_PROHIBIT_MALLOC
#include "esehttpd.h"

#ifdef DEBUG_HEAP

#define MCALLERID_MAX 10000
static const char *mcallerid_map[MCALLERID_MAX];
static size_t refcounts[MCALLERID_MAX];
static size_t mcallerid_map_len = 0;
#define DEBUGINFO_SIZE (sizeof (char *) * 2)

static size_t
x_debug_get_mcallerid (const char *mcaller)
{
  size_t i;
  for (i = 0; i < mcallerid_map_len; i++) {
    if (strcmp (mcallerid_map[i], mcaller) == 0)
      return i;
  }
  if (mcallerid_map_len >= MCALLERID_MAX - 1) {
    eh_log (EH_LOG_FATAL, "increase MCALLERID_MAX!");
    abort ();
  }
  mcallerid_map_len++;
  mcallerid_map[i] = mcaller;
  return i;
}

static void
x_debug_increment_refcount (const char *mcaller)
{
  size_t i;
  i = x_debug_get_mcallerid (mcaller);
  refcounts[i]++;
}

static void
x_debug_decrement_refcount (const char *mcaller, const char *fcaller)
{
  size_t i;
  i = x_debug_get_mcallerid (mcaller);
  if (refcounts[i] == 0) {
    eh_log (EH_LOG_FATAL, "invalid pointer [mcaller: %s mcaller: %s]",
	    mcaller ? mcaller : "", fcaller ? fcaller : "");
    abort ();
  }
  refcounts[i]--;
}

void *x_malloc_debug (const char *mcaller, size_t s)
{
  const char **r;
  r = (const char **)malloc (s + DEBUGINFO_SIZE);
  x_debug_increment_refcount (mcaller);
  r[0] = mcaller;
  r[1] = NULL;
  return r + 2;
}

void x_set_mcaller_debug (const char *mcaller, void *p)
{
  const char **r;
  r = (const char **)p;
  r -= 2;
  x_debug_decrement_refcount (r[0], mcaller);
  x_debug_increment_refcount (mcaller);
  r[0] = mcaller;
}

void x_set_fcaller_debug (const char *fcaller, void *p)
{
  const char **r;
  r = (const char **)p;
  r -= 2;
  r[1] = fcaller;
}

void x_free_debug (const char *fcaller, void *p)
{
  const char **r;
  r = (const char **)p;
  r -= 2;
  if (r[1])
    fcaller = r[1];
  x_debug_decrement_refcount (r[0], fcaller);
  free (r);
}

void *x_realloc_debug (const char *caller, void *p, size_t s)
{
  void *r;
  if (s == 0) {
    if (p != NULL) {
      x_free_debug (caller, p);
    }
    return NULL;
  } else {
    if (p == NULL) {
      r = x_malloc_debug (caller, s);
    } else {
      const char **vp;
      vp = (const char **)p;
      vp -= 2;
      vp = realloc (vp, s + DEBUGINFO_SIZE);
      if (vp == NULL) {
	eh_log (EH_LOG_FATAL, "realloc returns NULL");
	abort ();
      }
      r = (void *)(vp + 2);
    }
    return r;
  }
}

char *
x_strdup_printf_debug (const char *mcaller, const char *format, ...)
{
  char *buffer;
  const char **r;
  int len;
  va_list ap;
  va_start (ap, format);
  if ((len = vasprintf (&buffer, format, ap)) < 0) {
    eh_log (EH_LOG_FATAL, "vasprintf: returns NULL");
    abort ();
  }
  va_end (ap);
  buffer = (char *)realloc (buffer, len + 1 + DEBUGINFO_SIZE);
  memmove (buffer + DEBUGINFO_SIZE, buffer, len + 1);
  r = (const char **)buffer;
  x_debug_increment_refcount (mcaller);
  r[0] = mcaller;
  r[1] = NULL;
  return (char *)(r + 2);
}

char *
x_strndup_debug (const char *mcaller, const char *str, int n)
{
  char *val;
  val = (char *)x_malloc_debug (mcaller, n + 1);
  memcpy (val, str, n);
  val[n] = '\0';
  return val;
}

char *
x_strdup_debug (const char *mcaller, const char *str)
{
  return x_strndup_debug (mcaller, str, strlen (str));
}

void
x_str_append_debug (const char *mcaller, char **str, const char *appendstr)
{
  size_t len1, len2;
  len1 = (*str) ? strlen (*str) : 0;
  len2 = strlen (appendstr);
  *str = (char *)x_realloc_debug (mcaller, *str, len1 + len2 + 1);
  memcpy ((*str) + len1, appendstr, len2 + 1);
}

void
x_append_printf_debug (const char *mcaller, char **str,
		       const char *format, ...)
{
  char *buffer;
  va_list ap;
  va_start (ap, format);
  if (vasprintf (&buffer, format, ap) < 0) {
    eh_log (EH_LOG_FATAL, "vasprintf: returns NULL");
    abort ();
  }
  va_end (ap);
  x_str_append_debug (mcaller, str, buffer);
  free (buffer);
}

void
x_mem_check_debug (void)
{
  size_t i, n;
  eh_log (EH_LOG_INFO, "heap: checking leaks...");
  for (i = 0, n = 0; i < mcallerid_map_len; i++) {
    if (refcounts[i]) {
      eh_log (EH_LOG_FATAL, "heap: leaks %lu, mcaller: %s",
	      refcounts[i], mcallerid_map[i]);
      n++;
    }
  }
  if (n == 0) {
    eh_log (EH_LOG_INFO, "heap: no leaks.");
  }
}

#else /* DEBUG_HEAP */

void *x_malloc_nodebug (size_t s)
{
  void *r;
  r = malloc (s);
  if (!r) {
    eh_log (EH_LOG_FATAL, "malloc returns NULL");
    abort ();
  }
  return r;
}

void x_free_nodebug (void *p)
{
  free (p);
}

void *x_realloc_nodebug (void *p, size_t s)
{
  void *r;
  if (s == 0) {
    if (p != NULL) {
      free (p);
    }
    return NULL;
  } else {
    if (p == NULL) {
      r = malloc (s);
    } else {
      r = realloc (p, s);
    }
    if (r == NULL) {
      eh_log (EH_LOG_FATAL, "realloc/malloc returns NULL");
      abort ();
    }
    return r;
  }
}

char *
x_strdup_printf_nodebug (const char *format, ...)
{
  char *buffer;
  va_list ap;
  va_start (ap, format);
  if (vasprintf (&buffer, format, ap) < 0) {
    eh_log (EH_LOG_FATAL, "vasprintf: returns NULL");
    abort ();
  }
  va_end (ap);
  return buffer;
}

char *
x_strndup_nodebug (const char *str, int n)
{
  char *val;
  val = (char *)x_malloc (n + 1);
  memcpy (val, str, n);
  val[n] = '\0';
  return val;
}

char *
x_strdup_nodebug (const char *str)
{
  return x_strndup (str, strlen (str));
}

void
x_str_append_nodebug (char **str, const char *appendstr)
{
  size_t len1, len2;
  len1 = (*str) ? strlen (*str) : 0;
  len2 = strlen (appendstr);
  *str = (char *)x_realloc (*str, len1 + len2 + 1);
  memcpy ((*str) + len1, appendstr, len2 + 1);
}

void
x_append_printf_nodebug (char **str, const char *format, ...)
{
  char *buffer;
  va_list ap;
  va_start (ap, format);
  if (vasprintf (&buffer, format, ap) < 0) {
    eh_log (EH_LOG_FATAL, "vasprintf: returns NULL");
    abort ();
  }
  va_end (ap);
  x_str_append_nodebug (str, buffer);
  free (buffer);
}

#endif /* DEBUG_HEAP */
