/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

typedef struct {
  int fd;
  void *ptr;
  size_t size;
} eh_mapped_file_t;

static void
eh_mapped_file_free (void *data)
{
  eh_mapped_file_t *mmf = (eh_mapped_file_t *)data;
  eh_debug ("");
  munmap (mmf->ptr, mmf->size);
  x_free (mmf);
}

static void
str_free (void *str)
{
  x_free (str);
}

static void
eh_connection_append_wvec_304 (eh_connection_t *ec, eh_request_t *er)
{
  /* 304 response must have Content-Location: field. body must be empty. */
  char *str;
  eh_debug ("304 %s", er->filename);
  str = x_strdup_printf ("HTTP/1.1 304 Not Modified\r\n"
			 "Connection: %s\r\n"
			 "Content-Location: %s\r\n"
			 "\r\n",
			 ec->rconn == eh_rconn_keepalive ?
			 "Keep-Alive" : "Close",
			 er->url);
  eh_connection_append_wvec (ec, (void *)str, strlen (str), str_free, str);
  eh_accesslog_set_response_status (&ec->accesslog, 304, 0);
}

static void
eh_connection_append_wvec_412 (eh_connection_t *ec, eh_request_t *er)
{
  char *str;
  eh_debug ("412 %s", er->filename);
  str = x_strdup_printf ("HTTP/1.1 412 Precondition Failed\r\n"
			 "Connection: %s\r\n"
			 "\r\n",
			 ec->rconn == eh_rconn_keepalive ?
			 "Keep-Alive" : "Close"
			 );
  eh_connection_append_wvec (ec, (void *)str, strlen (str), str_free, str);
  eh_accesslog_set_response_status (&ec->accesslog, 412, 0);
}

static void
eh_connection_append_wvec_fcent (eh_connection_t *ec, eh_request_t *er,
				 eh_filecache_entry_t *fcent)
{
  eh_filecache_entry_wvec_ref (fcent);
  if (er->method == eh_method_get) {
    /* get request */
    eh_connection_append_wvec (ec, fcent->header, fcent->header_len,
			       NULL, NULL);
    eh_connection_append_wvec (ec, fcent->body, fcent->statbuf.st_size,
			       eh_filecache_entry_wvec_unref, fcent);
    eh_accesslog_set_response_status (&ec->accesslog,
				      200, fcent->statbuf.st_size);
  } else {
    /* head request */
    eh_connection_append_wvec (ec, fcent->header, fcent->header_len,
			       eh_filecache_entry_wvec_unref, fcent);
    eh_accesslog_set_response_status (&ec->accesslog, 200, 0);
  }
}

static void *
eh_try_mmap (eh_connection_t *ec, eh_request_t *er, int fd)
{
  void *mptr;
  if ((mptr = mmap (NULL, er->statbuf.st_size, PROT_READ, MAP_SHARED, fd, 0))
	== (void *)-1) {
    eh_connection_append_wvec_response (ec, er->method, "503", NULL, NULL, 0);
  }
  return mptr;
}

static int
eh_parse_range (eh_request_t *er, size_t content_length,
		size_t *range_start, size_t *range_end)
{
  const char *rangestr;
  int r;
  size_t s = 0, e = 0;
  assert (content_length > 0);
  rangestr = er->headers.predef.range;
  if (strncmp (rangestr, "bytes=", 6) != 0)
    return -1;
  rangestr += 6;
  r = eh_parse_sizestr (rangestr, &s);
  if (r < 0) {
    return -1;
  } else if (r == 0) {
    /* suffix-byte-range-spec */
    eh_debug ("subbix-byte-range-spec");
    if (rangestr[0] != '-')
      return -1;
    rangestr++;
    r = eh_parse_sizestr (rangestr, &e);
    if (r <= 0)
      return -1;
    if (rangestr[r] != '\0')
      return -1;              /* possibly multiple range */
    if (e >= content_length)
      return -1;              /* longer than content-length. reply 200 */
    if (e == 0)
      return 1;               /* 'bytes=-0'. reply 416 */
    *range_start = content_length - e;
    *range_end = content_length;
  } else {
    /* byte-range-spec */
    size_t e1;
    eh_debug ("byte-range-spec");
    if (rangestr[r] != '-')
      return -1;
    rangestr += r + 1;
    r = eh_parse_sizestr (rangestr, &e);
    if (r < 0)
      return -1;
    if (rangestr[r] != '\0')
      return -1;              /* possibly multiple range */
    if (s >= content_length)
      return 1;               /* empty range. reply 416 */
    if (r == 0) {
      /* 'bytes=aaa-' */
      e = content_length - 1;
    } else {
      /* 'bytes=aaa-bbb' */
      if (e >= content_length)
	e = content_length - 1;
    }
    if (e < s)
      return -1;              /* invalid range. reply 200 */
    e1 = e + 1;
    if (e1 < e)
      return -1;              /* overflow */
    *range_start = s;
    *range_end = e1;
  }
  return 0;
}

static char *
eh_make_header_200_noconn (const char *last_modified, const char *content_type,
			   size_t content_length)
{
  const char tmpl_200[] =
    "HTTP/1.1 200 OK\r\n"
    "Last-Modified: %s\r\n"
    "Content-Type: %s\r\n"
    "Content-Length: %lu\r\n\r\n";
  return x_strdup_printf (tmpl_200,
			  last_modified, content_type, content_length);
}

static char *
eh_make_header_206_200 (eh_connection_t *ec, eh_request_t *er,
			const char *last_modified_str, size_t datelen,
			const char *content_type,
			size_t content_length, size_t *range_start,
			size_t *range_end, int *statuscode_r)
{
  int if_range_failed = 0;
  eh_headers_t *const headers = &er->headers;
  *range_start = 0;
  *range_end = content_length;
  *statuscode_r = 200;
  if (headers->predef.range &&
      headers->predef.if_range &&
      strncmp (headers->predef.if_range, last_modified_str, datelen) != 0) {
    if_range_failed = 1;
  }
  if (headers->predef.range &&
      if_range_failed == 0 &&
      content_length > 0 &&
      eh_parse_range (er, content_length, range_start, range_end) == 0) {
    const char tmpl_206[] =
      "HTTP/1.1 206 Partial Content\r\n"
      "Connection: %s\r\n"
      "Date: %s\r\n"                      /* RFC says Date is a must. why? */
      "Content-Location: %s\r\n"
      "Content-Range: %lu-%lu/%lu\r\n\r\n";
    /* TODO: reply 416 */
    *statuscode_r = 206;
    return x_strdup_printf (tmpl_206,
			    ec->rconn == eh_rconn_keepalive ?
			    "Keep-Alive" : "Close",
			    eh_fd_get_time_string (), er->url,
			    *range_start, *range_end - 1,
			    content_length);
    
  } else {
    const char tmpl_200[] =
      "HTTP/1.1 200 OK\r\n"
      "Connection: %s\r\n"
      "Last-Modified: %s\r\n"
      "Content-Type: %s\r\n"
      "Content-Length: %lu\r\n\r\n";
    return x_strdup_printf (tmpl_200,
			    ec->rconn == eh_rconn_keepalive ?
			    "Keep-Alive" : "Close",
			    last_modified_str, content_type, content_length);
  }
}

void
eh_connection_append_wvec_file (eh_connection_t *ec, eh_request_t *er)
{
  eh_filecache_entry_t *fcent = NULL;
  eh_headers_t *const headers = &er->headers;
  char last_modified_str[80];
  size_t datelen = 0;
  size_t fcthreshold;
  const char *mime_type;
  
  /* append http response to the write vector, with minimal headers */
  int fd = -1;
  
  if (er->method != eh_method_get && er->method != eh_method_head) {
    eh_connection_append_wvec_response (ec, er->method, "405", NULL, NULL, 0);
    goto finish;
  }

  fcent = er->fcent_ref;
  /* if we need 'Connection: foo' field, we can't use filecache because
     cached entry must not contain it. */
  if (er->need_connection_field == 0 &&
      fcent != NULL && headers->predef.range == NULL) {
    /* this file is already cached */
    eh_debug ("already cached");
    if (fcent->statbuf.st_mtime == er->statbuf.st_mtime) {
      /* cache entry is up to date */
      eh_filecache_entry_set_last_stat_call_time (fcent, eh_fd_get_time ());
      /* check If-Modified-Since -> 304 */
      if (headers->predef.if_modified_since) {
	datelen = eh_rfc1123_date (last_modified_str, 80,
				   &er->statbuf.st_mtime);
	if (strncmp (headers->predef.if_modified_since, last_modified_str,
		     datelen) == 0) {
	  eh_connection_append_wvec_304 (ec, er);
	  goto finish;
	}
      }
      /* check If-Unmodified-Since -> 412 */
      if (headers->predef.if_unmodified_since) {
	datelen = eh_rfc1123_date (last_modified_str, 80,
				   &er->statbuf.st_mtime);
	if (strncmp (headers->predef.if_unmodified_since, last_modified_str,
		     datelen) != 0) {
	  eh_connection_append_wvec_412 (ec, er);
	  goto finish;
	}
      }
      /* send the cache entry */
      eh_connection_append_wvec_fcent (ec, er, fcent);
      goto finish;
    }
    /* disk file is newer than cache. need reloading */
  } else {
    eh_debug ("%s: not cached (or no-cache is specified)", er->filename);
  }
  
  if ((fd = open (er->filename, O_RDONLY)) < 0) {
    if (errno == EMFILE || errno == ENFILE) {
      eh_connection_append_wvec_response (ec, er->method, "503",
					  NULL, NULL, 0);
    } else {
      eh_connection_append_wvec_response (ec, er->method, "404",
					  NULL, NULL, 0);
    }
    goto finish;
  }
  
  /* this file is not cached yet */
  
  if (datelen == 0)
    datelen = eh_rfc1123_date (last_modified_str, 80, &er->statbuf.st_mtime);
  /* check If-Modified-Since -> 304 */
  if (headers->predef.if_modified_since &&
      strncmp (headers->predef.if_modified_since, last_modified_str,
	       datelen) == 0) {
    eh_connection_append_wvec_304 (ec, er);
    goto finish;
  }
  /* check If-Unmodified-Since -> 412 */
  if (headers->predef.if_unmodified_since &&
      strncmp (headers->predef.if_unmodified_since, last_modified_str,
	       datelen) != 0) {
    eh_connection_append_wvec_412 (ec, er);
    goto finish;
  }
  mime_type = er->mime_type;
  assert (mime_type);
  eh_debug ("%s: mimetype: %s", er->filename, mime_type);
  fcthreshold = ec->app_backref->filecache_threshold;
  if (er->econf_dir_ref->option_nofilecache == 0 &&
      (fcthreshold == 0 || er->statbuf.st_size <= (int)fcthreshold) &&
      er->need_connection_field == 0 &&
      headers->predef.range == NULL) {
    /* this file can be cached. insert to the filecache */
    eh_filecache_entry_t *ent;
    void *mptr;
    char *hstr;
    if ((mptr = eh_try_mmap (ec, er, fd)) == (void *)-1)
      goto finish;
    hstr = eh_make_header_200_noconn (last_modified_str, mime_type,
				      er->statbuf.st_size);
    ent = eh_filecache_insert (ec->app_backref->fcache, er->filename, hstr,
			       strlen (hstr), (char *)mptr,
			       &er->statbuf,
			       eh_fd_get_time (), er->mime_type);
    eh_debug ("inserted to filecache: %s", er->filename);
    /* mptr and hstr will be freed by filecache. don't free them now */
    eh_connection_append_wvec_fcent (ec, er, ent);
  } else {
    /* this file will not be cached. send it directly */
    eh_debug ("not inserted to filecache: %s", er->filename);
    if (er->method == eh_method_head) {
      /* head request */
      char *hstr;
      size_t offset_s, offset_e;
      int statuscode = 200;
      hstr = eh_make_header_206_200 (ec, er, last_modified_str, datelen,
				     mime_type, er->statbuf.st_size,
				     &offset_s, &offset_e, &statuscode);
      eh_connection_append_wvec (ec, (void *)hstr, strlen (hstr), str_free,
				 hstr);
      eh_accesslog_set_response_status (&ec->accesslog, statuscode, 0);
    } else {
      /* get request */
      void *mptr;
      char *hstr;
      size_t offset_s, offset_e;
      int statuscode = 200;
      eh_mapped_file_t *mmf;
      if ((mptr = eh_try_mmap (ec, er, fd)) == (void *)-1)
	goto finish;
      hstr = eh_make_header_206_200 (ec, er, last_modified_str, datelen,
				     mime_type, er->statbuf.st_size,
				     &offset_s, &offset_e, &statuscode);
      mmf = (eh_mapped_file_t *)x_malloc (sizeof (*mmf));
      mmf->fd = -1;
      mmf->ptr = mptr;
      mmf->size = er->statbuf.st_size;
      eh_connection_append_wvec (ec, (void *)hstr, strlen (hstr), str_free,
				 hstr);
      eh_connection_append_wvec (ec, (char *)mptr + offset_s,
				 offset_e - offset_s,
				 eh_mapped_file_free, mmf);
      eh_accesslog_set_response_status (&ec->accesslog, statuscode,
					er->statbuf.st_size);
    }
  }
  
 finish:
  if (fd >= 0) close (fd);
  return;
}

