
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <signal.h>
#include <assert.h>
#include <fcntl.h>

#define DEBUG_DO(x)

#define NUM_PROCESS 512
#define HOSTNAMELEN 1024
#define LBUFLEN 1024
#define RSH_PATH "/usr/bin/rsh"
#define RSH "rsh"

typedef struct {
  int fd[2]; /* 0 for stdout, 1 for stderr */
  char *hostname;
  char *linebuffer[2];
  int linebuffer_offset[2];
  struct timeval timeout;
  pid_t pid;
} rsh_process_t;

static char *hosts_file = NULL;
static char *rsh_path = RSH_PATH;
static char *rsh_command = RSH;
static int num_process = NUM_PROCESS;
static int quiet = 0;
static int timeout = 0;
static int from_stdin = 1;
static char **hostname_list = NULL;

static void usage (const char *argv0)
{
  printf ("Usage: %s [OPTIONS] COMMAND [ARGS ...]\n", argv0);
  printf ("Options:\n");
  printf ("\t-n NUM\n");
  printf ("\t-t TIMEOUT\n");
  printf ("\t-c RSH_COMMAND\n");
  printf ("\t-h HOSTS_FILE\n");
  printf ("\t-l HOSTNAME_LIST\n");
  printf ("\t-s\n");
  printf ("\t-q\n");
  exit (1);
}

static void *x_malloc (size_t s)
{
  void *ptr;
  for (;;) {
    ptr = malloc(s);
    if (ptr) return ptr;
    usleep (100000);
  }
  return ptr;
}

static void
do_fork_rsh (int argc, char **argv, char *host, pid_t *pid_r,
	     int *fd0_r, int *fd1_r)
{
  pid_t pid;
  int i;
  int devnull, fds_0[2], fds_1[2];
  
  DEBUG_DO (printf ("do_fork_rsh host = %s\n", host));

  devnull = open ("/dev/null", O_RDONLY);
  if (devnull < 0) {
    perror ("/dev/null");
    exit (-1);
  }

  *pid_r = 0;
  if (pipe (fds_0) < 0) { /* for stdout of rsh */
    perror ("pipe");
    exit (-1);
  }
  if (pipe (fds_1) < 0) { /* for stderr of rsh */
    perror ("pipe");
    exit (-1);
  }

  DEBUG_DO (printf ("do_fork_rsh fds_0 = %d %d, fds_1 = %d %d\n",
		    fds_0[0], fds_0[1], fds_1[0], fds_1[1]));
  
  DEBUG_DO (printf ("do_fork_rsh devnull = %d\n", devnull));

  pid = fork ();
  if (pid < 0) {
    /* FIXME: keep alive even if fork() is failed */
    perror ("fork");
    exit (-1);
  } else if (pid == 0) {
    /* child process: execute rsh */
    char **real_argv;
    real_argv = x_malloc (sizeof (*real_argv) * (argc + 5));
    real_argv[0] = rsh_command;
    real_argv[1] = host;
    for (i = 0; i < argc; i++)
      real_argv[i + 2] = argv[i];
    real_argv[i + 2] = NULL;

    close (fds_0[0]);
    close (fds_1[0]);
    
    close (1);
    dup2 (fds_0[1], 1);
    close (fds_0[1]);

    close (2);
    dup2 (fds_1[1], 2);
    close (fds_0[1]);

    close (0);
    dup2 (devnull, 0);
    
    execve (rsh_path, real_argv, NULL);
    perror ("execve");
    exit (-1);
  }
  close (fds_0[1]);
  close (fds_1[1]);
  close (devnull);
  *pid_r = pid;
  *fd0_r = fds_0[0];
  *fd1_r = fds_1[0];
  
  DEBUG_DO (printf ("do_fork_rsh fd = %d, %d\n", fds_0[0], fds_1[0]));
  
}

static void read_host (FILE *fp_hosts, rsh_process_t *ps,
		       int argc, char **argv)
{
  int len;
  struct timezone tz;
  
  ps->timeout.tv_sec = 0;
  ps->timeout.tv_usec = 0;
  ps->pid = 0;
  
  do {
    if (hostname_list) {
      /* read hostname from hostname_list[] */
      static char **hosts_p = NULL;
      if (hosts_p == NULL)
	hosts_p = hostname_list;
      if (!*hosts_p) {
	ps->fd[0] = ps->fd[1] = -1; /* done */
	return;
      }
      strncpy (ps->hostname, *hosts_p, HOSTNAMELEN);
      DEBUG_DO (printf ("read_host hostname = %s\n", ps->hostname));
      hosts_p++;
    } else {
      /* read hostname from fp_hosts */
      if (fgets(ps->hostname, HOSTNAMELEN, fp_hosts) == NULL) {
	ps->fd[0] = ps->fd[1] = -1; /* done */
	return;
      }
    }
    len = strlen (ps->hostname);
  } while (len == 0);
  
  if (ps->hostname[len - 1] == '\n')
    ps->hostname[len - 1] = '\0';
  DEBUG_DO (printf ("hostname = %s\n", ps->hostname));
  do_fork_rsh (argc, argv, ps->hostname, &ps->pid, &ps->fd[0], &ps->fd[1]);
  if (timeout > 0) {
    if (gettimeofday (&ps->timeout, &tz) < 0) {
      perror ("gettimeofday");
      exit (-1);
    }
    ps->timeout.tv_sec += timeout;
  }
}

static int fill_linebuffer (rsh_process_t *ps, int fdnum)
{
  char *read_top, *p, *topofline;
  int readlen, readlen_max, new_offset;

  DEBUG_DO (printf ("fill_linebuffer %d\n", fdnum));

  /* read and append to the linebuffer */
  read_top = ps->linebuffer[fdnum] + ps->linebuffer_offset[fdnum];
  readlen_max = LBUFLEN - ps->linebuffer_offset[fdnum];
  do {
    readlen = read (ps->fd[fdnum], read_top, readlen_max);
  } while (readlen < 0 && (errno == EINTR || errno == ERESTART));
  if (readlen < 0) {
    perror ("read");
    DEBUG_DO (printf ("read() error\n"));
    return readlen;
  }

  /* if linebuffer contains newline characters, write each lines to stdout. */
  for (topofline = ps->linebuffer[fdnum],
	 p = memchr (read_top, '\n', readlen); p;
       p = memchr (topofline, '\n', read_top + readlen - topofline)) {
    if (!quiet)
      fprintf (stdout, "%s%s ", ps->hostname, fdnum ? "::" : ":");
    fwrite (topofline, p - topofline + 1, 1, stdout);
    topofline = p + 1;
  }
  
  new_offset = ps->linebuffer_offset[fdnum] + readlen;

  /* if some lines have written, remove them from linebuffer */
  if (topofline != ps->linebuffer[fdnum]) {
    int movelen;
    movelen = ps->linebuffer[fdnum] + new_offset - topofline;
    assert(movelen >= 0);
    if (movelen > 0)
      memmove (ps->linebuffer[fdnum], topofline, movelen);
    new_offset = movelen;
  }
  
  if (new_offset == LBUFLEN || readlen == 0) {
    /* buffer is full or EOF. we need to flush */
    if (new_offset > 0) {
      if (!quiet)
	fprintf (stdout, "%s%s ", ps->hostname, fdnum ? "::" : ":");
      fwrite (ps->linebuffer[fdnum], new_offset, 1, stdout);
      fputc ('\n', stdout);
    }
    new_offset = 0;
  }
  ps->linebuffer_offset[fdnum] = new_offset;
  DEBUG_DO (printf ("fill_linebuffer: readlen = %d\n", readlen));
  return readlen;
}

static void zombie_hunter (int x)
{
  pid_t pid;
  int status;
  while ((pid = waitpid (-1, &status, WNOHANG)) > 0);
}

static int forall (int argc, char **argv)
{
  FILE *fp_hosts = NULL;
  rsh_process_t *ps;
  int i, numavailable;
  struct sigaction act;
  struct timeval now;
  struct timezone tz;

  sigemptyset (&act.sa_mask);
  act.sa_handler = zombie_hunter;
  act.sa_flags = SA_RESTART;
  sigaction (SIGCHLD, &act, NULL);
  if (hostname_list == NULL) {
    if (from_stdin)
      fp_hosts = stdin;
    else
      fp_hosts = fopen (hosts_file, "r");
    if (fp_hosts == NULL) {
      perror (hosts_file);
      exit (-1);
    }
  }
  ps = x_malloc (sizeof (*ps) * num_process);
  for (i = 0; i < num_process; i++) {
    ps[i].fd[0] = -1;
    ps[i].fd[1] = -1;
    ps[i].hostname = x_malloc (HOSTNAMELEN + 1);
    ps[i].linebuffer[0] = x_malloc (LBUFLEN + 1);
    ps[i].linebuffer[1] = x_malloc (LBUFLEN + 1);
    ps[i].linebuffer_offset[0] = 0;
    ps[i].linebuffer_offset[1] = 0;
    ps[i].hostname[HOSTNAMELEN] = '\0';
  }
  if (gettimeofday (&now, &tz) < 0) {
    perror ("gettimeofday");
    exit (-1);
  }

  /* fill fd_list */
  for (i = 0; i < num_process; i++) {
    read_host (fp_hosts, &ps[i], argc, argv);
  }
  /* select loop */
  for (;;) {
    int fdmax;
    struct timeval first_timeout;
    fd_set readfds;

    /* initialize readfds and timeout */
    FD_ZERO (&readfds);
    first_timeout = now;
    first_timeout.tv_sec += timeout;
    for (i = 0, fdmax = -1; i < num_process; i++) {
      if (ps[i].fd[0] >= 0) {
	FD_SET (ps[i].fd[0], &readfds);
	if (ps[i].fd[0] > fdmax)
	  fdmax = ps[i].fd[0];
      }
      if (ps[i].fd[1] >= 0) {
	FD_SET (ps[i].fd[1], &readfds);
	if (ps[i].fd[1] > fdmax)
	  fdmax = ps[i].fd[1];
      }
      if ((ps[i].fd[0] >= 0 || ps[i].fd[1] >= 0) && timeout) {
	if (timercmp (&first_timeout, &ps[i].timeout, >))
	  first_timeout = ps[i].timeout;
      }
    }

    /* if there is no file to watch, exit now */
    if (fdmax < 0) {
      DEBUG_DO (printf ("no more process. exiting\n"));
      break; /* no more process. exit now. */
    }

    /* do select () */
    if (timeout && timercmp (&now, &first_timeout, >)) {
      numavailable = 0; /* timeout already. we need not to do select () */
    } else if (timeout > 0) {
      struct timeval select_timeout;
      timersub (&first_timeout, &now, &select_timeout);
      numavailable = select (fdmax + 1, &readfds, NULL, NULL, &select_timeout);
    } else {
      numavailable = select (fdmax + 1, &readfds, NULL, NULL, NULL);
    }
    
    if (numavailable < 0) {
      if (errno != EINTR && errno != ERESTART)
	perror ("select");
      continue;
    } else if (numavailable > 0) {
      for (i = 0; i < num_process; i++) {
	if (ps[i].fd[0] >= 0 && FD_ISSET (ps[i].fd[0], &readfds)) {
	  DEBUG_DO (printf ("process number %d: read0 fd=%d\n", i,
			    ps[i].fd[0]));
	  if (fill_linebuffer (&ps[i], 0) == 0) {
	    DEBUG_DO (printf ("process number %d: EOF0\n", i));
	    close (ps[i].fd[0]);
	    ps[i].fd[0] = -1;
	  }
	}
	if (ps[i].fd[1] >= 0 && FD_ISSET (ps[i].fd[1], &readfds)) {
	  DEBUG_DO (printf ("process number %d: read1 fd=%d\n", i,
			    ps[i].fd[1]));
	  if (fill_linebuffer (&ps[i], 1) == 0) {
	    DEBUG_DO (printf ("process number %d: EOF0\n", i));
	    close (ps[i].fd[1]);
	    ps[i].fd[1] = -1;
	  }
	}
	if (ps[i].fd[0] < 0 && ps[i].fd[1] < 0) {
	  DEBUG_DO (printf ("process number %d: next\n", i));
	  read_host (fp_hosts, &ps[i], argc, argv);
	}
      }
    } else {
      /* timeout */
      if (gettimeofday (&now, &tz) < 0) {
	perror ("gettimeofday");
	exit (-1);
      }
      for (i = 0; i < num_process; i++) {
	if ((ps[i].fd[0] < 0 && ps[i].fd[1] < 0)
	    || timercmp (&now, &ps[i].timeout, <))
	  continue;
	if (!quiet) {
	  fprintf (stdout, "%s::: timeout\n", ps[i].hostname);
	}
	DEBUG_DO (printf ("process number %d: timeout\n", i));
	close (ps[i].fd[0]);
	close (ps[i].fd[1]);
	kill (ps[i].pid, SIGKILL);
	read_host (fp_hosts, &ps[i], argc, argv);
      }
    }
  }
  return 0;
}

void parse_host_list (const char *str)
{
  const char *s;
  int len, i, num;
  len = strlen (str);
  for (i = 0, num = 1; i < len; i++)
    if (str[i] == ',') num++;
  DEBUG_DO (printf ("parse_host_list str = %s\n", str));
  hostname_list = x_malloc ((num + 2) * sizeof (char *));
  for (i = 0, s = str; i < num; i++) {
    int j;
    j = strcspn (s, ",");
    if (j < 1) break;
    hostname_list[i] = x_malloc (j + 1);
    memcpy (hostname_list[i], s, j);
    hostname_list[i][j] = '\0';
    s += j + 1;
  }
  hostname_list[i] = NULL;
  DEBUG_DO (printf ("parse_host_list i = %d\n", i));
}

int main (int argc, char **argv)
{
  int i;
  if (argc < 2)
    usage (argv[0]);
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '-') {
      if (argv[i][1] != 0 && argv[i][2] != 0)
	usage (argv[0]);
      switch (argv[i][1]) {
      case 'n':
	if (i + 1 >= argc)
	  usage (argv[0]);
	num_process = atoi (argv[i + 1]);
	if (num_process == 0)
	  usage (argv[0]);
	i++;
	break;
      case 't':
	if (i + 1 >= argc)
	  usage (argv[0]);
	timeout = atoi (argv[i + 1]);
	if (timeout == 0)
	  usage (argv[0]);
	i++;
	break;
      case 'h':
	if (i + 1 >= argc)
	  usage (argv[0]);
	hosts_file = argv[i + 1];
	from_stdin = 0;
	i++;
	break;
      case 'l':
	if (i + 1 >= argc)
	  usage (argv[0]);
	parse_host_list (argv[i + 1]);
	i++;
	break;
      case 'c':
	if (i + 1 >= argc)
	  usage (argv[0]);
	{
	  char *p;
	  rsh_path = argv[i + 1];
	  p = strrchr (rsh_path, '/');
	  if (p) {
	    rsh_command = p + 1;
	  } else {
	    rsh_command = rsh_path;
	  }
	}
	i++;
	break;
      case 'q':
	quiet = 1;
	break;
      case 's':
	from_stdin = 1;
	break;
      case '-':
	i++;
	goto break2;
      default:
	usage (argv[0]);
      }
    } else {
      break;
    }
  }
 break2:
  if (i == argc)
    usage (argv[0]);
  forall (argc - i, argv + i);
  return 0;
}

