/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "esehttpd.h"
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

int debug = 0;
int num_requests = 0;

static void
eh_app_prepare_vserver (eh_app_t *app, eh_config_vhost_t *vhost,
			const char *hostname)
{
  int i;
  
  for (i = 0; i < app->num_vserver; i++) {
    if (vhost->servaddr.sin_addr.s_addr
	== app->vserver[i].servaddr.sin_addr.s_addr)
      break;
  }
  if (i < app->num_vserver) {
    /* this virtual host and vserver[i] shares the same IP address */
    eh_vserver_attach_namevhost (&app->vserver[i], vhost, hostname);
    return;
  }
  /* this virtual host has own IP address */
  app->num_vserver++;
  app->vserver = (eh_vserver_t *)
    x_realloc (app->vserver, app->num_vserver * sizeof (eh_vserver_t));
  eh_vserver_init (&app->vserver[i], app, vhost, hostname);
}

static void
eh_app_cb_prepare_vserver (eh_strhash_entry_t *ent, void *data)
{
  eh_app_t *app = (eh_app_t *)data;
  eh_config_vhost_t *vhost = (eh_config_vhost_t *)ent->val;
  eh_app_prepare_vserver (app, vhost, (const char *)ent->key);
}

void
eh_app_init (eh_app_t *app, const char *conf_file)
{
  eh_config_global_t *econf;
  int i;

  memset (app, 0, sizeof (*app));
  app->rlimit_nofile = 0;

  econf = eh_config_global_new (conf_file);
  app->econf = econf;
  if (eh_config_global_prepare_tables (econf)) {
    eh_config_global_log_errors (econf);
    eh_config_global_delete (econf);
    eh_log (EH_LOG_FATAL, "failed to load config file");
    exit (-1);
  }

  app->fcache = eh_filecache_new (econf->filecachesize);
  app->filecache_threshold = econf->filecachethreshold;
  eh_log (EH_LOG_INFO, "maximum filesize to be cached: %d %s",
	  app->filecache_threshold,
	  app->filecache_threshold == 0 ? "(unlimited)" : "");

  eh_app_prepare_vserver (app, econf->default_vhost, econf->bindaddress);
  eh_strhash_foreach (econf->vhost_ht, eh_app_cb_prepare_vserver, app);
  for (i = 0; i < app->num_vserver; i++)
    eh_vserver_init_ssl (&app->vserver[i]);
}

void
eh_app_start_servers (eh_app_t *app)
{
  eh_config_global_t *econf;
  struct rlimit rlim;
  int max_files;
  int i;

  econf = app->econf;
  
  max_files = econf->maxfiles;
  if (max_files) {
    int max_files_old;
    Syscall_perror (getrlimit (RLIMIT_NOFILE, &rlim), "getrlimit", exit (1));
    max_files_old = rlim.rlim_cur;
    if (max_files < 100)
      max_files = 100;
    rlim.rlim_cur = max_files;
    rlim.rlim_max = max_files;
    Syscall_perror (setrlimit (RLIMIT_NOFILE, &rlim), "setrlimit", exit (1));
    eh_log (EH_LOG_INFO, "maximum number of files: %d -> %d",
	    max_files_old, max_files);
  }
  Syscall_perror (getrlimit (RLIMIT_NOFILE, &rlim), "getrlimit", exit (1));
  app->rlimit_nofile = rlim.rlim_cur;
  if (app->rlimit_nofile < 10) {
    eh_log (EH_LOG_FATAL, "RLIMIT_NOFILE: too small (%d)", app->rlimit_nofile);
    exit (1);
  }

  for (i = 0; i < app->num_vserver; i++)
    eh_vserver_start_servers (&app->vserver[i]);

  app->num_child = 1;
  app->num_child = econf->startservers;
  if (app->num_child < 1 || app->num_child > 256)
    app->num_child = 1;
  eh_log (EH_LOG_INFO, "number of worker processes: %d", app->num_child);

  assert (app->child == NULL);
  app->child = (pid_t *)x_malloc (app->num_child *sizeof (pid_t));
  app->my_worker_id = -1;
  
  if (app->num_child > 1) {
    /* open accesslog files for each workers. accesslog file must be open
       before we drop root privilege, so only the parent process can do it.
    */
    for (i = 0; i < app->num_child; i++) {
      eh_config_global_prepare_accesslog (app->econf, i);
    }
  } else {
    eh_config_global_prepare_accesslog (app->econf, -1);
  }
}

static void
eh_app_setuid (uid_t uid, gid_t gid)
{
  eh_log (EH_LOG_INFO, "group id: %lu", (unsigned long)gid);
  Syscall_perror (setgid (gid), "setgid", exit (1));
  if (getegid () == 0) {
    eh_log (EH_LOG_FATAL, "GID is 0. Set 'Group' in config file");
    exit (1);
  }
  eh_log (EH_LOG_INFO, "user id: %lu", (unsigned long)uid);
  Syscall_perror (setuid (uid), "setuid", exit (1));
  if (geteuid () == 0) {
    eh_log (EH_LOG_FATAL, "UID is 0. Set 'User' in config file");
    exit (1);
  }
}

static void
eh_app_check_user (eh_app_t *app, uid_t *uid_r, gid_t *gid_r)
{
  struct group *gent;
  struct passwd *pent;
  const char *gname, *pname;
  
  gname = app->econf->group ? app->econf->group : "nobody";
  gent = getgrnam (gname);
  if (gent == NULL) {
    eh_log (EH_LOG_FATAL, "group %s not found in /etc/group", gname);
    exit (1);
  }
  *gid_r = gent->gr_gid;

  pname = app->econf->user ? app->econf->user : "nobody";
  pent = getpwnam (pname);
  if (pent == NULL) {
    eh_log (EH_LOG_FATAL, "user %s not found in /etc/passwd", pname);
    exit (1);
  }
  *uid_r = pent->pw_uid;
}

void
eh_app_chroot_setuid (eh_app_t *app)
{
  uid_t uid = -1;
  gid_t gid = -1;
  eh_app_check_user (app, &uid, &gid);
  if (app->econf->change_root_fullpath) {
    eh_log (EH_LOG_INFO, "root directory: %s",
	    app->econf->change_root_fullpath);
    Syscall_perror (chroot (app->econf->change_root_fullpath),
		    "chroot", exit (1));
    Syscall_perror (chdir ("/"), "chdir", exit (1));
  }
  eh_app_setuid (uid, gid);
  eh_fd_getpid ();
}

static void
eh_app_global_timer_callback (void *data)
{
  eh_app_t *app = (eh_app_t *)data;
  eh_debug ("global timer callback");
  eh_config_global_flush_accesslog (app->econf);
}

static void
eh_app_set_worker (eh_app_t *app)
{
  int i;
  eh_config_global_set_worker_id (app->econf, app->my_worker_id,
				  app->num_child);
  eh_fd_set_global_timer_callback (eh_app_global_timer_callback, app);
  for (i = 0; i < app->num_vserver; i++)
    eh_vserver_set_worker (&app->vserver[i]);
}

static void
eh_app_set_manager_rtsig (eh_app_t *app)
{
  int i;
  for (i = 0; i < app->num_vserver; i++)
    eh_vserver_set_manager (&app->vserver[i], 1); /* handle events */
}

static void
eh_app_on_childexit (pid_t pid, int status, void *data)
{
  int i;
  eh_app_t *app = (eh_app_t *)data;
  for (i = 0; i < app->num_child; i++) {
    if (pid == app->child[i]) {
      app->child[i] = -1;
      if (WIFEXITED (status) && WEXITSTATUS (status) == 0) {
	/* don't fork */
	eh_log (EH_LOG_INFO, "child exited");
      } else {
	eh_log (EH_LOG_INFO, "forking a new worker process");
	if (eh_app_fork_worker (app, i) == 0) {
	  /* child process */
	  eh_fd_start_timer (1000);
	}
      }
      break;
    }
  }
  /* return to the event loop */
}

static void
eh_app_set_manager_poll (eh_app_t *app)
{
  int i;
  eh_fd_set_childexit_callback (eh_app_on_childexit, app);
  for (i = 0; i < app->num_vserver; i++)
    eh_vserver_set_manager (&app->vserver[i], 0); /* don't handle events */
}

pid_t
eh_app_fork_worker (eh_app_t *app, int worker_num)
{
  pid_t pid;
  pid = fork ();
  if (pid == 0) {
    /* child */
    int i;
    eh_fd_getpid ();
    eh_log_getpid ();
    for (i = 0; i < app->num_child; i++)
      app->child[i] = 0;
    app->my_worker_id = worker_num;
    eh_app_set_worker (app);
    return 0;
  } else if (pid > 0) {
    /* parent */
    app->child[worker_num] = pid;
  } else if (pid < 0) {
    eh_log_perror (EH_LOG_FATAL, "fork");
  }
  return pid;
}

void
eh_app_start_workers (eh_app_t *app) {
  int i;
  
  if (app->num_child > 1) {
    /* multi-process server */
    for (i = 0; i < app->num_child; i++) {
      // FIXME: test
      eh_app_set_manager_poll (app);
      if (eh_app_fork_worker (app, i) == 0) {
	/* child */
	return;
      }
    }
    /* parent */
    assert (app->my_worker_id == -1);
    switch (eh_fd_get_multiplexing_method ()) {
    case eh_fd_multiplexing_method_rtsignal:
      /* RT-signals type server. parent process does round-robin the
	 'ready-to-accept' signals. */
      eh_app_set_manager_rtsig (app);
      break;
    default:
      /* poll() type server. parent process waits for sigchld, and
	 re-fork a child. */
      eh_app_set_manager_poll (app);
    }
  } else {
    /* single-process server. the process itself work as a worker. */
    eh_app_set_worker (app);
  }
}

int
eh_app_select_loop (eh_app_t *app)
{
  eh_debug ("");
  eh_fd_start_timer (1000);
  return eh_fd_check_and_dispatch_events (1000);
}

void
eh_app_increment_script_eval_count (eh_app_t *app)
{
  if (app->num_child < 2)
    return;
  if (++app->script_eval_count >= app->econf->scriptevalmax &&
      app->econf->scriptevalmax > 0) {
    eh_fd_schedule_graceful_shutdown (15);
  }
}

void
eh_app_discard (eh_app_t *app)
{
  int i;
  for (i = 0; i < app->num_vserver; i++)
    eh_vserver_discard (&app->vserver[i]);
  x_free (app->vserver);
  eh_config_global_delete (app->econf);
  app->econf = NULL;
  eh_filecache_delete (app->fcache);
  app->fcache = NULL;
  if (app->child) {
#if 0
    if (app->my_worker_id == -1) {
      int i;
      for (i = 0; i < app->num_child; i++) {
	int status;
	waitpid (app->child[i], &status, WNOHANG);
      }
    }
#endif
    x_free (app->child);
  }
}

