/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

static int
eh_connection_check_passwd (eh_connection_t *ec, eh_request_t *er,
			    eh_config_limit_t *ecl)
{
  char *decoded = NULL;
  char *p;
  const char *authstr;
  const char *correct_crypted_passwd;
  const char *crypt_retval;
  const char *user_name, *user_passwd;
  char salt[13];
  int salt_len;
  int retval = 1;
  authstr = er->headers.predef.authorization;
  if (strcspn (authstr, " \t") != 5 ||
      strncasecmp (authstr, "Basic", 5) != 0) {
    retval = 1;
    goto finish;
  }
  authstr += 5;
  authstr += strspn (authstr, " \t");
  decoded = x_strdup (authstr);
  if (eh_base64_decode (decoded) < 0) {
    retval = 1;
    goto finish;
  }
  eh_debug ("decoded = %s", decoded);
  user_name = decoded;
  p = strchr (decoded, ':');
  if (p == NULL) {
    retval = 1;
    goto finish;
  }
  *p++ = '\0';
  user_passwd = p;
  eh_request_set_remote_user (er, user_name);
  eh_accesslog_set_remote_user (&ec->accesslog, user_name);
  eh_debug ("user_name = %s, user_passwd = %s", user_name, user_passwd);
  correct_crypted_passwd = (const char *)
    eh_strhash_find (ecl->passwd_ht, user_name, NULL);
  if (correct_crypted_passwd == NULL) {
    eh_debug ("user %s not found in passwd file", user_name);
    retval = 1;
    goto finish;
  }
  salt_len = 2;
  if (strlen (correct_crypted_passwd) > 12 &&
      correct_crypted_passwd[0] == '$') {
    salt_len = 12;
  }
  memcpy (salt, correct_crypted_passwd, salt_len);
  salt[salt_len] = '\0';
  /* crypt() is not multithread safe */
  crypt_retval = crypt (user_passwd, salt);
  retval = strcmp (correct_crypted_passwd, crypt_retval);
  eh_debug ("correct passwd: '%s', user supplied: '%s'",
	    correct_crypted_passwd, crypt_retval);

 finish:
  if (decoded)
    x_free (decoded);
  return retval;
}

static int
eh_connection_check_auth_internal (eh_connection_t *ec, eh_request_t *req,
				   char **authname_r)
{
  eh_config_dir_t *ecd;
  eh_config_limit_t *ecl_matched;

  ecd = req->econf_dir_ref;
  ecl_matched = req->econf_limit_ref;
  if (ecl_matched == NULL)
    ecl_matched = ecd->default_filesmatch->default_limit;
  
  if (!ecl_matched->auth_required)
    return 0;
  if (ecd->authrequiressl && ec->sslcon == NULL)
    return 1;
  *authname_r = ecd->authname ? x_strdup (ecd->authname) : x_strdup ("AUTH");
  if (req->headers.predef.authorization == NULL)
    return 1;
  return eh_connection_check_passwd (ec, req, ecl_matched);
#if 0
  int allowoverride;
  allowoverride = eh_config_vhost_dir_get_allowoverride (req.econf_vhost_ref,
							 req.filename);
  return eh_connection_check_htaccess (ec, req, allowoverride);
#endif
}

static int
eh_connection_check_dir_index (eh_connection_t *ec, eh_request_t *req,
			       eh_config_dir_t *ecd)
{
  if (!S_ISDIR (req->statbuf.st_mode))
    return 0;
  if (ecd == NULL || ecd->option_indexes == 0)
    return 1;
  return 0;
}

int
eh_connection_check_auth (eh_connection_t *ec, eh_request_t *req)
{
  char *authname = NULL;
  char *header;
  int r;
  eh_config_dir_t *ecd;
  ecd = req->econf_dir_ref;
  r = eh_connection_check_dir_index (ec, req, ecd);
  if (r) {
    eh_connection_append_wvec_response (ec, req->method, "403", NULL, NULL, 0);
    return 1;
  }
  if (ecd == NULL) {
    return 0;
  }
  r = eh_connection_check_auth_internal (ec, req, &authname);
  if (r == 0)
    goto finish;
  if (ecd->authrequiressl && ec->sslcon == NULL) {
    eh_connection_append_wvec_response (ec, req->method, "403", NULL, NULL, 0);
    r = 1;
    goto finish;
  }
  if (authname == NULL)
    authname = x_strdup ("WWW Authenticate");
  header = x_strdup_printf ("WWW-Authenticate: Basic realm=\"%s\"\r\n",
			    authname);
  eh_connection_append_wvec_response (ec, req->method, "401", header, NULL, 0);
  x_free (header);
  
 finish:
  if (authname)
    x_free (authname);
  return r;
}
